/* 
 * Copyright (C) 2000-2008 the xine project
 * 
 * This file is part of xine, a unix video player.
 * 
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include "common.h"
#include "kbindings_common.h"


#undef TRACE_KBINDINGS

#define  KBEDIT_NOOP           0
#define  KBEDIT_ALIASING       1
#define  KBEDIT_EDITING        2

typedef struct {
  kbinding_t           *kbt;

  xitk_window_t        *xwin;
  xitk_widget_list_t   *widget_list;
  int                   running;
  int                   visible;

  xitk_widget_t        *browser;
  kbinding_entry_t     *ksel;
  
  xitk_widget_t        *alias;
  xitk_widget_t        *edit;
  xitk_widget_t        *delete;
  xitk_widget_t        *save;
  xitk_widget_t        *done;
  xitk_widget_t        *grab;

  xitk_widget_t        *comment;
  xitk_widget_t        *key;

  xitk_widget_t        *ctrl;
  xitk_widget_t        *meta;
  xitk_widget_t        *mod3;
  xitk_widget_t        *mod4;
  xitk_widget_t        *mod5;

  int                   num_entries;
  char                **entries;
  char                **shortcuts;

  int                   grabbing;
  int                   action_wanted; /* See KBEDIT_ defines */

  xitk_register_key_t   kreg;
} _kbedit_t;

static _kbedit_t    *kbedit = NULL;

#define WINDOW_WIDTH        530
#define WINDOW_HEIGHT       456
#define MAX_DISP_ENTRIES    11

#define DEFAULT_DISPLAY_MODE   1
#define LIRC_DISPLAY_MODE      2

/*
  Remap file entries syntax are:
  ...  
  WindowReduce {
      key = less
      modifier = none
  }

  Alias {
      entry = WindowReduce
      key = w
      modifier = control
  }
  ...
  WindowReduce action key is <less>, with no modifier usage.
  There is an alias: C-w (control w> keystroke will have same
                     action than WindowReduce.

  modifier entry is optional.
  modifier key can be multiple:
 ...
  WindowReduce {
      key = less
      modifier = control, meta
  }
  ...
  Here, WindowReduce action keystroke will be CM-<

  Key modifier name are:
    none, control, ctrl, meta, alt, mod3, mod4, mod5.

  shift,lock and numlock modifier are voluntary not handled.

*/


static void _kbindings_display_kbindings_to_stream(kbinding_t *kbt, int mode, FILE *stream) {
  char               buf[256];
  kbinding_entry_t **k;
  int          i;

  if(kbt == NULL) {
    xine_error(_("OOCH: key binding table is NULL.\n"));
    return;
  }

  switch(mode) {
  case LIRC_DISPLAY_MODE:
    fprintf(stream, "##\n# xine key bindings.\n"
		    "# Automatically generated by %s version %s.\n##\n\n", PACKAGE, VERSION);
    
    for(k = kbt->entry, i = 0; k[i]->action != NULL; i++) {
      if(!k[i]->is_alias) {
	snprintf(buf, sizeof(buf), "# %s\n", k[i]->comment);
	strlcat(buf, "begin\n"
		     "\tremote = xxxxx\n"
		     "\tbutton = xxxxx\n"
		     "\tprog   = xine\n"
		     "\trepeat = 0\n", sizeof(buf));
	snprintf(buf+strlen(buf), sizeof(buf)-strlen(buf), "\tconfig = %s\n", k[i]->action);
	strlcat(buf, "end\n\n", sizeof(buf));
	fputs(buf, stream);
      }
    }
    fprintf(stream, "##\n# End of xine key bindings.\n##\n");
    break;
    
  case DEFAULT_DISPLAY_MODE:
  default:
    fprintf(stream, "##\n# xine key bindings.\n"
		    "# Automatically generated by %s version %s.\n##\n\n", PACKAGE, VERSION);

    for(k = kbt->entry, i = 0; k[i]->action != NULL; i++) {
      snprintf(buf, sizeof(buf), "# %s\n", k[i]->comment);

      if(k[i]->is_alias) {
	sprintf(buf+strlen(buf), "Alias {\n\tentry = %s\n", k[i]->action);
      }
      else
	sprintf(buf+strlen(buf), "%s {\n", k[i]->action);

      sprintf(buf+strlen(buf), "\tkey = %s\n\tmodifier = ", k[i]->key);
      if(k[i]->modifier == KEYMOD_NOMOD)
	strlcat(buf, "none, ", sizeof(buf));
      if(k[i]->modifier & KEYMOD_CONTROL)
	strlcat(buf, "control, ", sizeof(buf));
      if(k[i]->modifier & KEYMOD_META)
	strlcat(buf, "meta, ", sizeof(buf));
      if(k[i]->modifier & KEYMOD_MOD3)
	strlcat(buf, "mod3, ", sizeof(buf));
      if(k[i]->modifier & KEYMOD_MOD4)
	strlcat(buf, "mod4, ", sizeof(buf));
      if(k[i]->modifier & KEYMOD_MOD5)
	strlcat(buf, "mod5, ", sizeof(buf));
      buf[strlen(buf) - 2] = '\n';
      buf[strlen(buf) - 1] = '\0';
      strlcat(buf, "}\n\n", sizeof(buf));
      fputs(buf, stream);
    }
    fprintf(stream, "##\n# End of xine key bindings.\n##\n");
    break;
  }

}
/*
 * Display all key bindings from kbt key binding table.
 */
static void _kbindings_display_kbindings(kbinding_t *kbt, int mode) {
  _kbindings_display_kbindings_to_stream(kbt, mode, stdout);  
}

/*
 * Convert a modifier to key binding modifier style.
 */
static void kbindings_convert_modifier(int mod, int *modifier) {
  *modifier = KEYMOD_NOMOD;
  
  ABORT_IF_NULL(modifier);

  if(mod & MODIFIER_NOMOD)
    *modifier = KEYMOD_NOMOD;
  if(mod & MODIFIER_CTRL)
    *modifier |= KEYMOD_CONTROL;
  if(mod & MODIFIER_META)
    *modifier |= KEYMOD_META;
  if(mod & MODIFIER_MOD3)
    *modifier |= KEYMOD_MOD3;
  if(mod & MODIFIER_MOD4)
    *modifier |= KEYMOD_MOD4;
  if(mod & MODIFIER_MOD5)
    *modifier |= KEYMOD_MOD5;
}

/*
 * Save current key binding table to keymap file.
 */
void kbindings_save_kbinding(kbinding_t *kbt) {
  FILE   *f;
  
  if((f = fopen(gGui->keymap_file, "w+")) == NULL) {
    return;
  }
  else {
    _kbindings_display_kbindings_to_stream(kbt, DEFAULT_DISPLAY_MODE, f);  
    fclose(f);
  }
}

/*
 * Return a duplicated table from kbt.
 */
static kbinding_t *_kbindings_duplicate_kbindings(kbinding_t *kbt) {
  int         i;
  kbinding_t *k;
  
  ABORT_IF_NULL(kbt);
  
  k = (kbinding_t *) calloc(1, sizeof(kbinding_t));

  for(i = 0; kbt->entry[i]->action != NULL; i++) {
    k->entry[i]            = (kbinding_entry_t *) calloc(1, sizeof(kbinding_entry_t));
    k->entry[i]->comment   = strdup(kbt->entry[i]->comment);
    k->entry[i]->action    = strdup(kbt->entry[i]->action);
    k->entry[i]->action_id = kbt->entry[i]->action_id;
    k->entry[i]->key       = strdup(kbt->entry[i]->key);
    k->entry[i]->modifier  = kbt->entry[i]->modifier;
    k->entry[i]->is_alias  = kbt->entry[i]->is_alias;
    k->entry[i]->is_gui    = kbt->entry[i]->is_gui;
  }

  k->entry[i]            = (kbinding_entry_t *) calloc(1, sizeof(kbinding_t));
  k->entry[i]->comment   = NULL;
  k->entry[i]->action    = NULL;
  k->entry[i]->action_id = 0;
  k->entry[i]->key       = NULL;
  k->entry[i]->modifier  = 0;
  k->entry[i]->is_alias  = 0;
  k->entry[i]->is_gui    = 0;
  k->num_entries         = i + 1;
  
  return k;
}

/*
 * Free key binding table kbt, then set it to default table.
 */
void kbindings_reset_kbinding(kbinding_t *kbt) {
  
  ABORT_IF_NULL(kbt);
  
  _kbindings_free_bindings_no_kbt(kbt);
  _kbindings_init_to_default_no_kbt(kbt);
}

/*
 * This could be used to create a default key binding file
 * with 'xine --keymap > $HOME/.xine_keymap'
 */
void kbindings_display_default_bindings(void) {
  kbinding_t *kbt = NULL;
  
  kbt = _kbindings_init_to_default();
  
  _kbindings_display_kbindings(kbt, DEFAULT_DISPLAY_MODE);
  _kbindings_free_bindings(kbt);
}

/*
 * This could be used to create a default key binding file
 * with 'xine --keymap=lirc > $HOME/.lircrc'
 */
void kbindings_display_default_lirc_bindings(void) {
  kbinding_t *kbt = NULL;
  
  kbt = _kbindings_init_to_default();
  
  _kbindings_display_kbindings(kbt, LIRC_DISPLAY_MODE);
  _kbindings_free_bindings(kbt);
}

/*
 * This could be used to create a key binding file from key binding 
 * object k, with 'xine --keymap > $HOME/.xine_keymap'
 */
void kbindings_display_current_bindings(kbinding_t *kbt) {

  ABORT_IF_NULL(kbt);

  _kbindings_display_kbindings(kbt, DEFAULT_DISPLAY_MODE);
}

/*
 * Return action id from key binding kbt entry.
 */
action_id_t kbindings_get_action_id(kbinding_entry_t *kbt) {

  if(kbt == NULL)
    return ACTID_NOKEY;

  return kbt->action_id;
}

static char *_kbindings_get_shortcut_from_kbe(kbinding_entry_t *kbe) {
  static char shortcut[32];

  if(kbe) {
    shortcut[0] = 0;
    
    if(gGui->shortcut_style == 0) {
      if(kbe->modifier & KEYMOD_CONTROL)
	snprintf(shortcut+strlen(shortcut), sizeof(shortcut)-strlen(shortcut), "%s+", _("Ctrl"));
      if(kbe->modifier & KEYMOD_META)
	snprintf(shortcut+strlen(shortcut), sizeof(shortcut)-strlen(shortcut), "%s+", _("Alt"));
      if(kbe->modifier & KEYMOD_MOD3)
	strlcat(shortcut, "M3+", sizeof(shortcut));
      if(kbe->modifier & KEYMOD_MOD4)
	strlcat(shortcut, "M4+", sizeof(shortcut));
      if(kbe->modifier & KEYMOD_MOD5)
	strlcat(shortcut, "M5+", sizeof(shortcut));
    }
    else {
      if(kbe->modifier & KEYMOD_CONTROL)
	strlcat(shortcut, "C-", sizeof(shortcut));
      if(kbe->modifier & KEYMOD_META)
	strlcat(shortcut, "M-", sizeof(shortcut));
      if(kbe->modifier & KEYMOD_MOD3)
	strlcat(shortcut, "M3-", sizeof(shortcut));
      if(kbe->modifier & KEYMOD_MOD4)
	strlcat(shortcut, "M4-", sizeof(shortcut));
      if(kbe->modifier & KEYMOD_MOD5)
	strlcat(shortcut, "M5-", sizeof(shortcut));
    }
    
    strlcat(shortcut, kbe->key, sizeof(shortcut));
    
    return shortcut;
  }
  return NULL;
}

char *kbindings_get_shortcut(kbinding_t *kbt, char *action) {
  kbinding_entry_t  *k;
  static char        shortcut[32];
  
  if(kbt) {
    if(action && (k = kbindings_lookup_action(kbt, action))) {
      if(strcmp(k->key, "VOID")) {
	snprintf(shortcut, sizeof(shortcut), "%c%s%c", '[', _kbindings_get_shortcut_from_kbe(k), ']');
	return shortcut;
      }
    }
  }
  return NULL;
}

/*
 * Try to find and entry in key binding table matching with key and modifier value.
 */
static kbinding_entry_t *kbindings_lookup_binding(kbinding_t *kbt, const char *key, int modifier) {
  kbinding_entry_t *kret = NULL, *k;
  int               i;

  if((key == NULL) || (kbt == NULL))
    return NULL;

#ifdef TRACE_KBINDINGS
  printf("Looking for: '%s' [", key);
  if(modifier == KEYMOD_NOMOD)
    printf("none, ");
  if(modifier & KEYMOD_CONTROL)
    printf("control, ");
  if(modifier & KEYMOD_META)
    printf("meta, ");
  if(modifier & KEYMOD_MOD3)
    printf("mod3, ");
  if(modifier & KEYMOD_MOD4)
    printf("mod4, ");
  if(modifier & KEYMOD_MOD5)
    printf("mod5, ");
  printf("\b\b]\n");
#endif

  /* Be case sensitive */
  for(i = 0, k = kbt->entry[0]; kbt->entry[i]->action != NULL; i++, k = kbt->entry[i]) {
    if(k && k->key && strlen(k->key) && ((!(strcmp(k->key, key))) && (modifier == k->modifier))) {
      kret = k;
      goto __found;
    }
  }

  /* Not case sensitive */
  /*
  for(i = 0, k = kbt[0]; kbt[i]->action != NULL; i++, k = kbt[i]) {
    if((!(strcasecmp(k->key, key))) && (modifier == k->modifier))
      return k;
  }
  */
  /* Last chance */
  for(i = 0, k = kbt->entry[0]; kbt->entry[i]->action != NULL; i++, k = kbt->entry[i]) {
    if(k && k->key && strlen(k->key) && ((!(strcmp(k->key, key))) && (k->modifier == KEYMOD_NOMOD))) {
      kret = k;
      break;
    }
  }

 __found:

  /* Keybinding unknown */
  return kret;
}

/* Convert X(Button|Key)(Press|Release) events into string identifier. */
static int xevent2id(XEvent *event, int *modifier, char *buf, int size) {
  int    mod;
  KeySym mkey;
  char  *keySym;

  if (event == NULL)
    return -1;

  switch (event->type) {
  case ButtonPress:
  case ButtonRelease:
    (void) xitk_get_key_modifier(event, &mod);
    kbindings_convert_modifier(mod, modifier);
    snprintf(buf, size, "XButton_%d", event->xbutton.button);
    return 0;

  case KeyPress:
  case KeyRelease:
    (void) xitk_get_key_modifier(event, &mod);
    kbindings_convert_modifier(mod, modifier);
    mkey = xitk_get_key_pressed(event);

    switch (mkey) {
      default:
        XLockDisplay (event->xany.display);
        keySym = XKeysymToString(mkey);
        XUnlockDisplay (event->xany.display);
        if (keySym != NULL) {
          strlcpy(buf, keySym, size);
          return 0;
        }
      case 0: /* Key without assigned KeySymbol */
      case XK_VoidSymbol:
        /* For keys without assigned KeySyms. */
        snprintf(buf, size, "XKey_%d", event->xkey.keycode);
        return 0;
    }

  default:
    memset(buf, 0, size);
    return -1;
  }
}

/*
 * Handle key event from an XEvent.
 */
void kbindings_handle_kbinding(kbinding_t *kbt, XEvent *event) {
  int               modifier;
  char              buf[256];
  kbinding_entry_t *k;

  if(!gGui->kbindings_enabled || (kbt == NULL) || (event == NULL))
    return;

  if (xevent2id(event, &modifier, buf, sizeof(buf)))
    return;
  k = kbindings_lookup_binding(kbt, buf, modifier);
  if(k && !(gGui->no_gui && k->is_gui))
    gui_execute_action_id(k->action_id);
#if 0  /* DEBUG */
  else
    printf("%s unhandled\n", kbuf);
#endif
}

/*
 * ***** Key Binding Editor ******
 */
/*
 * return 1 if key binding editor is ON
 */
int kbedit_is_running(void) {

  if(kbedit != NULL)
    return kbedit->running;

  return 0;
}

/*
 * Return 1 if setup panel is visible
 */
int kbedit_is_visible(void) {

  if(kbedit != NULL) {
    if(gGui->use_root_window)
      return xitk_is_window_visible(gGui->display, xitk_window_get_window(kbedit->xwin));
    else
      return kbedit->visible && xitk_is_window_visible(gGui->display, xitk_window_get_window(kbedit->xwin));
  }

  return 0;
}

/*
 * Raise kbedit->xwin
 */
void kbedit_raise_window(void) {
  
  if(kbedit != NULL)
    raise_window(xitk_window_get_window(kbedit->xwin), kbedit->visible, kbedit->running);
}

/*
 * Hide/show the kbedit panel
 */
void kbedit_toggle_visibility (xitk_widget_t *w, void *data) {
  if(kbedit != NULL)
    toggle_window(xitk_window_get_window(kbedit->xwin), kbedit->widget_list,
		  &kbedit->visible, kbedit->running);
}

static void kbedit_create_browser_entries(void) {
  xitk_font_t  *fs;
  int           i;
  
  if(kbedit->num_entries) {
    for(i = 0; i < kbedit->num_entries; i++) {
      free((char *)kbedit->entries[i]);
      free((char *)kbedit->shortcuts[i]);
    }
    free((char **)kbedit->entries);
    free((char **)kbedit->shortcuts);
  }
  
  fs = xitk_font_load_font(gGui->display, br_fontname);
  xitk_font_set_font(fs, (XITK_WIDGET_LIST_GC(kbedit->widget_list)));
  
  kbedit->entries   = (char **) calloc(kbedit->kbt->num_entries, sizeof(char *));
  kbedit->shortcuts = (char **) calloc(kbedit->kbt->num_entries, sizeof(char *));
  kbedit->num_entries = kbedit->kbt->num_entries - 1;
  
  for(i = 0; i < kbedit->num_entries; i++) {
    char  buf[2048];
    char *sc = _kbindings_get_shortcut_from_kbe(kbedit->kbt->entry[i]);
    char  shortcut[32];
        
    snprintf(shortcut, sizeof(shortcut), "%c%s%c", '[', (sc ? sc : "VOID"), ']');
    
    if(kbedit->kbt->entry[i]->is_alias)
      snprintf(buf, sizeof(buf), "@{%s}", kbedit->kbt->entry[i]->comment);
    else
      strlcpy(buf, kbedit->kbt->entry[i]->comment, sizeof(buf));
    
    kbedit->entries[i]   = strdup(buf);
    kbedit->shortcuts[i] = strdup(shortcut);
  }

  kbedit->entries[i]   = NULL;
  kbedit->shortcuts[i] = NULL;

  xitk_font_unload_font(fs);
}

static void kbedit_display_kbinding(const char *action, kbinding_entry_t *kbe) {

  if(action && kbe) {

    xitk_label_change_label(kbedit->comment, action);
    xitk_label_change_label(kbedit->key, kbe->key);
    
    xitk_checkbox_set_state(kbedit->ctrl, (kbe->modifier & KEYMOD_CONTROL) ? 1 : 0);
    xitk_checkbox_set_state(kbedit->meta, (kbe->modifier & KEYMOD_META) ? 1 : 0);
    xitk_checkbox_set_state(kbedit->mod3, (kbe->modifier & KEYMOD_MOD3) ? 1 : 0);
    xitk_checkbox_set_state(kbedit->mod4, (kbe->modifier & KEYMOD_MOD4) ? 1 : 0);
    xitk_checkbox_set_state(kbedit->mod5, (kbe->modifier & KEYMOD_MOD5) ? 1 : 0);
  }
}

/*
 *
 */
static void kbedit_select(int s) {

  xitk_enable_and_show_widget(kbedit->alias);
  xitk_enable_and_show_widget(kbedit->edit);
  xitk_enable_and_show_widget(kbedit->delete);
  
  kbedit->ksel = kbedit->kbt->entry[s];
  kbedit_display_kbinding(kbedit->entries[s], kbedit->kbt->entry[s]);
}

/*
 *
 */
static void kbedit_unset(void) {

  if(xitk_labelbutton_get_state(kbedit->alias))
    xitk_labelbutton_set_state(kbedit->alias, 0);
  
  if(xitk_labelbutton_get_state(kbedit->edit))
    xitk_labelbutton_set_state(kbedit->edit, 0);
  
  xitk_disable_widget(kbedit->alias);
  xitk_disable_widget(kbedit->edit);
  xitk_disable_widget(kbedit->delete);
  xitk_disable_widget(kbedit->grab);
  kbedit->ksel = NULL;

  xitk_label_change_label(kbedit->comment, _("Nothing selected"));
  xitk_label_change_label(kbedit->key, _("None"));
  
  xitk_checkbox_set_state(kbedit->ctrl, 0);
  xitk_checkbox_set_state(kbedit->meta, 0);
  xitk_checkbox_set_state(kbedit->mod3, 0);
  xitk_checkbox_set_state(kbedit->mod4, 0);
  xitk_checkbox_set_state(kbedit->mod5, 0);
}

/*
 * Check for redundancy.
 * return: -2 on failure (null pointer passed)
 *         -1 on success
 *         >=0 if a redundant entry found (bkt array entry num).
 */
static int bkedit_check_redundancy(kbinding_t *kbt, kbinding_entry_t *kbe) {
  int ret = -1;
  
  if(kbt && kbe) {
    int i;
    
    for(i = 0; kbt->entry[i]->action != NULL; i++) {
      if((!strcmp(kbt->entry[i]->key, kbe->key)) &&
	 (kbt->entry[i]->modifier == kbe->modifier) &&
	 (strcasecmp(kbt->entry[i]->key, "void"))) {
	return i;
      }
    }
  }
  else
    ret = -2;
  
  return ret;
}

/*
 *
 */
static void kbedit_exit(xitk_widget_t *w, void *data) {

  if(kbedit) {
    window_info_t wi;
    
    kbedit->running = 0;
    kbedit->visible = 0;
    
    if((xitk_get_window_info(kbedit->kreg, &wi))) {
      config_update_num("gui.kbedit_x", wi.x);
      config_update_num("gui.kbedit_y", wi.y);
      WINDOW_INFO_ZERO(&wi);
    }
    
    xitk_unregister_event_handler(&kbedit->kreg);
    
    xitk_destroy_widgets(kbedit->widget_list);
    xitk_window_destroy_window(gGui->imlib_data, kbedit->xwin);
    
    kbedit->xwin = NULL;
    xitk_list_free((XITK_WIDGET_LIST_LIST(kbedit->widget_list)));
    
    XLockDisplay(gGui->display);
    XFreeGC(gGui->display, (XITK_WIDGET_LIST_GC(kbedit->widget_list)));
    XUnlockDisplay(gGui->display);
    
    XITK_WIDGET_LIST_FREE(kbedit->widget_list);
    
    {
      int i;
      
      for(i = 0; i < kbedit->num_entries; i++) {
	free((char *)kbedit->entries[i]);
	free((char *)kbedit->shortcuts[i]);
      }
      
      free((char **)kbedit->entries);
      free((char **)kbedit->shortcuts);
    }
    
    kbindings_free_kbinding(&kbedit->kbt);

    free(kbedit);
    kbedit = NULL;
    gGui->ssaver_enabled = 1;

    try_to_set_input_focus(gGui->video_window);
  }
}

/*
 *
 */
static void kbedit_sel(xitk_widget_t *w, void *data, int s) {

  if(s >= 0)
    kbedit_select(s);
}

/*
 * Create an alias from the selected entry.
 */
static void kbedit_alias(xitk_widget_t *w, void *data, int state) {

  xitk_labelbutton_set_state(kbedit->edit, 0);

  if(state) {
    xitk_enable_widget(kbedit->grab);
    kbedit->action_wanted = KBEDIT_ALIASING;
  }
  else {
    xitk_disable_widget(kbedit->grab);
    kbedit->action_wanted = KBEDIT_NOOP;
  }
}

/*
 * Change shortcut, should take care about reduncancy.
 */
static void kbedit_edit(xitk_widget_t *w, void *data, int state) {

  xitk_labelbutton_set_state(kbedit->alias, 0);

  if(state) {
    xitk_enable_widget(kbedit->grab);
    kbedit->action_wanted = KBEDIT_EDITING;
  }
  else {
    xitk_disable_widget(kbedit->grab);
    kbedit->action_wanted = KBEDIT_NOOP;
  }

}

/*
 * Remove selected entry, but alias ones only.
 */
static void kbedit_delete(xitk_widget_t *w, void *data) {
  int s = xitk_browser_get_current_selected(kbedit->browser);

  xitk_labelbutton_set_state(kbedit->alias, 0);
  xitk_labelbutton_set_state(kbedit->edit, 0);
  xitk_disable_widget(kbedit->grab);
  
  /* We can delete alias entries, only */
  if(s >= 0) {
    
    if(kbedit->kbt->entry[s]->is_alias) {
      int i;
      
      
      xitk_browser_release_all_buttons(kbedit->browser);
      
      free(kbedit->kbt->entry[s]->comment);
      free(kbedit->kbt->entry[s]->action);
      free(kbedit->kbt->entry[s]->key);
      free(kbedit->kbt->entry[s]);
      
      
      for(i = s; s < kbedit->num_entries; s++)
	kbedit->kbt->entry[s] = kbedit->kbt->entry[s + 1];

      kbedit->kbt->entry[s]->comment = NULL;
      kbedit->kbt->entry[s]->action  = NULL;
      kbedit->kbt->entry[s]->key     = NULL;
      
      kbedit->kbt->num_entries--;
      
      kbedit_create_browser_entries();
      
      xitk_browser_update_list(kbedit->browser, 
			       (const char* const*) kbedit->entries, 
			       (const char* const*) kbedit->shortcuts, kbedit->num_entries, xitk_browser_get_current_start(kbedit->browser));
    }
    else
      xine_error(_("You can only delete alias entries."));
    
  }
}

/*
 * Reset to xine's default table.
 */
static void kbedit_reset(xitk_widget_t *w, void *data) {

  xitk_labelbutton_set_state(kbedit->alias, 0);
  xitk_labelbutton_set_state(kbedit->edit, 0);
  xitk_disable_widget(kbedit->grab);

  kbindings_reset_kbinding(kbedit->kbt);
  kbedit_create_browser_entries();
  xitk_browser_update_list(kbedit->browser, 
			   (const char* const*) kbedit->entries, 
			   (const char* const*) kbedit->shortcuts, kbedit->num_entries, xitk_browser_get_current_start(kbedit->browser));
}

/*
 * Save keymap file, then set global table to hacked one 
 */
static void kbedit_save(xitk_widget_t *w, void *data) {
  xitk_labelbutton_set_state(kbedit->alias, 0);
  xitk_labelbutton_set_state(kbedit->edit, 0);
  xitk_disable_widget(kbedit->grab);
  
  kbindings_free_kbinding(&gGui->kbindings);
  gGui->kbindings = _kbindings_duplicate_kbindings(kbedit->kbt);
  kbindings_save_kbinding(gGui->kbindings);
}

/*
 * Forget and dismiss kbeditor 
 */
void kbedit_end(void) {
  kbedit_exit(NULL, NULL);
}

static void kbedit_accept_yes(xitk_widget_t *w, void *data, int state) {
  kbinding_entry_t *kbe = (kbinding_entry_t *) data;
  
  switch(kbedit->action_wanted) {
    
  case KBEDIT_ALIASING:
    if(kbedit->kbt->num_entries >= MAX_ENTRIES) {
      xine_error(_("No more space for additional entries!"));
      return;
    }

    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->comment   = strdup(kbedit->ksel->comment);
    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->action    = strdup(kbedit->ksel->action);
    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->action_id = kbedit->ksel->action_id;
    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->key       = strdup(kbe->key);
    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->modifier  = kbe->modifier;
    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->is_alias  = 1;
    kbedit->kbt->entry[kbedit->kbt->num_entries - 1]->is_gui    = kbe->is_gui;

    kbedit->kbt->entry[kbedit->kbt->num_entries]            = (kbinding_entry_t *) calloc(1, sizeof(kbinding_t));
    kbedit->kbt->entry[kbedit->kbt->num_entries]->comment   = NULL;
    kbedit->kbt->entry[kbedit->kbt->num_entries]->action    = NULL;
    kbedit->kbt->entry[kbedit->kbt->num_entries]->action_id = 0;
    kbedit->kbt->entry[kbedit->kbt->num_entries]->key       = NULL;
    kbedit->kbt->entry[kbedit->kbt->num_entries]->modifier  = 0;
    kbedit->kbt->entry[kbedit->kbt->num_entries]->is_alias  = 0;
    kbedit->kbt->entry[kbedit->kbt->num_entries]->is_gui    = 0;
    
    kbedit->kbt->num_entries++;
    
    kbedit_create_browser_entries();
    xitk_browser_update_list(kbedit->browser, 
			     (const char* const*) kbedit->entries, 
			     (const char* const*) kbedit->shortcuts, kbedit->num_entries, xitk_browser_get_current_start(kbedit->browser));
    break;
    
  case KBEDIT_EDITING:
    kbedit->ksel->key = (char *) realloc(kbedit->ksel->key, sizeof(char *) * (strlen(kbe->key) + 1));
    strcpy(kbedit->ksel->key, kbe->key);
    kbedit->ksel->modifier = kbe->modifier;
    
    kbedit_create_browser_entries();
    xitk_browser_update_list(kbedit->browser, 
			     (const char* const*) kbedit->entries, 
			     (const char* const*) kbedit->shortcuts, kbedit->num_entries, xitk_browser_get_current_start(kbedit->browser));
    break;
  }

  kbedit->action_wanted = KBEDIT_NOOP;
  
  SAFE_FREE(kbe->comment);
  SAFE_FREE(kbe->action);
  SAFE_FREE(kbe->key);
  SAFE_FREE(kbe);
  kbedit_unset();
}

static void kbedit_accept_no(xitk_widget_t *w, void *data, int state) {
  kbinding_entry_t *kbe = (kbinding_entry_t *) data;
  
  kbedit->action_wanted = KBEDIT_NOOP;
  
  xitk_browser_update_list(kbedit->browser, 
			   (const char* const*) kbedit->entries, 
			   (const char* const*) kbedit->shortcuts, kbedit->num_entries, xitk_browser_get_current_start(kbedit->browser));
  SAFE_FREE(kbe->comment);
  SAFE_FREE(kbe->action);
  SAFE_FREE(kbe->key);
  SAFE_FREE(kbe);
  kbedit_unset();
}

/*
 * Grab key binding.
 */
static void kbedit_grab(xitk_widget_t *w, void *data) {
  char              *olbl;
  XEvent             xev;
  int                mod, modifier;
  xitk_window_t     *xwin;
  kbinding_entry_t  *kbe;
  int                redundant;
  
  /* We are already grabbing keybinding */
  if(kbedit->grabbing)
    return;

  olbl = strdup(xitk_labelbutton_get_label(kbedit->grab));
 
  kbedit->grabbing = 1;
  
  kbe = (kbinding_entry_t *) calloc(1, sizeof(kbinding_entry_t));
  kbe->comment   = strdup(kbedit->ksel->comment);
  kbe->action    = strdup(kbedit->ksel->action);
  kbe->action_id = kbedit->ksel->action_id;
  kbe->is_alias  = kbedit->ksel->is_alias;
  kbe->is_gui    = kbedit->ksel->is_gui;
  
  xitk_labelbutton_change_label(kbedit->grab, _("Press Keyboard Keys..."));
  XLockDisplay(gGui->display);
  XSync(gGui->display, False);
  XUnlockDisplay(gGui->display);

  {
    int x, y, w, h;

    xitk_get_window_position(gGui->display, 
			     (xitk_window_get_window(kbedit->xwin)), &x, &y, &w, &h);
        
    xwin = xitk_window_create_dialog_window(gGui->imlib_data, 
					    _("Event Receiver Window:  Press keyboard keys to bind..."),
					    x, y, w, h);

    set_window_states_start((xitk_window_get_window(xwin)));
  }
  
  XLockDisplay(gGui->display);
  XRaiseWindow(gGui->display, (xitk_window_get_window(xwin)));
  XMapWindow(gGui->display, (xitk_window_get_window(xwin)));
  XUnlockDisplay(gGui->display);

  try_to_set_input_focus(xitk_window_get_window(xwin));

  do {
    /* Although only release events are evaluated, we must also grab the corresponding press */
    /* events to hide them from the other GUI windows and prevent unexpected side effects.   */
    XMaskEvent(gGui->display, ButtonPressMask | ButtonReleaseMask | KeyPressMask | KeyReleaseMask, &xev);
  } while ((xev.type != KeyRelease && xev.type != ButtonRelease) ||
	   xev.xany.window != xitk_window_get_window(xwin));
  
  (void) xitk_get_key_modifier(&xev, &mod);
  kbindings_convert_modifier(mod, &modifier);

  kbe->modifier = modifier;
  

  {
    char buf[256];
    xevent2id(&xev, &kbe->modifier, buf, sizeof(buf));
    kbe->key = strdup(buf);
    kbe->modifier &= ~MODIFIER_NUML;
  }
  
  xitk_labelbutton_change_label(kbedit->grab, olbl);

  xitk_window_destroy_window(gGui->imlib_data, xwin);
  
  XLockDisplay(gGui->display);
  XSync(gGui->display, False);
  XUnlockDisplay(gGui->display);

  kbedit->grabbing = 0;
  
  if((redundant = bkedit_check_redundancy(kbedit->kbt, kbe)) == -1) {
    char shortcut[32];
    
    kbedit_display_kbinding(xitk_label_get_label(kbedit->comment), kbe);
    
    snprintf(shortcut, sizeof(shortcut), "%c%s%c", '[', _kbindings_get_shortcut_from_kbe(kbe), ']');
    
    /* Ask if user wants to store new shortcut */
    xitk_window_dialog_yesno_with_width(gGui->imlib_data, _("Accept?"),
					kbedit_accept_yes, 
					kbedit_accept_no, 
					(void *) kbe, 400, ALIGN_CENTER,
					_("Store %s as\n'%s' key binding ?"),
					shortcut, kbedit->ksel->comment);
  }
  else {
    /* error, redundant */
    if(redundant >= 0) {
      xine_error(_("This key binding is redundant with action:\n\"%s\".\n"),
		 kbedit->kbt->entry[redundant]->comment);
    }
  }
  
  free(olbl);
  xitk_labelbutton_set_state(kbedit->alias, 0);
  xitk_labelbutton_set_state(kbedit->edit, 0);
  xitk_disable_widget(kbedit->grab);
}

/*
 *
 */
static void kbedit_handle_event(XEvent *event, void *data) {
  
  switch(event->type) {
    
  case KeyPress:
    if(xitk_get_key_pressed(event) == XK_Escape)
      kbedit_exit(NULL, NULL);
    else
      gui_handle_event(event, data);
    break;

  case KeyRelease: {
    xitk_widget_t *w;

    if((!kbedit) || (kbedit && !kbedit->widget_list))
      return;
    
    w = xitk_get_focused_widget(kbedit->widget_list);

    if((w && (w != kbedit->done) && (w != kbedit->save))
       && (xitk_browser_get_current_selected(kbedit->browser) < 0)) {
      kbedit_unset();
    }
  }
  break;

  case ButtonRelease: {
    xitk_widget_t *w;

    if((!kbedit) || (kbedit && !kbedit->widget_list))
      return;

    w = xitk_get_focused_widget(kbedit->widget_list);
    
    if((w && (w != kbedit->done) && (w != kbedit->save))
       && (xitk_browser_get_current_selected(kbedit->browser) < 0)) {
      kbedit_unset();
    }

  }
  break;
  
  }
}

void kbedit_reparent(void) {
  if(kbedit)
    reparent_window((xitk_window_get_window(kbedit->xwin)));
}

/*
 *
 */
void kbedit_window(void) {
  int                        x, y, y1;
  GC                         gc;
  xitk_pixmap_t             *bg;
  xitk_labelbutton_widget_t  lb;
  xitk_label_widget_t        l;
  xitk_browser_widget_t      br;
  xitk_checkbox_widget_t     cb;
  int                        btnw = 80;
  int                        fontheight;
  xitk_font_t               *fs;
  xitk_widget_t             *w;
  
  x = xine_config_register_num(__xineui_global_xine_instance, "gui.kbedit_x", 
			       80,
			       CONFIG_NO_DESC,
			       CONFIG_NO_HELP,
			       CONFIG_LEVEL_DEB,
			       CONFIG_NO_CB,
			       CONFIG_NO_DATA);
  y = xine_config_register_num(__xineui_global_xine_instance, "gui.kbedit_y",
			       80,
			       CONFIG_NO_DESC,
			       CONFIG_NO_HELP,
			       CONFIG_LEVEL_DEB,
			       CONFIG_NO_CB,
			       CONFIG_NO_DATA);
  
  kbedit = (_kbedit_t *) calloc(1, sizeof(_kbedit_t));

  kbedit->kbt           = _kbindings_duplicate_kbindings(gGui->kbindings);
  kbedit->action_wanted = KBEDIT_NOOP;
  kbedit->xwin          = xitk_window_create_dialog_window(gGui->imlib_data,
							   _("Key Binding Editor"),
							   x, y, WINDOW_WIDTH, WINDOW_HEIGHT);
  set_window_states_start((xitk_window_get_window(kbedit->xwin)));

  XLockDisplay (gGui->display);
  gc = XCreateGC(gGui->display, 
		 (xitk_window_get_window(kbedit->xwin)), None, None);
  XUnlockDisplay (gGui->display);

  kbedit->widget_list      = xitk_widget_list_new();

  xitk_widget_list_set(kbedit->widget_list, WIDGET_LIST_LIST, (xitk_list_new()));
  xitk_widget_list_set(kbedit->widget_list, 
		       WIDGET_LIST_WINDOW, (void *) (xitk_window_get_window(kbedit->xwin)));
  xitk_widget_list_set(kbedit->widget_list, WIDGET_LIST_GC, gc);
  
  bg = xitk_image_create_xitk_pixmap(gGui->imlib_data, WINDOW_WIDTH, WINDOW_HEIGHT);
  
  XLockDisplay (gGui->display);
  XCopyArea(gGui->display, (xitk_window_get_background(kbedit->xwin)), bg->pixmap,
	    bg->gc, 0, 0, WINDOW_WIDTH, WINDOW_HEIGHT, 0, 0);
  XUnlockDisplay (gGui->display);
  
  x = 15;
  y = 34;
  
  draw_rectangular_inner_box(gGui->imlib_data, bg, x, y,
			     (WINDOW_WIDTH - 30 - 1), (MAX_DISP_ENTRIES * 20 + 16 + 10 - 1));

  y += MAX_DISP_ENTRIES * 20 + 16 + 10 + 30;
  y1 = y; /* remember for later */
  draw_outter_frame(gGui->imlib_data, bg, 
		    _("Binding Action"), hboldfontname,
		    x, y, 
		    (WINDOW_WIDTH - 30), 45);

  y += 45 + 3;
  draw_outter_frame(gGui->imlib_data, bg, 
		    _("Key"), hboldfontname,
		    x, y, 
		    120, 45);

  draw_outter_frame(gGui->imlib_data, bg, 
		    _("Modifiers"), hboldfontname,
		    x + 130, y, 
		    (WINDOW_WIDTH - (x + 130) - 15), 45);


  xitk_window_change_background(gGui->imlib_data, kbedit->xwin, bg->pixmap,
				WINDOW_WIDTH, WINDOW_HEIGHT);
  
  xitk_image_destroy_xitk_pixmap(bg);

  kbedit_create_browser_entries();

  y = 34;

  XITK_WIDGET_INIT(&br, gGui->imlib_data);
  
  br.arrow_up.skin_element_name    = NULL;
  br.slider.skin_element_name      = NULL;
  br.arrow_dn.skin_element_name    = NULL;
  br.browser.skin_element_name     = NULL;
  br.browser.max_displayed_entries = MAX_DISP_ENTRIES;
  br.browser.num_entries           = kbedit->num_entries;
  br.browser.entries               = (const char* const*)kbedit->entries;
  br.callback                      = kbedit_sel;
  br.dbl_click_callback            = NULL;
  br.parent_wlist                  = kbedit->widget_list;
  br.userdata                      = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
			   (kbedit->browser = 
			    xitk_noskin_browser_create(kbedit->widget_list, &br,
						       (XITK_WIDGET_LIST_GC(kbedit->widget_list)),
						       x + 5, y + 5,
						       WINDOW_WIDTH - (30 + 10 + 16), 20,
						       16, br_fontname)));
  xitk_enable_and_show_widget(kbedit->browser);

  xitk_browser_set_alignment(kbedit->browser, ALIGN_LEFT);
  xitk_browser_update_list(kbedit->browser, 
			   (const char* const*) kbedit->entries, 
			   (const char* const*) kbedit->shortcuts, kbedit->num_entries, xitk_browser_get_current_start(kbedit->browser));

  y = y1 - 30 + 4;

  XITK_WIDGET_INIT(&lb, gGui->imlib_data);
  
  lb.button_type       = RADIO_BUTTON;
  lb.label             = _("Alias");
  lb.align             = ALIGN_CENTER;
  lb.callback          = NULL;
  lb.state_callback    = kbedit_alias;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
		   (kbedit->alias = 
		    xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, btnw, 23,
						   "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(kbedit->alias);
  
  x += btnw + 4;

  lb.button_type       = RADIO_BUTTON;
  lb.label             = _("Edit");
  lb.align             = ALIGN_CENTER;
  lb.callback          = NULL;
  lb.state_callback    = kbedit_edit;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
		   (kbedit->edit = 
		    xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, btnw, 23,
						   "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(kbedit->edit);
  
  x += btnw + 4;

  lb.button_type       = CLICK_BUTTON;
  lb.label             = _("Delete");
  lb.align             = ALIGN_CENTER;
  lb.callback          = kbedit_delete; 
  lb.state_callback    = NULL;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
		   (kbedit->delete = 
		    xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, btnw, 23,
						   "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(kbedit->delete);

  x += btnw + 4;

  lb.button_type       = CLICK_BUTTON;
  lb.label             = _("Save");
  lb.align             = ALIGN_CENTER;
  lb.callback          = kbedit_save; 
  lb.state_callback    = NULL;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
		   (kbedit->save = 
		    xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, btnw, 23,
						   "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(kbedit->save);

  x += btnw + 4;

  lb.button_type       = CLICK_BUTTON;
  lb.label             = _("Reset");
  lb.align             = ALIGN_CENTER;
  lb.callback          = kbedit_reset; 
  lb.state_callback    = NULL;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
   (w = xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, btnw, 23,
				       "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(w);

  x += btnw + 4;
  
  lb.button_type       = CLICK_BUTTON;
  lb.label             = _("Done");
  lb.align             = ALIGN_CENTER;
  lb.callback          = kbedit_exit; 
  lb.state_callback    = NULL;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
		   (kbedit->done = 
		    xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, btnw, 23,
						   "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(kbedit->done);

  x = 15;
  
  XITK_WIDGET_INIT(&l, gGui->imlib_data);

  fs = xitk_font_load_font(gGui->display, hboldfontname);
  xitk_font_set_font(fs, (XITK_WIDGET_LIST_GC(kbedit->widget_list)));
  fontheight = xitk_font_get_string_height(fs, " ");
  xitk_font_unload_font(fs);
  
  y = y1 + (45 / 2);                /* Checkbox                     */
  y1 = y - ((fontheight - 10) / 2); /* Text, v-centered to ckeckbox */

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = "Binding Action";
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
			   (kbedit->comment =
			    xitk_noskin_label_create(kbedit->widget_list, &l, x + 10, y1,
						     WINDOW_WIDTH - 50 - 2, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(kbedit->comment);

  y += 45 + 3;
  y1 += 45 + 3;

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = "THE Key";
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
			   (kbedit->key =
			    xitk_noskin_label_create(kbedit->widget_list, &l, x + 10, y1,
						     100, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(kbedit->key);
  
  XITK_WIDGET_INIT(&cb, gGui->imlib_data);

  x += 130 + 10;

  cb.callback          = NULL;
  cb.userdata          = NULL;
  cb.skin_element_name = NULL;
  xitk_list_append_content ((XITK_WIDGET_LIST_LIST(kbedit->widget_list)),
			    (kbedit->ctrl = 
			     xitk_noskin_checkbox_create(kbedit->widget_list, &cb, x, y, 10, 10)));
  xitk_show_widget(kbedit->ctrl);
  xitk_disable_widget(kbedit->ctrl);


  x += 15;

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = _("ctrl");
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
   (w = xitk_noskin_label_create(kbedit->widget_list, &l, x, y1,
				 40, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(w);

  x += 55;

  cb.callback          = NULL;
  cb.userdata          = NULL;
  cb.skin_element_name = NULL;
  xitk_list_append_content ((XITK_WIDGET_LIST_LIST(kbedit->widget_list)),
			    (kbedit->meta = 
			     xitk_noskin_checkbox_create(kbedit->widget_list, &cb, x, y, 10, 10)));
  xitk_show_widget(kbedit->meta);
  xitk_disable_widget(kbedit->meta);


  x += 15;

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = _("meta");
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
   (w = xitk_noskin_label_create(kbedit->widget_list, &l, x, y1,
				 40, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(w);

  x += 55;

  cb.callback          = NULL;
  cb.userdata          = NULL;
  cb.skin_element_name = NULL;
  xitk_list_append_content ((XITK_WIDGET_LIST_LIST(kbedit->widget_list)),
			    (kbedit->mod3 = 
			     xitk_noskin_checkbox_create(kbedit->widget_list, &cb, x, y, 10, 10)));
  xitk_show_widget(kbedit->mod3);
  xitk_disable_widget(kbedit->mod3);


  x += 15;

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = _("mod3");
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
   (w = xitk_noskin_label_create(kbedit->widget_list, &l, x, y1,
				 40, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(w);

  x += 55;

  cb.callback          = NULL;
  cb.userdata          = NULL;
  cb.skin_element_name = NULL;
  xitk_list_append_content ((XITK_WIDGET_LIST_LIST(kbedit->widget_list)),
			    (kbedit->mod4 = 
			     xitk_noskin_checkbox_create(kbedit->widget_list, &cb, x, y, 10, 10)));
  xitk_show_widget(kbedit->mod4);
  xitk_disable_widget(kbedit->mod4);


  x += 15;

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = _("mod4");
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
   (w = xitk_noskin_label_create(kbedit->widget_list, &l, x, y1,
				 40, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(w);

  x += 55;

  cb.callback          = NULL;
  cb.userdata          = NULL;
  cb.skin_element_name = NULL;
  xitk_list_append_content ((XITK_WIDGET_LIST_LIST(kbedit->widget_list)),
			    (kbedit->mod5 = 
			     xitk_noskin_checkbox_create(kbedit->widget_list, &cb, x, y, 10, 10)));
  xitk_show_widget(kbedit->mod5);
  xitk_disable_widget(kbedit->mod5);


  x += 15;

  l.window            = (XITK_WIDGET_LIST_WINDOW(kbedit->widget_list));
  l.gc                = (XITK_WIDGET_LIST_GC(kbedit->widget_list));
  l.skin_element_name = NULL;
  l.label             = _("mod5");
  l.callback          = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
   (w = xitk_noskin_label_create(kbedit->widget_list, &l, x, y1,
				 40, fontheight, hboldfontname)));
  xitk_enable_and_show_widget(w);

  x = 15;
  y = WINDOW_HEIGHT - (23 + 15);

  lb.button_type       = CLICK_BUTTON;
  lb.label             = _("Grab");
  lb.align             = ALIGN_CENTER;
  lb.callback          = kbedit_grab; 
  lb.state_callback    = NULL;
  lb.userdata          = NULL;
  lb.skin_element_name = NULL;
  xitk_list_append_content((XITK_WIDGET_LIST_LIST(kbedit->widget_list)), 
	   (kbedit->grab = 
	    xitk_noskin_labelbutton_create(kbedit->widget_list, &lb, x, y, WINDOW_WIDTH - 30, 23,
					   "Black", "Black", "White", hboldfontname)));
  xitk_enable_and_show_widget(kbedit->grab);

  kbedit_unset();
  
  kbedit->kreg = xitk_register_event_handler("kbedit", 
					     (xitk_window_get_window(kbedit->xwin)),
					     kbedit_handle_event,
					     NULL,
					     NULL,
					     kbedit->widget_list,
					     NULL);
  

  gGui->ssaver_enabled = 0;
  kbedit->visible      = 1;
  kbedit->running      = 1;
  kbedit_raise_window();

  try_to_set_input_focus(xitk_window_get_window(kbedit->xwin));
}
