/*
**************************************************************************
                                 description
                             --------------------
    copyright            : (C) 2002 by Andreas Zehender
    email                : zehender@kde.org
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/


#include "pmprismedit.h"
#include "pmprism.h"
#include "pmvectoredit.h"
#include "pmlineedits.h"

#include <kdebug.h>
#include "pmglobals.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <klocale.h>
#include <kdialog.h>
#include <kiconloader.h>
#include <kmessagebox.h>

PMPrismEdit::PMPrismEdit( QWidget* parent, const char* name )
      : Base( parent, name )
{
   m_pDisplayedObject = 0;
   m_edits.setAutoDelete( true );
   m_addButtons.setAutoDelete( true );
   m_removeButtons.setAutoDelete( true );
   m_lastSplineType = 0;
}

PMPrismEdit::~PMPrismEdit( )
{
   m_edits.clear( );
   m_addButtons.clear( );
   m_removeButtons.clear( );
}

void PMPrismEdit::createTopWidgets( )
{
   Base::createTopWidgets( );

   QHBoxLayout* hl = new QHBoxLayout( topLayout( ) );
   hl->addWidget( new QLabel( i18n( "Spline type:" ), this ) );
   m_pSplineType = new QComboBox( false, this );
   m_pSplineType->insertItem( i18n( "Linear Spline" ) );
   m_pSplineType->insertItem( i18n( "Quadratic Spline" ) );
   m_pSplineType->insertItem( i18n( "Cubic Spline" ) );
   m_pSplineType->insertItem( i18n( "Bezier Spline" ) );
   hl->addWidget( m_pSplineType );

   hl = new QHBoxLayout( topLayout( ) );
   hl->addWidget( new QLabel( i18n( "Sweep type:" ), this ) );
   m_pSweepType = new QComboBox( false, this );
   m_pSweepType->insertItem( i18n( "Linear Sweep" ) );
   m_pSweepType->insertItem( i18n( "Conic Sweep" ) );
   hl->addWidget( m_pSweepType );

   connect( m_pSplineType, SIGNAL( activated( int ) ),
            SLOT( slotTypeChanged( int ) ) );
   connect( m_pSweepType, SIGNAL( activated( int ) ),
            SLOT( slotSweepChanged( int ) ) );

   hl = new QHBoxLayout( topLayout( ) );
   QGridLayout* gl = new QGridLayout( hl, 2, 2 );
   gl->addWidget( new QLabel( i18n( "Height 1:" ), this ), 0, 0 );
   m_pHeight1 = new PMFloatEdit( this );
   gl->addWidget( m_pHeight1, 0, 1 );
   connect( m_pHeight1, SIGNAL( dataChanged( ) ), SIGNAL( dataChanged( ) ) );
   
   gl->addWidget( new QLabel( i18n( "Height 2:" ), this ), 1, 0 );
   m_pHeight2 = new PMFloatEdit( this );
   gl->addWidget( m_pHeight2, 1, 1 );
   connect( m_pHeight2, SIGNAL( dataChanged( ) ), SIGNAL( dataChanged( ) ) );
   hl->addStretch( 1 );
}

void PMPrismEdit::createBottomWidgets( )
{
   //topLayout( )->addWidget( new QLabel( i18n( "Prism Points:" ), this ) );
   m_pEditWidget = new QWidget( this );
   topLayout( )->addWidget( m_pEditWidget );
   m_pOpen = new QCheckBox( i18n( "Open" ), this );
   topLayout( )->addWidget( m_pOpen );
   m_pSturm = new QCheckBox( i18n( "Sturm" ), this );
   topLayout( )->addWidget( m_pSturm );
   connect( m_pSturm, SIGNAL( clicked( ) ), SIGNAL( dataChanged( ) ) );
   connect( m_pOpen, SIGNAL( clicked( ) ), SIGNAL( dataChanged( ) ) );

   Base::createBottomWidgets( );
}

void PMPrismEdit::displayObject( PMObject* o )
{
   if( o->isA( PMTPrism ) )
   {
      bool readOnly = o->isReadOnly( );
      m_pDisplayedObject = ( PMPrism* ) o;

      switch( m_pDisplayedObject->splineType( ) )
      {
         case PMPrism::LinearSpline:
            m_pSplineType->setCurrentItem( 0 );
            break;
         case PMPrism::QuadraticSpline:
            m_pSplineType->setCurrentItem( 1 );
            break;
         case PMPrism::CubicSpline:
            m_pSplineType->setCurrentItem( 2 );
            break;
         case PMPrism::BezierSpline:
            m_pSplineType->setCurrentItem( 3 );
            break;
      }
      m_pSplineType->setEnabled( !readOnly );
      switch( m_pDisplayedObject->sweepType( ) )
      {
         case PMPrism::LinearSweep:
            m_pSweepType->setCurrentItem( 0 );
            break;
         case PMPrism::ConicSweep:
            m_pSweepType->setCurrentItem( 1 );
            break;
      }
      m_pHeight1->setValue( m_pDisplayedObject->height1( ) );
      m_pHeight1->setReadOnly( readOnly );
      m_pHeight2->setValue( m_pDisplayedObject->height2( ) );
      m_pHeight2->setReadOnly( readOnly );
      m_pSweepType->setEnabled( !readOnly );
      m_pSturm->setChecked( m_pDisplayedObject->sturm( ) );
      m_pSturm->setEnabled( !readOnly );
      m_pOpen->setChecked( m_pDisplayedObject->open( ) );
      m_pOpen->setEnabled( !readOnly );
      displayPoints( m_pDisplayedObject->points( ) );

      Base::displayObject( o );
   }
   else
      kdError( PMArea ) << "PMPrismEdit: Can't display object\n";
}

void PMPrismEdit::displayPoints( const QValueList< QValueList<PMVector> >& sp )
{
   bool readOnly = m_pDisplayedObject->isReadOnly( );

   // (re)create the edit widget if neccessary
   createEdits( sp );
   
   QValueList< QValueList<PMVector> >::ConstIterator spit = sp.begin( );
   QPtrListIterator< QPtrList<PMVectorEdit> > seit( m_edits );

   // display the points
   for( ; ( spit != sp.end( ) ) && *seit; ++spit, ++seit )
   {
      QValueList<PMVector>::ConstIterator pit = ( *spit ).begin( );
      QPtrListIterator<PMVectorEdit> eit( **seit );

      for( ; ( pit != ( *spit ).end( ) ) && *eit; ++pit, ++eit )
      {
         ( *eit )->setVector( *pit );
         ( *eit )->setReadOnly( readOnly );
      }
   }

   // enable/disable the edits
   QPtrListIterator< QPtrList<QPushButton> > sbit1( m_addButtons );
   for( ; *sbit1; ++sbit1 )
   {
      QPtrListIterator<QPushButton> bit( **sbit1 );
      for( ; *bit; ++bit )
         ( *bit )->setEnabled( !readOnly );
   }
   QPtrListIterator< QPtrList<QPushButton> > sbit2( m_removeButtons );
   for( ; *sbit2; ++sbit2 )
   {
      QPtrListIterator<QPushButton> bit( **sbit2 );
      for( ; *bit; ++bit )
         ( *bit )->setEnabled( !readOnly );
   }
   QPtrListIterator<QPushButton> bit1( m_subPrismAddButtons );
   for( ; *bit1; ++bit1 )
      ( *bit1 )->setEnabled( !readOnly );
   QPtrListIterator<QPushButton> bit2( m_subPrismRemoveButtons );
   for( ; *bit2; ++bit2 )
      ( *bit2 )->setEnabled( !readOnly );
}

void PMPrismEdit::createEdits( const QValueList< QValueList<PMVector> >& sp )
{
   // test if the current edits can be reused
   bool reuse = true;
   int st = m_pSplineType->currentItem( );

   if( m_labels.count( ) == 0 )               // no edits created
      reuse = false;
   else if( m_lastSplineType != st )          // labels %1=%2 would be wrong
      reuse = false;
   else if( sp.count( ) != m_edits.count( ) ) // different number of sub prisms
      reuse = false;
   else
   {
      QValueList< QValueList<PMVector> >::ConstIterator spit = sp.begin( );
      QPtrListIterator< QPtrList<PMVectorEdit> > eit( m_edits );
      for( ; ( spit != sp.end( ) ) && *eit && reuse; ++spit, ++eit )
         if( ( *spit ).count( ) != ( *eit )->count( ) )
            reuse = false; // different number of points in sub prism
   }
   m_lastSplineType = st;

   if( !reuse )
   {
      deleteEdits( );
      
      // count number of needed lines
      QValueList< QValueList<PMVector> >::ConstIterator spit = sp.begin( );
      int lines = 1;
      for( ; spit != sp.end( ); ++spit )
      {
         if( st != 3 )
            lines += ( *spit ).count( ) + 4;
         else
            lines += ( *spit ).count( ) / 3 * 4 + 3;
      }
      
      QGridLayout* gl = new QGridLayout( m_pEditWidget, lines, 4,
                                         0, KDialog::spacingHint( ) );
      QPixmap addPixmap = SmallIcon( "pmaddpoint" );
      QPixmap removePixmap = SmallIcon( "pmremovepoint" );
      QPixmap addPrismPixmap = SmallIcon( "pmaddsubprism" );
      QLabel* label;
      QPushButton* button;
      int lnr = 0;
      int spnr = 0;
      
      for( spit = sp.begin( ); spit != sp.end( ); ++spit, spnr++ )
      {
         // create all edits for one sub prism
         int pnr;
         QValueList<PMVector>::ConstIterator pit = ( *spit ).begin( );
         QPtrList<PMVectorEdit>* editList = new QPtrList<PMVectorEdit>;
         QPtrList<QPushButton>* addList = new QPtrList<QPushButton>;
         QPtrList<QPushButton>* removeList = new QPtrList<QPushButton>;
         
         label = new QLabel( i18n( "Sub prism %1:" ).arg( spnr + 1 ),
                             m_pEditWidget );
         gl->addMultiCellWidget( label, lnr, lnr, 0, 1 );
         m_labels.append( label );
         label->show( );
         
         button = new QPushButton( m_pEditWidget );
         button->setPixmap( addPrismPixmap );
         m_subPrismAddButtons.append( button );
         connect( button, SIGNAL( clicked( ) ), SLOT( slotAddSubPrism( ) ) );
         gl->addWidget( button, lnr, 2 );
         button->show( );
         QToolTip::add( button, i18n( "Add Sub Prism" ) );
         
         button = new QPushButton( m_pEditWidget );
         button->setPixmap( removePixmap );
         m_subPrismRemoveButtons.append( button );
         connect( button, SIGNAL( clicked( ) ), SLOT( slotRemoveSubPrism( ) ) );
         gl->addWidget( button, lnr, 3 );
         button->show( );
         QToolTip::add( button, i18n( "Remove Sub Prism" ) );
         
         lnr++;
         
         button = new QPushButton( m_pEditWidget );
         button->setPixmap( addPixmap );
         addList->append( button );
         connect( button, SIGNAL( clicked( ) ), SLOT( slotAddPoint( ) ) );
         gl->addWidget( button, lnr, 2 );
         button->show( );
         QToolTip::add( button, i18n( "Add new point" ) );
         
         lnr++;
         
         int refa = 0, refb = ( *spit ).count( ) - 1;
         switch( m_pSplineType->currentItem( ) )
         {
            case 0:
               break;
            case 1:
               refa++;
               break;
            case 2:
               refa++;
               refb--;
               break;
            case 3:
               refb = ( refb + 1 ) / 3 * 4;
               break;
         }

         for( pnr = 0; pit != ( *spit ).end( ); ++pit )
         {
            // create the edits for one point
            PMVectorEdit* edit;
            
            label = new QLabel( QString( "%1:" ).arg( pnr + 1 ), m_pEditWidget );
            gl->addWidget( label, lnr, 0 );
            label->show( );
            m_labels.append( label );
            
            edit = new PMVectorEdit( "x", "z", m_pEditWidget );
            connect( edit, SIGNAL( dataChanged( ) ), SIGNAL( dataChanged( ) ) );
            editList->append( edit );
            gl->addWidget( edit, lnr, 1 );
            edit->show( );
            
            button = new QPushButton( m_pEditWidget );
            button->setPixmap( addPixmap );
            addList->append( button );
            connect( button, SIGNAL( clicked( ) ), SLOT( slotAddPoint( ) ) );
            gl->addWidget( button, lnr, 2 );
            button->show( );
            QToolTip::add( button, i18n( "Add new point" ) );
         
            button = new QPushButton( m_pEditWidget );
            button->setPixmap( removePixmap );
            removeList->append( button );
            connect( button, SIGNAL( clicked( ) ), SLOT( slotRemovePoint( ) ) );
            gl->addWidget( button, lnr, 3 );
            button->show( );
            QToolTip::add( button, i18n( "Remove point" ) );

            if( st != 3 )
            {
               if( pnr == refb )
               {
                  pnr++;
                  lnr++;
                  label = new QLabel( QString( "%1 = %2" ).arg( pnr + 1 )
                                      .arg( refa + 1 ), m_pEditWidget );
                  gl->addMultiCellWidget( label, lnr, lnr, 0, 1 );
                  label->show( );
                  m_labels.append( label );
               }
            }
            else
            {
               if( ( ( pnr + 2 ) %4 ) == 0 )
               {
                  pnr++;
                  lnr++;
                  label = new QLabel( QString( "%1 = %2" ).arg( pnr + 1 )
                                      .arg( ( pnr + 1 ) % refb + 1 ),
                                      m_pEditWidget );
                  gl->addMultiCellWidget( label, lnr, lnr, 0, 1 );
                  label->show( );
                  m_labels.append( label );
               }
            }
            pnr++;
            lnr++;
         }
         m_edits.append( editList );
         m_addButtons.append( addList );
         m_removeButtons.append( removeList );

         // One extra line
         gl->addRowSpacing( lnr, 8 );
         lnr++;
      }

      label = new QLabel( i18n( "New sub prism" ), m_pEditWidget );
      gl->addMultiCellWidget( label, lnr, lnr, 0, 1 );
      m_labels.append( label );
      label->show( );
         
      button = new QPushButton( m_pEditWidget );
      button->setPixmap( addPrismPixmap );
      m_subPrismAddButtons.append( button );
      connect( button, SIGNAL( clicked( ) ), SLOT( slotAddSubPrism( ) ) );
      gl->addWidget( button, lnr, 2 );
      button->show( );
      QToolTip::add( button, i18n( "Append Sub Prism" ) );
      lnr++;
   }
}

void PMPrismEdit::deleteEdits( )
{
   QPtrListIterator< QPtrList<PMVectorEdit> > elit( m_edits );
   for( ; elit.current( ); ++elit )
   {
      elit.current( )->setAutoDelete( true );
      elit.current( )->clear( );
   }
   m_edits.clear( );
   QPtrListIterator< QPtrList<QPushButton> > blit( m_addButtons );
   for( ; blit.current( ); ++blit )
   {
      blit.current( )->setAutoDelete( true );
      blit.current( )->clear( );
   }
   m_addButtons.clear( );
   QPtrListIterator< QPtrList<QPushButton> > rlit( m_removeButtons );
   for( ; rlit.current( ); ++rlit )
   {
      rlit.current( )->setAutoDelete( true );
      rlit.current( )->clear( );
   }
   m_removeButtons.clear( );
   m_labels.setAutoDelete( true );
   m_labels.clear( );
   m_labels.setAutoDelete( false );
   m_subPrismAddButtons.setAutoDelete( true );
   m_subPrismAddButtons.clear( );
   m_subPrismAddButtons.setAutoDelete( false );
   m_subPrismRemoveButtons.setAutoDelete( true );
   m_subPrismRemoveButtons.clear( );
   m_subPrismRemoveButtons.setAutoDelete( false );
   
   if( m_pEditWidget->layout( ) )
      delete m_pEditWidget->layout( );
}

QValueList< QValueList<PMVector> > PMPrismEdit::splinePoints( )
{
   QPtrListIterator< QPtrList<PMVectorEdit> > it( m_edits );
   QValueList< QValueList<PMVector> > values;
   QValueList<PMVector> subPrism;
   
   for( ; it.current( ); ++it )
   {
      subPrism.clear( );
      QPtrListIterator<PMVectorEdit> sit( *it.current( ) );
      for( ; sit.current( ); ++sit )
         subPrism.append( sit.current( )->vector( ) );
      values.append( subPrism );
   }
   
   return values;
}

void PMPrismEdit::saveContents( )
{
   if( m_pDisplayedObject )
   {
      m_pDisplayedObject->setPoints( splinePoints( ) );

      switch( m_pSplineType->currentItem( ) )
      {
         case 0:
            m_pDisplayedObject->setSplineType( PMPrism::LinearSpline );
            break;
         case 1:
            m_pDisplayedObject->setSplineType( PMPrism::QuadraticSpline );
            break;
         case 2:
            m_pDisplayedObject->setSplineType( PMPrism::CubicSpline );
            break;
         case 3:
            m_pDisplayedObject->setSplineType( PMPrism::BezierSpline );
            break;
      }
      switch( m_pSweepType->currentItem( ) )
      {
         case 0:
            m_pDisplayedObject->setSweepType( PMPrism::LinearSweep );
            break;
         case 1:
            m_pDisplayedObject->setSweepType( PMPrism::ConicSweep );
            break;
      }
      m_pDisplayedObject->setSturm( m_pSturm->isChecked( ) );
      m_pDisplayedObject->setOpen( m_pOpen->isChecked( ) );
      m_pDisplayedObject->setHeight1( m_pHeight1->value( ) );
      m_pDisplayedObject->setHeight2( m_pHeight2->value( ) );
      Base::saveContents( );
   }
}

bool PMPrismEdit::isDataValid( )
{
   QPtrListIterator< QPtrList<PMVectorEdit> > it( m_edits );
   for( ; it.current( ); ++it )
   {
      QPtrListIterator<PMVectorEdit> sit( *it.current( ) );
      for( ; sit.current( ); ++sit )
         if( !sit.current( )->isDataValid( ) )
            return false;
   }

   for( it.toFirst( ); it.current( ); ++it )
   {
      int np = it.current( )->count( );
      switch( m_pSplineType->currentItem( ) )
      {
         case 0:
            if( np < 3 )
            {
               KMessageBox::error( this, i18n( "Linear splines need at least 4 points." ),
                                   i18n( "Error" ) );
               return false;
            }
            break;
         case 1:
            if( np < 4 )
            {
               KMessageBox::error( this, i18n( "Quadratic splines need at least 5 points." ),
                                   i18n( "Error" ) );
               return false;
            }
            break;
         case 2:
            if( np < 5 )
            {
               KMessageBox::error( this, i18n( "Cubic splines need at least 6 points." ),
                                   i18n( "Error" ) );
               return false;
            }
            break;
         case 3:
            if( ( np < 3 ) || ( ( np % 3 ) != 0 ) )
            {
               KMessageBox::error( this, i18n( "Bezier splines need 4 points for each segment." ),
                                   i18n( "Error" ) );
               return false;
            }
            break;
      }
   }
   return Base::isDataValid( );
}

void PMPrismEdit::slotTypeChanged( int )
{
   displayPoints( splinePoints( ) );
   emit dataChanged( );
   emit sizeChanged( );
}

void PMPrismEdit::slotSweepChanged( int )
{
   emit dataChanged( );
}

void PMPrismEdit::slotAddPoint( )
{
   QPushButton* button = ( QPushButton* ) sender( );
   if( button )
   {
      QValueList< QValueList<PMVector> > points = splinePoints( );
      QPtrListIterator< QPtrList<QPushButton> > seit( m_addButtons );
      QValueList<QValueList<PMVector> >::Iterator spit = points.begin( );
      bool found = false;

      for( ; ( spit != points.end( ) ) && *seit && !found; ++spit, ++seit )
      {  
         int index = ( *seit )->findRef( button );
         if( index >= 0 )
         {
            found = true;
            QValueListIterator<PMVector> it = ( *spit ).at( index );
            PMVector newPoint( 2 );
            if( index == 0 )
               newPoint = ( *spit ).first( );
            else
            {
               --it;
               newPoint = *it;
               ++it;
               if( it != ( *spit ).end( ) )
                  newPoint = ( newPoint + *it ) / 2;
            }
            ( *spit ).insert( it, newPoint );
         }
      }
      if( found )
      {
         displayPoints( points );
         emit dataChanged( );
         emit sizeChanged( );
      }
   }
}

void PMPrismEdit::slotRemovePoint( )
{
   QPushButton* button = ( QPushButton* ) sender( );
   if( button )
   {
      QValueList< QValueList<PMVector> > points = splinePoints( );
      QPtrListIterator< QPtrList<QPushButton> > seit( m_removeButtons );
      QValueList<QValueList<PMVector> >::Iterator spit = points.begin( );
      bool found = false;

      for( ; ( spit != points.end( ) ) && *seit && !found; ++spit, ++seit )
      {  
         int index = ( *seit )->findRef( button );
         if( index >= 0 )
         {
            found = true;
            QValueListIterator<PMVector> it = ( *spit ).at( index );
            if( ( *spit ).count( ) > 1 )
               ( *spit ).remove( it );
         }
      }
      if( found )
      {
         displayPoints( points );
         emit dataChanged( );
         emit sizeChanged( );
      }
   }
}

void PMPrismEdit::slotAddSubPrism( )
{
   if( m_pSplineType->currentItem( ) == 3 )
   {
      KMessageBox::information( this, i18n( "Sub prisms do not work with "
                                            "bezier splines in pov-ray 3.1" ),
                                i18n( "Warning" ), "subPrismWithBezierSplines" );
   }
   
   QPushButton* button = ( QPushButton* ) sender( );
   if( button )
   {
      int index = m_subPrismAddButtons.findRef( button );
      if( index >= 0 )
      {
         QValueList< QValueList<PMVector> > points = splinePoints( );
         QValueList< QValueList<PMVector> >::Iterator it = points.at( index );
         QValueList<PMVector> newSubPrism;

         if( it != points.begin( ) )
         {
            --it;
            newSubPrism = *it;
            ++it;
            
            // find middle point
            PMVector mid( 2 );
            int num = 0;
            QValueList<PMVector>::Iterator pit = newSubPrism.begin( );
            for( ; pit != newSubPrism.end( ); ++pit, ++num )
               mid += *pit;
            if( num > 0 )
               mid /= num;
            for( pit = newSubPrism.begin( ); pit != newSubPrism.end( ); ++pit )
               *pit = ( *pit - mid ) * 0.8 + mid;
         }
         else
            newSubPrism = *it;

         points.insert( it, newSubPrism );
         displayPoints( points );
         emit dataChanged( );
         emit sizeChanged( );
      }
   }
}

void PMPrismEdit::slotRemoveSubPrism( )
{
   QPushButton* button = ( QPushButton* ) sender( );
   if( button )
   {
      int index = m_subPrismRemoveButtons.findRef( button );
      if( index >= 0 )
      {
         QValueList< QValueList<PMVector> > points = splinePoints( );
         QValueList< QValueList<PMVector> >::Iterator it = points.at( index );

         if( points.count( ) > 1 )
         {
            points.remove( it );
            displayPoints( points );
            emit dataChanged( );
            emit sizeChanged( );
         }
      }
   }
}

#include "pmprismedit.moc"
