% This -*- slang -*- file defines syntax highlighting and 
% indentation routines for Perl
% ------------------------------------------------------------------------
% Changes:
% The original author: Lars Marowsky-Bree <lmb@pointer.in-minden.de>
%
% 2001-01-10  Mark Olesen <mark.olesen@gmx.de>
% - modified a few keywords, added a Perl specific keymap
% - indentation mode improved and now includes a POD mode
% - added begin/end/mark/prev/next 'chunk' (sub or pod)
%
% NB: no automatic brace '{' handling!  otherwise constructs such as
%   '@foo = @{$arrayref}', and '%foo = %{$hashref}' are a nightmare!
% ------------------------------------------------------------------------
%{{{ default values for Perl custom variables
% override these default values in ~/.jedrc
% %!%+
%\variable{Perl_Continued_Offset}
%\synopsis{Perl_Continued_Offset}
%\usage{Integer Perl_Continued_Offset = 2;}
%\description
% This variable controls the indentation of statements that are continued
% onto the next line as in the following example:
%#v+
%  if (something) {
%    continued_statement ();
%  } else {
%    another_continued_statement ();
%  }
%#v-
%\seealso{C_CONTINUED_OFFSET, Perl_Indent}
%!%-
custom_variable ("Perl_Continued_Offset", 2);
%!%+
%\variable{Perl_Indent}
%\synopsis{Perl_Indent}
%\usage{Integer Perl_Indent = 4;}
%\description
% This value determines the number of columns the current line is indented
% past the previous line containing an opening \var{'{'} character.
%\seealso{C_INDENT, Perl_Continued_Offset}
%!%-
custom_variable ("Perl_Indent", 4);
%}}}
%{{{ create/initialize key map
$1 = "perl";
!if (keymap_p ($1)) make_keymap ($1);
definekey ("indent_line",      "\t",   $1);
definekey ("perl_beg_chunk",  "\e^A", $1);
definekey ("perl_end_chunk",  "\e^E", $1);
definekey ("perl_mark_chunk", "\e^H", $1);
definekey ("perl_next_chunk", "\e^N", $1);
definekey ("perl_prev_chunk", "\e^P", $1);
definekey ("perl_mark_matching", "\e^M", $1);
definekey ("perl_format_paragraph", "\eq", $1);
% definekey ("perl_newline_and_indent", "\r", $1);
definekey ("perl_indent_region", "\e\t", $1);	% override 'move-to-tab'
%}}} key map
%{{{ create/initialize syntax tables
$1 = "perl";
create_syntax_table ($1);
define_syntax ("#", "", '%', $1);		% single line comment
define_syntax ("=pod", "=cut", '%', $1);	% multiline comments
define_syntax ("([{", ")]}", '(', $1);
define_syntax ('\'', '"', $1);
define_syntax ('"', '"', $1);
define_syntax ('\\', '\\', $1);
define_syntax ("$0-9A-Z_a-z", 'w', $1);     % words
define_syntax ("-+.0-9_xa-fA-F", '0', $1);   % Numbers
define_syntax (",;.?:", ',', $1);
define_syntax ("%-+/&*=<>|!~^", '+', $1);
set_syntax_flags ($1, 0x10|0x80);
%}}} syntax table

% this experiment didn't work
% set_syntax_flags ($1, 0x10|0x80|0x02);
% set_fortran_comment_chars ($1, "=");

% with DFA =pod / =cut directives are not recognized

#ifdef HAS_DFA_SYNTAX %{{{
%%% DFA_CACHE_BEGIN %%%
static define setup_dfa_callback (name)
{
   dfa_enable_highlight_cache("perl.dfa", name);
   dfa_define_highlight_rule("#.*$", "comment", name);
   dfa_define_highlight_rule("([\\$%&@\\*]|\\$#)[A-Za-z_0-9]+", "normal", name);
   dfa_define_highlight_rule(strcat("\\$([_\\./,\"\\\\#\\*\\?\\]\\[;!@:\\$<>\\(\\)",
				"%=\\-~\\^\\|&`'\\+]|\\^[A-Z])"), "normal", name);
   dfa_define_highlight_rule("[A-Za-z_][A-Za-z_0-9]*", "Knormal", name);
   dfa_define_highlight_rule("[0-9]+(\\.[0-9]+)?([Ee][\\+\\-]?[0-9]*)?", "number",
			 name);
   dfa_define_highlight_rule("0[xX][0-9A-Fa-f]*", "number", name);
   dfa_define_highlight_rule("[\\(\\[\\{\\<\\>\\}\\]\\),;\\.\\?:]", "delimiter", name);
   dfa_define_highlight_rule("[%\\-\\+/&\\*=<>\\|!~\\^]", "operator", name);
   dfa_define_highlight_rule("-[A-Za-z]", "keyword0", name);
   dfa_define_highlight_rule("'[^']*'", "string", name);
   dfa_define_highlight_rule("'[^']*$", "string", name);
   dfa_define_highlight_rule("\"([^\"\\\\]|\\\\.)*\"", "string", name);
   dfa_define_highlight_rule("\"([^\"\\\\]|\\\\.)*\\\\?$", "string", name);
   dfa_define_highlight_rule("m?/([^/\\\\]|\\\\.)*/[gio]*", "string", name);
   dfa_define_highlight_rule("m/([^/\\\\]|\\\\.)*\\\\?$", "string", name);
   dfa_define_highlight_rule("s/([^/\\\\]|\\\\.)*(/([^/\\\\]|\\\\.)*)?/[geio]*",
			 "string", name);
   dfa_define_highlight_rule("s/([^/\\\\]|\\\\.)*(/([^/\\\\]|\\\\.)*)?\\\\?$",
			 "string", name);
   dfa_define_highlight_rule("(tr|y)/([^/\\\\]|\\\\.)*(/([^/\\\\]|\\\\.)*)?/[cds]*",
			 "string", name);
   dfa_define_highlight_rule("(tr|y)/([^/\\\\]|\\\\.)*(/([^/\\\\]|\\\\.)*)?\\\\?$",
			 "string", name);
   dfa_define_highlight_rule(".", "normal", name);
   dfa_build_highlight_table (name);
}
dfa_set_init_callback (&setup_dfa_callback, "perl");
%%% DFA_CACHE_END %%%
%}}}
#endif

%{{{ keywords
% Type 0 keywords
() = define_keywords ($1,
		      "doiflcmynoqqqwqxtruc",
		      2);
() = define_keywords ($1,
		      "abschrcosdieeofexpforhexintlogmapoctordpoppos" +
		      "refsinsubtieusevec",
		      3);
() = define_keywords ($1,
		      "bindcarpchopdumpeachelseevalexecexitforkgetcglobgoto" +
		      "grepjoinkeyskilllastlinknextopenpackpipepush" +
		      "randreadrecvredoseeksendsortsqrtstattelltime" +
		      "waitwarn",
		      4);
() = define_keywords ($1,
		      "alarmatan2blesschdirchmodchompchownclosecroakcrypt" +
		      "elsiffcntlflockindexioctllocallstatmkdirprint" +
		      "resetrmdirsemopshiftsleepsplitsrandstudytimes" +
		      "umaskundefuntieuntilutimewhilewrite",
		      5);
() = define_keywords ($1,
		      "acceptcallerchrootdeleteexistsfilenogmtime" +
		      "importlengthlistenmsgctlmsggetmsgrcvmsgsnd" +
		      "printfrenamereturnrindexscalarselectsemctl" +
		      "semgetshmctlshmgetsocketsplicesubstrsystem" +
		      "unlessunlinkunpackvalues",
		      6);
() = define_keywords ($1,
		      "binmodeconnectdefinedforeachgetpgrpgetppid" +
		      "lcfirstopendirpackagereaddirrequirereverse" +
		      "seekdirsetpgrpshmreadsprintfsymlinksyscall" +
		      "sysreadtelldirucfirstunshiftwaitpid",
		      7);
() = define_keywords ($1,
		      "closedirendgrentendpwentformline" +
		      "getgrentgetgrgidgetgrnamgetlogingetpwent" +
		      "getpwnamgetpwuidreadlinksetgrentsetpwent" +
		      "shmwriteshutdownsyswritetruncate",
		      8);
() = define_keywords ($1,
		      "endnetentgetnetentlocaltimequotemeta" +
		      "rewinddirsetnetentwantarray",
		      9);

() = define_keywords ($1,
		      "endhostentendserventgethostentgetservent" +
		      "getsockoptsethostentsetserventsetsockoptsocketpair",
		      10);
() = define_keywords ($1,
		      "endprotoentgetpeernamegetprioritygetprotoent" +
		      "getsocknamesetprioritysetprotoent",
		      11);
() = define_keywords ($1,
		      "getnetbyaddrgetnetbyname",
		      12);
() = define_keywords ($1,
		      "gethostbyaddrgethostbynamegetservbynamegetservbyport",
		      13);
() = define_keywords ($1,
		      "getprotobyname",
		      14);
() = define_keywords ($1,
		      "getprotobynumber",
		      16);

% Type 1 keywords - use for operator like keywords
() = define_keywords_n ($1, "eqgegtleltneor", 2, 1);
() = define_keywords_n ($1, "andcmpnotxor", 3, 1);
%}}} keywords

% return 0/1:
% we are / are not in pod (Perl plain old documentation)
static define in_pod ()
{
    variable beg = "=pod", mid = "=head", end = "=cut";
    variable here = what_line ();

    push_spot ();
    EXIT_BLOCK { pop_spot (); }

    bol ();
    if (looking_at_char ('=')) {
	if (orelse {looking_at (end)} {looking_at (beg)} {looking_at (mid)})
	  return 1;	% pod!
    }

    variable x = bol_bsearch (beg);
    !if (x)  x = bol_bsearch (mid);

    if (x) {
	if (bol_fsearch (end)) {
	    if (here <= what_line ()) return 1;	% pod!
	}
    }
    return 0;
}

private define perl_parse_to_point ()
{
    variable ptp = -2;
    if (in_pod()) return ptp;	% treat pod like a giant comment
    ptp = parse_to_point();
    !if (ptp) {
	push_spot();
	bol_skip_white();
	if ('#' == what_char()) ptp = -2;
	pop_spot();
    }
    return ptp;
}

% adapted (stolen) from cmode.sl
static define perl_indent_to (n)
{
    push_spot();
    bol_skip_white ();
    if (what_column != n) {
	bol_trim ();
	n--;
	whitespace (n);
    }
    pop_spot();
}

public define perl_indent_line()
{
    variable ch;	% the first character
    variable col = 1;	% default to indent on first column
    variable indent_ok = 0;

    push_spot();
    % on exit: restore position and indent to the prescribed 'col'
    EXIT_BLOCK {
	goto_spot ();
	perl_indent_to (col);
	bskip_white ();
	col = bolp();
	pop_spot ();
	if (col) bol_skip_white();	% (start of line)
    }

    % ---------------------------------------------------------------------
    % how to indent Perl POD (Perl plain old documentation)
    % simple indent mode, assumes the previous line is correctly indented
    % trims blank lines (since this gives pod problems otherwise)
    %----------------------------------------------------------------------
    bol();
    ch = what_char ();	% the first character
    if (in_pod()) {
	if (ch == '=') return;
	do {
	    eol_trim (); bol_skip_white();
	    if (eolp()) return;	% don't indent blanks!
	    col = what_column();	% best guess - leave here
	} while (up_1());
	return;
    }

    % ---------------------------------------------------------------------
    % indent normal Perl
    %----------------------------------------------------------------------
    bol_skip_white();
    ch = what_char ();	% the first character

    % If inside a () indent to level of opening '(' + 1
    % Ex:
    %  foo (bar
    %       baz(fum
    %           foz))
    if (find_matching_delimiter(')') == 1) {
	col = what_column() + 1;
	indent_ok = 1;
    }

    goto_spot (); bol();	% (original position : start of line)

    % could skip this ...
    % parse_to_point() doesn't seem to work with the perl mode anyhow
    
    % If strings continue through lines, indent them to the first column.
    % Ex:
    %    "foo
    % bar
    % baz"
    %    foobar();
    if (perl_parse_to_point() == -1) {
	col = 1;
	return;
    }

    goto_spot (); bol ();	% (original position : start of line)


    if (andelse 
	{find_matching_delimiter('}') == 1}
	  {not (blooking_at("#{{")) }	% '}' did not match a '#{{{' fold
	) {
	% (original position : start of line : start of last '{' line)
	bol_skip_white();
	col = what_column();	% Indent to last '{'

	% variable p;
	% p = perl_parse_to_point();
	% message (sprintf ("ptp = %d", p));

	if (ch == '}')
	  indent_ok = 1;
	else
	  col += Perl_Indent;	% Indent to last '{' + Perl_Indent
    }
    if (ch == '{') indent_ok = 1; % don't need any extra offset

    if (indent_ok) return;

    goto_spot ();	% (original position)

    % Find previous non-comment line
    do {
	% Start of file, pretty hard to find context ;-)
	!if (up_1 ()) {
	    bol();
	    break;
	}
	bol_skip_white();
	!if (eolp()) go_right_1();
    } while (perl_parse_to_point() == -2);
    %trim();
    eol();

    % Find last non-comment character
    variable ptp;
    while (ptp = perl_parse_to_point (), (ptp == -2)) {
	!if (left(1)) break;	% Oops?
    }

    bskip_white ();
    ch = ';';			% default final character
    !if (bolp()) {
	go_left_1 ();
	if (perl_parse_to_point() != -2) ch = what_char ();
	% flush (sprintf ("end char = %c", ch));
    }
    
    % commas are annoying:
    % either I can properly indent something like this:
    %   print "This is a ",
    %     "multi-line",
    %     " print \n";
    % or something like this:
    %  %hash =
    %   (
    %    key1 =>
    %    {
    %      subkey1 => 'foo',
    %      subkey2 => 'bar',
    %    },
    %    key2 =>
    %    {
    %      subkey1 => 'bar',
    %      subkey2 => 'foo',
    %    }
    %  );
    % but I apparently cannot do both!

    bol_skip_white();
    %    !if (is_substr("(,;{}", char(ch))) col += Perl_Continued_Offset;
    !if (is_substr("(;{}", char(ch))) col += Perl_Continued_Offset;
}

define perl_indent_region ()
{
    variable line = 0;
    variable nlines;
    push_spot ();

    if (markp ()) {
	check_region (0);	% canonical region
    } else {
	bob (); push_mark (); eob ();	% mark whole buffer
    }
    narrow ();
    nlines = what_line ();	% number of lines
    bob ();      		% start here

    do {
	line++;
	eol_trim ();
	bol ();
	% skip the comment
	if (looking_at ("=pod") or looking_at ("=head")) {
	    while (down_1 ()) {
		line++;
		if (looking_at ("=cut")) break;
		eol_trim ();
	    }
	    continue;
	}
	skip_white ();

	% skip the comment
	if (looking_at ("#")) {
	    indent_line ();
	    continue;
	}

	indent_line ();
	flush (sprintf ("processed %d/%d lines.", line, nlines));
    } while (down_1 ());
    trim_buffer ();
    widen ();
    flush (sprintf ("processed %d/%d lines.", line, nlines));
    pop_spot ();
}

% provide a few simple routines to move thru
% and mark 'chunks' (WEB notation), where a 'chunk' may either
% be a text chunk (in this case '=pod/=cut')
% or a code chunk (in this case 'sub foo { ... }')
define perl_beg_chunk ()
{
    variable beg;
    if (in_pod()) beg = "=pod"; else beg = "sub";
    eol(); !if (bol_bsearch (beg)) error ("Top of '" + beg + "'not found.");
}

define perl_end_chunk ()
{
    perl_beg_chunk ();
    if (in_pod()) {
	variable end = "=cut";
	bol(); !if (bol_fsearch(end)) error (end + " not found");
    } else {
	() = fsearch_char ('{');
	call ("goto_match");
    }
}

define perl_mark_chunk ()
{
    perl_beg_chunk ();
    push_visible_mark ();
    perl_end_chunk ();
    eol ();
    exchange_point_and_mark ();
}

static define perl_prev_next_chunk (dirfun)
{
    push_mark ();
    while (@dirfun()) {
	bol();
	if (looking_at ("=pod") or looking_at ("sub")) {
	    pop_mark_0 ();
	    return;
	}
    }
    pop_mark_1 ();
}

define perl_prev_chunk () { perl_prev_next_chunk (&up_1); }
define perl_next_chunk () { perl_prev_next_chunk (&down_1); }

define perl_mark_matching ()
{
    variable beg = "([{", end = ")]}";
    variable ch = what_char();
    
    ERROR_BLOCK {
	pop_mark_1 ();
    }

    if (is_substr (beg, char (ch))) {
	push_visible_mark ();
	if (1 != find_matching_delimiter (ch))
	  error ("matching delimiter not found");
	go_right_1 ();
	exchange_point_and_mark ();
    } else if (is_substr (end, char (ch))) {
	push_visible_mark ();
	if (1 != find_matching_delimiter (ch)) 
	  error ("matching delimiter not found");
	exchange_point_and_mark ();
    }
}

static define perl_init_menu (menu)
{
    menu_append_item (menu, "&Top of Function",	"perl_beg_chunk");
    menu_append_item (menu, "&End of Function",	"perl_end_chunk");
    menu_append_item (menu, "&Mark Function",	"perl_mark_chunk");
}

static variable Perlmode_Comment    = "# ";
static variable Perlmode_CommentLen = 2;

% this is mostly really annoying when bound to '\r'
define perl_newline_and_indent ()
{
    if (bolp ()) {
	newline ();
	indent_line ();
	return;
    }

    push_spot ();
    bol_skip_white ();
    if (looking_at (Perlmode_Comment)) {
	variable col = what_column ();
	col -= Perlmode_CommentLen;
	pop_spot ();
	newline ();
	insert_spaces (col);
	insert (Perlmode_Comment);
	return;
    }
    pop_spot ();

    newline ();
    indent_line ();
}

% adapted (stolen) from cmisc.sl - formats a comment in Perl mode
static variable Perlmode_Fill_Chars = "";
define perl_paragraph_sep ()
{
    if (strlen (Perlmode_Fill_Chars)) return 0;
    push_spot ();
    bol_skip_white ();
    if (looking_at (Perlmode_Comment)) {
	go_right (Perlmode_CommentLen);
	skip_white ();
	if (looking_at ("@ ")) eol ();	% don't wrap this line
    }
   
   eolp () or (-2 != parse_to_point ());
   pop_spot ();
}

define perl_format_paragraph ()
{
    variable col;
    
    % !if (is_c_mode ()) return;
    Perlmode_Fill_Chars = "";
    if (perl_paragraph_sep ()) return;
    push_spot (); push_spot (); push_spot ();
    while (not(perl_paragraph_sep ())) {
	!if (up_1 ()) break;
    }
    if (perl_paragraph_sep ()) go_down_1 ();
    push_mark ();
    pop_spot ();
    
    while (not(perl_paragraph_sep ())) {
	!if (down_1 ()) break;
    }
    if (perl_paragraph_sep ()) go_up_1 ();
    narrow ();
    pop_spot ();
    bol ();
    push_mark ();
    skip_white ();
    if (looking_at (Perlmode_Comment)) go_right (Perlmode_CommentLen);

    Perlmode_Fill_Chars = bufsubstr ();
    col = what_column ();
    bob ();
    do {
	bol_trim ();
	if (looking_at (Perlmode_Comment)) deln (Perlmode_CommentLen);
    } while (down_1 ());
    WRAP -= col;
    call ("format_paragraph");
    WRAP += col;
    bob ();
    do {
	insert (Perlmode_Fill_Chars);
    } while (down_1 ());
       
    Perlmode_Fill_Chars = "";
    widen ();
    pop_spot ();
}

%!%+
%\function{perl_mode}
%\synopsis{perl_mode}
%\usage{Void perl_mode();}
%\description
% This is a mode that is dedicated to editing Perl language files
% including a bimodal Pod/Perl indentation mode.
% Functions that affect this mode include:
%#v+
%  function:             default binding:
%  indent_line                TAB
%  perl_beg_chunk             ESC Ctrl-A
%  perl_end_chunk             ESC Ctrl-E
%  perl_mark_chunk            ESC Ctrl-H
%  perl_next_chuck            ESC Ctrl-N
%  perl_prev_chunk            ESC Ctrl-P
%  perl_indent_region         ESC TAB
%  perl_format_paragraph      ESC q
%  perl_newline_and_indent    Ctrl-M (not bound)
%#v-
% Variables affecting this mode include:
%#v+
%  Perl_Indent
%  Perl_Continued_Offset
%#v-
% Hooks: \var{perl_mode_hook}
%!%-
define perl_mode ()
{
    variable mode = "perl";
    set_mode (mode, 4);
    use_keymap(mode);
#iftrue
    push_spot ();
    bob ();
    if ((0 == bol_fsearch ("=pod")) and (bol_fsearch ("=head")))
      define_syntax ("=head", "=cut", '%', mode);
    else
      define_syntax ("=pod", "=cut", '%', mode);
    pop_spot ();
#endif
    use_syntax_table (mode);
    set_buffer_hook ("par_sep", "perl_paragraph_sep");
    set_buffer_hook ("indent_hook", &perl_indent_line);
    mode_set_mode_info (mode, "fold_info", "#{{{\r#}}}\r\r");
    mode_set_mode_info (mode, "init_mode_menu", &perl_init_menu);
    run_mode_hooks("perl_mode_hook");
}
% ----------------------------------------------------------------- end-of-file
