/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: analyzer.cpp,v 1.52 2006/01/17 19:46:34 luks Exp $

----------------------------------------------------------------------------*/
#ifdef WIN32
#if _MSC_VER == 1200
#pragma warning(disable:4786)
#endif
#endif

#include <stdio.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include "../config.h"
#include "analyzer.h"
#include "tunepimp.h"
#include "submit.h"
#include "watchdog.h"

const int iDataFieldLen = 255;
const int iChunkSize = 8192;

#define DB printf("%s:%d\n", __FILE__, __LINE__);

//---------------------------------------------------------------------------

Analyzer::Analyzer(TunePimp       *tunePimpArg,
                   Plugins        *plugins, 
                   FileCache      *cacheArg,
                   SubmitInfo     *submitInfoArg,
                   WatchdogThread *watchdog) : Thread()
{
   tunePimp = tunePimpArg;
   this->plugins = plugins;
   cache = cacheArg;
   submitInfo = submitInfoArg;
   dog = watchdog;
   exitThread = false;
   sem = new Semaphore();
}

//---------------------------------------------------------------------------

Analyzer::~Analyzer(void)
{
   exitThread = true;
   sem->signal();
   join();
   delete sem;
}

//---------------------------------------------------------------------------

void Analyzer::wake(void)
{
    sem->signal();
}

//---------------------------------------------------------------------------

void Analyzer::threadMain(void)
{
    string  fileName, status, trm;
    Track  *track;

    dog->setAnalyzerThread(getId());
    setPriority(tunePimp->context.getAnalyzerPriority());
    for(; !exitThread;)
    {
        unsigned long  duration = 0;
        char          *ptr;
        Plugin        *plugin;
        TRMResult      ret = eOtherError;
        trm = "";

        track = cache->getNextItem(ePending);
        if (track == NULL)
        {
           sem->wait();
           continue;
        }

        track->lock();

        dog->setAnalyzerTask(cache->getFileIdFromTrack(track));
        
        track->getFileName(fileName);

        // Check to see if we pulled up a musicbrainz id
        Metadata data, id3;
        track->getServerMetadata(data);
        track->getLocalMetadata(id3);
        track->getTRM(trm);
        if (trm == string(ANALYZER_REDO_STRING))
        {
            data.clear();
            data.trackId = "";
            track->setServerMetadata(data);

            track->getLocalMetadata(id3);
            id3.trackId = "";
            track->setLocalMetadata(id3);
        }

        ptr = strrchr((char *)fileName.c_str(), '.');
        if (ptr)
            plugin = plugins->get(string(ptr), TP_PLUGIN_FUNCTION_DECODE);
        else
            plugin = NULL;

        if (plugin)
        {
            string err;

            track->unlock();
            status = "Analyzing " + fileName;
            tunePimp->setStatus(status);
            ret = calculateTRM(plugin, fileName, err, trm, duration);
            track->lock();

            // Check to make sure no one has messed with this track since we started
            if (track->getStatus() == ePending)
            {
                if (ret != eOk)
                {
                    tunePimp->setStatus("Failed to generate trm from " + fileName);
                    if (err.empty())
                        setError(track, ret);
                    else
                        track->setError(err);
                    track->setStatus(eError);
                }
                else
                    tunePimp->setStatus(string(" "));

                if (duration > 0)
                {
                    Metadata fileData;

                    track->getLocalMetadata(fileData);
                    fileData.duration = duration;
                    track->setLocalMetadata(fileData);
                }

                if (!trm.empty())
                {
                    if (trm.substr(14, 1) != string("4"))
                    {
                        string err;

                        err = string("Unable to retrieve TRM from fingerprint server.");
                        tunePimp->setStatus(err);
                        track->setStatus(eError);
                        track->setError(err);
                    }
                    else
                    {
                        if (strcmp("c457a4a8-b342-4ec9-8f13-b6bd26c0e400", trm.c_str()) == 0)
                        {
                            track->setStatus(eError);
                            setError(track, eSigServerBusy);
                            tunePimp->setStatus("TRM calculation failed: TRM server is too busy.");
                        }
                        else
                        {
                            track->setTRM(trm);
                            track->setStatus(eTRMLookup);
                        }
                    }
                }
            }
            else
            {
                string err;

                if (ptr)
                    err = string(ptr) + string(" is not a supported filetype.");
                else
                    err = string(fileName) + string(": cannot determine filetype.");
                tunePimp->setStatus(err);
                track->setStatus(eError);
                track->setError(err);
            }
        }
        track->unlock();
        tunePimp->wake(track);
        cache->release(track);
        dog->setAnalyzerTask(-1);
    }
}

//---------------------------------------------------------------------------

void Analyzer::setError(Track *track, TRMResult retVal)
{
    switch(retVal)
    {
        case eFileNotFound:
           track->setError("Audio file not found.");
           break;
        case eDecodeError:
           track->setError("Cannot decode audio file.");
           break;
        case eCannotConnect:
           track->setError("Cannot connect to the TRM signature server.");
           break;
        case eSigServerBusy:
           track->setError("The TRM signature server is too busy to process your request.");
           break;
        default:
           track->setError("Unknown error. Sorry, this program sucks.");
           break;
    }
}

//---------------------------------------------------------------------------

TRMResult Analyzer::calculateTRM(Plugin *plugin, const string &fileName, string &err, 
                                 string &trmId, unsigned long &duration)
{
    TRMResult  ret = eOk;
    void      *decode;
    trm_t      trm;

#ifdef WIN32
    musicbrainz_t mb = mb_New();
    mb_WSAInit(mb);
#endif

    // The only thing the try block should catch is something
    // we know nothing about or a segfault and even then only
    // under windows. Thus, exit the thread and let the
    // watchdog skip this file and start a new analyzer.
    try
    {
        unsigned int samplesPerSecond, bitsPerSample, channels;
        int      numRead;
        string   proxyServer, encoding;
        short    proxyPort;
        int      flags = 0;

        encoding = tunePimp->context.getFileNameEncoding();

        decode = plugin->decodeStart(fileName.c_str(), flags, encoding.c_str());
        if (!decode)
        {
            err = string(plugin->getError());
            ret = eDecodeError;
        }
        else
        {
            trm = trm_New ();

            tunePimp->getProxy(proxyServer, proxyPort);
            if (proxyServer.size() > 0 && proxyPort != 0)
                trm_SetProxy(trm, (char *)proxyServer.c_str(), proxyPort);

            if (!plugin->decodeInfo(decode, &duration, &samplesPerSecond, &bitsPerSample, &channels))
            {
                err = string(plugin->getError());
                ret = eDecodeError;
            }
            else
            {
                char *buffer;
                char  trmidRaw[32];
                char  trmid[37];

                trm_SetPCMDataInfo(trm, samplesPerSecond, channels, bitsPerSample);
                trm_SetSongLength(trm, duration / 1000);

                buffer = new char[iChunkSize];
                while(1)
                {
                    numRead = plugin->decodeRead(decode, buffer, iChunkSize);
                    if (numRead > 0)
                    {
                        if (trm_GenerateSignature(trm, buffer, numRead))
                            break;
                    }
                    else
                    {
                        if (numRead < 0)
                        {
                            err = string(plugin->getError());
                            ret = eDecodeError;
                        }
                        break;
                    }
                }
                delete [] buffer;

                if (ret == eOk)
                {
                    trmId = string("");
                    ret = trm_FinalizeSignature(trm, (char *)trmidRaw, NULL) ? eCannotConnect : eOk;
                    if (ret == eOk)
                    {
                        trm_ConvertSigToASCII(trm, (char *)trmidRaw, (char *)trmid);
                        trmId = string((char *)trmid);
                    }
                    else
                        ret = eCannotConnect;
                }
            }
            trm_Delete(trm);
            trm = NULL;
        }

        plugin->decodeEnd(decode);
    }
    catch(...)
    {
        terminate();
    }

#ifdef WIN32
    mb_WSAStop(mb);
    mb_Delete(mb);
#endif

    return ret;
}
