/*
 * cwdaemon - morse sounding daemon for the parallel or serial port
 * Copyright (C) 2002 -2003 Joop Stakenborg <pa4tu@amsat.org>
 *                       and many authors, see the AUTHORS file.
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Code parts were taken from cwbeacon by EA4RJ and morse by GW4PTS, see
 * notes below.....
 *
 * cwbeacon is part of the beaconProject, ea4rj@amsat.org
 * and it is based on the original:
 *
 * GW4PTS Morse tutor for Linux, by Alan Cox (morse.c 1.00, 21/09/93)
 *
 * This software is placed under the GNU software license
 *
 */

#include <cwdaemon.h>

/* we fork by default */
int forking = 1;

/* network vars */
int sin_len;
int socket_descriptor;
struct sockaddr_in k_sin;
int port = 6789;

/* morse defaults */
int morse_gap = 0;       /* TODO: Set with -d delay */
int morse_speed = 24;
int morse_tone = 1500;   /* 800 Hz */
int morse_sound = 1;     /* speaker on */
int wordmode = 0;
long int dot = 1000000;  /* dot length = unit */
long int dash = 3000000; /* dash length = 3-dot */
long int eldelay = 1000000;  /* pause between elements equals to dot time */
unsigned int ptt_delay = 0; /* default = off*/
int ptt_timer_running = 0;

struct timeval now,end,left;

#define MAXMORSE 4000
char morsetext[MAXMORSE];

morse morsetable[] = {
  {' ', {NIL, NIL, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'A', {DIH, DAH, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'B', {DAH, DIH, DIH, DIH, NIL, NIL, NIL, NIL}},
  {'C', {DAH, DIH, DAH, DIH, NIL, NIL, NIL, NIL}},
  {'D', {DAH, DIH, DIH, NIL, NIL, NIL, NIL, NIL}},
  {'E', {DIH, NIL, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'F', {DIH, DIH, DAH, DIH, NIL, NIL, NIL, NIL}},
  {'G', {DAH, DAH, DIH, NIL, NIL, NIL, NIL, NIL}},
  {'H', {DIH, DIH, DIH, DIH, NIL, NIL, NIL, NIL}},
  {'I', {DIH, DIH, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'J', {DIH, DAH, DAH, DAH, NIL, NIL, NIL, NIL}},
  {'K', {DAH, DIH, DAH, NIL, NIL, NIL, NIL, NIL}},
  {'L', {DIH, DAH, DIH, DIH, NIL, NIL, NIL, NIL}},
  {'M', {DAH, DAH, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'N', {DAH, DIH, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'O', {DAH, DAH, DAH, NIL, NIL, NIL, NIL, NIL}},
  {'P', {DIH, DAH, DAH, DIH, NIL, NIL, NIL, NIL}},
  {'Q', {DAH, DAH, DIH, DAH, NIL, NIL, NIL, NIL}},
  {'R', {DIH, DAH, DIH, NIL, NIL, NIL, NIL, NIL}},
  {'S', {DIH, DIH, DIH, NIL, NIL, NIL, NIL, NIL}},
  {'T', {DAH, NIL, NIL, NIL, NIL, NIL, NIL, NIL}},
  {'U', {DIH, DIH, DAH, NIL, NIL, NIL, NIL, NIL}},
  {'V', {DIH, DIH, DIH, DAH, NIL, NIL, NIL, NIL}},
  {'W', {DIH, DAH, DAH, NIL, NIL, NIL, NIL, NIL}},
  {'X', {DAH, DIH, DIH, DAH, NIL, NIL, NIL, NIL}},
  {'Y', {DAH, DIH, DAH, DAH, NIL, NIL, NIL, NIL}},
  {'Z', {DAH, DAH, DIH, DIH, NIL, NIL, NIL, NIL}},
  {'1', {DIH, DAH, DAH, DAH, DAH, NIL, NIL, NIL}},
  {'2', {DIH, DIH, DAH, DAH, DAH, NIL, NIL, NIL}},
  {'3', {DIH, DIH, DIH, DAH, DAH, NIL, NIL, NIL}},
  {'4', {DIH, DIH, DIH, DIH, DAH, NIL, NIL, NIL}},
  {'5', {DIH, DIH, DIH, DIH, DIH, NIL, NIL, NIL}},
  {'6', {DAH, DIH, DIH, DIH, DIH, NIL, NIL, NIL}},
  {'7', {DAH, DAH, DIH, DIH, DIH, NIL, NIL, NIL}},
  {'8', {DAH, DAH, DAH, DIH, DIH, NIL, NIL, NIL}},
  {'9', {DAH, DAH, DAH, DAH, DIH, NIL, NIL, NIL}},
  {'0', {DAH, DAH, DAH, DAH, DAH, NIL, NIL, NIL}},
  {'?', {DIH, DIH, DAH, DAH, DIH, DIH, NIL, NIL}},
  {'/', {DAH, DIH, DIH, DAH, DIH, NIL, NIL, NIL}},
  {'*', {DIH, DAH, DIH, DAH, DIH, NIL, NIL, NIL}}, /* 'AR', end of message */
  {'.', {DIH, DAH, DIH, DAH, DIH, DAH, NIL, NIL}},
  {'=', {DAH, DIH, DIH, DIH, DAH, NIL, NIL, NIL}}, /* BT */
  {'<', {DIH, DIH, DIH, DAH, DIH, DAH, NIL, NIL}}, /* SK */
  {'(', {DAH, DIH, DAH, DAH, DIH, NIL, NIL, NIL}}, /* KN */
  {'!', {DIH, DIH, DIH, DAH, DIH, NIL, NIL, NIL}}, /* SN */
  {'&', {DIH, DAH, DIH, DIH, DIH, NIL, NIL, NIL}}, /* AS */
  {'>', {DAH, DIH, DIH, DIH, DAH, DIH, DAH, NIL}}, /* BK */
  {',', {DAH, DAH, DIH, DIH, DAH, DAH, NIL, NIL}},
  {0xC4,{DIH, DAH, DIH, DAH, NIL, NIL, NIL, NIL}}, /* A umlaut */
  {0xE4,{DIH, DAH, DIH, DAH, NIL, NIL, NIL, NIL}},
  {0XD6,{DAH, DAH, DAH, DIH, NIL, NIL, NIL, NIL}}, /* O umlaut */
  {0xF6,{DAH, DAH, DAH, DIH, NIL, NIL, NIL, NIL}},
  {0xDC,{DIH, DIH, DAH, DAH, NIL, NIL, NIL, NIL}}, /* U umlaut */
  {0xFC,{DIH, DIH, DAH, DAH, NIL, NIL, NIL, NIL}},
  {0XC5,{DIH, DAH, DAH, DIH, DAH, NIL, NIL, NIL}}, /* Ringed A */
  {0xE5,{DIH, DAH, DAH, DIH, DAH, NIL, NIL, NIL}},
  {0xD1,{DAH, DAH, DIH, DAH, DAH, NIL, NIL, NIL}}, /* N tilde */
  {0xF1,{DAH, DAH, DIH, DAH, DAH, NIL, NIL, NIL}},
  {0, {NIL, NIL, NIL, NIL, NIL, NIL, NIL, NIL}}	/* END MARKER */
};

cwdevice cwdevice_ttys = {
  init : ttys_init,
  free : ttys_free,
  reset : ttys_reset,
  cw_on   : ttys_cw_on,
  cw_off  : ttys_cw_off,
  ptt_on  : ttys_ptt_on,
  ptt_off : ttys_ptt_off,
  ssbway_soundcard : NULL,
  ssbway_mic : NULL,
  base : 0x3f8
};

cwdevice cwdevice_lp = {
  init : lp_init,
  free : lp_free,
  reset : lp_reset,
  cw_on   : lp_cw_on,
  cw_off  : lp_cw_off,
  ptt_on  : lp_ptt_on,
  ptt_off : lp_ptt_off,
  ssbway_soundcard : lp_ssbway_soundcard,
  ssbway_mic : lp_ssbway_mic,
  base : 0x378
};

cwdevice *cwdev = &cwdevice_lp;

/*
  int weight [-5...0...+5]
  0 = dot/space ratio 1:3, element space = dot time
  -5...-1 dot and dash length decreased and element spacing increased
  this is "light" weighting, sounds often choppy
  +1...+5 dot and dash length increased and element spacing decreased
  this is "heavy" weighting, sounds more like music ;-)
*/
void
set_weighting(int weight)
{
  dot = eldelay = 1000000; /* default values */

  if ((weight > -6) && (weight < 6))
  {
     dot = 1000000 + (weight * 100000);   /* elements */
     eldelay = 1000000 - (weight * 100000); /* delay  */
  }
  dash = 3 * dot; /* always 3 times the dot length */
}

/* catch ^C when running in foreground */
RETSIGTYPE
catchint (int signal)
{
  cwdev->free(cwdev);
  printf ("%s: Exiting\n", PACKAGE);
  exit (0);
}

/* print error message to the console or syslog if we are forked */
void
errmsg (char *info, ...)
{
  va_list ap;
  char s[1025];

  va_start(ap, info);
  vsnprintf(s, 1024, info, ap);
  va_end(ap);

  if (forking)
    syslog (LOG_ERR, "%s\n", s);
  else
    printf ("%s: %s failed: %s\n", PACKAGE, s, strerror (errno));
}

/* print only debug message to the console */
void
debug (char *info, ...)
{
  va_list ap;
  char s[1025];

  va_start(ap, info);
  vsnprintf(s, 1024, info, ap);
  va_end(ap);

  if (!forking)
    printf ("%s: %s \n", PACKAGE, s);
}

/* use nanosleep instead of usleep */
void
udelay (unsigned long us)
{
  struct timespec sleeptime, dummy;

  sleeptime.tv_sec = 0;
  sleeptime.tv_nsec = us * 1000;

  if (nanosleep (&sleeptime, &dummy) == -1)
    errmsg ("Nanosleep");
}

/*
   this is sample code shameless :-) stolen from
   http://www.erlenstar.demon.co.uk/unix/faq_8.html
*/

/* some simple timing utilities */

void timer_add(struct timeval *tv, long secs, long usecs)
{
    tv->tv_sec += secs;
    if ((tv->tv_usec += usecs) >= 1000000)
    {
        tv->tv_sec += tv->tv_usec / 1000000;
        tv->tv_usec %= 1000000;
    }
}

/* Set *RES = *A - *B, returning the sign of the result */

int timer_sub(struct timeval *res, const struct timeval *a, const struct timeval *b)
{
    long sec = a->tv_sec - b->tv_sec;
    long usec = a->tv_usec - b->tv_usec;

    if (usec < 0)
        usec += 1000000, --sec;

    res->tv_sec = sec;
    res->tv_usec = usec;

    return (sec < 0) ? (-1) : ((sec == 0 && usec == 0) ? 0 : 1);
}


void tune (int onoff)
{
   int fd;

   if (onoff)
   {
      if(ptt_delay)
      {
         cwdev->ptt_on(cwdev);
         debug("PTT (TUNE) on");
         /* TOD */
         udelay(ptt_delay);
      }
      if (cwdev->cw_on)
      {
         cwdev->cw_on(cwdev);
         debug("CW (TUNE) on");
      }
      else debug("CW on unimplemented");
      /* sidetone on if morseton !=0 */
      fd = open ("/dev/tty1", O_WRONLY);
      if (fd != -1)
      {
         if (ioctl (fd, KIOCSOUND, morse_tone) != 0)
         {
            errmsg ("Ioctl console speaker");
            exit(1);
         }
         close (fd);
      }
      else
      {
         errmsg ("Open console speaker");
         exit(1);
      }

   }
   else
   {
      if(ptt_delay)
      {
         gettimeofday(&end, NULL);
         timer_add(&end, 0, dash/morse_speed);
         ptt_timer_running = 1;
      }

      if (cwdev->cw_off)
      {
         cwdev->cw_off(cwdev);
         debug("CW (TUNE) off");
      }
      else debug("CW off unimplemented");

      fd = open ("/dev/tty1", O_WRONLY);
      if (fd != -1)
      {
         if (ioctl (fd, KIOCSOUND, 0) != 0)
         {
            errmsg ("Ioctl console speaker");
            exit(1);
         }
         close (fd);
      }
      else
      {
         errmsg ("Open console speaker");
         exit(1);
      }
      /* delayed ptt off */
      if (1 == ptt_timer_running)
      {
         while (1)
         {
            udelay (1000); /*prevent 100% CPU */
            gettimeofday (&now, NULL);
            if (timer_sub (&left, &end, &now) <= 0)
            {
               cwdev->ptt_off (cwdev);
               debug ("PTT /TUNE) off");
               ptt_timer_running = 0;
	       break;
            }
         }
      }
   }
}
/* convert character to morse by searching the morsetable */
morse *
chartomorse (char c)
{
  int ct = 0;

  while (morsetable[ct].code)
    {
      if (morsetable[ct].code == c)
        return (&morsetable[ct]);
      ct++;
    }
  return (NULL);
}

/* output to console and serial/parallel port */
void
playbeep (long duration)
{
  int fd;

  if (morse_sound)
    {
      signal (SIGINT, SIG_IGN);
      signal (SIGQUIT, SIG_IGN);
      fd = open ("/dev/tty1", O_WRONLY);
      if (fd != -1)
        {
          if (ioctl (fd, KIOCSOUND, morse_tone) != 0)
            {
              errmsg ("Ioctl console speaker");
              exit (1);
            }
          close (fd);
        }
      else
        {
          errmsg ("Open console speaker");
          exit (1);
        }
    }

  cwdev->cw_on(cwdev);
  udelay (duration / morse_speed);

  if (morse_sound)
    {
      fd = open ("/dev/tty1", O_WRONLY);
      if (fd != -1)
        {
          if (ioctl (fd, KIOCSOUND, 0) != 0)
            {
              errmsg ("Ioctl console speaker");
              exit (1);
            }
          close (fd);
        }
      else
        {
          errmsg ("Open console speaker");
          exit (1);
        }
      if (forking)
        signal (SIGINT, SIG_DFL);
      else
        signal (SIGINT, catchint);
      signal (SIGQUIT, SIG_DFL);
    }
    cwdev->cw_off(cwdev);
}

/* play dot, dash and use a delay */
void
playmorse (morse * m)
{
  int d = 0;

  while (d < 8 && m->data[d] != NIL)
    {
      if (m->data[d] == DIH)
        playbeep (dot);
      if (m->data[d] == DAH)
        playbeep (dash);
      d++;
      udelay (eldelay / morse_speed); /* element spacing */
    }  /* morse signs delay */
  udelay ((dash + ((dot / 10) * morse_gap)) / morse_speed);
  morse_gap = 0; /* gap for one sign only */
 }

/*
 * watch the socket and if there is an escape character check what it is,
 * otherwise play morse. Return 0 with escape charcters and empty messages.
 */
int
recv_code (void)
{
  char message[257];
  ssize_t recv_rc;
  int speed = 0;
  int weight = 0;

  recv_rc = recvfrom (socket_descriptor, message, sizeof (message) - 1, 0,
    (struct sockaddr *) &k_sin, &sin_len);

  if (recv_rc == -1 && errno != EAGAIN)
    {
      debug ("Recvfrom");
      exit (1);
    }

  if (recv_rc > 0)
    {
      if (message[0] != 27)
        { /* no ESCAPE */
          message[recv_rc] = '\0';
          debug ("Message: %s, length: %d", message, recv_rc);
          if ((strlen (message) + strlen (morsetext)) <= MAXMORSE - 1)
            strcat (morsetext, message);
          return 1;
        }
      else
        { /* check ESCAPE characters */
          switch ((int) message[1])
            {
              case '0': /* reset  all values */
                morsetext[0] = '\0';
                morse_gap = 0;
                morse_speed = 24;
                morse_tone = 1500;
                morse_sound = 1;
                wordmode = 0;
                cwdev->reset(cwdev);
                debug ("Reset all values");
              break;
              case '2': /*speed */
                speed = atoi (message + 2);
                if (speed > 4)	/* prevent divide by zero */
                  morse_speed = speed;
                debug ("Speed: %d wpm", morse_speed);
              break;
              case '3': /* tone */
                if (atoi (message + 2) != 0)
                  {
                     morse_tone = (12000 / (atoi (message + 2)) * 100);
                     morse_sound = 1;
                  }
                else /* sidetone off */
		{	
                  morse_sound = 0;
		  morse_tone  = 0;
                } 
                if (morse_sound == 1)
                  debug ("Tone: %s Hz", message + 2);
                else
                  debug ("Tone off");
              break;
              case '4':  /* message abort */
                message[0] = '\0';
                morsetext[0] = '\0';
                debug ("Message abort");
              break;
              case '5':  /* exit */
                cwdev->free(cwdev);
                errno = 0;
                errmsg ("Sender has told me to end the connection");
                exit (0);
              break;
              case '6': /* set uninterruptable */
                message[0] = '\0';
                morsetext[0] = '\0';
                wordmode = 1;
                debug ("Wordmode set");
              break;
              case '7': /* set weighting */
                weight = atoi (message + 2);
                if ((weight > -6) && (weight < 6)) /* only allowed range */
                  set_weighting (weight);
                debug("Weight: %d", weight);
                break;
             case '8': /* device type */
                if (!strcmp (message + 2, "ttyS0"))
                    {
                        cwdev = &cwdevice_ttys;
                        cwdev->base = 0x3f8;
                        cwdev->desc = "ttyS0";
                      }
                else if (!strcmp (message + 2, "ttyS1"))
                    {
                        cwdev = &cwdevice_ttys;
                        cwdev->base = 0x2f8;
                        cwdev->desc = "ttyS1";
                    }
                else if (!strcmp (message + 2, "lp0"))
                    {
                        cwdev = &cwdevice_lp;
                        cwdev->base = 0x378;
                        cwdev->desc = "lp0";
                    }
                else if (!strcmp (message + 2, "lp1"))
                    {
                        cwdev = &cwdevice_lp;
                        cwdev->base = 0x278;
                        cwdev->desc = "lp1";
                    }
                else if (!strcmp (message + 2, "lp2"))
                    {
                        cwdev = &cwdevice_lp;
                        cwdev->base = 0x3bc;
                        cwdev->desc = "lp2";
                    }
                break;
              case '9': /* base port number (decimal or 0xhexadecimal) */
                debug("Closed port 0x%x, %d dec", cwdev->base, cwdev->base);
                cwdev->free(cwdev);
                cwdev->base = strtol(message + 2, NULL, 0);
                cwdev->init(cwdev);
                debug("Setting base port number to 0x%x, %d", cwdev->base, cwdev->base);
              break;
              case 'a': /* PTT keying on or off */
                if (atoi (message + 2))
                {
                  if (cwdev->ptt_on){
                    cwdev->ptt_on(cwdev);
                    debug("PTT on");
                  }
                  else debug("PTT on unimplemented");
                }
                else
                {
                  if (cwdev->ptt_off){
                    cwdev->ptt_off(cwdev);
                    debug("PTT off");
                  }
                  else debug("PTT off unimplemented");
                }
              break;
              case 'b': /* SSB way */
                if (atoi (message + 2))
                {
                  if (cwdev->ssbway_soundcard){
                    cwdev->ssbway_soundcard(cwdev);
                    debug("SSB way set to SOUNDCARD", PACKAGE);
                  }
                  else debug("SOUNDCARD unimplemented");
                }
                else
                {
                  if (cwdev->ssbway_mic){
                    cwdev->ssbway_mic(cwdev);
                    debug("SSB way set to MIC");
                  }
                  else debug("MICROPHONE unimplemented");
                }
              break;
              case 'c': /* Tune keying on or off */
                  tune (atoi (message + 2));
              break;
              case 'd': /* set ptt delay (TOD, Turn On Delay) */
                ptt_delay = atoi (message + 2);
                if ((ptt_delay >= 0) && (ptt_delay < 51)) /* only allowed range 0..50 ms*/
                   ptt_delay*=1000; /* 0 = ptt delay off */
                else
                  ptt_delay=50000; /* longest delay (secure) if parameter wrong */
                debug("PTT delay(TOD): %d ms", ptt_delay/1000);
                if(ptt_delay)
                  ;
                else
                {
                  if (cwdev->ptt_off)
                     {
                        cwdev->ptt_off(cwdev);
                        debug("PTT off");
                     }
                     else
                        debug("PTT off unimplemented");
                }
                break;
            }
          return 0;
        }
    }
  return 0;
}

/* check every character for speed increase or decrease, convert other
   characters to morse and play them */
void
playmorsestring (char *x)
{

  int i = 0;
  /* stop ptt timer */
  if(ptt_delay)
  {
      cwdev->ptt_on(cwdev);
      debug("PTT on");
      /* TOD */
      udelay(ptt_delay);
  }
  while (*x)
    {
      char c = islower (*x) ? toupper (*x) : *x;
      if ((c == '+') || (c == '-'))
        { /* speed in- & decrease */
          if ((c == '+') && (morse_speed <= 58))
            morse_speed += 2;
          if ((c == '-') && (morse_speed >= 10))
            morse_speed -= 2;
        }
      else if (c == '~')
        morse_gap = 15; /* half dash time additional for the next char */
      else
        {
          morse *m = chartomorse (c);
          if (m != NULL)
            playmorse (m);
        }
      x++;
      i++;
      if (i >= strlen (morsetext))
        {
          i = 0;
          break;
        }
      if (wordmode == 0)
        recv_code ();
    }
  morsetext[0] = '\0';
  /* start ptt off timer */
  if(ptt_delay)
  {
      gettimeofday(&end, NULL);
      timer_add(&end, 0, dash/morse_speed);
      ptt_timer_running = 1;
  }
}

/* parse the command line and check for options, do some error checking */
void
parsecommandline (int argc, char *argv[])
{
  int p;
  int w = 0;
  int base = -1;

  cwdev = &cwdevice_lp;
  cwdev->base = 0x378;
  cwdev->desc = "lp0";

  while ((p = getopt (argc, argv, "d:hl:np:s:t:vw:")) != -1)
    {
      switch (p)
        {
          case ':':
          case '?':
          case 'h':
            printf ("Usage: %s [option]...\n", PACKAGE);
            printf ("       -d <device>  ");
            printf ("Use a different device ");
            printf ("(ttyS0|1 or lp0|1|2, default = lp0)\n");
            printf ("       -h           ");
            printf ("Display this help and exit\n");
            printf ("       -l <port>    ");
            printf ("Use a different serial or parallel base port number\n");
            printf ("                    (defaults: 0x3f8 and 0x378)\n");
            printf ("       -n           ");
            printf ("Do not fork and print debug information to stdout\n");
            printf ("       -p <port>    ");
            printf ("Use a different UDP port number (> 1023, default = 6789)\n");
            printf ("       -s <speed>   ");
            printf ("Set morse speed (> 4, default = 24)\n");
            printf ("       -t <time>    ");
            printf ("Set PTT delay (0 ... 50 ms, default = 0)\n");
            printf ("       -v           ");
            printf ("Output version information and exit\n");
            printf ("       -w <weight>  ");
            printf ("Set weighting (-5 ... +5, default = 0)\n");
            exit (0);
          case 'd':
            if (!strcmp (optarg, "ttyS0"))
	          {
   	            cwdev = &cwdevice_ttys;
                cwdev->base = 0x3f8;
                cwdev->desc = "ttyS0";
		      }
            else if (!strcmp (optarg, "ttyS1"))
	          {
  	            cwdev = &cwdevice_ttys;
                cwdev->base = 0x2f8;
                cwdev->desc = "ttyS1";
          	  }
            else if (!strcmp (optarg, "lp0"))
              {
   	            cwdev = &cwdevice_lp;
                cwdev->base = 0x378;
                cwdev->desc = "lp0";
              }
            else if (!strcmp (optarg, "lp1"))
              {
   	            cwdev = &cwdevice_lp;
                cwdev->base = 0x378;
                cwdev->desc = "lp1";
              }
            else if (!strcmp (optarg, "lp2"))
              {
   	            cwdev = &cwdevice_lp;
                cwdev->base = 0x3bc;
                cwdev->desc = "lp2";
              }
            else
              {
                printf ("%s: Wrong device, ", PACKAGE);
                printf ("use one of ttyS0, ttyS1, lp0, lp1 or lp2\n");
                exit (1);
              }
          break;
          case 'l':
            base = strtol(optarg, NULL, 0);
            debug("Setting base port number to 0x%x, %d dec\n", base, base);
          break;
          case 'n':
            forking = 0;
            printf ("%s: Not forking...\n", PACKAGE);
          break;
          case 'p':
            if ((port = atoi (optarg)) <= 0)
              {
                printf ("%s: Invalid port number - %s\n", PACKAGE, optarg);
                exit (1);
              }
            if (port < 1024)
              {
                printf ("%s: Port should be larger than 1023\n", PACKAGE);
                exit (1);
              }
          break;
          case 's':
            morse_speed = atoi (optarg);
            if (morse_speed < 5)
            {
               printf ("Wrong speed, use a value larger than 4\n");
               exit(1);
            }
          break;
          case 't':
            ptt_delay = atoi (optarg);
            if (ptt_delay < 0 || ptt_delay > 50)
            {
               printf ("Wrong PTT delay value, must be between 0 and 50 ms\n");
               exit(1);
            }
            else
              ptt_delay*=1000;
          break;
          case 'v':
            printf ("%s version %s\n", PACKAGE, VERSION);
            exit (0);
          case 'w':
            w = atoi (optarg);
            if ((w > 5) || (w < -5))
            {
               printf ("Wrong weighting value, use -5 ... +5\n");
               exit(1);
            }
            set_weighting (w);
          break;
        }
    }
    if (base!=-1)
        cwdev->base = base;
}

/* main program: fork, open network connection and go into an endless loop
   waiting for something to happen on the UDP port */
int
main (int argc, char *argv[])
{
  pid_t pid, sid;
  int bind_rc, close_rc;
  long save_file_flags;

  parsecommandline (argc, argv);

  debug ("Device used: %s at 0x%x, %d", cwdev->desc, cwdev->base, cwdev->base);

  if (forking)
    {
      pid = fork ();
      if (pid < 0)
        {
          printf ("%s: Fork failed: %s\n", PACKAGE, strerror (errno));
          exit (1);
        }

      if (pid > 0)
        exit (0);

      openlog ("netkeyer", LOG_PID, LOG_DAEMON);
      if ((sid = setsid ()) < 0)
        {
          syslog (LOG_ERR, "%s\n", "setsid");
          exit (1);
        }
      if ((chdir ("/")) < 0)
        {
          syslog (LOG_ERR, "%s\n", "chdir");
          exit (1);
        }
      umask (0);
      close (STDIN_FILENO);
      close (STDOUT_FILENO);
      close (STDERR_FILENO);
    }
  else
    {
      debug ("Press ^C to quit");
      signal (SIGINT, catchint);
    }

  cwdev->init(cwdev);

  bzero (&k_sin, sizeof (k_sin));
  k_sin.sin_family = AF_INET;
  k_sin.sin_addr.s_addr = htonl (INADDR_ANY);
  k_sin.sin_port = htons (port);
  sin_len = sizeof (k_sin);

  socket_descriptor = socket (AF_INET, SOCK_DGRAM, 0);
  if (socket_descriptor == -1)
    {
      errmsg ("Socket open");
      exit (1);
    }

  bind_rc =
    bind (socket_descriptor, (struct sockaddr *) &k_sin, sizeof (k_sin));
  if (bind_rc == -1)
    {
      errmsg ("Bind");
      exit (1);
    }

  save_file_flags = fcntl (socket_descriptor, F_GETFL);
  save_file_flags |= O_NONBLOCK;

  if (fcntl (socket_descriptor, F_SETFL, save_file_flags) == -1)
    {
      errmsg ("Trying non-blocking");
      exit (1);
    }

  morsetext[0] = '\0';
  while (1)
    {
      udelay (1000); /*prevent 100% CPU */
      if (recv_code ())
        playmorsestring (morsetext);
      /* check for ptt off timer */
      if (1 == ptt_timer_running)
        {
          gettimeofday (&now, NULL);
          if (timer_sub (&left, &end, &now) <= 0)
            {
              cwdev->ptt_off (cwdev);
              debug ("PTT off");
              ptt_timer_running = 0;
            }
        }
    }

  cwdev->free(cwdev);
  close_rc = close (socket_descriptor);
  if (close_rc == -1)
    {
      errmsg ("Close socket");
      exit (1);
    }
  exit (0);
}
