/*
 * Copyright (C) 2001-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: main.c,v 1.116.2.4 2005/07/31 16:49:36 dsalt Exp $
 *
 * gtk2 ui for xine
 *
 * main
 */

#include "globals.h"

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <X11/Xlib.h>

#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>

#include "engine.h"
#include "player.h"
#include "gtkvideo.h"
#include "infobar.h"
#include "wizards.h"
#include "skin_window.h"
#include "noskin_window.h"
#include "server.h"
#include "open_mrl.h"
#include "log_window.h"
#include "preferences.h"
#include "key_events.h"
#include "mediamarks.h"
#include "playlist.h"
#include "settings.h"
#include "snapshot.h"
#include "play_item.h"
#include "lirc.h"
#include "stream_info.h"
#include "utils.h"
#include "vis.h"
#include "post.h"

/*
 * globals
 */

int             verbosity;
GtkWidget      *app=NULL;
infobar_t      *bar=NULL;
GdkPixbuf      *gxine_logo = NULL;
gchar          *plugindir, *bindir, *logodir, *pixmapdir, *icondir, *miscdir;
gchar          *video_driver_id=NULL;
gchar          *audio_driver_id=NULL;

void gxine_try_remote (int argc, char *argv[], int enqueue) {

  if (!server_client_connect()) {
    return;
  }

  /*
   * pass on files to play
   */

  if (!enqueue)
    server_client_send ("playlist_clear();\n");

  if (optind<argc) {
    int i;

    for (i=optind; i<argc; i++) {

      char *mrl;

      mrl = argv[i];

      if ((mrl[0] != '/') && !strstr (mrl, ":/")) {

	/* make filename an absolute pathname */

	char  buf[1024];
	char *pathname;
	int   l1, l2;

	getcwd (buf, 1024);

	l1 = strlen(buf);
	l2 = strlen(mrl);
	
	pathname = g_malloc (l1+l2+2);
	memcpy (pathname, buf, l1);
	pathname[l1]='/';
	memcpy (&pathname[l1+1], mrl, l2);
	pathname[l1+l2+1]=0;

	mrl = pathname;
      }	

      server_client_send ("playlist_add (\"");
      server_client_send (mrl);
      server_client_send ("\");\n");
    }

    if (!enqueue)
      server_client_send ("playlist_play (0)\n");
  }

  exit (0);
}

static GtkWidget *splash = NULL;

static void splash_show (void) {

  gchar     *pathname;
  GtkWidget *img;

  gdk_threads_enter();  

  splash = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_window_set_decorated (GTK_WINDOW (splash), FALSE);

  gtk_window_set_type_hint (GTK_WINDOW (splash),
			    GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
  gtk_window_set_position (GTK_WINDOW (splash), GTK_WIN_POS_CENTER_ALWAYS);
  gtk_window_set_keep_above (GTK_WINDOW (splash), TRUE);

  pathname = g_strconcat (pixmapdir, "/splash.png", NULL);
  img = gtk_image_new_from_file (pathname);
  g_free (pathname);

  gtk_container_add (GTK_CONTAINER (splash),img);

  gtk_widget_show_all (splash);

  while (gtk_events_pending ()) 
    gtk_main_iteration ();

  gdk_threads_leave();  
}

#ifndef GLIB_USES_SYSTEM_MALLOC
#include <assert.h>
gpointer xmalloc (gsize size)
{
  gpointer p = malloc (size);
  if (size)
    assert (p != NULL);
  return p;
}

gpointer xrealloc (gpointer p, gsize size)
{
  p = realloc (p, size);
  if (size)
    assert (p != NULL);
  return p;
}
#endif

/* Initialise some engine settings from saved preferences */
static gboolean post_init_configure (void)
{
  xine_cfg_entry_t entry;
  GtkVideo *v = GTK_VIDEO(gtv);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace", &entry))
    gtk_video_set_tvtime (v, entry.str_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace_enable",
				&entry))
    gtk_video_set_deinterlace (v, entry.num_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.video", &entry))
    gtk_video_set_post_plugins_video (v, entry.str_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.audio", &entry))
    gtk_video_set_post_plugins_audio (v, entry.str_value, audio_port);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.video_enable", &entry))
    gtk_video_set_use_post_plugins_video (v, entry.num_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.audio_enable", &entry))
    gtk_video_set_use_post_plugins_audio (v, entry.num_value, audio_port);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.audio_visualisation",
				&entry))
    vis_set (v, entry.enum_values[entry.num_value], &audio_port);

  return FALSE;
}

int main(int argc, char* argv[])
{
  int opt, enqueue;
  gboolean show_splash = TRUE;

#ifndef GLIB_USES_SYSTEM_MALLOC
  GMemVTable vtab = { xmalloc, xrealloc, free, NULL, NULL, NULL };
  g_mem_set_vtable (&vtab);
#endif

  /*
   * init glib/gdk/gtk thread safe/aware
   */

#ifdef ENABLE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  /* The next two lines prevent GTK errors caused by gettext's conversion of
   * text from UTF-8. However, apparently we shouldn't use
   * bind_textdomain_codeset...
   */
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  bind_textdomain_codeset (LIB_PACKAGE, "UTF-8");
  textdomain (PACKAGE);
#endif

  g_thread_init (NULL);
  gdk_threads_init ();
  gtk_init(&argc, &argv);
  gtk_window_set_auto_startup_notification (FALSE);

  /*
   * init paths here. defaults are compiled in and may
   * be overwritten by environment variables
   */

  plugindir = getenv("GXINE_PLUGINDIR");
  if (!plugindir) plugindir = g_strconcat(GXINE_PLUGINDIR, NULL);
  bindir = getenv("GXINE_BINDIR");
  if (!bindir) bindir = g_strconcat(GXINE_BINDIR, NULL);
  logodir = getenv("GXINE_LOGODIR");
  if (!logodir) logodir = g_strconcat(GXINE_LOGODIR, NULL);
  pixmapdir = getenv("GXINE_PIXMAPDIR");
  if (!pixmapdir) pixmapdir = g_strconcat(GXINE_PIXMAPDIR, NULL);
  icondir = getenv("GXINE_ICONDIR");
  if (!icondir) icondir = g_strconcat(GXINE_ICONDIR, NULL);
  miscdir = getenv("GXINE_MISCDIR");
  if (!miscdir) miscdir = g_strconcat(GXINE_MISCDIR, NULL);

  /*
   * parse command line arguments
   */

  verbosity = 0;
  enqueue = 0;
  while ((opt = getopt(argc, argv, "veV:A:S")) > 0) {
    switch (opt) {
    case 'v':
      verbosity++;
      break;
    case 'e':
      enqueue = 1;
      break;
    case 'V':
      video_driver_id = optarg;
      break;
    case 'A':
      audio_driver_id = optarg;
      break;
    case 'S':
      show_splash = FALSE;
      break;
    default:
      printf (_("usage: %s [-v] [-e] [-V <video_driver>] [-A <audio_driver>] [-S] mrls...\n\n"), 
	      argv[0]);
      exit (1);
    }
  }

  /*
   * make sure ~/.gxine exists
   */
  {
    char *fname = g_strconcat (g_get_home_dir(), "/.gxine", NULL);
    mkdir (fname, 0700);
    g_free (fname);
  }

  /*
   * find out if gxine is already running, if so
   * just pass on the files to play
   */

  gxine_try_remote (argc, argv, enqueue);

  /* start using X... */
  if (!XInitThreads ()) {
    printf (_("gtkvideo: XInitThreads failed - looks like you don't have a thread-safe xlib.\n"));
    return 2;
  }

  server_setup ();

  /*
   * a splash screen for the impatient
   */

  if (show_splash)
    splash_show ();

  /*
   * init xine, set up skript engine, main window
   */

  gdk_threads_enter ();
  while (gtk_events_pending ())
    gtk_main_iteration ();

  {
    char *pathname = g_strconcat (icondir, "/gxine-logo.png", NULL);
    gxine_logo = gdk_pixbuf_new_from_file (pathname, NULL);
    g_free (pathname);
  }
  gtk_window_set_default_icon (gxine_logo);

  engine_init ();
  player_init ();

  while (gtk_events_pending ())
    gtk_main_iteration ();
  gtk_window_set_auto_startup_notification (TRUE);
#ifdef EXP_STUFF
  create_skin_window () ;
#else
  noskin_main_init ();
#endif

  while (gtk_events_pending ())
    gtk_main_iteration ();

  /*
   * create all dialogs (invisible)
   */

  file_dialog_init ();
  utils_init       ();
  open_mrl_init    ();
  log_window_init  ();
  preferences_init ();
  playlist_init    ();
  settings_init    ();
  mediamarks_init  ();
  play_item_init   ();
  key_events_init  ();
  gxine_lirc_init  ();
  snapshot_init    ();
  ui_init	   ();
  post_init	   ();
  stream_info_init ();
  wizards_init     ();

  if (splash)
    gtk_widget_destroy (splash);

  /*
   * wizards (first run only)
   */

  run_wizards (FALSE);
  gdk_threads_leave ();

  /*
   * argument parsing
   */

  if (optind<argc) {
    int i;
    int first = -1;

    if (!enqueue)
      playlist_clear();

    for (i=optind; i<argc; i++) {

      if (first<0) {
	first = playlist_add_mrl (argv[i], -1);
      } else {
	playlist_add_mrl (argv[i], -1);
      }
    }

    playlist_play (first);
  } else
    playlist_logo ();

  post_init_configure ();
  g_idle_add ((GSourceFunc) ui_post_init, NULL);
  server_start ();

  gdk_threads_enter();
  gtk_main();
  gdk_threads_leave();

  return 0;
}
