/*
  Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this library; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
*/

#include <ksimpleconfig.h>
#include <kglobal.h>

#include "wificonfig.h"

static QStringList speedList;
static QStringList wifiModeList;
static QStringList cryptoModeList;
static QStringList powerModeList;

Key::Key()
  : m_isString( false )
{
}

Key::Key( const QString &key )
{
  setKey( key );
}

QString Key::rawKey() const
{
  QString s = m_key;

  if ( m_isString )
    s = "s:" + s;

  return s;
}

void Key::setKey( const QString &key )
{
  if ( key.left( 2 ) == "s:" )
  {
    m_isString = true;
    m_key = key.mid( 2 );
  }
  else
  {
    m_isString = false;
    m_key = key;
  }
}

IfConfig::IfConfig()
{
  speedList << "Auto" << "1M" << "2M" << "5.5M" << "6M" << "9M" << "11M" << "12M" << "18M" << "24M" << "36M" << "48M" << "54M";
  wifiModeList << "Ad-Hoc" << "Managed" << "Repeater"
               << "Master" << "Secondary";
  cryptoModeList << "Open" << "Restricted";
  powerModeList << "All" << "UnicastOnly" << "MulticastOnly";
}

void IfConfig::load( KConfig *config, int i )
{
  QString entry;

  QString group = QString( "Configuration %1" ).arg( i+1 );
  config->setGroup( group );

  m_networkName = config->readEntry( "NetworkName" );
  entry = config->readEntry( "WifiMode", "Managed" );
  wifimodeFromString( entry );
  entry =  config->readEntry( "Speed", "Auto" );
  speedFromString( entry );
  m_runScript = config->readBoolEntry( "RunScript", false );
  m_connectScript = config->readEntry( "ScriptName" );

  m_useCrypto = config->readBoolEntry( "UseCrypto", false );
  entry = config->readEntry( "CryptoMode", "Open" );
  cryptomodeFromString( entry );
  m_activeKey = config->readNumEntry( "ActiveKey", 1 );
  m_keys[ 0 ].setKey( config->readEntry( "Key1" ) );
  m_keys[ 1 ].setKey( config->readEntry( "Key2" ) );
  m_keys[ 2 ].setKey( config->readEntry( "Key3" ) );
  m_keys[ 3 ].setKey( config->readEntry( "Key4" ) );

  m_keys[ 0 ].setIsString( config->readBoolEntry( "Key1_String" ) );
  m_keys[ 1 ].setIsString( config->readBoolEntry( "Key2_String" ) );
  m_keys[ 2 ].setIsString( config->readBoolEntry( "Key3_String" ) );
  m_keys[ 3 ].setIsString( config->readBoolEntry( "Key4_String" ) );

  m_pmEnabled = config->readBoolEntry( "PMEnabled", false );
  entry = config->readEntry( "PMMode", "All" );
  powermodeFromString( entry );
  m_sleepTimeout = config->readNumEntry( "SleepTimeout", 30 );
  m_wakeupPeriod = config->readNumEntry( "WakeupPeriod", 20 );
}

void IfConfig::save( KConfig *config, int i )
{
  QString group = QString( "Configuration %1" ).arg( i+1 );
  config->setGroup( group );

  config->writeEntry( "NetworkName", m_networkName );
  config->writeEntry( "WifiMode", wifimodeAsString() );
  config->writeEntry( "Speed", speedAsString() );
  config->writeEntry( "RunScript", m_runScript );
  config->writeEntry( "ScriptName", m_connectScript );

  config->writeEntry( "UseCrypto", m_useCrypto );
  config->writeEntry( "CryptoMode", cryptomodeAsString() );
  config->writeEntry( "ActiveKey", m_activeKey );
  config->writeEntry( "Key1", m_keys[ 0 ].key() );
  config->writeEntry( "Key2", m_keys[ 1 ].key() );
  config->writeEntry( "Key3", m_keys[ 2 ].key() );
  config->writeEntry( "Key4", m_keys[ 3 ].key() );

  config->writeEntry( "Key1_String", m_keys[ 0 ].isString() );
  config->writeEntry( "Key2_String", m_keys[ 1 ].isString() );
  config->writeEntry( "Key3_String", m_keys[ 2 ].isString() );
  config->writeEntry( "Key4_String", m_keys[ 3 ].isString() );

  config->writeEntry( "PMEnabled", m_pmEnabled );
  config->writeEntry( "PMMode", powermodeAsString() );
  config->writeEntry( "SleepTimeout", m_sleepTimeout );
  config->writeEntry( "WakeupPeriod", m_wakeupPeriod );
}

WifiConfig *WifiConfig::m_instance = 0;

WifiConfig *WifiConfig::instance()
{
  if ( m_instance == 0 )
    m_instance = new WifiConfig();

  return m_instance;
}

WifiConfig::WifiConfig()
{
  //m_config = KGlobal::config();
  m_config = new KSimpleConfig( "kcmwifirc" );

  load();
}

WifiConfig::~WifiConfig()
{
}

void WifiConfig::load()
{
  m_config->setGroup( "General" );

  m_usePreset = m_config->readBoolEntry( "UsePreset", false );
  m_presetConfig = m_config->readNumEntry( "PresetConfig", 1 )-1;
  m_interface = m_config->readEntry( "Interface", "" );

  for ( int i=0; i<4; ++i )
  {
    m_ifConfig[ i ].load( m_config, i );
  }
}

void WifiConfig::save()
{
  m_config->setGroup( "General" );

  m_config->writeEntry( "UsePreset", m_usePreset );
  m_config->writeEntry( "PresetConfig", m_presetConfig+1 );
  m_config->writeEntry( "Interface", m_interface );

  for ( int i=0; i<4; ++i )
  {
    m_ifConfig[ i ].save( m_config, i );
  }

  m_config->sync();
}

QString IfConfig::speedAsString()
{
  return speedList[ m_speed ];
}

QString IfConfig::wifimodeAsString()
{
  return wifiModeList[ m_wifiMode ];
}

QString IfConfig::cryptomodeAsString()
{
  return cryptoModeList[ m_cryptoMode ];
}

QString IfConfig::powermodeAsString()
{
  return powerModeList[ m_pmMode ];
}

int IfConfig::activeKeyId()
{
  return m_activeKey;
}

Key IfConfig::activeKey()
{
  return m_keys[ m_activeKey-1 ];
}

void IfConfig::speedFromString( const QString &s )
{
  m_speed = ( IfConfig::Speed )speedList.findIndex( s );
}

void IfConfig::wifimodeFromString( const QString &s )
{
  m_wifiMode = ( IfConfig::WifiMode )wifiModeList.findIndex( s );
}

void IfConfig::cryptomodeFromString( const QString &s )
{
  m_cryptoMode = ( IfConfig::CryptoMode )cryptoModeList.findIndex( s );
}

void IfConfig::powermodeFromString( const QString &s )
{
  m_pmMode = ( IfConfig::PowerMode )powerModeList.findIndex( s );
}

