/*
 * Copyright © 2001 Red Hat, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Red Hat not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  Red Hat makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * RED HAT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL RED HAT
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Owen Taylor, Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *          Olivier Fourdan: adapted to the "multi-channel" concept
 *	        Benedikt Meurer: Bugfixes/speedups.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xmd.h>

#include <glib.h>

#include <libxfce4util/i18n.h>

#include "mcs-channel.h"
#include "mcs-client.h"

struct _McsClient
{
    Display *display;
    int screen;
    McsNotifyFunc notify;
    McsWatchFunc watch;
    void *cb_data;

    Window mcs_manager_window;
    Atom manager_atom;
    Atom selection_atom;

    McsChannelList *channels;
};

/* Forward decl */
static void read_settings(McsClient *, const gchar *);

McsChannel *
mcs_client_add_channel(McsClient * client, const gchar *channel_name)
{
    McsChannel *channel;
    McsChannelList *channels;
    McsChannelList *new_elemt;

    g_return_val_if_fail(channel_name != NULL, NULL);
    g_return_val_if_fail(client != NULL, NULL);

    if(!g_ascii_strncasecmp(channel_name, "SETTINGS", strlen("SETTINGS")))
    {
        g_warning(
            _("Adding a standard mcs channel is not allowed\n"
              "Client channel not created\n"));
        return(NULL);
    }

    if ((channel = _mcs_channel_lookup(client->channels, channel_name)) != NULL)
        return(channel);

    if ((channel = _mcs_channel_new(channel_name, client->display)) == NULL) {
        g_warning(_("Unable to create a new MCS channel\n"));
        return(NULL);
    }

    channels = client->channels;
    new_elemt = g_new(McsChannelList, 1);
    new_elemt->channel = channel;
    new_elemt->next = NULL;

    if (channels)
    {
        for (; channels->next; channels = channels->next)
            ;

        channels->next = new_elemt;
    }
    else
    {
        client->channels = new_elemt;
    }

    read_settings(client, channel_name);

    return channel;
}

void
mcs_client_delete_channel(McsClient *client, const gchar *channel_name)
{
    McsChannel *channel;

    g_return_if_fail(channel_name != NULL);
    g_return_if_fail(client != NULL);

    if ((channel = _mcs_channel_lookup(client->channels, channel_name)) != NULL)
        _mcs_channel_delete(&client->channels, channel);
}

static void
notify_changes(McsClient *client, McsList *old_list, const gchar *channel_name)
{
    McsList *old_iter = old_list;
    McsList *new_iter;
    McsChannel *channel;

    if(!client->notify)
        return;

    g_return_if_fail(channel_name != NULL);
    g_return_if_fail(client != NULL);

    if ((channel = _mcs_channel_lookup(client->channels, channel_name)) == NULL)
        return;

    new_iter = channel->settings;

    while(old_iter || new_iter)
    {
        int cmp;

        if(old_iter && new_iter)
        {
            cmp = strcmp(old_iter->setting->name, new_iter->setting->name);
            if(!cmp)
                cmp = strcmp(old_iter->setting->channel_name, new_iter->setting->channel_name);
        }
        else if(old_iter)
            cmp = -1;
        else
            cmp = 1;

        if(cmp < 0)
        {
            client->notify(old_iter->setting->name, old_iter->setting->channel_name, MCS_ACTION_DELETED, NULL, client->cb_data);
        }
        else if(cmp == 0)
        {
            if(!mcs_setting_equal(old_iter->setting, new_iter->setting))
                client->notify(old_iter->setting->name, old_iter->setting->channel_name, MCS_ACTION_CHANGED, new_iter->setting, client->cb_data);
        }
        else
        {
            client->notify(new_iter->setting->name, new_iter->setting->channel_name, MCS_ACTION_NEW, new_iter->setting, client->cb_data);
        }

        if(old_iter)
            old_iter = old_iter->next;
        if(new_iter)
            new_iter = new_iter->next;
    }
}

static int ignore_errors(Display * display, XErrorEvent * event)
{
    return True;
}

static gchar local_byte_order = '\0';

#define BYTES_LEFT(buffer) ((buffer)->data + (buffer)->len - (buffer)->pos)

static McsResult fetch_card16(McsBuffer * buffer, CARD16 * result)
{
    CARD16 x;

    if(BYTES_LEFT(buffer) < 2)
        return MCS_ACCESS;

    x = *(CARD16 *) buffer->pos;
    buffer->pos += 2;

    if(buffer->byte_order == local_byte_order)
        *result = x;
    else
        *result = (x << 8) | (x >> 8);

    return MCS_SUCCESS;
}

static McsResult fetch_ushort(McsBuffer * buffer, unsigned short *result)
{
    CARD16 x;
    McsResult r;

    r = fetch_card16(buffer, &x);
    if(r == MCS_SUCCESS)
        *result = x;

    return r;
}

static McsResult fetch_card32(McsBuffer * buffer, CARD32 * result)
{
    CARD32 x;

    if(BYTES_LEFT(buffer) < 4)
        return MCS_ACCESS;

    x = *(CARD32 *) buffer->pos;
    buffer->pos += 4;

    if(buffer->byte_order == local_byte_order)
        *result = x;
    else
        *result = (x << 24) | ((x & 0xff00) << 8) | ((x & 0xff0000) >> 8) | (x >> 24);

    return MCS_SUCCESS;
}

static McsResult fetch_card8(McsBuffer * buffer, CARD8 * result)
{
    if(BYTES_LEFT(buffer) < 1)
        return MCS_ACCESS;

    *result = *(CARD8 *) buffer->pos;
    buffer->pos += 1;

    return MCS_SUCCESS;
}

#define MCS_PAD(n,m) ((n + m - 1) & (~(m-1)))

static McsList *parse_settings(guchar *data, size_t len, const gchar *channel_name)
{
    McsBuffer buffer;
    McsResult result = MCS_SUCCESS;
    McsList *settings = NULL;
    CARD32 serial;
    CARD32 n_entries;
    CARD32 i;
    McsSetting *setting = NULL;

    g_return_val_if_fail(channel_name != NULL, NULL);

    local_byte_order = mcs_byte_order();

    buffer.pos = buffer.data = data;
    buffer.len = len;

    result = fetch_card8(&buffer, (gchar *)&buffer.byte_order);
    if(buffer.byte_order != MSBFirst && buffer.byte_order != LSBFirst)
    {
        fprintf(stderr, _("Invalid byte order in MCS property\n"));
        result = MCS_FAILED;
        goto out;
    }

    buffer.pos += 3;

    result = fetch_card32(&buffer, &serial);
    if(result != MCS_SUCCESS)
        goto out;

    result = fetch_card32(&buffer, &n_entries);
    if(result != MCS_SUCCESS)
        goto out;

    for(i = 0; i < n_entries; i++)
    {
        CARD8 type;
        CARD16 name_len;
        CARD32 v_int;
        size_t pad_len;

        result = fetch_card8(&buffer, &type);
        if(result != MCS_SUCCESS)
            goto out;

        buffer.pos += 1;

        result = fetch_card16(&buffer, &name_len);
        if(result != MCS_SUCCESS)
            goto out;

        pad_len = MCS_PAD(name_len, 4);
        if(BYTES_LEFT(&buffer) < pad_len)
        {
            result = MCS_ACCESS;
            goto out;
        }

        setting = g_malloc(sizeof *setting);
        if(!setting)
        {
            result = MCS_NO_MEM;
            goto out;
        }
        setting->type = MCS_TYPE_INT;   /* No allocated memory */

        setting->name = g_malloc(name_len + 1);
        if(!setting->name)
        {
            result = MCS_NO_MEM;
            goto out;
        }

        memcpy(setting->name, buffer.pos, name_len);

        setting->name[name_len] = '\0';
        setting->channel_name = g_ascii_strup(channel_name, -1);
        buffer.pos += pad_len;

        result = fetch_card32(&buffer, &v_int);
        if(result != MCS_SUCCESS)
            goto out;
        setting->last_change_serial = v_int;

        switch (type)
        {
            case MCS_TYPE_INT:
                result = fetch_card32(&buffer, &v_int);
                if(result != MCS_SUCCESS)
                    goto out;

                setting->data.v_int = (INT32) v_int;
                break;
            case MCS_TYPE_STRING:
                result = fetch_card32(&buffer, &v_int);
                if(result != MCS_SUCCESS)
                    goto out;

                pad_len = MCS_PAD(v_int, 4);
                if(v_int + 1 == 0 ||    /* Guard against wrap-around */
                   BYTES_LEFT(&buffer) < pad_len)
                {
                    result = MCS_ACCESS;
                    goto out;
                }

                setting->data.v_string = g_malloc(v_int + 1);
                if(!setting->data.v_string)
                {
                    result = MCS_NO_MEM;
                    goto out;
                }

                memcpy(setting->data.v_string, buffer.pos, v_int);
                setting->data.v_string[v_int] = '\0';
                buffer.pos += pad_len;

                break;
            case MCS_TYPE_COLOR:
                result = fetch_ushort(&buffer, &setting->data.v_color.red);
                if(result != MCS_SUCCESS)
                    goto out;
                result = fetch_ushort(&buffer, &setting->data.v_color.green);
                if(result != MCS_SUCCESS)
                    goto out;
                result = fetch_ushort(&buffer, &setting->data.v_color.blue);
                if(result != MCS_SUCCESS)
                    goto out;
                result = fetch_ushort(&buffer, &setting->data.v_color.alpha);
                if(result != MCS_SUCCESS)
                    goto out;

                break;
            default:
                /* Quietly ignore unknown types */
                break;
        }

        setting->type = type;

        result = mcs_list_insert(&settings, setting);
        if(result != MCS_SUCCESS)
            goto out;

        setting = NULL;
    }

  out:

    if(result != MCS_SUCCESS)
    {
        switch (result)
        {
            case MCS_NO_MEM:
                fprintf(stderr, _("Out of memory reading MCS property\n"));
                break;
            case MCS_ACCESS:
                fprintf(stderr, _("Invalid MCS property (read off end)\n"));
                break;
            case MCS_DUPLICATE_ENTRY:
                fprintf(stderr, _("Duplicate MCS entry for '%s'\n"),
				setting->name);
            case MCS_FAILED:
            case MCS_SUCCESS:
            case MCS_NO_ENTRY:
            case MCS_NO_CHANNEL:
            default:
                break;
        }

        if(setting)
            mcs_setting_free(setting);

        mcs_list_free(settings);
        settings = NULL;

    }

    return settings;
}

static void
read_settings(McsClient *client, const gchar *channel_name)
{
    Atom type;
    int format;
    McsChannel *channel;
    McsList *old_list;
    unsigned long n_items;
    unsigned long bytes_after;
    guchar *data;
    int result;

    int (*old_handler) (Display *, XErrorEvent *);

    g_return_if_fail(client != NULL);
    g_return_if_fail(channel_name != NULL);

    if ((channel = _mcs_channel_lookup(client->channels, channel_name)) == NULL)
        return;

    old_list = channel->settings;

    channel->settings = NULL;

    old_handler = XSetErrorHandler(ignore_errors);
    result = XGetWindowProperty(client->display, client->mcs_manager_window, channel->channel_atom, 0, LONG_MAX, False, channel->channel_atom, &type, &format, &n_items, &bytes_after, &data);
    XSetErrorHandler(old_handler);

    if(result == Success && type == channel->channel_atom)
    {
        if(format != 8)
        {
            fprintf(stderr, _("Invalid format for MCS property %d"), format);
        }
        else
        {
            channel->settings = parse_settings(data, n_items, channel_name);
        }

        XFree(data);
    }

    notify_changes(client, old_list, channel_name);
    mcs_list_free(old_list);
}

static void add_events(Display * display, Window window, long mask)
{
    XWindowAttributes attr;

    XGetWindowAttributes(display, window, &attr);
    XSelectInput(display, window, attr.your_event_mask | mask);
}

static void check_manager_window(McsClient * client)
{
    if(client->mcs_manager_window && client->watch)
        client->watch(client->mcs_manager_window, False, 0, client->cb_data);

    XGrabServer(client->display);

    client->mcs_manager_window = XGetSelectionOwner(client->display, client->selection_atom);
    if(client->mcs_manager_window)
        XSelectInput(client->display, client->mcs_manager_window, PropertyChangeMask | StructureNotifyMask);

    XUngrabServer(client->display);
    XFlush(client->display);

    if(client->mcs_manager_window && client->watch)
        client->watch(client->mcs_manager_window, True, PropertyChangeMask | StructureNotifyMask, client->cb_data);
}

McsClient *
mcs_client_new(Display *display, int screen, McsNotifyFunc notify,
        McsWatchFunc watch, void *cb_data)
{
    McsClient *client;
    gchar *buffer;

    if ((client = g_new(McsClient, 1)) == NULL)
        return(NULL);

    client->display = display;
    client->screen = screen;
    client->notify = notify;
    client->watch = watch;
    client->cb_data = cb_data;

    client->mcs_manager_window = None;
    client->channels = NULL;

    buffer = g_strdup_printf("_MCS_S%d", screen);
    client->selection_atom = XInternAtom(display, buffer, False);
    client->manager_atom = XInternAtom(display, "MCS_MANAGER", False);
    g_free(buffer);

    /* Select on StructureNotify so we get MANAGER events */
    add_events(display, RootWindow(display, screen), StructureNotifyMask);

    if(client->watch)
        client->watch(RootWindow(display, screen), True, StructureNotifyMask, client->cb_data);

    check_manager_window(client);

    return client;
}

void
mcs_client_destroy(McsClient *client)
{
    g_return_if_fail(client != NULL);

    if(client->watch)
        client->watch(RootWindow(client->display, client->screen), False, 0, client->cb_data);
    if(client->mcs_manager_window && client->watch)
        client->watch(client->mcs_manager_window, False, 0, client->cb_data);

    while(client->channels != NULL) {
        if (!(client->channels->channel) ||
                !(client->channels->channel->channel_name))
            g_warning(_("Bogus MCS client channels"));
        else
            _mcs_channel_delete(&client->channels, client->channels->channel);
    }

    g_free(client);
}

McsResult mcs_client_get_setting(McsClient * client, const gchar *name, const gchar *channel_name, McsSetting ** setting)
{
    McsChannel *channel;
    McsSetting *search;

    g_return_val_if_fail (client != NULL, MCS_FAILED);
    g_return_val_if_fail (setting != NULL, MCS_FAILED);
    g_return_val_if_fail (name != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);

    if ((channel = _mcs_channel_lookup(client->channels, channel_name)) == NULL)
        return(MCS_NO_CHANNEL);

    search = mcs_list_lookup(channel->settings, name);

    if(search)
    {
        *setting = mcs_setting_copy(search);
        return *setting ? MCS_SUCCESS : MCS_NO_MEM;
    }
    else
        return MCS_NO_ENTRY;
}

Bool mcs_client_process_event(McsClient * client, XEvent * xev)
{
    g_return_val_if_fail (client != NULL, False);

    if(xev->xany.window == RootWindow(client->display, client->screen))
    {
        if(xev->xany.type == ClientMessage && xev->xclient.message_type == client->manager_atom && xev->xclient.data.l[1] == client->selection_atom)
        {
            check_manager_window(client);
            return True;
        }
    }
    else if(xev->xany.window == client->mcs_manager_window)
    {
        if(xev->xany.type == DestroyNotify)
        {
            check_manager_window(client);
            return True;
        }
        else if(xev->xany.type == PropertyNotify)
        {
            Atom type = xev->xclient.message_type;
            McsChannelList *channels = client->channels;

            while(channels)
            {
                if(type == channels->channel->channel_atom)
                {
                    read_settings(client, channels->channel->channel_name);
                    return True;
                }
                channels = channels->next;
            }
        }
    }

    return False;
}

void mcs_client_show(Display * display, int screen, const gchar *message)
{
    Window w;
    static Atom show_atom = None;
    Atom selection_atom = None;
    gchar *buffer;

    g_return_if_fail (message != NULL);

    XGrabServer(display);
    buffer = g_strdup_printf("_MCS_S%d", screen);
    selection_atom = XInternAtom(display, buffer, False);
    g_free(buffer);
    show_atom = XInternAtom(display, "SHOW", False);
    
    if((w = XGetSelectionOwner(display, selection_atom)))
    {
        XChangeProperty(display, w, show_atom, show_atom, 8, PropModeReplace, (guchar *)message, strlen(message));
    }

    XUngrabServer(display);
    XFlush(display);
}

gboolean mcs_client_check_manager(Display * display, int screen, const gchar *manager_command)
{
    GError *error = NULL;
    
    g_return_val_if_fail (manager_command != NULL, FALSE);
    
    if (mcs_manager_check_running(display, screen) < MCS_MANAGER_MULTI_CHANNEL)
    {
	if ((!manager_command) || (!strlen(manager_command)) || (!g_spawn_command_line_sync(manager_command, NULL, NULL, NULL, &error)))
	{
	    if (error)
	    {
	        g_critical(_("Error starting settings manager: %s\n"),
				error->message);
	        g_error_free(error);
	    }
	    return FALSE;
	}
    }
    return TRUE;
}
