/*
 This file is part of GNU Taler
 (C) 2022-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { j2s, TalerCorebankApiClient } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  BankService,
  ExchangeService,
  FakebankService,
  getTestHarnessPaytoForLabel,
  GlobalTestState,
  HarnessExchangeBankAccount,
  setupDb,
} from "../harness/harness.js";

/**
 * Test feature reporting of exchange entries (p2p payments, shopping URL)
 * in the wallet.
 */
export async function runWalletExchangeFeaturesTest(t: GlobalTestState) {
  const db = await setupDb(t);

  const bc = {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  };

  const bank: BankService = await FakebankService.create(t, bc);

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const receiverName = "Exchange";
  const exchangeBankUsername = "exchange";
  const exchangeBankPassword = "mypw-password";
  const exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);
  const wireGatewayApiBaseUrl = new URL(
    `accounts/${exchangeBankUsername}/taler-wire-gateway/`,
    bank.corebankApiBaseUrl,
  ).href;

  const exchangeBankAccount: HarnessExchangeBankAccount = {
    wireGatewayApiBaseUrl,
    wireGatewayAuth: {
      username: exchangeBankUsername,
      password: exchangeBankPassword,
    },
    accountPaytoUri: exchangePaytoUri,
  };

  await exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  exchange.addCoinConfigList(defaultCoinConfig.map((x) => x("TESTKUDOS")));

  await exchange.modifyConfig(async (f) => {
    f.setString(
      "exchange",
      "shopping_url",
      "https://taler-shopping.example.com",
    );
    f.setString("exchange", "DISABLE_DIRECT_DEPOSIT", "yes");
  });

  await exchange.start({
    skipGlobalFees: true,
  });

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "wallet",
    persistent: true,
  });

  console.log("setup done!");

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:5",
  });

  await wres.withdrawalFinishedCond;

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(bal));

  t.assertDeepEqual(bal.balances[0].disableDirectDeposits, true);
  t.assertDeepEqual(bal.balances[0].disablePeerPayments, true);
}

runWalletExchangeFeaturesTest.suites = ["wallet"];
