/*
 * Copyright (C) 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>

#include "fileutil.h"

static void file_error(char *progname, const char *error, char *file);
static void internal_error(char *progname, const char *error);

#define OUTPUT_EXTENSION ".html"

char *
read_file(char *progname, char *fileName)
{
  FILE *file;
  char *buffer = 0;
  long int char_count;

  // we open it as binary otherwise we may experience problems under
  // Windows system: when we fread, the number of char read can be
  // less then char_count, and thus we'd get an error...
  if ( (file = fopen(fileName,"rb") ) == 0 )	// The file does not exist :(
    file_error (progname, "Error operning", fileName);
  else
    {
      // let's go to the end of the file...
      if (fseek (file, 0, SEEK_END) != 0)
        file_error (progname, "Error positioning", fileName);

      // ...to read the dimension
      char_count = ftell (file);
      if (char_count < 0)
        file_error (progname, "Error reading position", fileName);

      buffer = (char *) malloc (char_count +1);
      if (! buffer)
        internal_error (progname, "Memory allocation failed");

      // let's go back to the start
      rewind (file);

      if (fread ((void *) buffer, 1, char_count, file) < (size_t) char_count)
        file_error (progname, "read error", fileName);
      buffer[char_count] = '\0';

      fclose (file);
    }

  return buffer;
}

void
file_error(char *progname, const char *error, char *file)
{
  cerr << progname << ": " << error << ", file " << file << endl;
  exit (1);
}

void
internal_error(char *progname, const char *error)
{
  cerr << progname << ": Internal error: " << error << endl;
  exit (1);
}

// output file name = <outputDir> + input file name + ".html"
char *
createOutputFileName( char *inputFileName, char *outputDir )
{
  char *input_file_name;
  char path_separator = '/';

  if (! outputDir)
    input_file_name = inputFileName;
  else
    {
      char *file_name; // without the path
      file_name = strrchr (inputFileName, '/');
      
      if (! file_name) // try with DOS separator
        {
          file_name = strrchr (inputFileName, '\\');
          if (file_name)
            path_separator = '\\';
        }


      if (file_name)
        input_file_name = &file_name[1];
      else
        input_file_name = inputFileName;
    }

  int lenght =  strlen(input_file_name) + 
    strlen(OUTPUT_EXTENSION) + 1 ;
  int output_dir_lenght = 0;
  
  if (outputDir)
    {
      output_dir_lenght = strlen (outputDir);
      lenght += output_dir_lenght + 1; // one is for the additional /
    }

  char *outputFileName = new char[lenght] ;
  outputFileName[0] = '\0';

  if (outputDir)
    {
      strcat (outputFileName, outputDir);
      outputFileName[output_dir_lenght] = path_separator;
      outputFileName[output_dir_lenght + 1] = '\0';
    }
  strcat (outputFileName, input_file_name);
  strcat (outputFileName, OUTPUT_EXTENSION);

  return outputFileName ;
}
