"======================================================================
|
|   Smalltalk GUI namespace browser
|
|
 ======================================================================"


"======================================================================
|
| Copyright 2002 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================
"

ClassHierarchyBrowser subclass: #NamespaceBrowser
     instanceVariableNames: 'curNamespace byCategory namespacesMap namespaces 
			     categories'
     classVariableNames: ''
     poolDictionaries: ''
     category: 'Graphics-Browser'
!

!NamespaceBrowser methodsFor: 'initializing'!

open
    byCategory := false.
    super open
!

initialize
    self updateNamespaceList
!

createUpperPanesIn: upper
    self createNamespaceListIn: upper.
    super createUpperPanesIn: upper!

createNamespaceListIn: upper
    | list |
    upper addChildView: ( (list := PList new: 'Namespaces' in: upper)
	  initialize;
	  data: self;
	  "Register three types of messages"
	  stateChange: #namespaceList;
	  changedSelection: #newNamespaceSelection;
	  handleUserChange: #namespaceSelection:;
	  listMsg: #namespaceList;
	  hiliteItemInitMsg: #namespaceString;
	  menuInit: (self blueButtonMenuForNamespaces: list);
	  yourself).

    self layoutUpperPaneElement: list blox num: -1!

layoutUpperPaneElement: blox num: n
    blox x: 150 * n + 150 y: 0 width: 150 height: 200.
! !

!NamespaceBrowser methodsFor: 'overriding'!

currentNamespace
    ^byCategory ifTrue: [ Namespace current ] ifFalse: [ curNamespace ]
!

!NamespaceBrowser methodsFor: 'accessing'!

namespaceList
    ^byCategory ifTrue: [ categories ] ifFalse: [ namespaces ]
!

readNamespaces
    | stack top indent namespace subspaces |
    stack := OrderedCollection new.
    namespacesMap := Dictionary new: 17.
    namespaces := OrderedCollection new.
    subspaces := {Smalltalk}, RootNamespace allInstances.
    [
	subspaces isNil ifFalse: [
	    top := stack addLast: (subspaces 
		asSortedCollection: [ :a :b | a name <= b name ]).
	].
	[ top isEmpty ] whileTrue: [
	    stack removeLast.
	    stack isEmpty ifTrue: [ ^self ].
	    top := stack last.
	].
	namespace := top removeFirst.
	subspaces := namespace subspaces.

	indent := String new: stack size - 1 withAll: Character space. 
	namespacesMap at: indent, namespace name put: namespace.
	namespaces add: indent, namespace name.
    ] repeat.
!

readCategories
    categories := Set new.
    Class allSubclassesDo: [ :each |
	each isMetaclass ifTrue: [ 
	    categories add: each instanceClass category ].
    ].
    categories := categories asSortedCollection
!

namespaceSelection: assoc
    | name |
    name := assoc value.
    curNamespace := name isNil | byCategory
	ifTrue: [ name ]
	ifFalse: [ namespacesMap at: name ].

    self updateClassList
!

namespaceString
"Return name of selected class indented by 'n' spaces, where 'n' is the number
of class' superclasses"
     | spaces |
     curNamespace isNil ifTrue: [^nil].
     byCategory ifTrue: [ ^curNamespace ].

     spaces := String new: curNamespace allSuperspaces size.
     spaces atAllPut: Character space.
     ^spaces, curNamespace name
!

byCategory
    "If categories are being viewed, return true"
    ^byCategory
!

byCategory: aBoolean
"Change instance/class representation and record data state changes"
    byCategory = aBoolean ifTrue: [^self].
    curNamespace := nil.
    byCategory := aBoolean.
    self updateNamespaceList
! !

!NamespaceBrowser methodsFor: 'class list blue button menu'!

fileoutName
    byCategory ifTrue: [ ^curNamespace ].
    ^(curNamespace nameIn: Smalltalk) asString
	replaceAll: Character space with: $-;
	yourself!

fileOutNamespace: listView
    "File out a description of the currently selected namespace"
    | oldCurrent className fileName stream |
    curNamespace isNil ifTrue: [^listView beep].
    fileName := self fileoutDir, self fileoutName, '.st'.
    fileName := Prompter saveFileName: 'File out namespace' default: fileName in: listView.
    fileName isEmpty ifTrue: [ ^self ].

    stream := FileStream open: fileName mode: FileStream write.

    byCategory ifFalse: [
	curNamespace superspace isNil
	    ifFalse: [
        	stream
		    nextPutAll: (curNamespace superspace nameIn: Smalltalk);
		    nextPutAll: ' addSubspace: #';
		    nextPutAll: curNamespace name;
		    nextPutAll: '!'; nl;
		    nextPutAll: 'Namespace current: ';
		    nextPutAll: (curNamespace nameIn: Smalltalk);
		    nextPutAll: '!'; nl; nl
	    ]
	    ifTrue: [
		stream
		    nextPutAll: 'Namespace current: (RootNamespace new: #';
		    nextPutAll: (curNamespace nameIn: Smalltalk);
		    nextPutAll: '!'; nl; nl
	    ].

	oldCurrent := Namespace current.
	Namespace current: curNamespace.
    ].

    classList do: [ :each | 
        (each trimSeparators includes: $( ) ifFalse: [
    	(shownClasses at: each) fileOutOn: stream ].
    ].

    byCategory ifFalse: [
	Namespace current: oldCurrent.
        stream nextPutAll: 'Namespace current: Smalltalk!'; nl
    ].

    stream close.
    self setFileoutDirFromFile: fileName
!

renameNamespace: listView
"Rename currently selected namespace"
     | classes oldName newName prompter oldAssoc newNamespace |
     curNamespace isNil ifTrue: [^listView beep].
     oldName := self namespaceString trimSeparators.

     "Prompt user for new name"
     prompter := Prompter message: ('Rename namespace: ' , curClass name) 
	  in: listView.
     prompter response = '' ifTrue: [^self].

     self byCategory ifTrue: [
	shownClasses do: [ :each | each category: prompter response ].
	self updateNamespaceList.
        ^self
     ].

     oldName := oldName asSymbol.
     newName := prompter response asSymbol.
     (newName at: 1) isUppercase 
	 ifFalse: [^self error: 'Namespace name must begin with an uppercase letter'].

     (curNamespace includesKey: newName) 
          ifTrue: [^self error: newName , ' already exists' ].

     "Save old Association"
     oldAssoc := curNamespace superspace isNil
	ifTrue: [ Smalltalk associationAt: oldName ]
	ifFalse: [ curNamespace superspace associationAt: oldName ].

     "Rename the namespace now"
     newNamespace := curNamespace superspace isNil
	ifTrue: [ RootNamespace new: newName asSymbol ]
	ifFalse: [ curNamespace superspace addSubspace: newName asSymbol ].

     curNamespace definedKeys do: [ :each |
	newNamespace add: (curNamespace associationAt: each) ]. 

     curNamespace superspace isNil
	 ifTrue: [ Smalltalk removeKey: oldName ]
	 ifFalse: [ curNamespace superspace removeKey: oldName ].

     "Notify programmer of all references to renamed subspace"
     classes := SortedCollection new.
     CompiledMethod allInstancesDo:  [ :method |
          ((method refersTo: oldAssoc) or: [method refersTo: oldAssoc key])
	      ifTrue: [ classes add: method printString]].

     classes isEmpty
	   ifFalse:[ModalDialog new alertMessage: ('Rename all references to 
		    namespace ' , oldName , Character nl asSymbol , 'to the new name: ', 
		    newName ) in: listView.
	       MethodSetBrowser new openOn: classes title: ('References to ', 
		    oldName) selection: oldName].


"Update class list"
     self updateNamespaceList.
!

topClasses
    ^self topMetas collect: [ :each | each instanceClass ]
!

topMetas
    curNamespace isNil ifTrue: [ ^#() ].
    ^byCategory
	ifTrue: [ Class allSubclasses select: [ :each |
	    each category = curNamespace ] ]
	ifFalse: [ Class allSubclasses select: [ :each |
	    each environment = curNamespace ] ]
!

updateNamespaceList
"Invoked from class list pane popup.  Update class list pane through the 
change/update mechanism"

    byCategory 
	ifTrue: [ self readCategories ]
	ifFalse: [ self readNamespaces ].

    self changeState: #namespaceList.
    self updateClassList.
!

namespaces: namespaceList
    namespaceList canChangeState ifFalse: [^self].
    self byCategory: false
!

categories: namespaceList
    namespaceList canChangeState ifFalse: [^self].
    self byCategory: true
!

blueButtonMenuForNamespaces: theView
"Install popup for namespace list popup"
     ^(PopupMenu new: theView label: '') selectors: 
    #(('File out...' #fileOutNamespace: #theView) 
     ('Update' #updateNamespaceList ) () ('Namespaces' #namespaces: #theView) 
     ('Categories'  #categories: #theView )
     () ('Rename...' #renameNamespace: #theView ))
     receiver: self
     argument: theView
! !
