#!/usr/bin/env python
#
# Copyright 2003 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from GnuRadio import *
import wx
import fftsink
import os
import eng_notation

class usrp_FlowGraph (gr_FlowGraph):
    def __init__ (self):
        gr_FlowGraph.__init__ (self)
        self.usrp = None

    def _set_usrp (self, usrp):
        self.usrp = usrp

    # modifiers: delegate
    def set_rx_freq (self, chan, freq):
        print "set_rx_freq (%d, %g)" % (chan, freq)
        return self.usrp.set_rx_freq (chan, freq)
    
    def set_adc_clk_div (self, div):
        return self.usrp.set_adc_clk_div (div)

    def set_decim_rate (self, rate):
        return self.usrp.set_decim_rate (rate)
    
    # accessors: delegate
    def get_rx_freq (self, chan):
        return self.usrp.get_rx_freq (chan)

    def get_adc_clk_div (self):
        return self.usrp.get_adc_clk_div ()

    def get_decim_rate (self):
        return self.usrp.get_decim_rate ()

    def get_oscillator_freq (self):
        return self.usrp.get_oscillator_freq ()


    
def build_graph (win_parent):
    adc_clk_div = 2
    decim_rate =  16
    rx_freq = 0

    which_board = 0
    channel_selection_bitmask = 0x1
    osc_freq = 125e6
    
    fg = usrp_FlowGraph ()

    usrp = GrUsrpSource (which_board, channel_selection_bitmask, adc_clk_div, decim_rate)
    fg._set_usrp (usrp)
    
    usrp.set_rx_freq (0, rx_freq)
    sample_rate = osc_freq / adc_clk_div / decim_rate
    print "adc_clk_div = %d  decim_rate = %d  rx_freq = %g" % (adc_clk_div, decim_rate, rx_freq)
    print "adc_clk_rate = %g  usb_sample_rate = %g" % (osc_freq / adc_clk_div, sample_rate)

    dst, win = fftsink.makeFFTSinkC (fg, win_parent, "USRP", 512, sample_rate)

    fg.connect (usrp, dst)

    return (fg, win)


# ----------------------------------------------------------------


if __name__ == '__main__':

    class TestPanel (wx.Panel):
        def __init__ (self, parent, frame):
            wx.Panel.__init__ (self, parent, -1)

            self.frame = frame

            vbox = wx.BoxSizer (wx.VERTICAL)

            self.fg, fft_win = build_graph (self)
            vbox.Add (fft_win, 1, wx.EXPAND)

            hbox = wx.BoxSizer (wx.HORIZONTAL)

            hbox.Add (1, 1, 1, wx.EXPAND)
            hbox.Add (wx.StaticText (self, -1, "Set Rx Freqency: "), 0, wx.ALIGN_CENTER)
            self.tc_freq = wx.TextCtrl (self, -1, "", style=wx.TE_PROCESS_ENTER)
            hbox.Add (self.tc_freq, 0, wx.ALIGN_CENTER)
            hbox.Add (1, 1, 1, wx.EXPAND)

            vbox.Add (hbox, 0, wx.EXPAND)
            
            self.sizer = vbox

            self.SetSizer (self.sizer)
            self.SetAutoLayout (True)
            self.sizer.Fit (self)
            wx.EVT_CLOSE (self, self.OnCloseWindow)
            wx.EVT_TEXT_ENTER (self, self.tc_freq.GetId(), self.EvtTextEnter)
            self.UpdateStatusBar ()

        def UpdateStatusBar (self):
            adc_clk_div = self.fg.get_adc_clk_div ()
            decim_rate = self.fg.get_decim_rate ()
            rx_freq = self.fg.get_rx_freq (0)
            sample_rate = self.fg.get_oscillator_freq () / adc_clk_div / decim_rate
            msg = "adc_clk_div: %d  adc_clk: %s decim_rate: %3d  sample_rate: %s  rx_freq: %s" % (
                adc_clk_div,
                eng_notation.num_to_str (self.fg.get_oscillator_freq () / adc_clk_div),
                decim_rate,
                eng_notation.num_to_str (sample_rate),
                eng_notation.num_to_str (rx_freq))
            
            self.frame.GetStatusBar().SetStatusText (msg, 0)

        def EvtTextEnter(self, event):
            str = event.GetString ()
            self.tc_freq.Clear ()
            self.fg.set_rx_freq (0, eng_notation.str_to_num (str))
            self.UpdateStatusBar ()
        
        def OnCloseWindow (self, event):
            self.fg.stop ()
            self.Destroy ()


    class MainFrame (wx.Frame):
        def __init__ (self):
            wx.Frame.__init__(self, None, -1, "Usrp Rx FFT")

            self.CreateStatusBar ()
            mainmenu = wx.MenuBar ()
            menu = wx.Menu ()
            menu.Append (200, 'E&xit', 'Get outta here!')
            mainmenu.Append (menu, "&File")
            self.SetMenuBar (mainmenu)
            wx.EVT_MENU (self, 200, self.OnExit)
            self.panel = TestPanel (self, self)
            wx.EVT_CLOSE (self, self.OnCloseWindow)

        def OnCloseWindow (self, event):
            self.Destroy ()

        def OnExit (self, event):
            self.Close (True)


    class TestApp (wx.App):
        def OnInit(self):
            print "TestApp: pid = ", os.getpid ()
            frame = MainFrame ()
            frame.Show (True)
            self.SetTopWindow (frame)
            print "FlowGraph: ", frame.panel.fg
            frame.panel.fg.start ()
            return True

    app = TestApp (0)
    app.MainLoop ()


# ----------------------------------------------------------------
