;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: OPAL; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Close all connections to the X server by saying:
;;;     (opal:Disconnect-Garnet)
;;;
;;; While the connection to the X server is closed, you may
;;; save a core image of Garnet.  To save a core image:
;;;   In CMU Common Lisp say        (ext:save-lisp filename)
;;;   In Allegro Lisp say           (excl:dumplisp)
;;;   In Lucid Lisp the command is  (disksave filename)
;;;
;;; Reopen all connections to the X server by saying:
;;;     (opal:Reconnect-Garnet)
;;;
#|
CHANGE LOG:
        21-Sep-92 amickish No longer necessary to call notice-items-changed on
                      menubars, due to reimplementation of :submenu-window-list
                      in MENUBAR gadget.
        22-Jun-92 ECP It is necessary to call notice-items-changed on
                      menubars during the execution of reconnect-garnet.
	19-Jun-92 ECP In reconnect-garnet, turn off asynchronous error reports.
	29-May-92 ECP/KY Determine display number and screen number from
		      full display name, by calling initialize-x11-values.
		      If you call disconnect-garnet when already disconnected,
		      or reconnect-garnet when already reconnected, exit.
	25-May-92 ECP Check that elements of *all-windows* and
		      *all-windows-which-have-been-closed* have not
		      been destroyed.
	 6-May-92 ECP Only call main-event-loop-process in reconnect-garnet
		      if it had been halted in disconnect-garnet.
	16-Apr-92 ECP Call launch-main-event-loop-process at end of
		      reconnect-garnet.
        30-Mar-92 amickish  Changed funcalls of :update method to update call;
                      Changed the way *all-the-windows* is computed in
                      Disconnect-Garnet.
        25-Mar-92 amickish  Get-Values ---> G-Value
	23-Mar-92 ECP In reconnect-windows, must update all the windows,
			not just the visible ones.
	20-Mar-92 ECP Moved exports to defs.lisp.  Use process routines.
	11-Mar-92 ECP Added references to kr::*constants-disabled*
		      When reinitializing colors, just call g-value,
		      not s-value.
	17-Feb-92 ECP Added *auxilliary-reconnect-routines*
        31-Jan-92 ECP Eliminated *display-name-to-display-mapping*.
	24-Jan-92 ECP reinitialized text objects in reconnect-garnet.
	26-Mar-91 ECP kcl patch
        24-Mar-91 ECP Fixed bug involving reconnect to a color screen.
         7-Mar-91 ECP The question of whether the screen is color or
                      black-and-white is now determined inside
                      initialize-default-x-values in defs.lisp.
	14-Feb-91 ECP More changes to color for connect and disconnect
         8-Feb-91 ECP Added :color-p slot to opal:color to tell if
                      screen is black-and-white or color.
        11-Sep-90 ECP Get display name in allegro by (sys::getenv "DISPLAY")
                      Use (short-site-name) as an #+allegro alternative
                      to (machine-instance)
        15-Aug-90 ECP Yet more debugging.  Disconnect-garnet must
                      set windows :lineage slot to NIL.
                      Reconnect-garnet has an optional argument.
                      Call to initialize-default-x-values.
	14-Aug-90 ECP In reconnect-garnet, just explicitly update
			top level windows.
	10-Aug-90 ECP In reconnect-garnet, recompute display name.
	21-Mar-90 ECP Lots of debugging, as well as the above comments.
	9-Mar-90 ECP Released locally
|#

(in-package "OPAL" :use '("KR" "LISP"))

(defvar *gadgets-requiring-notice-items-changed* nil)
(defvar *all-the-windows* nil)
(defvar *all-windows-which-have-been-closed* nil)
(defvar *garnet-has-been-disconnected* nil)
#-cmu (defvar *main-event-loop-process-was-halted* nil)

(defun all-the-instances (x)
  (cons x (mapcan #'all-the-instances (g-local-value x :is-a-inv))))

(defun Disconnect-Garnet ()
  (when *garnet-has-been-disconnected*
    (return-from disconnect-garnet))
  #-cmu
  (when (opal:main-event-loop-process-running-p)
    (setq *main-event-loop-process-was-halted* t)
    (opal:kill-main-event-loop-process))
  #+cmu (ext:disable-clx-event-handling opal::*default-x-display*)
  (setq *all-the-windows*
	(let (result)
	  (maphash #'(lambda (x-window opal-window)
		       (declare (ignore x-window))
		       (push opal-window result))
		   opal::*drawable-to-window-mapping*)
	  result))
  (setq *all-windows-which-have-been-closed* nil)
  ;;; Make all the windows invisible.
  (dolist (w *all-the-windows*) 
    (when (and (kr:g-value w :visible)
	       (kr:g-value w :drawable))
       (push w *all-windows-which-have-been-closed*)
       (kr:s-value w :visible nil) 
       (update w)))  ; generalized update
  ;;; Remove all connections to X from the text objects,
  ;;; (even those hidden in the :update-slots-values slot!)
  (dolist (txt (all-the-instances opal:text))
    (when (kr:g-cached-value txt :xfont)
      (xlib:close-font (kr:g-cached-value txt :xfont))
      (when (kr:g-cached-value txt :update-slots-values)
        (setf (aref (kr:g-cached-value txt :update-slots-values)
		    opal::*text-xfont*)
	      :closed))))
  (dolist (fnt (all-the-instances opal:font-from-file))
    (kr:s-value fnt :display-xfont-plist nil))
  ;;; Remove all connections to X from the window objects.
  (clrhash opal::*drawable-to-window-mapping*)
  (dolist (w *all-the-windows*)
    (kr:s-value w :drawable nil)
    (kr:s-value w :lineage nil)
    (kr:s-value w :already-initialized-border-widths nil)
    (kr:s-value w :event-mask nil)
    (when (kr:g-cached-value w :display-info)
      (kr:s-value w :display-info nil)))
  ;;; Clear all colors.
  (dotimes (n *colormap-index-table-size*)
    (setf (aref *colormap-index-table* n) 0))
  (setq *garnet-has-been-disconnected* T)
)

;;; This is a list of init routines that are to be called whenever
;;; reconnect-garnet is called.  This will be used by the gestures handler.
(defparameter *auxilliary-reconnect-routines* ())

(defun Reconnect-Garnet (&optional display-name)

  (unless *garnet-has-been-disconnected*
    (return-from reconnect-garnet))

  (opal::initialize-x11-values  ;; defined in defs.lisp
    (or display-name (get-full-display-name)))
  (kr:s-value opal:window :display opal::*default-x-display-name*)
  (opal::set-draw-functions)	      ;; defined in basics.lisp

  (let ((kr::*constants-disabled* T))
    (s-value opal:color :color-p *is-this-a-color-screen?*))

  (when *is-this-a-color-screen?*
    (let ((indices (xlib:alloc-color-cells opal::*default-x-colormap* 1)))
      (setq *first-allocatable-colormap-index* (car indices))
      (xlib:free-colors opal::*default-x-colormap* indices)))

  (dolist (c (all-the-instances opal:color))
    (get-value c :xcolor)
    (mark-as-changed c :xcolor)
    (g-value c :colormap-index))

  (let ((kr::*constants-disabled* T))
    (dolist (txt (all-the-instances opal:text))
      (let ((vals (g-cached-value txt :update-slots-values)))
        (when (and vals (eq (aref vals opal::*text-xfont*) :closed))
          (setf (aref vals opal::*text-xfont*)
            (s-value txt :xfont
	      (opal::font-to-xfont (g-value txt :font)
		     opal::*default-x-display*)))))))
      
  (dolist (f *auxilliary-reconnect-routines*)
    (funcall f))

  (dolist (w *all-windows-which-have-been-closed*)
    (unless (already-been-destroyed w)
      (kr:s-value w :visible t)))
  (dolist (w *all-the-windows*)
    (unless (or (already-been-destroyed w)
		(kr:g-value w :parent))
      (update w T)))

  ;; The menubar gadget requires that notice-items-changed be called here.
;  (dolist (g *gadgets-requiring-notice-items-changed*)
;    (dolist (m-bar (cdr (all-the-instances g)))
;       (funcall (g-value m-bar :notice-items-changed) m-bar)))

  #+cmu
  (ext:enable-clx-event-handling opal::*default-x-display*
                                 #'inter::default-event-handler)

  #-cmu
  (when *main-event-loop-process-was-halted*
    (opal:launch-main-event-loop-process))

  (setf (xlib:display-report-asynchronous-errors
	   opal::*default-x-display*)
        nil)

  (setq *garnet-has-been-disconnected* nil)

  t)


