/*
 * ARCload (c) 2005 Stanislaw Skowronek
 */

#include <arclib/arc.h>
#include <arclib/stddef.h>
#include <arclib/stdio.h>
#include <arclib/stdlib.h>
#include <arclib/types.h>
#include <arclib/string.h>

#include "block.h"
#include "detect.h"

#define READBLOCK	256

#define STRING_LEN	1024
#define CONFIG_FILE	"/arc.cf"
#define DEFAULT_KERNEL	"/vmlinux"

#define T_NONE		0
#define T_TOKEN		1
#define T_STRING	2
#define T_COMMENT	3
#define T_LEFT		4
#define T_RIGHT		5
#define T_SEMI		6
#define T_ESCAPE	7

#define S_NONE		0
#define S_IMAGE		1
#define S_APPEND	2
#define S_MATCH		3
#define S_NOMATCH	4
#define S_IGNORE	5
#define S_TERM		6
#define S_SYSIMG	7
#define S_DESCR		8
#define S_COMMENT	9

static char *token_names[] = {
	"NONE", 
	"TOKEN", 
	"STRING", 
	"COMMENT", 
	"'{'", 
	"'}'", 
	"';'", 
	"ESCAPE"
};

static char *s_state_names[] = {
	"NONE", 
	"IMAGE", 
	"APPEND", 
	"MATCH", 
	"NOMATCH", 
	"IGNORE", 
	"TERM", 
	"SYSIMG",
	"DESCR",
	"COMMENT"
};

static int is_space(char ch)
{
	if((ch == '\n') || (ch == ' ') || (ch == '\t') || (ch == '\r'))
		return 1;
	return 0;
}

static int token_bomb(int token, char *text, int s_state, char *image, char *expect)
{
	printf("Invalid token %s", token_names[token]);

	if(text)
		printf(": \"%s\"", text);

	printf(" in state %s", s_state_names[s_state]);

	if(expect)
		printf(" (expected %s)", expect);

	printf(".\n\r");
	image[0] = 0;
	return 1;
}

static int is_label(char *text, char **label, int label_count)
{
	int i;
	for(i=0;i<label_count;i++)
		if(!strcmp(text,label[i]))
			return 1;
	return 0;
}

static int rx_token(int token, char *text, int *s_state, char **label, int n_label, int *label_count,
		    char *image, int n_image, char **append, int n_append, 
		    char *append_space, int *append_space_ptr, int n_append_space, 
		    int *append_count, char *descr, char *lastl, int *s_level, int *s_ignore,
		    char *system)
{
	int len;
	switch(*s_state) {
	case S_NONE:
		if(*s_level) {
			if((token != T_STRING) && (token != T_RIGHT))
				return token_bomb(token, text, *s_state, image, "STRING or '}'");

			if(token == T_RIGHT) {
				(*s_level)--;
				break;
			}
		} else {
			if(token != T_STRING)
				return token_bomb(token, text, *s_state, image, "STRING");
		}

		if(!strcmp(text, "detect")) {
			if(!is_label("force", label, *label_count)) {
				printf("Detecting system parameters...\r\n");
				(*label_count) = 0;
				detectsystem(label,n_label,label_count);
				if(!*label_count) {
					printf("Detection failed.\r\n");
					return 1;
				}
			}
			*s_state = S_TERM;
			break;
		}

		if(!strcmp(text, "image")) {
			image[0] = 0;
			*s_state = S_IMAGE;
			break;
		}

		if(!strcmp(text, "description")) {
			*s_state = S_DESCR;
			break;
		}

		if(!strcmp(text, "append")) {
			*s_state = S_APPEND;
			break;
		}

		if(!strcmp(text, "comment")) {
			*s_state = S_COMMENT;
			break;
		}

		if(!strcmp(text, "options")) {
			*append_count = 1;
			*append_space_ptr = 0;

			*s_state = S_APPEND;
			break;
		}

		if(!*label_count) {
			strcpy(lastl,text);
			descr[0] = 0;
			*s_state = S_MATCH;
			break;
		}

		if(is_label(text, label, *label_count)) {
			*s_state = S_MATCH;
			break;
		}

		*s_state = S_NOMATCH;
		break;

	case S_IMAGE:
	case S_SYSIMG:
		if(token != T_STRING)
			return token_bomb(token, text, *s_state, image, "STRING");

		if((*s_state == S_IMAGE) && !strcmp(text, "system")) {
			strcpy(image, system);
			*s_state = S_SYSIMG;
			break;
		}

		if((*s_state == S_IMAGE) && !strcmp(text, "absolute")) {
			*s_state = S_SYSIMG;
			break;
		}

		if(strlen(text) >= (n_image - strlen(image))) {
			printf("Error: N_IMAGE exceeded.\n\r");
			image[0] = 0;
			return 1;
		}

		strcat(image, text);
		*s_state = S_TERM;

		break;

	case S_APPEND:
		if((token != T_STRING) && (token != T_SEMI))
			return token_bomb(token, text, *s_state, image, "STRING or ';'");

		if(token == T_SEMI) {
			*s_state = S_NONE;
			break;
		}

		if(*append_count >= n_append) {
			printf("Error: N_APPEND exceeded.\n\r");
			image[0] = 0;
			return 1;
		}

		append[*append_count] = (append_space + *append_space_ptr);
		(*append_count)++;
		len = (strlen(text) + 1);

		if((*append_space_ptr + len) > n_append_space) {
			printf("Error: N_APPEND_SPACE exceeded.\n\r");
			image[0] = 0;
			return 1;
		}

		strcpy((append_space + *append_space_ptr), text);
		(*append_space_ptr) += len;
		break;

	case S_MATCH:
		if(token != T_LEFT && token != T_STRING)
			return token_bomb(token, text, *s_state, image, "'{' or STRING");
		if(token == T_LEFT) {
			(*s_level)++;
			*s_state = S_NONE;
		}
		break;

	case S_NOMATCH:
		if(token != T_LEFT && token != T_STRING)
			return token_bomb(token, text, *s_state, image, "'{' or STRING");
		if(token == T_LEFT) {
			(*s_level)++;
			*s_ignore = 1;
			*s_state = S_IGNORE;
		} else {
			if(is_label(text, label, *label_count))
				*s_state = S_MATCH;
		}
		break;

	case S_IGNORE:
		if(token == T_LEFT)
			(*s_ignore)++;
		if(token == T_RIGHT) {
			(*s_ignore)--;
			if(!*s_ignore) {
				(*s_level)--;
				*s_state = S_NONE;
			}
		}
		break;

	case S_TERM:
		if(token != T_SEMI)
			return token_bomb(token, text, *s_state, image, "';'");

		*s_state = S_NONE;
		break;

	case S_DESCR:
		if(token != T_STRING)
			return token_bomb(token, text, *s_state, image, "STRING");
		strcpy(descr, text);

		if(!*label_count) {
			for(len = 0; len < *s_level; len++)
				printf("    ");
			printf("%s",lastl);
			len = 4*(8-*s_level)-strlen(lastl);
			while((len--)>0)
				printf(" ");
			printf("%s\n\r", descr);
		}

		*s_state = S_TERM;
		break;

	case S_COMMENT:
		if(token != T_STRING)
			return token_bomb(token, text, *s_state, image, "STRING");

		if(!*label_count)
			printf(text);

		*s_state = S_TERM;
		break;
	}
	return 0;
}

#define RX_TOKEN(a,b)								\
	do {									\
		if(rx_token(a, b, &s_state, label, n_label, label_count, image,	\
			    n_image, append, n_append, append_space,		\
			    &append_space_ptr, n_append_space, append_count,	\
			    descr, lastl, &s_level, &s_ignore, system)) {	\
			bclose(&file);						\
			return;							\
		}								\
	} while(0)
	

void parseconfig(char **label, int n_label, int *label_count, char *image,
		 int n_image, char **append, int n_append, char *append_space,
		 int n_append_space, int *append_count, char *system)
{
	BDEV file;
	char str[STRING_LEN], descr[STRING_LEN], lastl[STRING_LEN];
	char buf[READBLOCK], ch;
	int block,i,cnt = 0, append_space_ptr = 0;
	int t_state = T_NONE, s_state = S_NONE, s_level = 0, s_ignore = 0;

	image[0] = 0;
	append[0] = image;
	*append_count = 1;
	strcpy(str, system);
	strcat(str, CONFIG_FILE);

	if(bopen(&file, str)) {
		printf("Opening %s failed - using default configuration.\n\r", str);
		strcpy(image, system);
		strcat(image, DEFAULT_KERNEL);
		return;
	}

	if(!*label)
		printf("Available configurations:\n\r");

	do {
		if((block=bread(&file, READBLOCK, buf))<0) {
			printf("Read failed - using default configuration.\n\r");
			strcpy(image, system);
			strcat(image, DEFAULT_KERNEL);
			*append_count = 1;
			bclose(&file);
			return;
		}

		for(i = 0; i < block; i++) {
			ch = buf[i];
			switch(t_state) {
			case T_NONE:
				if(ch == '#') {
					t_state = T_COMMENT;
					break;
				}

				if(ch == ';') {
					RX_TOKEN(T_SEMI, NULL);
					break;
				}

				if(ch == '{') {
					RX_TOKEN(T_LEFT, NULL);
					break;
				}

				if(ch == '}') {
					RX_TOKEN(T_RIGHT, NULL);
					break;
				}

				if(ch == '"') {
					t_state = T_STRING;
					cnt = 0;
					break;
				}

				if(!is_space(ch)) {
					t_state = T_TOKEN;
					str[0] = ch;
					cnt = 1;
					break;
				}

				break;
			case T_TOKEN:
				if(is_space(ch)) {
					t_state = T_NONE;
					str[cnt] = 0;
					RX_TOKEN(T_STRING, str);
					break;
				}

				if(ch == '#') {
					t_state = T_COMMENT;
					str[cnt] = 0;
					RX_TOKEN(T_STRING, str);
					break;
				}

				if(ch == '{') {
					t_state = T_NONE;
					str[cnt] = 0;
					RX_TOKEN(T_STRING, str);
					RX_TOKEN(T_LEFT, NULL);
					break;
				}

				if(ch == '}') {
					t_state = T_NONE;
					str[cnt] = 0;
					RX_TOKEN(T_STRING, str);
					RX_TOKEN(T_RIGHT, NULL);
					break;
				}

				if(ch == ';') {
					t_state = T_NONE;
					str[cnt] = 0;
					RX_TOKEN(T_STRING, str);
					RX_TOKEN(T_SEMI, NULL);
					break;
				}

				str[cnt] = ch;
				cnt++;

				if(cnt >= STRING_LEN) {
					printf("Error: STRING_LEN exceeded.\n\r");
					image[0] = 0;
					bclose(&file);
					return;
				}

				break;

			case T_STRING:
				if(ch == '"') {
					t_state = T_NONE;
					str[cnt] = 0;
					RX_TOKEN(T_STRING, str);
					break;
				}

				if(ch == '\\') {
					t_state = T_ESCAPE;
					break;
				}

				str[cnt] = ch;
				cnt++;

				if(cnt >= STRING_LEN) {
					printf("Error: STRING_LEN exceeded.\n\r");
					image[0] = 0;
					bclose(&file);
					return;
				}

				break;

			case T_COMMENT:
				if((ch == '\n') || (ch == '\r'))
					t_state = T_NONE;
				break;

			case T_ESCAPE:
				if(ch == 'n')
					ch = '\n';

				if(ch == 'r')
					ch = '\r';

				if(ch == 't')
					ch = '\t';

				if(ch == 'a')
					ch = '\a';

				if(ch == 'b')
					ch = '\b';

				str[cnt] = ch;
				cnt++;

				if(cnt >= STRING_LEN) {
					printf("Error: STRING_LEN exceeded.\n\r");
					image[0] = 0;
					bclose(&file);
					return;
				}

				t_state = T_STRING;
				break;
			}
		}
	} while (block == READBLOCK);
	if(s_state != S_NONE || s_level) {
		printf("Premature end of file.\n\r");
		image[0] = 0;
		bclose(&file);
		return;
	}
	if(!*label) {
		image[0] = 0;
		bclose(&file);
		return;
	}
	if(!*image)
		printf("No configuration found for this label.\n\r");
	bclose(&file);
}

static int is_sep(char ch)
{
	if((ch == '\n') || (ch == ' ') || (ch == '\t') || (ch == '\r') || (ch == '(') || (ch == ')') || (ch == ','))
		return 1;
	return 0;
}

void parselabel(char *text,
		char **label, int n_label, int *label_count,
		char *label_space, int n_label_space)
{
	int nch = 0, lsp = 0;
	*label_count = 0;

	while(*text) {
		if(is_sep(*text)) {
			if(nch) {
				label_space[lsp + nch] = 0;
				lsp += nch + 1;
				nch = 0;
			}
		} else {
			if(!nch) {
				if(*label_count >= n_label) {
					printf("Error: N_LABEL exceeded.\n");
					return;
				}
				label[*label_count] = label_space + lsp;
				(*label_count)++;
			}
			if(lsp + nch + 1 >= n_label_space) {
				printf("Error: N_LABEL_SPACE exceeded.\n");
				(*label_count)--;
				return;
			}
			label_space[lsp + nch] = *text;
			nch++;
		}
		text++;
	}

	if(nch)
		label_space[lsp + nch] = 0;
}
