/*
 * wxCHTDecoder
 * Copyright (C) 2004,2008 emuWorks
 * http://games.technoplaza.net/
 *
 * This file is part of wxCHTDecoder.
 *
 * wxCHTDecoder is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wxCHTDecoder is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with wxCHTDecoder; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

// $Id: chtdecoder.cc,v 1.6 2008/12/18 01:24:13 jdratlif Exp $

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wx/wxprec.h>

#ifndef WX_PRECOMP
   #include <wx/wx.h>
#endif

#include <wx/cmdline.h>

#ifndef USE_MSYS_CONSOLE
    #ifdef _WIN32
        #include <windows.h>
    #endif
#endif

#include <iostream>
#include <fstream>
#include <vector>

#include <sys/stat.h>

#include "chtdecoder.hh"
#include "exceptions/invalidchtfileexception.hh"
#include "tools/chtfilereader.hh"
#include "view/chtframe.hh"

using namespace wxCHTDecoder;

// static variable initialization
const wxString *CHTDecoder::APP_NAME = new wxString(wxT("wxCHTDecoder"));
const wxString *CHTDecoder::APP_VERSION = new wxString(wxT("1.5a"));
const wxString *CHTDecoder::APP_COPYRIGHT =
    new wxString(wxT("Copyright (C) 2004,2008 emuWorks"));
const wxString *CHTDecoder::APP_URL =
    new wxString(wxT("http://games.technoplaza.net/"));

const wxSize *CHTDecoder::APP_SIZE = new wxSize(640, 480);

namespace {
    const wxCmdLineEntryDesc CMDLINE_DESC[] = {
        { 
            wxCMD_LINE_SWITCH, wxT("h"), wxT("help"),
            wxT("display usage information"), wxCMD_LINE_VAL_STRING,
            wxCMD_LINE_OPTION_HELP
        },
          
        { 
            wxCMD_LINE_SWITCH, wxT("v"), wxT("version"),
            wxT("display version information"), wxCMD_LINE_VAL_STRING,
            wxCMD_LINE_OPTION_HELP 
        },
          
        {
            wxCMD_LINE_SWITCH, wxT("f"), wxT("force-open"),
            wxT("force opening of possibly invalid .CHT files")
        },
        
        {
            wxCMD_LINE_PARAM, 0, 0, wxT(".CHT file(s)"),
            wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_MULTIPLE
        },
        
        {
            wxCMD_LINE_NONE
        }
    };
}

void CHTDecoder::runTextMode() {
    wxCmdLineParser parser(argc, argv);
    const wxString logo = (*APP_NAME) + wxT(' ') + (*APP_VERSION) + wxT('\n') +
        (*APP_COPYRIGHT) + wxT(' ') + (*APP_URL) + wxT('\n');

    parser.SetDesc(CMDLINE_DESC);
    parser.SetLogo(logo);

    if (parser.Parse() != 0) {
        return;
    }

    bool force = false;

    if (parser.Found(wxT("f")) || parser.Found(wxT("force-open"))) {
        force = true;
    }

    int chtfiles = parser.GetParamCount();

    if (chtfiles == 0) {
        return;
    }

    CHTFileReader &reader = CHTFileReader::getInstance();
    struct stat filestats;

    #ifndef USE_MSYS_CONSOLE
        #ifdef _WIN32
            // make sure we have a console since windows subsystem doesn't
            // create or attach one by default -- this breaks it in msys, of
            // course... so use --enable-msys if you want to use it in msys
            AllocConsole();
            freopen("CONOUT$", "w", stdout);
            freopen("CONOUT$", "w", stderr);
        #endif
    #endif

    for (unsigned int i = 0; i < parser.GetParamCount(); i++) {
        wxString filename = parser.GetParam(i);

        try {
            std::vector<CHTCode> &codes = reader.readCHTFile(filename, force);

            filename += wxT(".txt");

            std::ofstream *out = 0;
            std::ostream *output = &std::cout;

            if (stat(filename.mb_str(), &filestats) == 0) {
                std::cerr << "error: " << filename.mb_str() <<
                    " already exists. Will output to console instead.\n";
            } else {
                out = new std::ofstream(filename.mb_str());

                if (!(*out)) {
                    std::cerr << "error: unable to open " << filename.mb_str()
                        << " for writing. Will output to console instead.\n";

                    delete out;
                    out = 0;
                } else {
                    output = out;
                }
            }

            for (size_t j = 0; j < codes.size(); j++) {
                CHTCode &code = codes[j];

                wxString str;
                str.Printf(wxT("%2d: %s\n"), (j + 1), code.toString().c_str());

                (*output) << str.mb_str();
            }

            if (out) {
                std::cout << "Wrote the decoded cheats to "
                    << filename.mb_str() << '.' << '\n';

                out->close();
                delete out;
            }
        } catch (InvalidCHTFileException e) {
            std::cerr << "error: " << filename.mb_str()
                << " is not a valid .CHT file.\n";
        }
    }

    #ifndef USE_MSYS_CONSOLE
        #ifdef _WIN32
            system("pause");
        #endif
    #endif
}

bool CHTDecoder::OnInit() {
    if (argc > 1) {
        runTextMode();

        return false;
    }

    CHTFrame *frame = new CHTFrame(*APP_NAME + wxT(' ') + *APP_VERSION,
                                   wxPoint(0, 0), *APP_SIZE);

    frame->Centre();
    frame->Show(true);
    SetTopWindow(frame);

    return true;
}

// wx MACRO to create startup methods
IMPLEMENT_APP(CHTDecoder);

