//
/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.


             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 


             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.


             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.


             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.


             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.


             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 



*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */


// File:  mech_aes.c
//
// Mechanisms for AES
//

#include <string.h>            // for memcmp() et al
#include <stdlib.h>

#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"
#include "trace.h"


//
//
CK_RV
aes_ecb_encrypt( SESSION           *sess,
                 CK_BBOOL           length_only,
                 ENCR_DECR_CONTEXT *ctx,
                 CK_BYTE           *in_data,
                 CK_ULONG           in_data_len,
                 CK_BYTE           *out_data,
                 CK_ULONG          *out_data_len )
{
   OBJECT       *key       = NULL;
   CK_RV         rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len) {
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   return ckm_aes_ecb_encrypt(in_data, in_data_len,
			      out_data, out_data_len, key);
}


//
//
CK_RV
aes_ecb_decrypt( SESSION           *sess,
                 CK_BBOOL           length_only,
                 ENCR_DECR_CONTEXT *ctx,
                 CK_BYTE           *in_data,
                 CK_ULONG           in_data_len,
                 CK_BYTE           *out_data,
                 CK_ULONG          *out_data_len)
{
   OBJECT       *key       = NULL;
   CK_RV         rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   // CKM_DES3_ECB requires the input data to be an integral
   // multiple of the block size
   //
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc; 
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len) {
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   return ckm_aes_ecb_decrypt(in_data, in_data_len,
                               out_data, out_data_len, key);
}


//
//
CK_RV
aes_cbc_encrypt( SESSION           *sess,
                 CK_BBOOL           length_only,
                 ENCR_DECR_CONTEXT *ctx,
                 CK_BYTE           *in_data,
                 CK_ULONG           in_data_len,
                 CK_BYTE           *out_data,
                 CK_ULONG          *out_data_len)
{
   OBJECT       *key       = NULL;
   CK_RV         rc;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // CKM_DES3_CBC requires the input data to be an integral
   // multiple of the block size
   //
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len) {
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   return ckm_aes_cbc_encrypt(in_data, in_data_len, out_data, out_data_len,
				ctx->mech.pParameter, key);
}

//
//
CK_RV
aes_cbc_decrypt( SESSION            *sess,
                 CK_BBOOL            length_only,
                 ENCR_DECR_CONTEXT  *ctx,
                 CK_BYTE            *in_data,
                 CK_ULONG            in_data_len,
                 CK_BYTE            *out_data,
                 CK_ULONG           *out_data_len)
{
   OBJECT       *key       = NULL;
   CK_RV         rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // CKM_DES3_CBC requires the input data to be an integral
   // multiple of the block size
   //
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len) {
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   return ckm_aes_cbc_decrypt(in_data, in_data_len, out_data, out_data_len,
                                ctx->mech.pParameter, key);
}


//
//
CK_RV
aes_cbc_pad_encrypt( SESSION           *sess,
                     CK_BBOOL           length_only,
                     ENCR_DECR_CONTEXT *ctx,
                     CK_BYTE           *in_data,
                     CK_ULONG           in_data_len,
                     CK_BYTE           *out_data,
                     CK_ULONG          *out_data_len)
{
   OBJECT       *key       = NULL;
   CK_BYTE      *clear     = NULL;
   CK_ULONG      padded_len;
   CK_RV         rc;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // DES3-CBC-PAD has no input length requirements
   //

   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   // compute the output length, accounting for padding
   //
   padded_len = AES_BLOCK_SIZE * (in_data_len / AES_BLOCK_SIZE + 1);

   if (length_only == TRUE) {
      *out_data_len = padded_len;
      return CKR_OK;
   }

   if (*out_data_len < padded_len) {
      *out_data_len = padded_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   clear = (CK_BYTE *)malloc( padded_len );
   if (!clear){
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      return CKR_HOST_MEMORY;
   }
   memcpy( clear, in_data, in_data_len );

   add_pkcs_padding( clear + in_data_len,
                     AES_BLOCK_SIZE,
                     in_data_len,
                     padded_len );

   rc = ckm_aes_cbc_encrypt(clear, padded_len, out_data, out_data_len,
                             ctx->mech.pParameter, key);

   free( clear );
   return rc;
}


//
//
CK_RV
aes_cbc_pad_decrypt( SESSION            *sess,
                     CK_BBOOL            length_only,
                     ENCR_DECR_CONTEXT  *ctx,
                     CK_BYTE            *in_data,
                     CK_ULONG            in_data_len,
                     CK_BYTE            *out_data,
                     CK_ULONG           *out_data_len)
{
   OBJECT       *key       = NULL;
   CK_BYTE      *clear     = NULL;
   CK_ULONG      padded_len;
   CK_RV         rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   //
   // no need to validate the input length since we'll pad as necessary
   //

   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   // we're decrypting so even with CBC-PAD, we should have an integral
   // number of block to decrypt
   //
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   // the amount of cleartext after stripping the padding will actually be less
   // than the input bytes...
   //
   padded_len = in_data_len;

   if (length_only == TRUE) {
      *out_data_len = padded_len;
      return CKR_OK;
   }

   clear = (CK_BYTE *)malloc( padded_len );
   if (!clear){
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      return CKR_HOST_MEMORY;
   }
   rc = ckm_aes_cbc_decrypt(in_data, in_data_len, clear, &padded_len,
                             ctx->mech.pParameter, key);

   if (rc == CKR_OK) {
      strip_pkcs_padding( clear, padded_len, out_data_len );
      memcpy( out_data, clear, *out_data_len );
   }

   free( clear );
   return rc;
}
//
//
CK_RV
aes_ctr_encrypt( SESSION           *sess,
                 CK_BBOOL           length_only,
		 ENCR_DECR_CONTEXT *ctx,
                 CK_BYTE           *in_data,
                 CK_ULONG           in_data_len,
                 CK_BYTE           *out_data,
                 CK_ULONG 	   *out_data_len )
{
   OBJECT *key = NULL;
   CK_RV rc;
   CK_AES_CTR_PARAMS *aesctr = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
       return CKR_OK;
   }

   if (*out_data_len < in_data_len) {
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   aesctr = (CK_AES_CTR_PARAMS *)ctx->mech.pParameter;

   return ckm_aes_ctr_encrypt(in_data, in_data_len, out_data, out_data_len,
                              (CK_BYTE *)aesctr->cb,
			      (CK_ULONG)aesctr->ulCounterBits, key);
}

//
//
CK_RV
aes_ctr_decrypt( SESSION           *sess,
                 CK_BBOOL           length_only,
		 ENCR_DECR_CONTEXT *ctx,
                 CK_BYTE           *in_data,
                 CK_ULONG           in_data_len,
                 CK_BYTE           *out_data,
                 CK_ULONG 	   *out_data_len )
{
   OBJECT *key = NULL;
   CK_RV rc;
   CK_AES_CTR_PARAMS *aesctr = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (in_data_len % AES_BLOCK_SIZE != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
       return CKR_OK;
   }

   if (*out_data_len < in_data_len) {
      *out_data_len = in_data_len;
       TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
       return CKR_BUFFER_TOO_SMALL;
   }

   aesctr = (CK_AES_CTR_PARAMS *)ctx->mech.pParameter;

   return ckm_aes_ctr_decrypt(in_data, in_data_len, out_data, out_data_len,
                              (CK_BYTE *)aesctr->cb,
			      (CK_ULONG)aesctr->ulCounterBits, key);
}

//
//
CK_RV
aes_ecb_encrypt_update( SESSION           *sess,
                        CK_BBOOL           length_only,
                        ENCR_DECR_CONTEXT *ctx,
                        CK_BYTE           *in_data,
                        CK_ULONG           in_data_len,
                        CK_BYTE           *out_data,
                        CK_ULONG          *out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * clear     = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad arguments\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }
      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      //
      remain  = (total % AES_BLOCK_SIZE);
      out_len = (total - remain);     // should always be at least 1 block

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      rc = object_mgr_find_in_map_nocache( ctx->key, &key );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      clear = (CK_BYTE *)malloc( out_len );
      if (!clear){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous encryption operation first
      //
      memcpy( clear,                context->data, context->len );
      memcpy( clear + context->len, in_data,       out_len - context->len );

      rc = ckm_aes_ecb_encrypt(clear, out_len, out_data, out_data_len, key);
      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // update the context buffer.  we already used the buffer's current
         // contents so we completely overwrite it
         //
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );

         context->len = remain;
      }

      free( clear );
      return rc;
   }
}


//
//
CK_RV
aes_ecb_decrypt_update( SESSION           *sess,
                        CK_BBOOL           length_only,
                        ENCR_DECR_CONTEXT *ctx,
                        CK_BYTE           *in_data,
                        CK_ULONG           in_data_len,
                        CK_BYTE           *out_data,
                        CK_ULONG          *out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * cipher    = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      //
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      rc = object_mgr_find_in_map_nocache( ctx->key, &key );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous decryption operation first
      //
      memcpy( cipher,                context->data, context->len );
      memcpy( cipher + context->len, in_data,       out_len - context->len );

      rc = ckm_aes_ecb_decrypt(cipher, out_len, out_data, out_data_len, key);
      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // copy the remaining 'new' input data to the context buffer
         //
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      }

      free( cipher );
      return rc;
   }
}


//
//
CK_RV
aes_cbc_encrypt_update( SESSION           *sess,
                        CK_BBOOL           length_only,
                        ENCR_DECR_CONTEXT *ctx,
                        CK_BYTE           *in_data,
                        CK_ULONG           in_data_len,
                        CK_BYTE           *out_data,
                        CK_ULONG          *out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * clear     = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      //
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      rc = object_mgr_find_in_map_nocache( ctx->key, &key );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      // these buffers need to be longword aligned
      //
      clear  = (CK_BYTE *)malloc( out_len );
      if (!clear){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous encryption operation first
      //
      memcpy( clear,                context->data, context->len );
      memcpy( clear + context->len, in_data,       out_len - context->len );

      rc = ckm_aes_cbc_encrypt(clear, out_len, out_data, out_data_len,
                                ctx->mech.pParameter, key);

      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // the new init_v is the last encrypted data block
         //
         memcpy( ctx->mech.pParameter, out_data + (*out_data_len - AES_BLOCK_SIZE), AES_BLOCK_SIZE );

         // copy the remaining 'new' input data to the context buffer
         //
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      }

      free( clear );
      return rc;
   }
}


//
//
CK_RV
aes_cbc_decrypt_update( SESSION           *sess,
                        CK_BBOOL           length_only,
                        ENCR_DECR_CONTEXT *ctx,
                        CK_BYTE           *in_data,
                        CK_ULONG           in_data_len,
                        CK_BYTE           *out_data,
                        CK_ULONG          *out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * cipher    = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   total = context->len + in_data_len;

   if (total < AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      //
      remain  = total % AES_BLOCK_SIZE;
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      rc = object_mgr_find_in_map_nocache( ctx->key, &key );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      // these buffers need to be longword aligned
      //
      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous decryption operation first
      //
      memcpy( cipher,                context->data, context->len );
      memcpy( cipher + context->len, in_data,       out_len - context->len );

      rc = ckm_aes_cbc_decrypt(cipher, out_len, out_data, out_data_len,
                                ctx->mech.pParameter, key);

      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // the new init_v is the last input data block
         //
         memcpy( ctx->mech.pParameter, cipher + (out_len - AES_BLOCK_SIZE), AES_BLOCK_SIZE );

         // copy the remaining 'new' input data to the context buffer
         //
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );

         context->len = remain;
      }

      free( cipher );
      return rc;
   }
}


//
//
CK_RV
aes_cbc_pad_encrypt_update( SESSION           *sess,
                            CK_BBOOL           length_only,
                            ENCR_DECR_CONTEXT *ctx,
                            CK_BYTE           *in_data,
                            CK_ULONG           in_data_len,
                            CK_BYTE           *out_data,
                            CK_ULONG          *out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * clear     = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   total     = (context->len + in_data_len);

   // note, this is subtly different from the other encrypt update routines
   //
   if (total <= AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;  // out_len is a multiple of DES_BLOCK_SIZE

      if (remain == 0) {
         remain     = AES_BLOCK_SIZE;
         out_len -= AES_BLOCK_SIZE;
      }

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      // at this point, we should have:
      //    1) remain != 0
      //    2) out_len != 0
      //
      rc = object_mgr_find_in_map_nocache( ctx->key, &key );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      // these buffers need to be longword aligned
      //
      clear = (CK_BYTE *)malloc( out_len );
      if (!clear){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous encryption operation first
      //
      memcpy( clear,                context->data, context->len );
      memcpy( clear + context->len, in_data,       out_len - context->len );

      //
      // we don't do padding during the update
      //
      rc = ckm_aes_cbc_encrypt(clear, out_len, out_data, out_data_len,
                                ctx->mech.pParameter, key);

      if (rc == CKR_OK) {
         // the new init_v is the last encrypted data block
         //
         memcpy( ctx->mech.pParameter, out_data + (*out_data_len - AES_BLOCK_SIZE), AES_BLOCK_SIZE );

         // copy the remaining 'new' input data to the temporary space
         //
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      }

      free( clear );
      return rc;
   }
}


//
//
CK_RV
aes_cbc_pad_decrypt_update( SESSION           *sess,
                            CK_BBOOL           length_only,
                            ENCR_DECR_CONTEXT *ctx,
                            CK_BYTE           *in_data,
                            CK_ULONG           in_data_len,
                            CK_BYTE           *out_data,
                            CK_ULONG          *out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * cipher    = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   // note, this is subtly different from the other decrypt update routines
   //
   if (total <= AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block + 1 byte
      //
      remain  = total % AES_BLOCK_SIZE;
      out_len = total - remain;

      if (remain == 0) {
         remain     = AES_BLOCK_SIZE;
         out_len -= AES_BLOCK_SIZE;
      }

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      // at this point, we should have:
      //    1) remain != 0
      //    2) out_len != 0
      //
      rc = object_mgr_find_in_map_nocache( ctx->key, &key );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      // these buffers need to be longword aligned
      //
      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous decryption operation first
      //
      memcpy( cipher,                context->data, context->len );
      memcpy( cipher + context->len, in_data,       out_len - context->len );

      rc = ckm_aes_cbc_decrypt(cipher, out_len, out_data, out_data_len,
                                ctx->mech.pParameter, key);

      if (rc == CKR_OK) {
         // the new init_v is the last input data block
         //
         memcpy( ctx->mech.pParameter, cipher + (out_len - AES_BLOCK_SIZE), AES_BLOCK_SIZE );

         // copy the remaining 'new' input data to the temporary space
         //
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      }

      free( cipher );
      return rc;
   }
}
//
//
CK_RV
aes_ctr_encrypt_update( SESSION                *sess,
                        CK_BBOOL 		 length_only,
			ENCR_DECR_CONTEXT 	*ctx,
                        CK_BYTE                 *in_data,
                        CK_ULONG 		 in_data_len,
                        CK_BYTE                 *out_data,
			CK_ULONG 		*out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * clear     = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;
   CK_AES_CTR_PARAMS *aesctr = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx -> context;
   total = (context->len + in_data_len);
   if( total < AES_BLOCK_SIZE){
     if(length_only == FALSE){
        memcpy( context->data + context->len, in_data, in_data_len);
	context->len += in_data_len;
     }
     *out_data_len = 0;
     return CKR_OK;
   }
   else
   {
     // we atleast have 1 block
     remain = (total % AES_BLOCK_SIZE);
     out_len = total -remain;
     if (length_only == TRUE){
	*out_data_len = out_len;
	 return CKR_OK;
     }
     rc = object_mgr_find_in_map_nocache(ctx->key, &key);
     if(rc != CKR_OK){
	TRACE_ERROR("Failed to find specified object.\n");
	return rc;
     }

     //these buffers need to be longword aligned
     clear = (CK_BYTE*) malloc (out_len);
     if (!clear){
	TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
        return CKR_HOST_MEMORY;
     }
     //copy all the leftover data  from the previous encryption operation first
     memcpy ( clear,		     context->data, context->len);
     memcpy ( clear+context->len, in_data, out_len - context->len);
     aesctr = (CK_AES_CTR_PARAMS *)ctx->mech.pParameter;
     rc = ckm_aes_ctr_encrypt(clear, out_len, out_data, out_data_len,
                               (CK_BYTE *)aesctr->cb,
			       (CK_ULONG)aesctr->ulCounterBits, key);
     if (rc == CKR_OK){
	*out_data_len = out_len;
	// copy the remaining 'new' input data to the context buffer
        if (remain != 0)
           memcpy( context->data, in_data + (in_data_len - remain), remain );
	context->len = remain;
      }

      free( clear );
      return rc;
   }
}
//
//
CK_RV
aes_ctr_decrypt_update( SESSION                 *sess,
                        CK_BBOOL 		 length_only,
			ENCR_DECR_CONTEXT 	*ctx,
                        CK_BYTE                 *in_data,
                        CK_ULONG 		 in_data_len,
                        CK_BYTE                 *out_data,
                        CK_ULONG 		*out_data_len )
{
   AES_CONTEXT  * context   = NULL;
   OBJECT       * key       = NULL;
   CK_BYTE      * clear     = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;
   CK_AES_CTR_PARAMS *aesctr = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx -> context;
   total = (context->len + in_data_len);
   if( total < AES_BLOCK_SIZE){
     if(length_only == FALSE){
       memcpy( context->data + context->len, in_data, in_data_len);
       context->len += in_data_len;
     }
     *out_data_len = 0;
     return CKR_OK;
   }
   else{
     // we atleast have 1 block
     remain = (total % AES_BLOCK_SIZE);
     out_len = total -remain;
     if (length_only == TRUE){
	*out_data_len = out_len;
	return CKR_OK;
     }
     rc = object_mgr_find_in_map_nocache(ctx->key, &key);
     if(rc != CKR_OK){
       TRACE_ERROR("Failed to find specified object.\n");
       return rc;
     }

     //these buffers need to be longword aligned
     clear = (CK_BYTE*) malloc (out_len);
     if (!clear){
	TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
	return CKR_HOST_MEMORY;
     }
     //copy all the leftover data  from the previous encryption operation first
     memcpy ( clear,		     context->data, context->len);
     memcpy ( clear+context->len, in_data, out_len - context->len);
     aesctr = (CK_AES_CTR_PARAMS *)ctx->mech.pParameter;
     rc = ckm_aes_ctr_decrypt(clear, out_len, out_data, out_data_len,
                              (CK_BYTE *)aesctr->cb,
			      (CK_ULONG)aesctr->ulCounterBits, key);
     if (rc == CKR_OK){
	*out_data_len = out_len;
	// copy the remaining 'new' input data to the context buffer
        if (remain != 0)
           memcpy( context->data, in_data + (in_data_len - remain), remain );
        context->len = remain;
     }

     free( clear );
     return rc;
     }
}
//
//
CK_RV
aes_ecb_encrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   AES_CONTEXT *context   = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //
   if (length_only)
      context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   // DES3-ECB does no padding so there had better not be
   // any data in the context buffer.  if there is it means
   // that the overall data length was not a multiple of the blocksize
   //
   if (context->len != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   *out_data_len = 0;
   return CKR_OK;
}


//
//
CK_RV
aes_ecb_decrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   AES_CONTEXT *context   = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //
   if (length_only)
      context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   // DES3-ECB does no padding so there had better not be
   // any data in the context buffer.  if there is it means
   // that the overall data length was not a multiple of the blocksize
   //
   if (context->len != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   *out_data_len = 0;
   return CKR_OK;
}


//
//
CK_RV
aes_cbc_encrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   AES_CONTEXT *context   = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //
   if (length_only)
      context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   // DES3-CBC does no padding so there had better not be
   // any data in the context buffer.  if there is it means
   // that the overall data length was not a multiple of the blocksize
   //
   if (context->len != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   *out_data_len = 0;
   return CKR_OK;
}


//
//
CK_RV
aes_cbc_decrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   AES_CONTEXT *context   = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //
   if (length_only)
      context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   if (context->len != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   *out_data_len = 0;
   return CKR_OK;
}


//
//
CK_RV
aes_cbc_pad_encrypt_final( SESSION           *sess,
                           CK_BBOOL           length_only,
                           ENCR_DECR_CONTEXT *ctx,
                           CK_BYTE           *out_data,
                           CK_ULONG          *out_data_len )
{
   AES_CONTEXT    *context   = NULL;
   OBJECT         *key       = NULL;
   CK_BYTE         clear[2*AES_BLOCK_SIZE];
   CK_ULONG        out_len;
   CK_RV           rc;


   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   context = (AES_CONTEXT *)ctx->context;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate one block of output
   //    if a full block is stored, we generate two blocks of output (one pad block)
   //
   if (context->len == AES_BLOCK_SIZE)
      out_len = 2 * AES_BLOCK_SIZE;
   else
      out_len = AES_BLOCK_SIZE;

   if (length_only == TRUE) {
      *out_data_len = out_len;
      return CKR_OK;
   }
   else {
      memcpy( clear, context->data, context->len );

      add_pkcs_padding( clear + context->len,
                        AES_BLOCK_SIZE,
                        context->len,
                        out_len );

      rc = ckm_aes_cbc_encrypt(clear, out_len, out_data, out_data_len,
                                ctx->mech.pParameter, key);
      
      return rc;
   }
}


//
//
CK_RV
aes_cbc_pad_decrypt_final( SESSION           *sess,
                           CK_BBOOL           length_only,
                           ENCR_DECR_CONTEXT *ctx,
                           CK_BYTE           *out_data,
                           CK_ULONG          *out_data_len )
{
   AES_CONTEXT    *context   = NULL;
   OBJECT         *key       = NULL;
   CK_BYTE         clear[AES_BLOCK_SIZE];
   CK_ULONG        out_len;
   CK_RV           rc;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   context = (AES_CONTEXT *)ctx->context;

   // there had better be a full block in the context buffer
   //
   if (context->len != AES_BLOCK_SIZE){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   // we don't know a priori how much data we'll be returning.  we won't
   // know until after we decrypt it and strip the padding.  it's possible
   // that we'll return nothing (the final block might be a padding block).
   //
   out_len = AES_BLOCK_SIZE;  // upper bound on what we'll return

   if (length_only == TRUE) {
      *out_data_len = out_len;
      return CKR_OK;
   }
   else {
      rc = ckm_aes_cbc_decrypt(context->data, AES_BLOCK_SIZE, clear, &out_len,
                                 ctx->mech.pParameter, key);

      if (rc == CKR_OK) {
         strip_pkcs_padding( clear, out_len, &out_len );

         if (out_len != 0)
            memcpy( out_data, clear, out_len );

         *out_data_len = out_len;
      }

      return rc;
   }
}

//
//
CK_RV
aes_ctr_encrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   AES_CONTEXT *context   = NULL;
   CK_AES_CTR_PARAMS *aesctr = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //
   if (length_only)
      context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   // DES3-CBC does no padding so there had better not be
   // any data in the context buffer.  if there is it means
   // that the overall data length was not a multiple of the blocksize
   //
   if (context->len != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   aesctr = (CK_AES_CTR_PARAMS *)ctx->mech.pParameter;
   //to check that the counter buffer doesnot overflow
   if ( ((CK_ULONG)aesctr->ulCounterBits) > ((CK_ULONG)aesctr->ulCounterBits+1) )
   {
	TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
	return CKR_DATA_LEN_RANGE;
   }

   *out_data_len = 0;
   return CKR_OK;
}

//
//
CK_RV
aes_ctr_decrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   AES_CONTEXT *context   = NULL;
   CK_AES_CTR_PARAMS *aesctr = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //
   if (length_only)
      context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   // DES3-CBC does no padding so there had better not be
   // any data in the context buffer.  if there is it means
   // that the overall data length was not a multiple of the blocksize
   //
   if (context->len != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   aesctr = (CK_AES_CTR_PARAMS *)ctx->mech.pParameter;
   //to check that the counter buffer doesnot overflow
   if ( ((CK_ULONG)aesctr->ulCounterBits) > ((CK_ULONG)aesctr->ulCounterBits+1) )
   {
	TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
	return CKR_DATA_LEN_RANGE;
   }
   *out_data_len = 0;
   return CKR_OK;
}

CK_RV
aes_ofb_encrypt( SESSION           * sess,
                 CK_BBOOL            length_only,
                 ENCR_DECR_CONTEXT * ctx,
                 CK_BYTE           * in_data,
                 CK_ULONG            in_data_len,
                 CK_BYTE           * out_data,
                 CK_ULONG          * out_data_len)
{
   CK_ULONG           rc;
   OBJECT           * key_obj  = NULL;

   if (!sess || !ctx || !in_data || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   rc = token_specific.t_aes_ofb(in_data, in_data_len, out_data,
                              key_obj, ctx->mech.pParameter, 1);

   if (rc != CKR_OK)
	TRACE_DEVEL("Token specific aes ofb encrypt failed.\n");
   return rc;
}

CK_RV
aes_ofb_encrypt_update( SESSION              * sess,
                        CK_BBOOL               length_only,
                        ENCR_DECR_CONTEXT    * ctx,
                        CK_BYTE              * in_data,
                        CK_ULONG               in_data_len,
                        CK_BYTE              * out_data,
                        CK_ULONG             * out_data_len)
{
   AES_CONTEXT  * context   = NULL;
   CK_BYTE      * cipher    = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;
   OBJECT       * key_obj  = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      if (*out_data_len < out_len){
         TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
         return CKR_BUFFER_TOO_SMALL;
      }

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous encryption operation first
      memcpy( cipher,                    context->data, context->len );
      memcpy( cipher + context->len, in_data, out_len - context->len );

      rc = token_specific.t_aes_ofb(cipher, out_len, out_data,
                              key_obj, ctx->mech.pParameter, 1);

      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // copy the remaining 'new' input data to the context buffer
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      } else
	    TRACE_DEVEL("Token specific aes ofb encrypt failed.\n");
      free( cipher );
      return rc;
   }
}

CK_RV
aes_ofb_encrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   OBJECT      *key_obj  = NULL;
   AES_CONTEXT *context  = NULL;
   CK_RV        rc;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate same length of output data
   //    if no data stored, no data can be returned (length zero)

   if (length_only == TRUE) {
      *out_data_len = context->len;
      return CKR_OK;
   }
   else {
      if (context->len == 0) {
         *out_data_len = 0;
         return CKR_OK;
      }

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
         TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      rc = token_specific.t_aes_ofb(context->data, context->len, out_data,
                              key_obj, ctx->mech.pParameter, 1);

      if (rc != CKR_OK)
	 TRACE_DEVEL("Token specific aes ofb encrypt failed.\n");

      *out_data_len = context->len;
      return rc;
   }
}

CK_RV
aes_ofb_decrypt( SESSION           * sess,
                 CK_BBOOL            length_only,
                 ENCR_DECR_CONTEXT * ctx,
                 CK_BYTE           * in_data,
                 CK_ULONG            in_data_len,
                 CK_BYTE           * out_data,
                 CK_ULONG          * out_data_len )
{
   CK_ULONG          rc;
   OBJECT           *key_obj  = NULL;

   if (!sess || !ctx || !in_data || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   if (length_only == TRUE) {
       *out_data_len = in_data_len;
       return CKR_OK;
   }

   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   rc = token_specific.t_aes_ofb(in_data, in_data_len, out_data,
                              key_obj, ctx->mech.pParameter, 0);

   if (rc != CKR_OK)
	TRACE_DEVEL("Token specific aes ofb decrypt failed.\n");
   return rc;
}

CK_RV
aes_ofb_decrypt_update( SESSION              * sess,
                        CK_BBOOL               length_only,
                        ENCR_DECR_CONTEXT    * ctx,
                        CK_BYTE              * in_data,
                        CK_ULONG               in_data_len,
                        CK_BYTE              * out_data,
                        CK_ULONG             * out_data_len)
{
   AES_CONTEXT  * context  = NULL;
   CK_BYTE      * cipher   = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;
   OBJECT       * key_obj  = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      if (*out_data_len < out_len){
         TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
         return CKR_BUFFER_TOO_SMALL;
      }

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous decryption operation first
      memcpy( cipher,                    context->data, context->len );
      memcpy( cipher + context->len, in_data, out_len - context->len );

      rc = token_specific.t_aes_ofb(cipher, out_len, out_data,
                              key_obj, ctx->mech.pParameter, 0);

      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // copy the remaining 'new' input data to the context buffer
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      } else
	 TRACE_DEVEL("Token specific aes ofb decrypt failed.\n");
      free( cipher );
      return rc;
   }
}

CK_RV
aes_ofb_decrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len )
{
   OBJECT      * key_obj = NULL;
   AES_CONTEXT * context = NULL;
   CK_RV         rc;

   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   // satisfy the compiler
   //if (length_only)
   //   context = NULL;

   context = (AES_CONTEXT *)ctx->context;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate same length of output data
   //    if no data stored, no data can be returned (length zero)

   if (length_only == TRUE) {
      *out_data_len = context->len;
      return CKR_OK;
   }
   else {
      if (context->len == 0) {
         *out_data_len = 0;
         return CKR_OK;
      }

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      rc = token_specific.t_aes_ofb(context->data, context->len, out_data,
                                           key_obj, ctx->mech.pParameter, 0);

      if (rc != CKR_OK)
         TRACE_DEVEL("Token specific aes ofb decrypt failed.\n");

      *out_data_len = context->len;
      return rc;
   }
}

CK_RV
aes_cfb_encrypt( SESSION           * sess,
                 CK_BBOOL            length_only,
                 ENCR_DECR_CONTEXT * ctx,
                 CK_BYTE           * in_data,
                 CK_ULONG            in_data_len,
                 CK_BYTE           * out_data,
                 CK_ULONG          * out_data_len,
                 CK_ULONG            cfb_len )
{
   CK_ULONG          rc;
   OBJECT           *key_obj = NULL;

   if (!sess || !ctx || !in_data || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   rc = token_specific.t_aes_cfb(in_data, in_data_len, out_data,
                     key_obj, ctx->mech.pParameter, cfb_len, 1);

   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes cfb encrypt failed.\n");
   return rc;
}

CK_RV
aes_cfb_encrypt_update( SESSION              * sess,
                        CK_BBOOL               length_only,
                        ENCR_DECR_CONTEXT    * ctx,
                        CK_BYTE              * in_data,
                        CK_ULONG               in_data_len,
                        CK_BYTE              * out_data,
                        CK_ULONG             * out_data_len,
                        CK_ULONG               cfb_len)
{
   AES_CONTEXT  * context  = NULL;
   CK_BYTE      * cipher   = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;
   OBJECT       * key_obj  = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < cfb_len) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      remain  = (total % cfb_len);
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      if (*out_data_len < out_len){
         TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
         return CKR_BUFFER_TOO_SMALL;
      }

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous encryption operation first
      memcpy( cipher,                    context->data, context->len );
      memcpy( cipher + context->len, in_data, out_len - context->len );

      rc = token_specific.t_aes_cfb(cipher, out_len, out_data,
                              key_obj, ctx->mech.pParameter, cfb_len, 1);

      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // copy the remaining 'new' input data to the context buffer
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      } else
         TRACE_DEVEL("Token specific aes cfb encrypt failed.\n");
      free( cipher );
      return rc;
   }
}

CK_RV
aes_cfb_encrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len,
                       CK_ULONG           cfb_len)
{
   OBJECT      *key_obj  = NULL;
   AES_CONTEXT *context  = NULL;
   CK_RV        rc;

   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate same length of output data
   //    if no data stored, no data can be returned (length zero)

   if (context->len == 0) {
      *out_data_len = 0;
      return CKR_OK;
   }

   if (length_only == TRUE) {
      *out_data_len = context->len;
      return CKR_OK;
   }
   else {
      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      rc = token_specific.t_aes_cfb(context->data, context->len, out_data,
                              key_obj, ctx->mech.pParameter, cfb_len, 1);

      if (rc != CKR_OK)
         TRACE_DEVEL("Token specific aes cfb encrypt failed.\n");

      *out_data_len = context->len;
      return rc;
   }
}

CK_RV
aes_cfb_decrypt( SESSION           * sess,
                 CK_BBOOL            length_only,
                 ENCR_DECR_CONTEXT * ctx,
                 CK_BYTE           * in_data,
                 CK_ULONG            in_data_len,
                 CK_BYTE           * out_data,
                 CK_ULONG          * out_data_len,
                 CK_ULONG            cfb_len )
{
   CK_ULONG          rc;
   OBJECT           *key_obj = NULL;

   if (!sess || !ctx || !in_data || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   if (length_only == TRUE) {
      *out_data_len = in_data_len;
      return CKR_OK;
   }

   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   rc = token_specific.t_aes_cfb(in_data, in_data_len, out_data,
                     key_obj, ctx->mech.pParameter, cfb_len, 0);

   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes cfb decrypt failed.\n");
   return rc;
}

CK_RV
aes_cfb_decrypt_update( SESSION              * sess,
                        CK_BBOOL               length_only,
                        ENCR_DECR_CONTEXT    * ctx,
                        CK_BYTE              * in_data,
                        CK_ULONG               in_data_len,
                        CK_BYTE              * out_data,
                        CK_ULONG             * out_data_len,
                        CK_ULONG               cfb_len)
{
   AES_CONTEXT  * context  = NULL;
   CK_BYTE      * cipher   = NULL;
   CK_ULONG       total, remain, out_len;
   CK_RV          rc;
   OBJECT       * key_obj  = NULL;

   if (!sess || !ctx || !out_data_len) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (AES_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < cfb_len) {
      if (length_only == FALSE) {
         memcpy( context->data + context->len, in_data, in_data_len );
         context->len += in_data_len;
      }

      *out_data_len = 0;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      remain  = (total % cfb_len);
      out_len = total - remain;

      if (length_only == TRUE) {
         *out_data_len = out_len;
         return CKR_OK;
      }

      if (*out_data_len < out_len){
         TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
         return CKR_BUFFER_TOO_SMALL;
      }

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous decryption operation first
      memcpy( cipher,                    context->data, context->len );
      memcpy( cipher + context->len, in_data, out_len - context->len );

      rc = token_specific.t_aes_cfb(cipher, out_len, out_data,
                              key_obj, ctx->mech.pParameter, cfb_len, 0);

      if (rc == CKR_OK) {
         *out_data_len = out_len;

         // copy the remaining 'new' input data to the context buffer
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      }
      else
         TRACE_DEVEL("Token specific aes cfb decrypt failed.\n");
      free( cipher );
      return rc;
   }
}

CK_RV
aes_cfb_decrypt_final( SESSION           *sess,
                       CK_BBOOL           length_only,
                       ENCR_DECR_CONTEXT *ctx,
                       CK_BYTE           *out_data,
                       CK_ULONG          *out_data_len,
                       CK_ULONG           cfb_len)
{
   OBJECT      *key_obj  = NULL;
   AES_CONTEXT *context  = NULL;
   CK_RV        rc;

   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_CONTEXT *)ctx->context;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate same length of output data
   //    if no data stored, no data can be returned (length zero)

   if (context->len == 0) {
      *out_data_len = 0;
      return CKR_OK;
   }

   if (length_only == TRUE) {
      *out_data_len = context->len;
      return CKR_OK;
   }
   else {
      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      rc = token_specific.t_aes_cfb(context->data, context->len, out_data,
                              key_obj, ctx->mech.pParameter, cfb_len, 0);

      if (rc != CKR_OK)
         TRACE_DEVEL("Token specific aes cfb decrypt failed.\n");

      *out_data_len = context->len;
      return rc;
   }
}


CK_RV
aes_mac_sign( SESSION              * sess,
              CK_BBOOL               length_only,
              SIGN_VERIFY_CONTEXT  * ctx,
              CK_BYTE              * in_data,
              CK_ULONG               in_data_len,
              CK_BYTE              * out_data,
              CK_ULONG             * out_data_len)
{
 CK_ULONG         rc;
 OBJECT           * key_obj  = NULL;
 CK_ULONG         mac_len;

   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   if (ctx->mech.pParameter)
      mac_len = *(CK_MAC_GENERAL_PARAMS *)ctx->mech.pParameter;
   else
      mac_len = AES_BLOCK_SIZE / 2;

   if (length_only == TRUE) {
      *out_data_len = mac_len;
      return CKR_OK;
   }

 if ( (in_data_len % AES_BLOCK_SIZE) != 0) {
    rc = aes_mac_sign_update(sess, ctx, in_data, in_data_len);
    if (rc != CKR_OK)
       return rc;

    rc = aes_mac_sign_final(sess, length_only, ctx, out_data, out_data_len);

    return rc;
 }
 else {

   if (*out_data_len < mac_len) {
      *out_data_len = mac_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   rc = token_specific.t_aes_mac(in_data, in_data_len,
        key_obj, ((AES_DATA_CONTEXT *)ctx->context)->iv);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes mac failed.\n");

   memcpy(out_data, ((AES_DATA_CONTEXT *)ctx->context)->iv, mac_len);
   *out_data_len = mac_len;

   return rc;
 }
}

CK_RV
aes_mac_sign_update ( SESSION              * sess,
                      SIGN_VERIFY_CONTEXT  * ctx,
                      CK_BYTE              * in_data,
                      CK_ULONG               in_data_len )
{
 CK_ULONG         rc;
 OBJECT           * key_obj  = NULL;
 AES_DATA_CONTEXT * context  = NULL;
 CK_BYTE          * cipher   = NULL;
 CK_ULONG         total, remain, out_len;

  if (!sess || !ctx) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
  }

   context = (AES_DATA_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      memcpy( context->data + context->len, in_data, in_data_len );
      context->len += in_data_len;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous signUpdate operation first
      memcpy( cipher,                context->data, context->len );
      memcpy( cipher + context->len, in_data,       out_len - context->len );

      rc = token_specific.t_aes_mac(cipher, out_len, key_obj, context->iv);

      if (rc == CKR_OK) {
         // copy the remaining 'new' input data to the context buffer
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      } else
         TRACE_DEVEL("Token specific aes mac failed.\n");

      free( cipher );
      return rc;
   }
}

CK_RV
aes_mac_sign_final( SESSION              * sess,
                    CK_BBOOL               length_only,
                    SIGN_VERIFY_CONTEXT  * ctx,
                    CK_BYTE              * out_data,
                    CK_ULONG             * out_data_len)
{
  CK_ULONG           rc = CKR_OK;
  CK_ULONG           mac_len;
  AES_DATA_CONTEXT * context  = NULL;
  OBJECT           * key_obj  = NULL;

   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_DATA_CONTEXT *)ctx->context;

   if (ctx->mech.pParameter)
      mac_len = *(CK_MAC_GENERAL_PARAMS *)ctx->mech.pParameter;
   else
      mac_len = AES_BLOCK_SIZE / 2;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate one block of output (with padding)
   //    if no data stored, we are done (take the cipher from previous round)

   if (length_only == TRUE) {
      *out_data_len = mac_len;
      return CKR_OK;
   }

   if (context->len > 0) {

      if (*out_data_len < mac_len) {
         *out_data_len = mac_len;
         TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
         return CKR_BUFFER_TOO_SMALL;
      }

      /* padding with '00' in case case we didn't reach block size */
      memset(context->data + context->len, 0x0, AES_BLOCK_SIZE - context->len);

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      rc = token_specific.t_aes_mac(context->data, AES_BLOCK_SIZE, key_obj, context->iv);
      if (rc != CKR_OK) {
         TRACE_DEVEL("Token Specific aes mac failed.\n");
         return rc;
      }
   }
   memcpy(out_data, context->iv, mac_len);
   *out_data_len = mac_len;

   return rc;
}

CK_RV
aes_mac_verify( SESSION              * sess,
                SIGN_VERIFY_CONTEXT  * ctx,
                CK_BYTE              * in_data,
                CK_ULONG               in_data_len,
                CK_BYTE              * out_data,
                CK_ULONG               out_data_len)
{
 CK_ULONG         rc;
 OBJECT           * key_obj  = NULL;
 CK_ULONG         mac_len;

   if (!sess || !ctx || !in_data || !out_data){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

 if ( (in_data_len % AES_BLOCK_SIZE) != 0) {
    rc = aes_mac_verify_update(sess, ctx, in_data, in_data_len);
    if (rc != CKR_OK)
       return rc;

    rc = aes_mac_verify_final(sess, ctx, out_data, out_data_len);
    return rc;
 }
 else {

   if (ctx->mech.pParameter)
      mac_len = *(CK_MAC_GENERAL_PARAMS *)ctx->mech.pParameter;
   else
      mac_len = AES_BLOCK_SIZE / 2;

   if (out_data_len != mac_len) {
           TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
           return CKR_SIGNATURE_LEN_RANGE;
   }

   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to find specified object.\n");
      return rc;
   }

   rc = token_specific.t_aes_mac(in_data, in_data_len,
        key_obj, ((AES_DATA_CONTEXT *)ctx->context)->iv);

   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes mac failed.\n");
   if (memcmp(
       out_data, ((AES_DATA_CONTEXT *)ctx->context)->iv, out_data_len) == 0) {
       return CKR_OK;
   }
   else
       return CKR_SIGNATURE_INVALID;
 }
}


CK_RV
aes_mac_verify_update( SESSION              * sess,
                       SIGN_VERIFY_CONTEXT  * ctx,
                       CK_BYTE              * in_data,
                       CK_ULONG               in_data_len)
{
 CK_ULONG         rc;
 OBJECT           * key_obj  = NULL;
 AES_DATA_CONTEXT * context  = NULL;
 CK_BYTE          * cipher   = NULL;
 CK_ULONG         total, remain, out_len;

   if (!sess || !ctx) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_DATA_CONTEXT *)ctx->context;

   total = (context->len + in_data_len);

   if (total < AES_BLOCK_SIZE) {
      memcpy( context->data + context->len, in_data, in_data_len );
      context->len += in_data_len;
      return CKR_OK;
   }
   else {
      // we have at least 1 block
      remain  = (total % AES_BLOCK_SIZE);
      out_len = total - remain;

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }

      cipher = (CK_BYTE *)malloc( out_len );
      if (!cipher){
         TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
         return CKR_HOST_MEMORY;
      }
      // copy any data left over from the previous signUpdate operation first
      memcpy( cipher,                context->data, context->len );
      memcpy( cipher + context->len, in_data,       out_len - context->len );

      rc = token_specific.t_aes_mac(cipher, out_len, key_obj, context->iv);
      if (rc == CKR_OK) {
         // copy the remaining 'new' input data to the context buffer
         if (remain != 0)
            memcpy( context->data, in_data + (in_data_len - remain), remain );
         context->len = remain;
      }
      else
         TRACE_DEVEL("Token specific aes mac failed.\n");

      free( cipher );
      return rc;
   }
}

CK_RV
aes_mac_verify_final( SESSION              * sess,
                      SIGN_VERIFY_CONTEXT  * ctx,
                      CK_BYTE              * signature,
                      CK_ULONG               signature_len)
{
 CK_ULONG           rc;
 OBJECT           * key_obj  = NULL;
 CK_ULONG           mac_len;
 AES_DATA_CONTEXT * context  = NULL;

   if (!sess || !ctx || !signature){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   context = (AES_DATA_CONTEXT *)ctx->context;

   if (ctx->mech.pParameter)
      mac_len = *(CK_MAC_GENERAL_PARAMS *)ctx->mech.pParameter;
   else
      mac_len = AES_BLOCK_SIZE / 2;

   // there will never be more than one block in the context buffer
   // so the amount of output is as follows:
   //    if less than 1 block stored, we generate one block of output (with padding)
   //    if no data stored, we are done (take the cipher from previous round)

   if (context->len > 0) {

      if (signature_len != mac_len) {
           TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
           return CKR_SIGNATURE_LEN_RANGE;
      }

      /* padding with '00' in case case we didn't reach block size */
      memset(context->data + context->len, 0x0, AES_BLOCK_SIZE - context->len);

      rc = object_mgr_find_in_map1( ctx->key, &key_obj );
      if (rc != CKR_OK){
	 TRACE_ERROR("Failed to find specified object.\n");
         return rc;
      }
      rc = token_specific.t_aes_mac(context->data, AES_BLOCK_SIZE, key_obj, context->iv);
      if (rc != CKR_OK) {
         TRACE_DEVEL("Token specific aes mac failed.\n");
         return rc;
      }
   }

   if (memcmp(signature, context->iv, signature_len) == 0) {
      return CKR_OK;
   }
   else
      return CKR_SIGNATURE_INVALID;
}

CK_RV aes_gcm_init(SESSION *sess, ENCR_DECR_CONTEXT *ctx, CK_MECHANISM *mech,
		   CK_OBJECT_HANDLE key, CK_BYTE direction)
{
	if (token_specific.t_aes_gcm_init == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	return token_specific.t_aes_gcm_init(sess, ctx, mech, key, direction);
}

CK_RV aes_gcm_encrypt(SESSION *sess, CK_BBOOL length_only,
		      ENCR_DECR_CONTEXT	*ctx, CK_BYTE *in_data,
		      CK_ULONG in_data_len, CK_BYTE *out_data,
		      CK_ULONG *out_data_len)
{
	CK_RV rc;
	CK_GCM_PARAMS *aesgcm = NULL;
	CK_ULONG tag_data_len;

	if (!sess || !ctx || !in_data || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	aesgcm = (CK_GCM_PARAMS *)ctx->mech.pParameter;
	tag_data_len = (aesgcm->ulTagBits + 7) / 8; /* round to full byte */


	if (length_only == TRUE) {
		*out_data_len = in_data_len + tag_data_len;
		return CKR_OK;
	}

	if (*out_data_len < in_data_len + tag_data_len) {
		*out_data_len = in_data_len + tag_data_len;
		TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
		return CKR_BUFFER_TOO_SMALL;
	}

	if (token_specific.t_aes_gcm == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_aes_gcm(sess, ctx , in_data, in_data_len,
				      out_data, out_data_len, 1);
	if (rc != CKR_OK)
		TRACE_ERROR("Token specific aes gcm encrypt failed:  %02lx\n",
			    rc);
	return rc;

}

CK_RV aes_gcm_encrypt_update(SESSION *sess, CK_BBOOL length_only,
			     ENCR_DECR_CONTEXT *ctx, CK_BYTE *in_data,
			     CK_ULONG in_data_len, CK_BYTE *out_data,
			     CK_ULONG *out_data_len)
{
	AES_GCM_CONTEXT *context = NULL;
	CK_ULONG total, remain, out_len;
	CK_RV rc;

	if (!sess || !ctx || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	context = (AES_GCM_CONTEXT *)ctx->context;
	total = (context->len + in_data_len);

	if (length_only) {
		if (total < AES_BLOCK_SIZE) {
			*out_data_len = 0;
			return CKR_OK;
		} else {
			remain = (total % AES_BLOCK_SIZE);
			out_len = total - remain;
			*out_data_len = out_len;
			TRACE_DEVEL("Length Only requested (%02ld bytes).\n",
				    *out_data_len);
			return CKR_OK;
		}
	}

	if (token_specific.t_aes_gcm_update == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_aes_gcm_update(sess, ctx, in_data, in_data_len,
					     out_data, out_data_len, 1);
	if (rc != CKR_OK)
		TRACE_ERROR("Token specific AES GCM EncryptUpdate failed: "
			    "%02lx\n", rc);
	return rc;
}

CK_RV aes_gcm_encrypt_final(SESSION *sess, CK_BBOOL length_only,
			    ENCR_DECR_CONTEXT *ctx, CK_BYTE *out_data,
			    CK_ULONG *out_data_len)
{
	CK_GCM_PARAMS *aesgcm = NULL;
	AES_GCM_CONTEXT	*context = NULL;
	CK_ULONG tag_data_len;
	CK_RV rc = CKR_OK;

	if (!sess || !ctx || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	context = (AES_GCM_CONTEXT *)ctx->context;

	aesgcm = (CK_GCM_PARAMS *)ctx->mech.pParameter;
	tag_data_len = (aesgcm->ulTagBits + 7) / 8; /* round to full byte */

	if (length_only) {
		if (context->len == 0) {
			*out_data_len = tag_data_len;
		} else {
			*out_data_len = context->len + tag_data_len;
		}
		return CKR_OK;
	}

	if (*out_data_len < context->len + tag_data_len) {
		TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
		return CKR_BUFFER_TOO_SMALL;
	}

	if (token_specific.t_aes_gcm_final == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_aes_gcm_final(sess, ctx, out_data, out_data_len, 1);
	if (rc != CKR_OK)
		TRACE_ERROR("Token specific AES GCM EncryptFinal failed: "
			    "%02lx\n", rc);
	return rc;
}

CK_RV aes_gcm_decrypt(SESSION *sess, CK_BBOOL length_only,
		      ENCR_DECR_CONTEXT	*ctx, CK_BYTE *in_data,
		      CK_ULONG in_data_len, CK_BYTE *out_data,
		      CK_ULONG *out_data_len)
{
	CK_GCM_PARAMS *aesgcm = NULL;
	CK_ULONG tag_data_len;
	CK_RV rc;

	if (!sess || !ctx || !in_data || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	aesgcm = (CK_GCM_PARAMS *)ctx->mech.pParameter;
	tag_data_len = (aesgcm->ulTagBits + 7) / 8; /* round to full byte */

	if (length_only == TRUE) {
		*out_data_len = in_data_len - tag_data_len;
		return CKR_OK;
	}

	if (*out_data_len < in_data_len - tag_data_len) {
		*out_data_len = in_data_len - tag_data_len;
		TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
		return CKR_BUFFER_TOO_SMALL;
	}

	if (token_specific.t_aes_gcm == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_aes_gcm(sess, ctx, in_data, in_data_len, out_data,
				      out_data_len, 0);
	if (rc != CKR_OK)
		TRACE_ERROR("Token specific aes gcm decrypt failed.\n");

	return rc;
}

CK_RV aes_gcm_decrypt_update(SESSION *sess, CK_BBOOL length_only,
			     ENCR_DECR_CONTEXT *ctx, CK_BYTE *in_data,
			     CK_ULONG in_data_len, CK_BYTE *out_data,
			     CK_ULONG *out_data_len)
{
	AES_GCM_CONTEXT	*context = NULL;
	CK_GCM_PARAMS *aesgcm = NULL;
	CK_ULONG total, remain, out_len;
	CK_ULONG tag_data_len;
	CK_RV rc;

	if (!sess || !ctx || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	/* Be aware that this part of incoming data could be the last chunk,
	 * that means it's tag data, not encrypted plaintext.
	 * Hence we'll keep at least tag data size in the context buffer */

	context = (AES_GCM_CONTEXT *)ctx->context;
	total = (context->len + in_data_len);

	aesgcm = (CK_GCM_PARAMS *)ctx->mech.pParameter;
	tag_data_len = (aesgcm->ulTagBits + 7) / 8; /* round to full byte */

	if (length_only) {
		if (total < AES_BLOCK_SIZE + tag_data_len) {
			*out_data_len = 0;
			return CKR_OK;
		} else {
			remain = ((total - tag_data_len) % AES_BLOCK_SIZE)
				   + tag_data_len;
			out_len = total - remain;
			*out_data_len = out_len;
			TRACE_DEVEL("Length Only requested (%02ld bytes).\n",
				    *out_data_len);
			return CKR_OK;
		}
	}

	if (token_specific.t_aes_gcm_update == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_aes_gcm_update(sess, ctx, in_data, in_data_len,
					     out_data, out_data_len, 0 );
	if (rc != CKR_OK)
		TRACE_ERROR("Token specific AES GCM DecryptUpdate "
			    "failed: %02lx\n", rc);
	return rc;
}

CK_RV aes_gcm_decrypt_final(SESSION *sess, CK_BBOOL length_only,
			    ENCR_DECR_CONTEXT *ctx, CK_BYTE *out_data,
			    CK_ULONG *out_data_len)
{
	AES_GCM_CONTEXT	*context = NULL;
	CK_RV rc = CKR_OK;

	if (!sess || !ctx || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	context = (AES_GCM_CONTEXT *)ctx->context;

	if (length_only) {
		if (context->len == 0) {
			*out_data_len = 0;
		} else {
			*out_data_len = context->len;
		}
		return CKR_OK;
	}

	if (token_specific.t_aes_gcm_final == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_aes_gcm_final(sess, ctx, out_data, out_data_len, 0);
	if (rc != CKR_OK)
		TRACE_ERROR("Token specific AES GCM DecryptFinal failed: "
			    "%02lx\n", rc);

	return rc;
}

//
// mechanisms
//


//
//
CK_RV
ckm_aes_key_gen( TEMPLATE *tmpl )
{

   CK_ATTRIBUTE     * opaque_attr   = NULL;
   CK_ATTRIBUTE     * value_attr    = NULL;
   CK_ATTRIBUTE     * key_type_attr = NULL;
   CK_ATTRIBUTE     * class_attr    = NULL;
   CK_ATTRIBUTE     * local_attr    = NULL;
   CK_ATTRIBUTE     * val_len_attr  = NULL;
   CK_BYTE          * aes_key       = NULL;
   CK_BYTE            dummy_key[AES_KEY_SIZE_256] = { 0, };
   CK_ULONG           rc;
   CK_ULONG	      key_size;
   CK_ULONG	      token_keysize;
   CK_BBOOL	      found	    = FALSE;


   found = template_attribute_find( tmpl, CKA_VALUE_LEN, &val_len_attr );
   if (found == FALSE)
      return CKR_TEMPLATE_INCONSISTENT;
      
   key_size = *(CK_ULONG *)val_len_attr->pValue;
   if (key_size != AES_KEY_SIZE_128 &&
       key_size != AES_KEY_SIZE_192 &&
       key_size != AES_KEY_SIZE_256) {
	   return CKR_ATTRIBUTE_VALUE_INVALID;
   }
   
   if (token_specific.t_aes_key_gen == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   if (token_specific.token_keysize)
	token_keysize = token_specific.token_keysize;
   else
	token_keysize = key_size;

   if ((aes_key = (CK_BYTE *)calloc(1,token_keysize)) == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      return CKR_HOST_MEMORY;
   }

   rc = token_specific.t_aes_key_gen(aes_key, token_keysize, key_size);
   
   if (rc != CKR_OK) 
      goto err;

   /* For secure-key keys put in CKA_IBM_OPAQUE 
    * and put dummy_key in CKA_VALUE.
    */
   if (token_specific.token_keysize) {
        opaque_attr = (CK_ATTRIBUTE *)malloc(sizeof(CK_ATTRIBUTE) + token_keysize);
        if (!opaque_attr) {
           TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
           rc = CKR_HOST_MEMORY;
           goto err;
        }
        opaque_attr->type = CKA_IBM_OPAQUE;
        opaque_attr->ulValueLen = token_keysize;
        opaque_attr->pValue = (CK_BYTE *)opaque_attr + sizeof(CK_ATTRIBUTE);
        memcpy(opaque_attr->pValue, aes_key, token_keysize);
        template_update_attribute(tmpl, opaque_attr);
   }

   value_attr    = (CK_ATTRIBUTE *)malloc(sizeof(CK_ATTRIBUTE) + key_size );
   key_type_attr = (CK_ATTRIBUTE *)malloc(sizeof(CK_ATTRIBUTE) + sizeof(CK_KEY_TYPE) );
   class_attr    = (CK_ATTRIBUTE *)malloc(sizeof(CK_ATTRIBUTE) + sizeof(CK_OBJECT_CLASS) );
   local_attr    = (CK_ATTRIBUTE *)malloc(sizeof(CK_ATTRIBUTE) + sizeof(CK_BBOOL) );

   if (!value_attr || !key_type_attr || !class_attr || !local_attr) {
      if (value_attr)    free( value_attr );
      if (key_type_attr) free( key_type_attr );
      if (class_attr)    free( class_attr );
      if (local_attr)    free( local_attr );

      TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
      rc = CKR_HOST_MEMORY;
      goto err;
   }

   value_attr->type         = CKA_VALUE;
   value_attr->ulValueLen   = key_size;
   value_attr->pValue       = (CK_BYTE *)value_attr + sizeof(CK_ATTRIBUTE);
   if (token_specific.token_keysize)
      memcpy( value_attr->pValue, dummy_key, key_size );
   else
      memcpy( value_attr->pValue, aes_key, key_size );
   free(aes_key);

   key_type_attr->type         = CKA_KEY_TYPE;
   key_type_attr->ulValueLen   = sizeof(CK_KEY_TYPE);
   key_type_attr->pValue       = (CK_BYTE *)key_type_attr + sizeof(CK_ATTRIBUTE);
   *(CK_KEY_TYPE *)key_type_attr->pValue = CKK_AES;

   class_attr->type         = CKA_CLASS;
   class_attr->ulValueLen   = sizeof(CK_OBJECT_CLASS);
   class_attr->pValue       = (CK_BYTE *)class_attr + sizeof(CK_ATTRIBUTE);
   *(CK_OBJECT_CLASS *)class_attr->pValue = CKO_SECRET_KEY;

   local_attr->type         = CKA_LOCAL;
   local_attr->ulValueLen   = sizeof(CK_BBOOL);
   local_attr->pValue       = (CK_BYTE *)local_attr + sizeof(CK_ATTRIBUTE);
   *(CK_BBOOL *)local_attr->pValue = TRUE;

   template_update_attribute( tmpl, value_attr );
   template_update_attribute( tmpl, key_type_attr );
   template_update_attribute( tmpl, class_attr );
   template_update_attribute( tmpl, local_attr );

   return CKR_OK;

err:
   if (aes_key)
	free(aes_key);
   return rc;
}


//
//
CK_RV
ckm_aes_ecb_encrypt( CK_BYTE     * in_data,
                     CK_ULONG      in_data_len,
                     CK_BYTE     * out_data,
                     CK_ULONG    * out_data_len,
		     OBJECT	 * key )
{
   CK_ULONG         rc;

   if (!in_data || !out_data || !key){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   if (token_specific.t_aes_ecb == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_aes_ecb(in_data,in_data_len,
				 out_data,out_data_len, key, 1);
   
   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes ecb encrypt failed.\n");
   return rc;
}

//
//
CK_RV
ckm_aes_ecb_decrypt( CK_BYTE     * in_data,
                     CK_ULONG      in_data_len,
                     CK_BYTE     * out_data,
                     CK_ULONG    * out_data_len,
                     OBJECT      * key )
{
   CK_ULONG         rc;


   if (!in_data || !out_data || !key){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }
   
   if (token_specific.t_aes_ecb == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_aes_ecb(in_data,in_data_len,
		      		 out_data,out_data_len, key, 0);
   
   if (rc != CKR_OK)
      TRACE_DEVEL("token specific aes ecb decrypt failed.\n");

   return rc;
}


//
//
CK_RV
ckm_aes_cbc_encrypt( CK_BYTE     * in_data,
                     CK_ULONG      in_data_len,
                     CK_BYTE     * out_data,
                     CK_ULONG    * out_data_len,
                     CK_BYTE     * init_v,
                     OBJECT      * key )
{
   CK_ULONG         rc;

   if (!in_data || !out_data || !init_v || !key){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (*out_data_len < in_data_len){
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }
   
   if (token_specific.t_aes_cbc == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_aes_cbc(in_data, in_data_len,
		      		 out_data,out_data_len, key, init_v, 1);

   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes cbc encrypt failed.\n");
   return rc;
}


//
//
CK_RV
ckm_aes_cbc_decrypt( CK_BYTE     * in_data,
                     CK_ULONG      in_data_len,
                     CK_BYTE     * out_data,
                     CK_ULONG    * out_data_len,
                     CK_BYTE     * init_v,
                     OBJECT      * key )
{
   CK_ULONG         rc;

   if (!in_data || !out_data || !init_v || !key ){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (*out_data_len < in_data_len){
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }
   
   if (token_specific.t_aes_cbc == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_aes_cbc(in_data, in_data_len,
		      		 out_data, out_data_len, key, init_v,0);
   
   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes cbc decrypt failed.\n");
   return rc;
}
//
//
CK_RV
ckm_aes_ctr_encrypt( CK_BYTE    *in_data,
                     CK_ULONG    in_data_len,
                     CK_BYTE    *out_data,
                     CK_ULONG 	*out_data_len,
                     CK_BYTE    *counterblock,
                     CK_ULONG    counter_width,
                     OBJECT  	*key )
{
   CK_ULONG         rc;
   if (!in_data || !out_data || !counterblock || !key){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (*out_data_len < in_data_len){
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }
   if (counter_width % 8 != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
      return CKR_MECHANISM_PARAM_INVALID;
   }

   if (token_specific.t_aes_ctr == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_aes_ctr(in_data, in_data_len,
                                 out_data, out_data_len, key,
				 counterblock, counter_width, 1);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific aes ctr encrypt failed.\n");
   return rc;
}
//
//
CK_RV
ckm_aes_ctr_decrypt( CK_BYTE       *in_data,
                     CK_ULONG       in_data_len,
                     CK_BYTE       *out_data,
                     CK_ULONG      *out_data_len,
                     CK_BYTE       *counterblock,
                     CK_ULONG       counter_width,
		     OBJECT        *key )
{
   CK_ULONG         rc;
   if (!in_data || !out_data || !counterblock || !key){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (*out_data_len < in_data_len){
      *out_data_len = in_data_len;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }
   if (counter_width % 8 != 0){
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
      return CKR_MECHANISM_PARAM_INVALID;
   }

   if (token_specific.t_aes_ctr == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_aes_ctr(in_data, in_data_len,
                                 out_data,out_data_len, key, 
				 counterblock, counter_width, 0);
   if (rc != CKR_OK)
      TRACE_ERROR("Token specific aes ctr decrypt failed.\n");
   return rc;
}
//
//
CK_RV
ckm_aes_wrap_format( CK_BBOOL    length_only,
                     CK_BYTE  ** data,
                     CK_ULONG  * data_len )
{
   CK_BYTE   * ptr = NULL;
   CK_ULONG    len1, len2;


   len1 = *data_len;

   // if the input key data isn't a multiple of the blocksize,
   // we pad with NULLs to the next blocksize multiple.
   //
   if (len1 % AES_BLOCK_SIZE != 0) {
      len2 = AES_BLOCK_SIZE * ((len1 / AES_BLOCK_SIZE) + 1);

      if (length_only == FALSE) {
         ptr = (CK_BYTE *)realloc(*data, len2);
         if (!ptr){
            TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
            return CKR_HOST_MEMORY;
         }
         else
            memset( ptr + len1, 0x0, (len2 - len1) );

         *data     = ptr;
         *data_len = len2;
      }
   }

   return CKR_OK;
}
