/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var isPositiveZero = require( './../../../../base/assert/is-positive-zero' );
var isNegativeZero = require( './../../../../base/assert/is-negative-zero' );
var pow = require( './../lib/x_is_zero.js' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof pow, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns `+infinity` if `y = -infinity`', function test( t ) {
	var v;

	v = pow( 0.0, NINF );
	t.equal( v, PINF, 'returns +infinity' );

	v = pow( -0.0, NINF );
	t.equal( v, PINF, 'returns +infinity' );

	t.end();
});

tape( 'the function returns `+0` if `y = +infinity`', function test( t ) {
	var v;

	v = pow( 0.0, PINF );
	t.equal( isPositiveZero( v ), true, 'returns +0' );

	v = pow( -0.0, PINF );
	t.equal( isPositiveZero( v ), true, 'returns +0' );

	t.end();
});

tape( 'the function returns `x` if `y` is an odd positive integer', function test( t ) {
	var v;

	v = pow( 0.0, 5 );
	t.equal( isPositiveZero( v ), true, 'returns +0' );

	v = pow( -0.0, 5 );
	t.equal( isNegativeZero( v ), true, 'returns -0' );

	t.end();
});

tape( 'the function returns `+0` if `y` is positive but not an odd integer', function test( t ) {
	var v;

	v = pow( 0.0, 4 );
	t.equal( isPositiveZero( v ), true, 'returns +0' );

	v = pow( -0.0, 3.14 );
	t.equal( isPositiveZero( v ), true, 'returns +0' );

	t.end();
});

tape( 'the function returns `+-infinity` if `y` is a negative odd integer', function test( t ) {
	var v;

	v = pow( 0.0, -5 );
	t.equal( v, PINF, 'returns +infinity' );

	v = pow( -0.0, -5 );
	t.equal( v, NINF, 'returns -infinity' );

	t.end();
});

tape( 'the function returns `+infinity` if `y` is negative but not an odd integer', function test( t ) {
	var v;

	v = pow( 0.0, -4 );
	t.equal( v, PINF, 'returns +infinity' );

	v = pow( -0.0, -3.14 );
	t.equal( v, PINF, 'returns +infinity' );

	t.end();
});
