/*
 * Copyright 2011  Heiko Hund <heikoh@users.sf.net>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * or see the LICENSE file that should have been distributed with this code.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Apache httpd SASL authentication module.
 *
 * Provides SASL username and password verification for HTTP Basic
 * Authentication. Uses Cyrus' libsasl2 backends for this task.
 * Based on mod_authn_file.c from the httpd-2.2.3 distribution.
 */

#include "ap_config.h"
#include "ap_provider.h"
#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"
#include "apr_strings.h"
#include "apr_tables.h"

#include "mod_auth.h"

#include <sasl/sasl.h>


/* Forward declaration of the module struct. */
module AP_MODULE_DECLARE_DATA authn_sasl_module;

/* Valid strings for AuthSaslPwcheckMethod */
#define NUM_PWCHECK_METHODS 3
static const char *pwcheck_methods[NUM_PWCHECK_METHODS] = {
	"saslauthd", "auxprop", "sasldb"
};

/* Default pwcheck_method for use with libsasl. */
static const char *default_pwcheck_method = "saslauthd";

/* Default auxprop_plugin for use with libsasl. */
static const char *default_auxprop_plugin = "sasldb";

/* Default service name for use with libsasl. */
static const char *default_service_name = "http";

/* The per-directory module config. */
typedef struct {
	const char *pwcheck_method;
	const char *sasldb_path;
	const char *service;
	apr_array_header_t *realms;
} authn_sasl_cfg_t;


/* authn_sasl_create_dir_config -
 * Create a default per-directory configuration. Set the default
 * service name and pwcheck_method.
 */
static void *
authn_sasl_create_dir_config(apr_pool_t *p, char *dirspec)
{
	authn_sasl_cfg_t *cfg = apr_pcalloc(p, sizeof(*cfg));

	cfg->service = default_service_name;
	cfg->pwcheck_method = default_pwcheck_method;

	return (void *) cfg;
}


/* set_pwcheck_method -
 * Store the user defined pwcheck_method(s) into the config.
 */
static const char *
set_pwcheck_method(cmd_parms *cmd, void *mconfig, const char *arg1, const char *arg2)
{
	authn_sasl_cfg_t *cfg = mconfig;
	const char *method_1 = NULL, *method_2 = NULL;
	int i;

	for (i = 0; i < NUM_PWCHECK_METHODS; ++i) {
		if (apr_strnatcmp(arg1, pwcheck_methods[i]) == 0)
			method_1 = pwcheck_methods[i];
		if (arg2 && apr_strnatcmp(arg2, pwcheck_methods[i]) == 0)
			method_2 = pwcheck_methods[i];
	}

	if (method_1 == NULL)
		return apr_pstrcat(cmd->pool, "Invalid SASL pwcheck method string: ", arg1, NULL);

	/* Convert "sasldb" to "auxprop" as expected by libsasl2 */
	if (method_1 == pwcheck_methods[2])
		method_1 = pwcheck_methods[1];
	if (method_2 == pwcheck_methods[2])
		method_2 = pwcheck_methods[1];

	if (arg2 == NULL) {
		cfg->pwcheck_method = method_1;
	}
	else {
		if (method_2 == NULL || method_1 == method_2) {
			return apr_pstrcat(cmd->pool, "Invalid SASL pwcheck method string: ", arg2, NULL);
		}
		cfg->pwcheck_method = apr_pstrcat(cmd->pool, method_1, " ", method_2, NULL);
	}

	return NULL;
}


/* set_service -
 * Store the user defined service name into the config.
 */
static const char *
set_service(cmd_parms *cmd, void *mconfig, const char *arg1)
{
	authn_sasl_cfg_t *cfg = mconfig;
	cfg->service = apr_pstrdup(cmd->pool, arg1);
	return NULL;
}


/* set_realm -
 * Store the allowed user realms into the config.
 */
static const char *
set_realm(cmd_parms *cmd, void *mconfig, const char *arg)
{
	authn_sasl_cfg_t *cfg = mconfig;
	if (cfg->realms == NULL) {
		cfg->realms = apr_array_make(cmd->pool, 1, sizeof(const char*)); 
	}
	*(const char**) apr_array_push(cfg->realms) = apr_pstrdup(cmd->pool, arg);
	return NULL;
}


/* set_sasldb_path -
 * Store the path to the sasldb file into the config.
 */
static const char *
set_sasldb_path(cmd_parms *cmd, void *mconfig, const char *arg1)
{
	authn_sasl_cfg_t *cfg = mconfig;
	cfg->sasldb_path = apr_pstrdup(cmd->pool, arg1);
	return NULL;
}


/* The config directives introduces by this module. */
static const command_rec authn_sasl_cmds[] = {
	AP_INIT_TAKE12("AuthSaslPwcheckMethod", set_pwcheck_method, NULL, OR_AUTHCFG,
	               "Set this to override libsasl's default 'pwcheck_method' used for "
	               "authentication. Valid values are 'sasldb' and 'saslauthd'."),
	AP_INIT_TAKE1("AuthSaslServiceName", set_service, NULL, OR_AUTHCFG,
	              "Set the service name to be used by libsasl during user authentication"),
	AP_INIT_TAKE1("AuthSaslAppname", set_service, NULL, OR_AUTHCFG,
	              "(DEPRECATED) Same semantics as AuthSaslServiceName, use that instead"),
	AP_INIT_ITERATE("AuthSaslRealm", set_realm, NULL, OR_AUTHCFG,
	                "Set the allowed user realms used by libsasl during user authentication"),
	AP_INIT_TAKE1("AuthSaslDbPath", set_sasldb_path, NULL, OR_AUTHCFG,
	              "Set the path to the sasldb file to use with pwcheck_method sasldb"),
	{ NULL, { NULL }, NULL, 0, 0, NULL }
};


/* authn_sasl_cb_getopt -
 * Callback function libsasl2 uses to get option values.
 * The value for option "pwcheck_method", "sasldb_path" and
 * "auxprop_plugin" are set here if requested by libsasl
 */
static int
authn_sasl_cb_getopt(void *mconfig, const char *pname, const char *opt, const char **res, unsigned *len)
{
	authn_sasl_cfg_t *cfg = mconfig;

	if (cfg->pwcheck_method && apr_strnatcmp(opt, "pwcheck_method") == 0) {
		*res = cfg->pwcheck_method;
		return SASL_OK;
	}
	else if (cfg->sasldb_path && apr_strnatcmp(opt, "sasldb_path") == 0) {
		*res = cfg->sasldb_path;
		return SASL_OK;
	}
	else if (apr_strnatcmp(opt, "auxprop_plugin") == 0) {
		*res = default_auxprop_plugin;
		return SASL_OK;
	}

	return SASL_CONTINUE;
}


/* authn_sasl_cb_log -
 * Callback function libsasl2 uses to send log lines
 * which are passed to the httpd log function here
 */
static int
authn_sasl_cb_log(void *req, int level, const char *message)
{
	request_rec *r = req;

	switch (level)
	{
	case SASL_LOG_NONE:
	case SASL_LOG_PASS:
		break; /* always ignore */

	case SASL_LOG_ERR:
	case SASL_LOG_FAIL:
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s", message);
		break;

	case SASL_LOG_WARN:
		ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "%s", message);
		break;

	case SASL_LOG_NOTE:
		ap_log_rerror(APLOG_MARK, APLOG_NOTICE, 0, r, "%s", message);
		break;

	case SASL_LOG_DEBUG:
	case SASL_LOG_TRACE:
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "%s", message);
		break;
	}

	return SASL_OK;
}


/* check_password -
 * Authentication backend provider callback function used by
 * mod_auth_basic to verify credentials. Uses functions from
 * libsasl2 to do the job.
 */
static authn_status
check_password(request_rec *r, const char *user, const char *pass)
{
	sasl_conn_t *sasl_conn;
	authn_status result = AUTH_GRANTED;

	authn_sasl_cfg_t *cfg = ap_get_module_config(r->per_dir_config, &authn_sasl_module);

	sasl_callback_t const cb[] = {
		{ SASL_CB_GETOPT, authn_sasl_cb_getopt, (void *) cfg },
		{ SASL_CB_LOG, authn_sasl_cb_log, (void *) r },
		{ SASL_CB_LIST_END, NULL, NULL }
	};

	const char *default_realm = NULL;

	if (cfg->realms) {
		const char *user_at = strchr(user, '@');

		if (user_at) {
			int i, match = 0;
			const char *user_realm = user_at + 1;

			for (i = 0; i < cfg->realms->nelts; ++i) {
				const char *realm = ((const char**) cfg->realms->elts)[i];
				if (strcmp(user_realm, realm) == 0) {
					match = 1;
					break;
				}
			}
			if (!match) {
				return AUTH_DENIED;
			}
		}
		else {
			default_realm = ((const char**) cfg->realms->elts)[0];
		}
	}

	if (sasl_server_new(cfg->service, NULL, default_realm, NULL, NULL, cb, 0, &sasl_conn) != SASL_OK ||
	    sasl_checkpass(sasl_conn, user, strlen(user), pass, strlen(pass)) != SASL_OK) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s", sasl_errdetail(sasl_conn));
		result = AUTH_DENIED;
	}

	sasl_dispose(&sasl_conn);

	return result;
}

static apr_status_t
authn_sasl_child_exit(void *data)
{
	sasl_done();
	return APR_SUCCESS;
}


/* authn_sasl_child_init -
 * Initialize libsasl2 once in every child process and register
 * a cleanup function to finalize it before the process exits.
 */
static void
authn_sasl_child_init(apr_pool_t *p, server_rec *s)
{
	sasl_server_init(NULL, NULL);
	apr_pool_cleanup_register(p, s, apr_pool_cleanup_null, authn_sasl_child_exit);
}


/* authn_sasl_register_hooks -
 * Registers an auth provider to be used with
 * mod_auth_basic and an init function to init sasl.
 */
static void
authn_sasl_register_hooks(apr_pool_t *p)
{
	static const authn_provider authn_sasl_provider = {
		check_password, NULL
	};

	ap_register_provider(p, AUTHN_PROVIDER_GROUP, "sasl", "0", &authn_sasl_provider);
	ap_hook_child_init(authn_sasl_child_init, NULL, NULL, APR_HOOK_MIDDLE);
}


module AP_MODULE_DECLARE_DATA authn_sasl_module = {
	STANDARD20_MODULE_STUFF,
	authn_sasl_create_dir_config,    /* per-directory config creator */
	NULL,                            /* dir config merger */
	NULL,                            /* server config creator */
	NULL,                            /* server config merger */
	authn_sasl_cmds,                 /* command table */
	authn_sasl_register_hooks        /* set up other request processing hooks */
};
