/*
 * KS X 1001 and MS CP949 implementation for libcharset.
 */

#include <assert.h>

#include "charset.h"
#include "internal.h"

/*
 * There is a KS X 1001 <-> Unicode character mapping available at
 * http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/KSC/KSX1001.TXT
 * 
 * but in fact the square (A1A1-FEFE) subset of the Microsoft CP949
 * mapping is superior, since it's slightly more up to date
 * (contains the Euro sign, for example). So this file actually
 * implements a mapping between the double-byte portion of CP949
 * and Unicode, and also provides ksx1001 conversion functions to
 * access that square subset.
 * 
 * The file I'm actually working from is at
 * http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP949.TXT
 */

/*
 * Mapping from KS X 1001 to Unicode. For these purposes KS X 1001
 * is considered to be a 128x192 square array, with coordinates
 * running from (0,0) to (127,191).
 * 
 * Generated by running the following sh+Perl over CP949.TXT:

cat CP949.TXT | \
perl -ne '$a{hex$1,hex$2}=$3 if /^0x(\S\S)(\S\S)\s+(0x\S+)\s/;' \
      -e 'END {for $r (0..127) { $o="    {"; for $c (0..191) {' \
      -e '$k=$a{$r+128,$c+64}; $k=" ERROR" unless defined $k;' \
      -e '$o .= ", " if $c > 0; (print "$o\n"), $o="    " if length $o > 70;' \
      -e '$o .= $k; } print "$o},\n"; }}'

 */

static const unsigned short cp949_forward[128][192] = {
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xAC02, 0xAC03, 0xAC05, 0xAC06, 0xAC0B, 0xAC0C, 0xAC0D, 0xAC0E, 
    0xAC0F, 0xAC18, 0xAC1E, 0xAC1F, 0xAC21, 0xAC22, 0xAC23, 0xAC25, 0xAC26, 
    0xAC27, 0xAC28, 0xAC29, 0xAC2A, 0xAC2B, 0xAC2E, 0xAC32, 0xAC33, 0xAC34, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xAC35, 0xAC36, 0xAC37, 
    0xAC3A, 0xAC3B, 0xAC3D, 0xAC3E, 0xAC3F, 0xAC41, 0xAC42, 0xAC43, 0xAC44, 
    0xAC45, 0xAC46, 0xAC47, 0xAC48, 0xAC49, 0xAC4A, 0xAC4C, 0xAC4E, 0xAC4F, 
    0xAC50, 0xAC51, 0xAC52, 0xAC53, 0xAC55,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xAC56, 0xAC57, 0xAC59, 0xAC5A, 0xAC5B, 0xAC5D, 0xAC5E, 
    0xAC5F, 0xAC60, 0xAC61, 0xAC62, 0xAC63, 0xAC64, 0xAC65, 0xAC66, 0xAC67, 
    0xAC68, 0xAC69, 0xAC6A, 0xAC6B, 0xAC6C, 0xAC6D, 0xAC6E, 0xAC6F, 0xAC72, 
    0xAC73, 0xAC75, 0xAC76, 0xAC79, 0xAC7B, 0xAC7C, 0xAC7D, 0xAC7E, 0xAC7F, 
    0xAC82, 0xAC87, 0xAC88, 0xAC8D, 0xAC8E, 0xAC8F, 0xAC91, 0xAC92, 0xAC93, 
    0xAC95, 0xAC96, 0xAC97, 0xAC98, 0xAC99, 0xAC9A, 0xAC9B, 0xAC9E, 0xACA2, 
    0xACA3, 0xACA4, 0xACA5, 0xACA6, 0xACA7, 0xACAB, 0xACAD, 0xACAE, 0xACB1, 
    0xACB2, 0xACB3, 0xACB4, 0xACB5, 0xACB6, 0xACB7, 0xACBA, 0xACBE, 0xACBF, 
    0xACC0, 0xACC2, 0xACC3, 0xACC5, 0xACC6, 0xACC7, 0xACC9, 0xACCA, 0xACCB, 
    0xACCD, 0xACCE, 0xACCF, 0xACD0, 0xACD1, 0xACD2, 0xACD3, 0xACD4, 0xACD6, 
    0xACD8, 0xACD9, 0xACDA, 0xACDB, 0xACDC, 0xACDD, 0xACDE, 0xACDF, 0xACE2, 
    0xACE3, 0xACE5, 0xACE6, 0xACE9, 0xACEB, 0xACED, 0xACEE, 0xACF2, 0xACF4, 
    0xACF7, 0xACF8, 0xACF9, 0xACFA, 0xACFB, 0xACFE, 0xACFF, 0xAD01, 0xAD02, 
    0xAD03, 0xAD05, 0xAD07, 0xAD08, 0xAD09, 0xAD0A, 0xAD0B, 0xAD0E, 0xAD10, 
    0xAD12, 0xAD13,  ERROR},
    { ERROR, 0xAD14, 0xAD15, 0xAD16, 0xAD17, 0xAD19, 0xAD1A, 0xAD1B, 0xAD1D, 
    0xAD1E, 0xAD1F, 0xAD21, 0xAD22, 0xAD23, 0xAD24, 0xAD25, 0xAD26, 0xAD27, 
    0xAD28, 0xAD2A, 0xAD2B, 0xAD2E, 0xAD2F, 0xAD30, 0xAD31, 0xAD32, 0xAD33, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xAD36, 0xAD37, 0xAD39, 
    0xAD3A, 0xAD3B, 0xAD3D, 0xAD3E, 0xAD3F, 0xAD40, 0xAD41, 0xAD42, 0xAD43, 
    0xAD46, 0xAD48, 0xAD4A, 0xAD4B, 0xAD4C, 0xAD4D, 0xAD4E, 0xAD4F, 0xAD51, 
    0xAD52, 0xAD53, 0xAD55, 0xAD56, 0xAD57,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xAD59, 0xAD5A, 0xAD5B, 0xAD5C, 0xAD5D, 0xAD5E, 0xAD5F, 
    0xAD60, 0xAD62, 0xAD64, 0xAD65, 0xAD66, 0xAD67, 0xAD68, 0xAD69, 0xAD6A, 
    0xAD6B, 0xAD6E, 0xAD6F, 0xAD71, 0xAD72, 0xAD77, 0xAD78, 0xAD79, 0xAD7A, 
    0xAD7E, 0xAD80, 0xAD83, 0xAD84, 0xAD85, 0xAD86, 0xAD87, 0xAD8A, 0xAD8B, 
    0xAD8D, 0xAD8E, 0xAD8F, 0xAD91, 0xAD92, 0xAD93, 0xAD94, 0xAD95, 0xAD96, 
    0xAD97, 0xAD98, 0xAD99, 0xAD9A, 0xAD9B, 0xAD9E, 0xAD9F, 0xADA0, 0xADA1, 
    0xADA2, 0xADA3, 0xADA5, 0xADA6, 0xADA7, 0xADA8, 0xADA9, 0xADAA, 0xADAB, 
    0xADAC, 0xADAD, 0xADAE, 0xADAF, 0xADB0, 0xADB1, 0xADB2, 0xADB3, 0xADB4, 
    0xADB5, 0xADB6, 0xADB8, 0xADB9, 0xADBA, 0xADBB, 0xADBC, 0xADBD, 0xADBE, 
    0xADBF, 0xADC2, 0xADC3, 0xADC5, 0xADC6, 0xADC7, 0xADC9, 0xADCA, 0xADCB, 
    0xADCC, 0xADCD, 0xADCE, 0xADCF, 0xADD2, 0xADD4, 0xADD5, 0xADD6, 0xADD7, 
    0xADD8, 0xADD9, 0xADDA, 0xADDB, 0xADDD, 0xADDE, 0xADDF, 0xADE1, 0xADE2, 
    0xADE3, 0xADE5, 0xADE6, 0xADE7, 0xADE8, 0xADE9, 0xADEA, 0xADEB, 0xADEC, 
    0xADED, 0xADEE, 0xADEF, 0xADF0, 0xADF1, 0xADF2, 0xADF3, 0xADF4, 0xADF5, 
    0xADF6, 0xADF7,  ERROR},
    { ERROR, 0xADFA, 0xADFB, 0xADFD, 0xADFE, 0xAE02, 0xAE03, 0xAE04, 0xAE05, 
    0xAE06, 0xAE07, 0xAE0A, 0xAE0C, 0xAE0E, 0xAE0F, 0xAE10, 0xAE11, 0xAE12, 
    0xAE13, 0xAE15, 0xAE16, 0xAE17, 0xAE18, 0xAE19, 0xAE1A, 0xAE1B, 0xAE1C, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xAE1D, 0xAE1E, 0xAE1F, 
    0xAE20, 0xAE21, 0xAE22, 0xAE23, 0xAE24, 0xAE25, 0xAE26, 0xAE27, 0xAE28, 
    0xAE29, 0xAE2A, 0xAE2B, 0xAE2C, 0xAE2D, 0xAE2E, 0xAE2F, 0xAE32, 0xAE33, 
    0xAE35, 0xAE36, 0xAE39, 0xAE3B, 0xAE3C,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xAE3D, 0xAE3E, 0xAE3F, 0xAE42, 0xAE44, 0xAE47, 0xAE48, 
    0xAE49, 0xAE4B, 0xAE4F, 0xAE51, 0xAE52, 0xAE53, 0xAE55, 0xAE57, 0xAE58, 
    0xAE59, 0xAE5A, 0xAE5B, 0xAE5E, 0xAE62, 0xAE63, 0xAE64, 0xAE66, 0xAE67, 
    0xAE6A, 0xAE6B, 0xAE6D, 0xAE6E, 0xAE6F, 0xAE71, 0xAE72, 0xAE73, 0xAE74, 
    0xAE75, 0xAE76, 0xAE77, 0xAE7A, 0xAE7E, 0xAE7F, 0xAE80, 0xAE81, 0xAE82, 
    0xAE83, 0xAE86, 0xAE87, 0xAE88, 0xAE89, 0xAE8A, 0xAE8B, 0xAE8D, 0xAE8E, 
    0xAE8F, 0xAE90, 0xAE91, 0xAE92, 0xAE93, 0xAE94, 0xAE95, 0xAE96, 0xAE97, 
    0xAE98, 0xAE99, 0xAE9A, 0xAE9B, 0xAE9C, 0xAE9D, 0xAE9E, 0xAE9F, 0xAEA0, 
    0xAEA1, 0xAEA2, 0xAEA3, 0xAEA4, 0xAEA5, 0xAEA6, 0xAEA7, 0xAEA8, 0xAEA9, 
    0xAEAA, 0xAEAB, 0xAEAC, 0xAEAD, 0xAEAE, 0xAEAF, 0xAEB0, 0xAEB1, 0xAEB2, 
    0xAEB3, 0xAEB4, 0xAEB5, 0xAEB6, 0xAEB7, 0xAEB8, 0xAEB9, 0xAEBA, 0xAEBB, 
    0xAEBF, 0xAEC1, 0xAEC2, 0xAEC3, 0xAEC5, 0xAEC6, 0xAEC7, 0xAEC8, 0xAEC9, 
    0xAECA, 0xAECB, 0xAECE, 0xAED2, 0xAED3, 0xAED4, 0xAED5, 0xAED6, 0xAED7, 
    0xAEDA, 0xAEDB, 0xAEDD, 0xAEDE, 0xAEDF, 0xAEE0, 0xAEE1, 0xAEE2, 0xAEE3, 
    0xAEE4, 0xAEE5,  ERROR},
    { ERROR, 0xAEE6, 0xAEE7, 0xAEE9, 0xAEEA, 0xAEEC, 0xAEEE, 0xAEEF, 0xAEF0, 
    0xAEF1, 0xAEF2, 0xAEF3, 0xAEF5, 0xAEF6, 0xAEF7, 0xAEF9, 0xAEFA, 0xAEFB, 
    0xAEFD, 0xAEFE, 0xAEFF, 0xAF00, 0xAF01, 0xAF02, 0xAF03, 0xAF04, 0xAF05, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xAF06, 0xAF09, 0xAF0A, 
    0xAF0B, 0xAF0C, 0xAF0E, 0xAF0F, 0xAF11, 0xAF12, 0xAF13, 0xAF14, 0xAF15, 
    0xAF16, 0xAF17, 0xAF18, 0xAF19, 0xAF1A, 0xAF1B, 0xAF1C, 0xAF1D, 0xAF1E, 
    0xAF1F, 0xAF20, 0xAF21, 0xAF22, 0xAF23,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xAF24, 0xAF25, 0xAF26, 0xAF27, 0xAF28, 0xAF29, 0xAF2A, 
    0xAF2B, 0xAF2E, 0xAF2F, 0xAF31, 0xAF33, 0xAF35, 0xAF36, 0xAF37, 0xAF38, 
    0xAF39, 0xAF3A, 0xAF3B, 0xAF3E, 0xAF40, 0xAF44, 0xAF45, 0xAF46, 0xAF47, 
    0xAF4A, 0xAF4B, 0xAF4C, 0xAF4D, 0xAF4E, 0xAF4F, 0xAF51, 0xAF52, 0xAF53, 
    0xAF54, 0xAF55, 0xAF56, 0xAF57, 0xAF58, 0xAF59, 0xAF5A, 0xAF5B, 0xAF5E, 
    0xAF5F, 0xAF60, 0xAF61, 0xAF62, 0xAF63, 0xAF66, 0xAF67, 0xAF68, 0xAF69, 
    0xAF6A, 0xAF6B, 0xAF6C, 0xAF6D, 0xAF6E, 0xAF6F, 0xAF70, 0xAF71, 0xAF72, 
    0xAF73, 0xAF74, 0xAF75, 0xAF76, 0xAF77, 0xAF78, 0xAF7A, 0xAF7B, 0xAF7C, 
    0xAF7D, 0xAF7E, 0xAF7F, 0xAF81, 0xAF82, 0xAF83, 0xAF85, 0xAF86, 0xAF87, 
    0xAF89, 0xAF8A, 0xAF8B, 0xAF8C, 0xAF8D, 0xAF8E, 0xAF8F, 0xAF92, 0xAF93, 
    0xAF94, 0xAF96, 0xAF97, 0xAF98, 0xAF99, 0xAF9A, 0xAF9B, 0xAF9D, 0xAF9E, 
    0xAF9F, 0xAFA0, 0xAFA1, 0xAFA2, 0xAFA3, 0xAFA4, 0xAFA5, 0xAFA6, 0xAFA7, 
    0xAFA8, 0xAFA9, 0xAFAA, 0xAFAB, 0xAFAC, 0xAFAD, 0xAFAE, 0xAFAF, 0xAFB0, 
    0xAFB1, 0xAFB2, 0xAFB3, 0xAFB4, 0xAFB5, 0xAFB6, 0xAFB7, 0xAFBA, 0xAFBB, 
    0xAFBD, 0xAFBE,  ERROR},
    { ERROR, 0xAFBF, 0xAFC1, 0xAFC2, 0xAFC3, 0xAFC4, 0xAFC5, 0xAFC6, 0xAFCA, 
    0xAFCC, 0xAFCF, 0xAFD0, 0xAFD1, 0xAFD2, 0xAFD3, 0xAFD5, 0xAFD6, 0xAFD7, 
    0xAFD8, 0xAFD9, 0xAFDA, 0xAFDB, 0xAFDD, 0xAFDE, 0xAFDF, 0xAFE0, 0xAFE1, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xAFE2, 0xAFE3, 0xAFE4, 
    0xAFE5, 0xAFE6, 0xAFE7, 0xAFEA, 0xAFEB, 0xAFEC, 0xAFED, 0xAFEE, 0xAFEF, 
    0xAFF2, 0xAFF3, 0xAFF5, 0xAFF6, 0xAFF7, 0xAFF9, 0xAFFA, 0xAFFB, 0xAFFC, 
    0xAFFD, 0xAFFE, 0xAFFF, 0xB002, 0xB003,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB005, 0xB006, 0xB007, 0xB008, 0xB009, 0xB00A, 0xB00B, 
    0xB00D, 0xB00E, 0xB00F, 0xB011, 0xB012, 0xB013, 0xB015, 0xB016, 0xB017, 
    0xB018, 0xB019, 0xB01A, 0xB01B, 0xB01E, 0xB01F, 0xB020, 0xB021, 0xB022, 
    0xB023, 0xB024, 0xB025, 0xB026, 0xB027, 0xB029, 0xB02A, 0xB02B, 0xB02C, 
    0xB02D, 0xB02E, 0xB02F, 0xB030, 0xB031, 0xB032, 0xB033, 0xB034, 0xB035, 
    0xB036, 0xB037, 0xB038, 0xB039, 0xB03A, 0xB03B, 0xB03C, 0xB03D, 0xB03E, 
    0xB03F, 0xB040, 0xB041, 0xB042, 0xB043, 0xB046, 0xB047, 0xB049, 0xB04B, 
    0xB04D, 0xB04F, 0xB050, 0xB051, 0xB052, 0xB056, 0xB058, 0xB05A, 0xB05B, 
    0xB05C, 0xB05E, 0xB05F, 0xB060, 0xB061, 0xB062, 0xB063, 0xB064, 0xB065, 
    0xB066, 0xB067, 0xB068, 0xB069, 0xB06A, 0xB06B, 0xB06C, 0xB06D, 0xB06E, 
    0xB06F, 0xB070, 0xB071, 0xB072, 0xB073, 0xB074, 0xB075, 0xB076, 0xB077, 
    0xB078, 0xB079, 0xB07A, 0xB07B, 0xB07E, 0xB07F, 0xB081, 0xB082, 0xB083, 
    0xB085, 0xB086, 0xB087, 0xB088, 0xB089, 0xB08A, 0xB08B, 0xB08E, 0xB090, 
    0xB092, 0xB093, 0xB094, 0xB095, 0xB096, 0xB097, 0xB09B, 0xB09D, 0xB09E, 
    0xB0A3, 0xB0A4,  ERROR},
    { ERROR, 0xB0A5, 0xB0A6, 0xB0A7, 0xB0AA, 0xB0B0, 0xB0B2, 0xB0B6, 0xB0B7, 
    0xB0B9, 0xB0BA, 0xB0BB, 0xB0BD, 0xB0BE, 0xB0BF, 0xB0C0, 0xB0C1, 0xB0C2, 
    0xB0C3, 0xB0C6, 0xB0CA, 0xB0CB, 0xB0CC, 0xB0CD, 0xB0CE, 0xB0CF, 0xB0D2, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB0D3, 0xB0D5, 0xB0D6, 
    0xB0D7, 0xB0D9, 0xB0DA, 0xB0DB, 0xB0DC, 0xB0DD, 0xB0DE, 0xB0DF, 0xB0E1, 
    0xB0E2, 0xB0E3, 0xB0E4, 0xB0E6, 0xB0E7, 0xB0E8, 0xB0E9, 0xB0EA, 0xB0EB, 
    0xB0EC, 0xB0ED, 0xB0EE, 0xB0EF, 0xB0F0,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB0F1, 0xB0F2, 0xB0F3, 0xB0F4, 0xB0F5, 0xB0F6, 0xB0F7, 
    0xB0F8, 0xB0F9, 0xB0FA, 0xB0FB, 0xB0FC, 0xB0FD, 0xB0FE, 0xB0FF, 0xB100, 
    0xB101, 0xB102, 0xB103, 0xB104, 0xB105, 0xB106, 0xB107, 0xB10A, 0xB10D, 
    0xB10E, 0xB10F, 0xB111, 0xB114, 0xB115, 0xB116, 0xB117, 0xB11A, 0xB11E, 
    0xB11F, 0xB120, 0xB121, 0xB122, 0xB126, 0xB127, 0xB129, 0xB12A, 0xB12B, 
    0xB12D, 0xB12E, 0xB12F, 0xB130, 0xB131, 0xB132, 0xB133, 0xB136, 0xB13A, 
    0xB13B, 0xB13C, 0xB13D, 0xB13E, 0xB13F, 0xB142, 0xB143, 0xB145, 0xB146, 
    0xB147, 0xB149, 0xB14A, 0xB14B, 0xB14C, 0xB14D, 0xB14E, 0xB14F, 0xB152, 
    0xB153, 0xB156, 0xB157, 0xB159, 0xB15A, 0xB15B, 0xB15D, 0xB15E, 0xB15F, 
    0xB161, 0xB162, 0xB163, 0xB164, 0xB165, 0xB166, 0xB167, 0xB168, 0xB169, 
    0xB16A, 0xB16B, 0xB16C, 0xB16D, 0xB16E, 0xB16F, 0xB170, 0xB171, 0xB172, 
    0xB173, 0xB174, 0xB175, 0xB176, 0xB177, 0xB17A, 0xB17B, 0xB17D, 0xB17E, 
    0xB17F, 0xB181, 0xB183, 0xB184, 0xB185, 0xB186, 0xB187, 0xB18A, 0xB18C, 
    0xB18E, 0xB18F, 0xB190, 0xB191, 0xB195, 0xB196, 0xB197, 0xB199, 0xB19A, 
    0xB19B, 0xB19D,  ERROR},
    { ERROR, 0xB19E, 0xB19F, 0xB1A0, 0xB1A1, 0xB1A2, 0xB1A3, 0xB1A4, 0xB1A5, 
    0xB1A6, 0xB1A7, 0xB1A9, 0xB1AA, 0xB1AB, 0xB1AC, 0xB1AD, 0xB1AE, 0xB1AF, 
    0xB1B0, 0xB1B1, 0xB1B2, 0xB1B3, 0xB1B4, 0xB1B5, 0xB1B6, 0xB1B7, 0xB1B8, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB1B9, 0xB1BA, 0xB1BB, 
    0xB1BC, 0xB1BD, 0xB1BE, 0xB1BF, 0xB1C0, 0xB1C1, 0xB1C2, 0xB1C3, 0xB1C4, 
    0xB1C5, 0xB1C6, 0xB1C7, 0xB1C8, 0xB1C9, 0xB1CA, 0xB1CB, 0xB1CD, 0xB1CE, 
    0xB1CF, 0xB1D1, 0xB1D2, 0xB1D3, 0xB1D5,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB1D6, 0xB1D7, 0xB1D8, 0xB1D9, 0xB1DA, 0xB1DB, 0xB1DE, 
    0xB1E0, 0xB1E1, 0xB1E2, 0xB1E3, 0xB1E4, 0xB1E5, 0xB1E6, 0xB1E7, 0xB1EA, 
    0xB1EB, 0xB1ED, 0xB1EE, 0xB1EF, 0xB1F1, 0xB1F2, 0xB1F3, 0xB1F4, 0xB1F5, 
    0xB1F6, 0xB1F7, 0xB1F8, 0xB1FA, 0xB1FC, 0xB1FE, 0xB1FF, 0xB200, 0xB201, 
    0xB202, 0xB203, 0xB206, 0xB207, 0xB209, 0xB20A, 0xB20D, 0xB20E, 0xB20F, 
    0xB210, 0xB211, 0xB212, 0xB213, 0xB216, 0xB218, 0xB21A, 0xB21B, 0xB21C, 
    0xB21D, 0xB21E, 0xB21F, 0xB221, 0xB222, 0xB223, 0xB224, 0xB225, 0xB226, 
    0xB227, 0xB228, 0xB229, 0xB22A, 0xB22B, 0xB22C, 0xB22D, 0xB22E, 0xB22F, 
    0xB230, 0xB231, 0xB232, 0xB233, 0xB235, 0xB236, 0xB237, 0xB238, 0xB239, 
    0xB23A, 0xB23B, 0xB23D, 0xB23E, 0xB23F, 0xB240, 0xB241, 0xB242, 0xB243, 
    0xB244, 0xB245, 0xB246, 0xB247, 0xB248, 0xB249, 0xB24A, 0xB24B, 0xB24C, 
    0xB24D, 0xB24E, 0xB24F, 0xB250, 0xB251, 0xB252, 0xB253, 0xB254, 0xB255, 
    0xB256, 0xB257, 0xB259, 0xB25A, 0xB25B, 0xB25D, 0xB25E, 0xB25F, 0xB261, 
    0xB262, 0xB263, 0xB264, 0xB265, 0xB266, 0xB267, 0xB26A, 0xB26B, 0xB26C, 
    0xB26D, 0xB26E,  ERROR},
    { ERROR, 0xB26F, 0xB270, 0xB271, 0xB272, 0xB273, 0xB276, 0xB277, 0xB278, 
    0xB279, 0xB27A, 0xB27B, 0xB27D, 0xB27E, 0xB27F, 0xB280, 0xB281, 0xB282, 
    0xB283, 0xB286, 0xB287, 0xB288, 0xB28A, 0xB28B, 0xB28C, 0xB28D, 0xB28E, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB28F, 0xB292, 0xB293, 
    0xB295, 0xB296, 0xB297, 0xB29B, 0xB29C, 0xB29D, 0xB29E, 0xB29F, 0xB2A2, 
    0xB2A4, 0xB2A7, 0xB2A8, 0xB2A9, 0xB2AB, 0xB2AD, 0xB2AE, 0xB2AF, 0xB2B1, 
    0xB2B2, 0xB2B3, 0xB2B5, 0xB2B6, 0xB2B7,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB2B8, 0xB2B9, 0xB2BA, 0xB2BB, 0xB2BC, 0xB2BD, 0xB2BE, 
    0xB2BF, 0xB2C0, 0xB2C1, 0xB2C2, 0xB2C3, 0xB2C4, 0xB2C5, 0xB2C6, 0xB2C7, 
    0xB2CA, 0xB2CB, 0xB2CD, 0xB2CE, 0xB2CF, 0xB2D1, 0xB2D3, 0xB2D4, 0xB2D5, 
    0xB2D6, 0xB2D7, 0xB2DA, 0xB2DC, 0xB2DE, 0xB2DF, 0xB2E0, 0xB2E1, 0xB2E3, 
    0xB2E7, 0xB2E9, 0xB2EA, 0xB2F0, 0xB2F1, 0xB2F2, 0xB2F6, 0xB2FC, 0xB2FD, 
    0xB2FE, 0xB302, 0xB303, 0xB305, 0xB306, 0xB307, 0xB309, 0xB30A, 0xB30B, 
    0xB30C, 0xB30D, 0xB30E, 0xB30F, 0xB312, 0xB316, 0xB317, 0xB318, 0xB319, 
    0xB31A, 0xB31B, 0xB31D, 0xB31E, 0xB31F, 0xB320, 0xB321, 0xB322, 0xB323, 
    0xB324, 0xB325, 0xB326, 0xB327, 0xB328, 0xB329, 0xB32A, 0xB32B, 0xB32C, 
    0xB32D, 0xB32E, 0xB32F, 0xB330, 0xB331, 0xB332, 0xB333, 0xB334, 0xB335, 
    0xB336, 0xB337, 0xB338, 0xB339, 0xB33A, 0xB33B, 0xB33C, 0xB33D, 0xB33E, 
    0xB33F, 0xB340, 0xB341, 0xB342, 0xB343, 0xB344, 0xB345, 0xB346, 0xB347, 
    0xB348, 0xB349, 0xB34A, 0xB34B, 0xB34C, 0xB34D, 0xB34E, 0xB34F, 0xB350, 
    0xB351, 0xB352, 0xB353, 0xB357, 0xB359, 0xB35A, 0xB35D, 0xB360, 0xB361, 
    0xB362, 0xB363,  ERROR},
    { ERROR, 0xB366, 0xB368, 0xB36A, 0xB36C, 0xB36D, 0xB36F, 0xB372, 0xB373, 
    0xB375, 0xB376, 0xB377, 0xB379, 0xB37A, 0xB37B, 0xB37C, 0xB37D, 0xB37E, 
    0xB37F, 0xB382, 0xB386, 0xB387, 0xB388, 0xB389, 0xB38A, 0xB38B, 0xB38D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB38E, 0xB38F, 0xB391, 
    0xB392, 0xB393, 0xB395, 0xB396, 0xB397, 0xB398, 0xB399, 0xB39A, 0xB39B, 
    0xB39C, 0xB39D, 0xB39E, 0xB39F, 0xB3A2, 0xB3A3, 0xB3A4, 0xB3A5, 0xB3A6, 
    0xB3A7, 0xB3A9, 0xB3AA, 0xB3AB, 0xB3AD,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB3AE, 0xB3AF, 0xB3B0, 0xB3B1, 0xB3B2, 0xB3B3, 0xB3B4, 
    0xB3B5, 0xB3B6, 0xB3B7, 0xB3B8, 0xB3B9, 0xB3BA, 0xB3BB, 0xB3BC, 0xB3BD, 
    0xB3BE, 0xB3BF, 0xB3C0, 0xB3C1, 0xB3C2, 0xB3C3, 0xB3C6, 0xB3C7, 0xB3C9, 
    0xB3CA, 0xB3CD, 0xB3CF, 0xB3D1, 0xB3D2, 0xB3D3, 0xB3D6, 0xB3D8, 0xB3DA, 
    0xB3DC, 0xB3DE, 0xB3DF, 0xB3E1, 0xB3E2, 0xB3E3, 0xB3E5, 0xB3E6, 0xB3E7, 
    0xB3E9, 0xB3EA, 0xB3EB, 0xB3EC, 0xB3ED, 0xB3EE, 0xB3EF, 0xB3F0, 0xB3F1, 
    0xB3F2, 0xB3F3, 0xB3F4, 0xB3F5, 0xB3F6, 0xB3F7, 0xB3F8, 0xB3F9, 0xB3FA, 
    0xB3FB, 0xB3FD, 0xB3FE, 0xB3FF, 0xB400, 0xB401, 0xB402, 0xB403, 0xB404, 
    0xB405, 0xB406, 0xB407, 0xB408, 0xB409, 0xB40A, 0xB40B, 0xB40C, 0xB40D, 
    0xB40E, 0xB40F, 0xB411, 0xB412, 0xB413, 0xB414, 0xB415, 0xB416, 0xB417, 
    0xB419, 0xB41A, 0xB41B, 0xB41D, 0xB41E, 0xB41F, 0xB421, 0xB422, 0xB423, 
    0xB424, 0xB425, 0xB426, 0xB427, 0xB42A, 0xB42C, 0xB42D, 0xB42E, 0xB42F, 
    0xB430, 0xB431, 0xB432, 0xB433, 0xB435, 0xB436, 0xB437, 0xB438, 0xB439, 
    0xB43A, 0xB43B, 0xB43C, 0xB43D, 0xB43E, 0xB43F, 0xB440, 0xB441, 0xB442, 
    0xB443, 0xB444,  ERROR},
    { ERROR, 0xB445, 0xB446, 0xB447, 0xB448, 0xB449, 0xB44A, 0xB44B, 0xB44C, 
    0xB44D, 0xB44E, 0xB44F, 0xB452, 0xB453, 0xB455, 0xB456, 0xB457, 0xB459, 
    0xB45A, 0xB45B, 0xB45C, 0xB45D, 0xB45E, 0xB45F, 0xB462, 0xB464, 0xB466, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB467, 0xB468, 0xB469, 
    0xB46A, 0xB46B, 0xB46D, 0xB46E, 0xB46F, 0xB470, 0xB471, 0xB472, 0xB473, 
    0xB474, 0xB475, 0xB476, 0xB477, 0xB478, 0xB479, 0xB47A, 0xB47B, 0xB47C, 
    0xB47D, 0xB47E, 0xB47F, 0xB481, 0xB482,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB483, 0xB484, 0xB485, 0xB486, 0xB487, 0xB489, 0xB48A, 
    0xB48B, 0xB48C, 0xB48D, 0xB48E, 0xB48F, 0xB490, 0xB491, 0xB492, 0xB493, 
    0xB494, 0xB495, 0xB496, 0xB497, 0xB498, 0xB499, 0xB49A, 0xB49B, 0xB49C, 
    0xB49E, 0xB49F, 0xB4A0, 0xB4A1, 0xB4A2, 0xB4A3, 0xB4A5, 0xB4A6, 0xB4A7, 
    0xB4A9, 0xB4AA, 0xB4AB, 0xB4AD, 0xB4AE, 0xB4AF, 0xB4B0, 0xB4B1, 0xB4B2, 
    0xB4B3, 0xB4B4, 0xB4B6, 0xB4B8, 0xB4BA, 0xB4BB, 0xB4BC, 0xB4BD, 0xB4BE, 
    0xB4BF, 0xB4C1, 0xB4C2, 0xB4C3, 0xB4C5, 0xB4C6, 0xB4C7, 0xB4C9, 0xB4CA, 
    0xB4CB, 0xB4CC, 0xB4CD, 0xB4CE, 0xB4CF, 0xB4D1, 0xB4D2, 0xB4D3, 0xB4D4, 
    0xB4D6, 0xB4D7, 0xB4D8, 0xB4D9, 0xB4DA, 0xB4DB, 0xB4DE, 0xB4DF, 0xB4E1, 
    0xB4E2, 0xB4E5, 0xB4E7, 0xB4E8, 0xB4E9, 0xB4EA, 0xB4EB, 0xB4EE, 0xB4F0, 
    0xB4F2, 0xB4F3, 0xB4F4, 0xB4F5, 0xB4F6, 0xB4F7, 0xB4F9, 0xB4FA, 0xB4FB, 
    0xB4FC, 0xB4FD, 0xB4FE, 0xB4FF, 0xB500, 0xB501, 0xB502, 0xB503, 0xB504, 
    0xB505, 0xB506, 0xB507, 0xB508, 0xB509, 0xB50A, 0xB50B, 0xB50C, 0xB50D, 
    0xB50E, 0xB50F, 0xB510, 0xB511, 0xB512, 0xB513, 0xB516, 0xB517, 0xB519, 
    0xB51A, 0xB51D,  ERROR},
    { ERROR, 0xB51E, 0xB51F, 0xB520, 0xB521, 0xB522, 0xB523, 0xB526, 0xB52B, 
    0xB52C, 0xB52D, 0xB52E, 0xB52F, 0xB532, 0xB533, 0xB535, 0xB536, 0xB537, 
    0xB539, 0xB53A, 0xB53B, 0xB53C, 0xB53D, 0xB53E, 0xB53F, 0xB542, 0xB546, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB547, 0xB548, 0xB549, 
    0xB54A, 0xB54E, 0xB54F, 0xB551, 0xB552, 0xB553, 0xB555, 0xB556, 0xB557, 
    0xB558, 0xB559, 0xB55A, 0xB55B, 0xB55E, 0xB562, 0xB563, 0xB564, 0xB565, 
    0xB566, 0xB567, 0xB568, 0xB569, 0xB56A,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB56B, 0xB56C, 0xB56D, 0xB56E, 0xB56F, 0xB570, 0xB571, 
    0xB572, 0xB573, 0xB574, 0xB575, 0xB576, 0xB577, 0xB578, 0xB579, 0xB57A, 
    0xB57B, 0xB57C, 0xB57D, 0xB57E, 0xB57F, 0xB580, 0xB581, 0xB582, 0xB583, 
    0xB584, 0xB585, 0xB586, 0xB587, 0xB588, 0xB589, 0xB58A, 0xB58B, 0xB58C, 
    0xB58D, 0xB58E, 0xB58F, 0xB590, 0xB591, 0xB592, 0xB593, 0xB594, 0xB595, 
    0xB596, 0xB597, 0xB598, 0xB599, 0xB59A, 0xB59B, 0xB59C, 0xB59D, 0xB59E, 
    0xB59F, 0xB5A2, 0xB5A3, 0xB5A5, 0xB5A6, 0xB5A7, 0xB5A9, 0xB5AC, 0xB5AD, 
    0xB5AE, 0xB5AF, 0xB5B2, 0xB5B6, 0xB5B7, 0xB5B8, 0xB5B9, 0xB5BA, 0xB5BE, 
    0xB5BF, 0xB5C1, 0xB5C2, 0xB5C3, 0xB5C5, 0xB5C6, 0xB5C7, 0xB5C8, 0xB5C9, 
    0xB5CA, 0xB5CB, 0xB5CE, 0xB5D2, 0xB5D3, 0xB5D4, 0xB5D5, 0xB5D6, 0xB5D7, 
    0xB5D9, 0xB5DA, 0xB5DB, 0xB5DC, 0xB5DD, 0xB5DE, 0xB5DF, 0xB5E0, 0xB5E1, 
    0xB5E2, 0xB5E3, 0xB5E4, 0xB5E5, 0xB5E6, 0xB5E7, 0xB5E8, 0xB5E9, 0xB5EA, 
    0xB5EB, 0xB5ED, 0xB5EE, 0xB5EF, 0xB5F0, 0xB5F1, 0xB5F2, 0xB5F3, 0xB5F4, 
    0xB5F5, 0xB5F6, 0xB5F7, 0xB5F8, 0xB5F9, 0xB5FA, 0xB5FB, 0xB5FC, 0xB5FD, 
    0xB5FE, 0xB5FF,  ERROR},
    { ERROR, 0xB600, 0xB601, 0xB602, 0xB603, 0xB604, 0xB605, 0xB606, 0xB607, 
    0xB608, 0xB609, 0xB60A, 0xB60B, 0xB60C, 0xB60D, 0xB60E, 0xB60F, 0xB612, 
    0xB613, 0xB615, 0xB616, 0xB617, 0xB619, 0xB61A, 0xB61B, 0xB61C, 0xB61D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB61E, 0xB61F, 0xB620, 
    0xB621, 0xB622, 0xB623, 0xB624, 0xB626, 0xB627, 0xB628, 0xB629, 0xB62A, 
    0xB62B, 0xB62D, 0xB62E, 0xB62F, 0xB630, 0xB631, 0xB632, 0xB633, 0xB635, 
    0xB636, 0xB637, 0xB638, 0xB639, 0xB63A,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB63B, 0xB63C, 0xB63D, 0xB63E, 0xB63F, 0xB640, 0xB641, 
    0xB642, 0xB643, 0xB644, 0xB645, 0xB646, 0xB647, 0xB649, 0xB64A, 0xB64B, 
    0xB64C, 0xB64D, 0xB64E, 0xB64F, 0xB650, 0xB651, 0xB652, 0xB653, 0xB654, 
    0xB655, 0xB656, 0xB657, 0xB658, 0xB659, 0xB65A, 0xB65B, 0xB65C, 0xB65D, 
    0xB65E, 0xB65F, 0xB660, 0xB661, 0xB662, 0xB663, 0xB665, 0xB666, 0xB667, 
    0xB669, 0xB66A, 0xB66B, 0xB66C, 0xB66D, 0xB66E, 0xB66F, 0xB670, 0xB671, 
    0xB672, 0xB673, 0xB674, 0xB675, 0xB676, 0xB677, 0xB678, 0xB679, 0xB67A, 
    0xB67B, 0xB67C, 0xB67D, 0xB67E, 0xB67F, 0xB680, 0xB681, 0xB682, 0xB683, 
    0xB684, 0xB685, 0xB686, 0xB687, 0xB688, 0xB689, 0xB68A, 0xB68B, 0xB68C, 
    0xB68D, 0xB68E, 0xB68F, 0xB690, 0xB691, 0xB692, 0xB693, 0xB694, 0xB695, 
    0xB696, 0xB697, 0xB698, 0xB699, 0xB69A, 0xB69B, 0xB69E, 0xB69F, 0xB6A1, 
    0xB6A2, 0xB6A3, 0xB6A5, 0xB6A6, 0xB6A7, 0xB6A8, 0xB6A9, 0xB6AA, 0xB6AD, 
    0xB6AE, 0xB6AF, 0xB6B0, 0xB6B2, 0xB6B3, 0xB6B4, 0xB6B5, 0xB6B6, 0xB6B7, 
    0xB6B8, 0xB6B9, 0xB6BA, 0xB6BB, 0xB6BC, 0xB6BD, 0xB6BE, 0xB6BF, 0xB6C0, 
    0xB6C1, 0xB6C2,  ERROR},
    { ERROR, 0xB6C3, 0xB6C4, 0xB6C5, 0xB6C6, 0xB6C7, 0xB6C8, 0xB6C9, 0xB6CA, 
    0xB6CB, 0xB6CC, 0xB6CD, 0xB6CE, 0xB6CF, 0xB6D0, 0xB6D1, 0xB6D2, 0xB6D3, 
    0xB6D5, 0xB6D6, 0xB6D7, 0xB6D8, 0xB6D9, 0xB6DA, 0xB6DB, 0xB6DC, 0xB6DD, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB6DE, 0xB6DF, 0xB6E0, 
    0xB6E1, 0xB6E2, 0xB6E3, 0xB6E4, 0xB6E5, 0xB6E6, 0xB6E7, 0xB6E8, 0xB6E9, 
    0xB6EA, 0xB6EB, 0xB6EC, 0xB6ED, 0xB6EE, 0xB6EF, 0xB6F1, 0xB6F2, 0xB6F3, 
    0xB6F5, 0xB6F6, 0xB6F7, 0xB6F9, 0xB6FA,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB6FB, 0xB6FC, 0xB6FD, 0xB6FE, 0xB6FF, 0xB702, 0xB703, 
    0xB704, 0xB706, 0xB707, 0xB708, 0xB709, 0xB70A, 0xB70B, 0xB70C, 0xB70D, 
    0xB70E, 0xB70F, 0xB710, 0xB711, 0xB712, 0xB713, 0xB714, 0xB715, 0xB716, 
    0xB717, 0xB718, 0xB719, 0xB71A, 0xB71B, 0xB71C, 0xB71D, 0xB71E, 0xB71F, 
    0xB720, 0xB721, 0xB722, 0xB723, 0xB724, 0xB725, 0xB726, 0xB727, 0xB72A, 
    0xB72B, 0xB72D, 0xB72E, 0xB731, 0xB732, 0xB733, 0xB734, 0xB735, 0xB736, 
    0xB737, 0xB73A, 0xB73C, 0xB73D, 0xB73E, 0xB73F, 0xB740, 0xB741, 0xB742, 
    0xB743, 0xB745, 0xB746, 0xB747, 0xB749, 0xB74A, 0xB74B, 0xB74D, 0xB74E, 
    0xB74F, 0xB750, 0xB751, 0xB752, 0xB753, 0xB756, 0xB757, 0xB758, 0xB759, 
    0xB75A, 0xB75B, 0xB75C, 0xB75D, 0xB75E, 0xB75F, 0xB761, 0xB762, 0xB763, 
    0xB765, 0xB766, 0xB767, 0xB769, 0xB76A, 0xB76B, 0xB76C, 0xB76D, 0xB76E, 
    0xB76F, 0xB772, 0xB774, 0xB776, 0xB777, 0xB778, 0xB779, 0xB77A, 0xB77B, 
    0xB77E, 0xB77F, 0xB781, 0xB782, 0xB783, 0xB785, 0xB786, 0xB787, 0xB788, 
    0xB789, 0xB78A, 0xB78B, 0xB78E, 0xB793, 0xB794, 0xB795, 0xB79A, 0xB79B, 
    0xB79D, 0xB79E,  ERROR},
    { ERROR, 0xB79F, 0xB7A1, 0xB7A2, 0xB7A3, 0xB7A4, 0xB7A5, 0xB7A6, 0xB7A7, 
    0xB7AA, 0xB7AE, 0xB7AF, 0xB7B0, 0xB7B1, 0xB7B2, 0xB7B3, 0xB7B6, 0xB7B7, 
    0xB7B9, 0xB7BA, 0xB7BB, 0xB7BC, 0xB7BD, 0xB7BE, 0xB7BF, 0xB7C0, 0xB7C1, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB7C2, 0xB7C3, 0xB7C4, 
    0xB7C5, 0xB7C6, 0xB7C8, 0xB7CA, 0xB7CB, 0xB7CC, 0xB7CD, 0xB7CE, 0xB7CF, 
    0xB7D0, 0xB7D1, 0xB7D2, 0xB7D3, 0xB7D4, 0xB7D5, 0xB7D6, 0xB7D7, 0xB7D8, 
    0xB7D9, 0xB7DA, 0xB7DB, 0xB7DC, 0xB7DD,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB7DE, 0xB7DF, 0xB7E0, 0xB7E1, 0xB7E2, 0xB7E3, 0xB7E4, 
    0xB7E5, 0xB7E6, 0xB7E7, 0xB7E8, 0xB7E9, 0xB7EA, 0xB7EB, 0xB7EE, 0xB7EF, 
    0xB7F1, 0xB7F2, 0xB7F3, 0xB7F5, 0xB7F6, 0xB7F7, 0xB7F8, 0xB7F9, 0xB7FA, 
    0xB7FB, 0xB7FE, 0xB802, 0xB803, 0xB804, 0xB805, 0xB806, 0xB80A, 0xB80B, 
    0xB80D, 0xB80E, 0xB80F, 0xB811, 0xB812, 0xB813, 0xB814, 0xB815, 0xB816, 
    0xB817, 0xB81A, 0xB81C, 0xB81E, 0xB81F, 0xB820, 0xB821, 0xB822, 0xB823, 
    0xB826, 0xB827, 0xB829, 0xB82A, 0xB82B, 0xB82D, 0xB82E, 0xB82F, 0xB830, 
    0xB831, 0xB832, 0xB833, 0xB836, 0xB83A, 0xB83B, 0xB83C, 0xB83D, 0xB83E, 
    0xB83F, 0xB841, 0xB842, 0xB843, 0xB845, 0xB846, 0xB847, 0xB848, 0xB849, 
    0xB84A, 0xB84B, 0xB84C, 0xB84D, 0xB84E, 0xB84F, 0xB850, 0xB852, 0xB854, 
    0xB855, 0xB856, 0xB857, 0xB858, 0xB859, 0xB85A, 0xB85B, 0xB85E, 0xB85F, 
    0xB861, 0xB862, 0xB863, 0xB865, 0xB866, 0xB867, 0xB868, 0xB869, 0xB86A, 
    0xB86B, 0xB86E, 0xB870, 0xB872, 0xB873, 0xB874, 0xB875, 0xB876, 0xB877, 
    0xB879, 0xB87A, 0xB87B, 0xB87D, 0xB87E, 0xB87F, 0xB880, 0xB881, 0xB882, 
    0xB883, 0xB884,  ERROR},
    { ERROR, 0xB885, 0xB886, 0xB887, 0xB888, 0xB889, 0xB88A, 0xB88B, 0xB88C, 
    0xB88E, 0xB88F, 0xB890, 0xB891, 0xB892, 0xB893, 0xB894, 0xB895, 0xB896, 
    0xB897, 0xB898, 0xB899, 0xB89A, 0xB89B, 0xB89C, 0xB89D, 0xB89E, 0xB89F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB8A0, 0xB8A1, 0xB8A2, 
    0xB8A3, 0xB8A4, 0xB8A5, 0xB8A6, 0xB8A7, 0xB8A9, 0xB8AA, 0xB8AB, 0xB8AC, 
    0xB8AD, 0xB8AE, 0xB8AF, 0xB8B1, 0xB8B2, 0xB8B3, 0xB8B5, 0xB8B6, 0xB8B7, 
    0xB8B9, 0xB8BA, 0xB8BB, 0xB8BC, 0xB8BD,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB8BE, 0xB8BF, 0xB8C2, 0xB8C4, 0xB8C6, 0xB8C7, 0xB8C8, 
    0xB8C9, 0xB8CA, 0xB8CB, 0xB8CD, 0xB8CE, 0xB8CF, 0xB8D1, 0xB8D2, 0xB8D3, 
    0xB8D5, 0xB8D6, 0xB8D7, 0xB8D8, 0xB8D9, 0xB8DA, 0xB8DB, 0xB8DC, 0xB8DE, 
    0xB8E0, 0xB8E2, 0xB8E3, 0xB8E4, 0xB8E5, 0xB8E6, 0xB8E7, 0xB8EA, 0xB8EB, 
    0xB8ED, 0xB8EE, 0xB8EF, 0xB8F1, 0xB8F2, 0xB8F3, 0xB8F4, 0xB8F5, 0xB8F6, 
    0xB8F7, 0xB8FA, 0xB8FC, 0xB8FE, 0xB8FF, 0xB900, 0xB901, 0xB902, 0xB903, 
    0xB905, 0xB906, 0xB907, 0xB908, 0xB909, 0xB90A, 0xB90B, 0xB90C, 0xB90D, 
    0xB90E, 0xB90F, 0xB910, 0xB911, 0xB912, 0xB913, 0xB914, 0xB915, 0xB916, 
    0xB917, 0xB919, 0xB91A, 0xB91B, 0xB91C, 0xB91D, 0xB91E, 0xB91F, 0xB921, 
    0xB922, 0xB923, 0xB924, 0xB925, 0xB926, 0xB927, 0xB928, 0xB929, 0xB92A, 
    0xB92B, 0xB92C, 0xB92D, 0xB92E, 0xB92F, 0xB930, 0xB931, 0xB932, 0xB933, 
    0xB934, 0xB935, 0xB936, 0xB937, 0xB938, 0xB939, 0xB93A, 0xB93B, 0xB93E, 
    0xB93F, 0xB941, 0xB942, 0xB943, 0xB945, 0xB946, 0xB947, 0xB948, 0xB949, 
    0xB94A, 0xB94B, 0xB94D, 0xB94E, 0xB950, 0xB952, 0xB953, 0xB954, 0xB955, 
    0xB956, 0xB957,  ERROR},
    { ERROR, 0xB95A, 0xB95B, 0xB95D, 0xB95E, 0xB95F, 0xB961, 0xB962, 0xB963, 
    0xB964, 0xB965, 0xB966, 0xB967, 0xB96A, 0xB96C, 0xB96E, 0xB96F, 0xB970, 
    0xB971, 0xB972, 0xB973, 0xB976, 0xB977, 0xB979, 0xB97A, 0xB97B, 0xB97D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xB97E, 0xB97F, 0xB980, 
    0xB981, 0xB982, 0xB983, 0xB986, 0xB988, 0xB98B, 0xB98C, 0xB98F, 0xB990, 
    0xB991, 0xB992, 0xB993, 0xB994, 0xB995, 0xB996, 0xB997, 0xB998, 0xB999, 
    0xB99A, 0xB99B, 0xB99C, 0xB99D, 0xB99E,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xB99F, 0xB9A0, 0xB9A1, 0xB9A2, 0xB9A3, 0xB9A4, 0xB9A5, 
    0xB9A6, 0xB9A7, 0xB9A8, 0xB9A9, 0xB9AA, 0xB9AB, 0xB9AE, 0xB9AF, 0xB9B1, 
    0xB9B2, 0xB9B3, 0xB9B5, 0xB9B6, 0xB9B7, 0xB9B8, 0xB9B9, 0xB9BA, 0xB9BB, 
    0xB9BE, 0xB9C0, 0xB9C2, 0xB9C3, 0xB9C4, 0xB9C5, 0xB9C6, 0xB9C7, 0xB9CA, 
    0xB9CB, 0xB9CD, 0xB9D3, 0xB9D4, 0xB9D5, 0xB9D6, 0xB9D7, 0xB9DA, 0xB9DC, 
    0xB9DF, 0xB9E0, 0xB9E2, 0xB9E6, 0xB9E7, 0xB9E9, 0xB9EA, 0xB9EB, 0xB9ED, 
    0xB9EE, 0xB9EF, 0xB9F0, 0xB9F1, 0xB9F2, 0xB9F3, 0xB9F6, 0xB9FB, 0xB9FC, 
    0xB9FD, 0xB9FE, 0xB9FF, 0xBA02, 0xBA03, 0xBA04, 0xBA05, 0xBA06, 0xBA07, 
    0xBA09, 0xBA0A, 0xBA0B, 0xBA0C, 0xBA0D, 0xBA0E, 0xBA0F, 0xBA10, 0xBA11, 
    0xBA12, 0xBA13, 0xBA14, 0xBA16, 0xBA17, 0xBA18, 0xBA19, 0xBA1A, 0xBA1B, 
    0xBA1C, 0xBA1D, 0xBA1E, 0xBA1F, 0xBA20, 0xBA21, 0xBA22, 0xBA23, 0xBA24, 
    0xBA25, 0xBA26, 0xBA27, 0xBA28, 0xBA29, 0xBA2A, 0xBA2B, 0xBA2C, 0xBA2D, 
    0xBA2E, 0xBA2F, 0xBA30, 0xBA31, 0xBA32, 0xBA33, 0xBA34, 0xBA35, 0xBA36, 
    0xBA37, 0xBA3A, 0xBA3B, 0xBA3D, 0xBA3E, 0xBA3F, 0xBA41, 0xBA43, 0xBA44, 
    0xBA45, 0xBA46,  ERROR},
    { ERROR, 0xBA47, 0xBA4A, 0xBA4C, 0xBA4F, 0xBA50, 0xBA51, 0xBA52, 0xBA56, 
    0xBA57, 0xBA59, 0xBA5A, 0xBA5B, 0xBA5D, 0xBA5E, 0xBA5F, 0xBA60, 0xBA61, 
    0xBA62, 0xBA63, 0xBA66, 0xBA6A, 0xBA6B, 0xBA6C, 0xBA6D, 0xBA6E, 0xBA6F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBA72, 0xBA73, 0xBA75, 
    0xBA76, 0xBA77, 0xBA79, 0xBA7A, 0xBA7B, 0xBA7C, 0xBA7D, 0xBA7E, 0xBA7F, 
    0xBA80, 0xBA81, 0xBA82, 0xBA86, 0xBA88, 0xBA89, 0xBA8A, 0xBA8B, 0xBA8D, 
    0xBA8E, 0xBA8F, 0xBA90, 0xBA91, 0xBA92,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBA93, 0xBA94, 0xBA95, 0xBA96, 0xBA97, 0xBA98, 0xBA99, 
    0xBA9A, 0xBA9B, 0xBA9C, 0xBA9D, 0xBA9E, 0xBA9F, 0xBAA0, 0xBAA1, 0xBAA2, 
    0xBAA3, 0xBAA4, 0xBAA5, 0xBAA6, 0xBAA7, 0xBAAA, 0xBAAD, 0xBAAE, 0xBAAF, 
    0xBAB1, 0xBAB3, 0xBAB4, 0xBAB5, 0xBAB6, 0xBAB7, 0xBABA, 0xBABC, 0xBABE, 
    0xBABF, 0xBAC0, 0xBAC1, 0xBAC2, 0xBAC3, 0xBAC5, 0xBAC6, 0xBAC7, 0xBAC9, 
    0xBACA, 0xBACB, 0xBACC, 0xBACD, 0xBACE, 0xBACF, 0xBAD0, 0xBAD1, 0xBAD2, 
    0xBAD3, 0xBAD4, 0xBAD5, 0xBAD6, 0xBAD7, 0xBADA, 0xBADB, 0xBADC, 0xBADD, 
    0xBADE, 0xBADF, 0xBAE0, 0xBAE1, 0xBAE2, 0xBAE3, 0xBAE4, 0xBAE5, 0xBAE6, 
    0xBAE7, 0xBAE8, 0xBAE9, 0xBAEA, 0xBAEB, 0xBAEC, 0xBAED, 0xBAEE, 0xBAEF, 
    0xBAF0, 0xBAF1, 0xBAF2, 0xBAF3, 0xBAF4, 0xBAF5, 0xBAF6, 0xBAF7, 0xBAF8, 
    0xBAF9, 0xBAFA, 0xBAFB, 0xBAFD, 0xBAFE, 0xBAFF, 0xBB01, 0xBB02, 0xBB03, 
    0xBB05, 0xBB06, 0xBB07, 0xBB08, 0xBB09, 0xBB0A, 0xBB0B, 0xBB0C, 0xBB0E, 
    0xBB10, 0xBB12, 0xBB13, 0xBB14, 0xBB15, 0xBB16, 0xBB17, 0xBB19, 0xBB1A, 
    0xBB1B, 0xBB1D, 0xBB1E, 0xBB1F, 0xBB21, 0xBB22, 0xBB23, 0xBB24, 0xBB25, 
    0xBB26, 0xBB27,  ERROR},
    { ERROR, 0xBB28, 0xBB2A, 0xBB2C, 0xBB2D, 0xBB2E, 0xBB2F, 0xBB30, 0xBB31, 
    0xBB32, 0xBB33, 0xBB37, 0xBB39, 0xBB3A, 0xBB3F, 0xBB40, 0xBB41, 0xBB42, 
    0xBB43, 0xBB46, 0xBB48, 0xBB4A, 0xBB4B, 0xBB4C, 0xBB4E, 0xBB51, 0xBB52, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBB53, 0xBB55, 0xBB56, 
    0xBB57, 0xBB59, 0xBB5A, 0xBB5B, 0xBB5C, 0xBB5D, 0xBB5E, 0xBB5F, 0xBB60, 
    0xBB62, 0xBB64, 0xBB65, 0xBB66, 0xBB67, 0xBB68, 0xBB69, 0xBB6A, 0xBB6B, 
    0xBB6D, 0xBB6E, 0xBB6F, 0xBB70, 0xBB71,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBB72, 0xBB73, 0xBB74, 0xBB75, 0xBB76, 0xBB77, 0xBB78, 
    0xBB79, 0xBB7A, 0xBB7B, 0xBB7C, 0xBB7D, 0xBB7E, 0xBB7F, 0xBB80, 0xBB81, 
    0xBB82, 0xBB83, 0xBB84, 0xBB85, 0xBB86, 0xBB87, 0xBB89, 0xBB8A, 0xBB8B, 
    0xBB8D, 0xBB8E, 0xBB8F, 0xBB91, 0xBB92, 0xBB93, 0xBB94, 0xBB95, 0xBB96, 
    0xBB97, 0xBB98, 0xBB99, 0xBB9A, 0xBB9B, 0xBB9C, 0xBB9D, 0xBB9E, 0xBB9F, 
    0xBBA0, 0xBBA1, 0xBBA2, 0xBBA3, 0xBBA5, 0xBBA6, 0xBBA7, 0xBBA9, 0xBBAA, 
    0xBBAB, 0xBBAD, 0xBBAE, 0xBBAF, 0xBBB0, 0xBBB1, 0xBBB2, 0xBBB3, 0xBBB5, 
    0xBBB6, 0xBBB8, 0xBBB9, 0xBBBA, 0xBBBB, 0xBBBC, 0xBBBD, 0xBBBE, 0xBBBF, 
    0xBBC1, 0xBBC2, 0xBBC3, 0xBBC5, 0xBBC6, 0xBBC7, 0xBBC9, 0xBBCA, 0xBBCB, 
    0xBBCC, 0xBBCD, 0xBBCE, 0xBBCF, 0xBBD1, 0xBBD2, 0xBBD4, 0xBBD5, 0xBBD6, 
    0xBBD7, 0xBBD8, 0xBBD9, 0xBBDA, 0xBBDB, 0xBBDC, 0xBBDD, 0xBBDE, 0xBBDF, 
    0xBBE0, 0xBBE1, 0xBBE2, 0xBBE3, 0xBBE4, 0xBBE5, 0xBBE6, 0xBBE7, 0xBBE8, 
    0xBBE9, 0xBBEA, 0xBBEB, 0xBBEC, 0xBBED, 0xBBEE, 0xBBEF, 0xBBF0, 0xBBF1, 
    0xBBF2, 0xBBF3, 0xBBF4, 0xBBF5, 0xBBF6, 0xBBF7, 0xBBFA, 0xBBFB, 0xBBFD, 
    0xBBFE, 0xBC01,  ERROR},
    { ERROR, 0xBC03, 0xBC04, 0xBC05, 0xBC06, 0xBC07, 0xBC0A, 0xBC0E, 0xBC10, 
    0xBC12, 0xBC13, 0xBC19, 0xBC1A, 0xBC20, 0xBC21, 0xBC22, 0xBC23, 0xBC26, 
    0xBC28, 0xBC2A, 0xBC2B, 0xBC2C, 0xBC2E, 0xBC2F, 0xBC32, 0xBC33, 0xBC35, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBC36, 0xBC37, 0xBC39, 
    0xBC3A, 0xBC3B, 0xBC3C, 0xBC3D, 0xBC3E, 0xBC3F, 0xBC42, 0xBC46, 0xBC47, 
    0xBC48, 0xBC4A, 0xBC4B, 0xBC4E, 0xBC4F, 0xBC51, 0xBC52, 0xBC53, 0xBC54, 
    0xBC55, 0xBC56, 0xBC57, 0xBC58, 0xBC59,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBC5A, 0xBC5B, 0xBC5C, 0xBC5E, 0xBC5F, 0xBC60, 0xBC61, 
    0xBC62, 0xBC63, 0xBC64, 0xBC65, 0xBC66, 0xBC67, 0xBC68, 0xBC69, 0xBC6A, 
    0xBC6B, 0xBC6C, 0xBC6D, 0xBC6E, 0xBC6F, 0xBC70, 0xBC71, 0xBC72, 0xBC73, 
    0xBC74, 0xBC75, 0xBC76, 0xBC77, 0xBC78, 0xBC79, 0xBC7A, 0xBC7B, 0xBC7C, 
    0xBC7D, 0xBC7E, 0xBC7F, 0xBC80, 0xBC81, 0xBC82, 0xBC83, 0xBC86, 0xBC87, 
    0xBC89, 0xBC8A, 0xBC8D, 0xBC8F, 0xBC90, 0xBC91, 0xBC92, 0xBC93, 0xBC96, 
    0xBC98, 0xBC9B, 0xBC9C, 0xBC9D, 0xBC9E, 0xBC9F, 0xBCA2, 0xBCA3, 0xBCA5, 
    0xBCA6, 0xBCA9, 0xBCAA, 0xBCAB, 0xBCAC, 0xBCAD, 0xBCAE, 0xBCAF, 0xBCB2, 
    0xBCB6, 0xBCB7, 0xBCB8, 0xBCB9, 0xBCBA, 0xBCBB, 0xBCBE, 0xBCBF, 0xBCC1, 
    0xBCC2, 0xBCC3, 0xBCC5, 0xBCC6, 0xBCC7, 0xBCC8, 0xBCC9, 0xBCCA, 0xBCCB, 
    0xBCCC, 0xBCCE, 0xBCD2, 0xBCD3, 0xBCD4, 0xBCD6, 0xBCD7, 0xBCD9, 0xBCDA, 
    0xBCDB, 0xBCDD, 0xBCDE, 0xBCDF, 0xBCE0, 0xBCE1, 0xBCE2, 0xBCE3, 0xBCE4, 
    0xBCE5, 0xBCE6, 0xBCE7, 0xBCE8, 0xBCE9, 0xBCEA, 0xBCEB, 0xBCEC, 0xBCED, 
    0xBCEE, 0xBCEF, 0xBCF0, 0xBCF1, 0xBCF2, 0xBCF3, 0xBCF7, 0xBCF9, 0xBCFA, 
    0xBCFB, 0xBCFD,  ERROR},
    { ERROR, 0xBCFE, 0xBCFF, 0xBD00, 0xBD01, 0xBD02, 0xBD03, 0xBD06, 0xBD08, 
    0xBD0A, 0xBD0B, 0xBD0C, 0xBD0D, 0xBD0E, 0xBD0F, 0xBD11, 0xBD12, 0xBD13, 
    0xBD15, 0xBD16, 0xBD17, 0xBD18, 0xBD19, 0xBD1A, 0xBD1B, 0xBD1C, 0xBD1D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBD1E, 0xBD1F, 0xBD20, 
    0xBD21, 0xBD22, 0xBD23, 0xBD25, 0xBD26, 0xBD27, 0xBD28, 0xBD29, 0xBD2A, 
    0xBD2B, 0xBD2D, 0xBD2E, 0xBD2F, 0xBD30, 0xBD31, 0xBD32, 0xBD33, 0xBD34, 
    0xBD35, 0xBD36, 0xBD37, 0xBD38, 0xBD39,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBD3A, 0xBD3B, 0xBD3C, 0xBD3D, 0xBD3E, 0xBD3F, 0xBD41, 
    0xBD42, 0xBD43, 0xBD44, 0xBD45, 0xBD46, 0xBD47, 0xBD4A, 0xBD4B, 0xBD4D, 
    0xBD4E, 0xBD4F, 0xBD51, 0xBD52, 0xBD53, 0xBD54, 0xBD55, 0xBD56, 0xBD57, 
    0xBD5A, 0xBD5B, 0xBD5C, 0xBD5D, 0xBD5E, 0xBD5F, 0xBD60, 0xBD61, 0xBD62, 
    0xBD63, 0xBD65, 0xBD66, 0xBD67, 0xBD69, 0xBD6A, 0xBD6B, 0xBD6C, 0xBD6D, 
    0xBD6E, 0xBD6F, 0xBD70, 0xBD71, 0xBD72, 0xBD73, 0xBD74, 0xBD75, 0xBD76, 
    0xBD77, 0xBD78, 0xBD79, 0xBD7A, 0xBD7B, 0xBD7C, 0xBD7D, 0xBD7E, 0xBD7F, 
    0xBD82, 0xBD83, 0xBD85, 0xBD86, 0xBD8B, 0xBD8C, 0xBD8D, 0xBD8E, 0xBD8F, 
    0xBD92, 0xBD94, 0xBD96, 0xBD97, 0xBD98, 0xBD9B, 0xBD9D, 0xBD9E, 0xBD9F, 
    0xBDA0, 0xBDA1, 0xBDA2, 0xBDA3, 0xBDA5, 0xBDA6, 0xBDA7, 0xBDA8, 0xBDA9, 
    0xBDAA, 0xBDAB, 0xBDAC, 0xBDAD, 0xBDAE, 0xBDAF, 0xBDB1, 0xBDB2, 0xBDB3, 
    0xBDB4, 0xBDB5, 0xBDB6, 0xBDB7, 0xBDB9, 0xBDBA, 0xBDBB, 0xBDBC, 0xBDBD, 
    0xBDBE, 0xBDBF, 0xBDC0, 0xBDC1, 0xBDC2, 0xBDC3, 0xBDC4, 0xBDC5, 0xBDC6, 
    0xBDC7, 0xBDC8, 0xBDC9, 0xBDCA, 0xBDCB, 0xBDCC, 0xBDCD, 0xBDCE, 0xBDCF, 
    0xBDD0, 0xBDD1,  ERROR},
    { ERROR, 0xBDD2, 0xBDD3, 0xBDD6, 0xBDD7, 0xBDD9, 0xBDDA, 0xBDDB, 0xBDDD, 
    0xBDDE, 0xBDDF, 0xBDE0, 0xBDE1, 0xBDE2, 0xBDE3, 0xBDE4, 0xBDE5, 0xBDE6, 
    0xBDE7, 0xBDE8, 0xBDEA, 0xBDEB, 0xBDEC, 0xBDED, 0xBDEE, 0xBDEF, 0xBDF1, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBDF2, 0xBDF3, 0xBDF5, 
    0xBDF6, 0xBDF7, 0xBDF9, 0xBDFA, 0xBDFB, 0xBDFC, 0xBDFD, 0xBDFE, 0xBDFF, 
    0xBE01, 0xBE02, 0xBE04, 0xBE06, 0xBE07, 0xBE08, 0xBE09, 0xBE0A, 0xBE0B, 
    0xBE0E, 0xBE0F, 0xBE11, 0xBE12, 0xBE13,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBE15, 0xBE16, 0xBE17, 0xBE18, 0xBE19, 0xBE1A, 0xBE1B, 
    0xBE1E, 0xBE20, 0xBE21, 0xBE22, 0xBE23, 0xBE24, 0xBE25, 0xBE26, 0xBE27, 
    0xBE28, 0xBE29, 0xBE2A, 0xBE2B, 0xBE2C, 0xBE2D, 0xBE2E, 0xBE2F, 0xBE30, 
    0xBE31, 0xBE32, 0xBE33, 0xBE34, 0xBE35, 0xBE36, 0xBE37, 0xBE38, 0xBE39, 
    0xBE3A, 0xBE3B, 0xBE3C, 0xBE3D, 0xBE3E, 0xBE3F, 0xBE40, 0xBE41, 0xBE42, 
    0xBE43, 0xBE46, 0xBE47, 0xBE49, 0xBE4A, 0xBE4B, 0xBE4D, 0xBE4F, 0xBE50, 
    0xBE51, 0xBE52, 0xBE53, 0xBE56, 0xBE58, 0xBE5C, 0xBE5D, 0xBE5E, 0xBE5F, 
    0xBE62, 0xBE63, 0xBE65, 0xBE66, 0xBE67, 0xBE69, 0xBE6B, 0xBE6C, 0xBE6D, 
    0xBE6E, 0xBE6F, 0xBE72, 0xBE76, 0xBE77, 0xBE78, 0xBE79, 0xBE7A, 0xBE7E, 
    0xBE7F, 0xBE81, 0xBE82, 0xBE83, 0xBE85, 0xBE86, 0xBE87, 0xBE88, 0xBE89, 
    0xBE8A, 0xBE8B, 0xBE8E, 0xBE92, 0xBE93, 0xBE94, 0xBE95, 0xBE96, 0xBE97, 
    0xBE9A, 0xBE9B, 0xBE9C, 0xBE9D, 0xBE9E, 0xBE9F, 0xBEA0, 0xBEA1, 0xBEA2, 
    0xBEA3, 0xBEA4, 0xBEA5, 0xBEA6, 0xBEA7, 0xBEA9, 0xBEAA, 0xBEAB, 0xBEAC, 
    0xBEAD, 0xBEAE, 0xBEAF, 0xBEB0, 0xBEB1, 0xBEB2, 0xBEB3, 0xBEB4, 0xBEB5, 
    0xBEB6, 0xBEB7,  ERROR},
    { ERROR, 0xBEB8, 0xBEB9, 0xBEBA, 0xBEBB, 0xBEBC, 0xBEBD, 0xBEBE, 0xBEBF, 
    0xBEC0, 0xBEC1, 0xBEC2, 0xBEC3, 0xBEC4, 0xBEC5, 0xBEC6, 0xBEC7, 0xBEC8, 
    0xBEC9, 0xBECA, 0xBECB, 0xBECC, 0xBECD, 0xBECE, 0xBECF, 0xBED2, 0xBED3, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBED5, 0xBED6, 0xBED9, 
    0xBEDA, 0xBEDB, 0xBEDC, 0xBEDD, 0xBEDE, 0xBEDF, 0xBEE1, 0xBEE2, 0xBEE6, 
    0xBEE7, 0xBEE8, 0xBEE9, 0xBEEA, 0xBEEB, 0xBEED, 0xBEEE, 0xBEEF, 0xBEF0, 
    0xBEF1, 0xBEF2, 0xBEF3, 0xBEF4, 0xBEF5,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBEF6, 0xBEF7, 0xBEF8, 0xBEF9, 0xBEFA, 0xBEFB, 0xBEFC, 
    0xBEFD, 0xBEFE, 0xBEFF, 0xBF00, 0xBF02, 0xBF03, 0xBF04, 0xBF05, 0xBF06, 
    0xBF07, 0xBF0A, 0xBF0B, 0xBF0C, 0xBF0D, 0xBF0E, 0xBF0F, 0xBF10, 0xBF11, 
    0xBF12, 0xBF13, 0xBF14, 0xBF15, 0xBF16, 0xBF17, 0xBF1A, 0xBF1E, 0xBF1F, 
    0xBF20, 0xBF21, 0xBF22, 0xBF23, 0xBF24, 0xBF25, 0xBF26, 0xBF27, 0xBF28, 
    0xBF29, 0xBF2A, 0xBF2B, 0xBF2C, 0xBF2D, 0xBF2E, 0xBF2F, 0xBF30, 0xBF31, 
    0xBF32, 0xBF33, 0xBF34, 0xBF35, 0xBF36, 0xBF37, 0xBF38, 0xBF39, 0xBF3A, 
    0xBF3B, 0xBF3C, 0xBF3D, 0xBF3E, 0xBF3F, 0xBF42, 0xBF43, 0xBF45, 0xBF46, 
    0xBF47, 0xBF49, 0xBF4A, 0xBF4B, 0xBF4C, 0xBF4D, 0xBF4E, 0xBF4F, 0xBF52, 
    0xBF53, 0xBF54, 0xBF56, 0xBF57, 0xBF58, 0xBF59, 0xBF5A, 0xBF5B, 0xBF5C, 
    0xBF5D, 0xBF5E, 0xBF5F, 0xBF60, 0xBF61, 0xBF62, 0xBF63, 0xBF64, 0xBF65, 
    0xBF66, 0xBF67, 0xBF68, 0xBF69, 0xBF6A, 0xBF6B, 0xBF6C, 0xBF6D, 0xBF6E, 
    0xBF6F, 0xBF70, 0xBF71, 0xBF72, 0xBF73, 0xBF74, 0xBF75, 0xBF76, 0xBF77, 
    0xBF78, 0xBF79, 0xBF7A, 0xBF7B, 0xBF7C, 0xBF7D, 0xBF7E, 0xBF7F, 0xBF80, 
    0xBF81, 0xBF82,  ERROR},
    { ERROR, 0xBF83, 0xBF84, 0xBF85, 0xBF86, 0xBF87, 0xBF88, 0xBF89, 0xBF8A, 
    0xBF8B, 0xBF8C, 0xBF8D, 0xBF8E, 0xBF8F, 0xBF90, 0xBF91, 0xBF92, 0xBF93, 
    0xBF95, 0xBF96, 0xBF97, 0xBF98, 0xBF99, 0xBF9A, 0xBF9B, 0xBF9C, 0xBF9D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xBF9E, 0xBF9F, 0xBFA0, 
    0xBFA1, 0xBFA2, 0xBFA3, 0xBFA4, 0xBFA5, 0xBFA6, 0xBFA7, 0xBFA8, 0xBFA9, 
    0xBFAA, 0xBFAB, 0xBFAC, 0xBFAD, 0xBFAE, 0xBFAF, 0xBFB1, 0xBFB2, 0xBFB3, 
    0xBFB4, 0xBFB5, 0xBFB6, 0xBFB7, 0xBFB8,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xBFB9, 0xBFBA, 0xBFBB, 0xBFBC, 0xBFBD, 0xBFBE, 0xBFBF, 
    0xBFC0, 0xBFC1, 0xBFC2, 0xBFC3, 0xBFC4, 0xBFC6, 0xBFC7, 0xBFC8, 0xBFC9, 
    0xBFCA, 0xBFCB, 0xBFCE, 0xBFCF, 0xBFD1, 0xBFD2, 0xBFD3, 0xBFD5, 0xBFD6, 
    0xBFD7, 0xBFD8, 0xBFD9, 0xBFDA, 0xBFDB, 0xBFDD, 0xBFDE, 0xBFE0, 0xBFE2, 
    0xBFE3, 0xBFE4, 0xBFE5, 0xBFE6, 0xBFE7, 0xBFE8, 0xBFE9, 0xBFEA, 0xBFEB, 
    0xBFEC, 0xBFED, 0xBFEE, 0xBFEF, 0xBFF0, 0xBFF1, 0xBFF2, 0xBFF3, 0xBFF4, 
    0xBFF5, 0xBFF6, 0xBFF7, 0xBFF8, 0xBFF9, 0xBFFA, 0xBFFB, 0xBFFC, 0xBFFD, 
    0xBFFE, 0xBFFF, 0xC000, 0xC001, 0xC002, 0xC003, 0xC004, 0xC005, 0xC006, 
    0xC007, 0xC008, 0xC009, 0xC00A, 0xC00B, 0xC00C, 0xC00D, 0xC00E, 0xC00F, 
    0xC010, 0xC011, 0xC012, 0xC013, 0xC014, 0xC015, 0xC016, 0xC017, 0xC018, 
    0xC019, 0xC01A, 0xC01B, 0xC01C, 0xC01D, 0xC01E, 0xC01F, 0xC020, 0xC021, 
    0xC022, 0xC023, 0xC024, 0xC025, 0xC026, 0xC027, 0xC028, 0xC029, 0xC02A, 
    0xC02B, 0xC02C, 0xC02D, 0xC02E, 0xC02F, 0xC030, 0xC031, 0xC032, 0xC033, 
    0xC034, 0xC035, 0xC036, 0xC037, 0xC038, 0xC039, 0xC03A, 0xC03B, 0xC03D, 
    0xC03E, 0xC03F,  ERROR},
    { ERROR, 0xC040, 0xC041, 0xC042, 0xC043, 0xC044, 0xC045, 0xC046, 0xC047, 
    0xC048, 0xC049, 0xC04A, 0xC04B, 0xC04C, 0xC04D, 0xC04E, 0xC04F, 0xC050, 
    0xC052, 0xC053, 0xC054, 0xC055, 0xC056, 0xC057, 0xC059, 0xC05A, 0xC05B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC05D, 0xC05E, 0xC05F, 
    0xC061, 0xC062, 0xC063, 0xC064, 0xC065, 0xC066, 0xC067, 0xC06A, 0xC06B, 
    0xC06C, 0xC06D, 0xC06E, 0xC06F, 0xC070, 0xC071, 0xC072, 0xC073, 0xC074, 
    0xC075, 0xC076, 0xC077, 0xC078, 0xC079,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC07A, 0xC07B, 0xC07C, 0xC07D, 0xC07E, 0xC07F, 0xC080, 
    0xC081, 0xC082, 0xC083, 0xC084, 0xC085, 0xC086, 0xC087, 0xC088, 0xC089, 
    0xC08A, 0xC08B, 0xC08C, 0xC08D, 0xC08E, 0xC08F, 0xC092, 0xC093, 0xC095, 
    0xC096, 0xC097, 0xC099, 0xC09A, 0xC09B, 0xC09C, 0xC09D, 0xC09E, 0xC09F, 
    0xC0A2, 0xC0A4, 0xC0A6, 0xC0A7, 0xC0A8, 0xC0A9, 0xC0AA, 0xC0AB, 0xC0AE, 
    0xC0B1, 0xC0B2, 0xC0B7, 0xC0B8, 0xC0B9, 0xC0BA, 0xC0BB, 0xC0BE, 0xC0C2, 
    0xC0C3, 0xC0C4, 0xC0C6, 0xC0C7, 0xC0CA, 0xC0CB, 0xC0CD, 0xC0CE, 0xC0CF, 
    0xC0D1, 0xC0D2, 0xC0D3, 0xC0D4, 0xC0D5, 0xC0D6, 0xC0D7, 0xC0DA, 0xC0DE, 
    0xC0DF, 0xC0E0, 0xC0E1, 0xC0E2, 0xC0E3, 0xC0E6, 0xC0E7, 0xC0E9, 0xC0EA, 
    0xC0EB, 0xC0ED, 0xC0EE, 0xC0EF, 0xC0F0, 0xC0F1, 0xC0F2, 0xC0F3, 0xC0F6, 
    0xC0F8, 0xC0FA, 0xC0FB, 0xC0FC, 0xC0FD, 0xC0FE, 0xC0FF, 0xC101, 0xC102, 
    0xC103, 0xC105, 0xC106, 0xC107, 0xC109, 0xC10A, 0xC10B, 0xC10C, 0xC10D, 
    0xC10E, 0xC10F, 0xC111, 0xC112, 0xC113, 0xC114, 0xC116, 0xC117, 0xC118, 
    0xC119, 0xC11A, 0xC11B, 0xC121, 0xC122, 0xC125, 0xC128, 0xC129, 0xC12A, 
    0xC12B, 0xC12E,  ERROR},
    { ERROR, 0xC132, 0xC133, 0xC134, 0xC135, 0xC137, 0xC13A, 0xC13B, 0xC13D, 
    0xC13E, 0xC13F, 0xC141, 0xC142, 0xC143, 0xC144, 0xC145, 0xC146, 0xC147, 
    0xC14A, 0xC14E, 0xC14F, 0xC150, 0xC151, 0xC152, 0xC153, 0xC156, 0xC157, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC159, 0xC15A, 0xC15B, 
    0xC15D, 0xC15E, 0xC15F, 0xC160, 0xC161, 0xC162, 0xC163, 0xC166, 0xC16A, 
    0xC16B, 0xC16C, 0xC16D, 0xC16E, 0xC16F, 0xC171, 0xC172, 0xC173, 0xC175, 
    0xC176, 0xC177, 0xC179, 0xC17A, 0xC17B,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC17C, 0xC17D, 0xC17E, 0xC17F, 0xC180, 0xC181, 0xC182, 
    0xC183, 0xC184, 0xC186, 0xC187, 0xC188, 0xC189, 0xC18A, 0xC18B, 0xC18F, 
    0xC191, 0xC192, 0xC193, 0xC195, 0xC197, 0xC198, 0xC199, 0xC19A, 0xC19B, 
    0xC19E, 0xC1A0, 0xC1A2, 0xC1A3, 0xC1A4, 0xC1A6, 0xC1A7, 0xC1AA, 0xC1AB, 
    0xC1AD, 0xC1AE, 0xC1AF, 0xC1B1, 0xC1B2, 0xC1B3, 0xC1B4, 0xC1B5, 0xC1B6, 
    0xC1B7, 0xC1B8, 0xC1B9, 0xC1BA, 0xC1BB, 0xC1BC, 0xC1BE, 0xC1BF, 0xC1C0, 
    0xC1C1, 0xC1C2, 0xC1C3, 0xC1C5, 0xC1C6, 0xC1C7, 0xC1C9, 0xC1CA, 0xC1CB, 
    0xC1CD, 0xC1CE, 0xC1CF, 0xC1D0, 0xC1D1, 0xC1D2, 0xC1D3, 0xC1D5, 0xC1D6, 
    0xC1D9, 0xC1DA, 0xC1DB, 0xC1DC, 0xC1DD, 0xC1DE, 0xC1DF, 0xC1E1, 0xC1E2, 
    0xC1E3, 0xC1E5, 0xC1E6, 0xC1E7, 0xC1E9, 0xC1EA, 0xC1EB, 0xC1EC, 0xC1ED, 
    0xC1EE, 0xC1EF, 0xC1F2, 0xC1F4, 0xC1F5, 0xC1F6, 0xC1F7, 0xC1F8, 0xC1F9, 
    0xC1FA, 0xC1FB, 0xC1FE, 0xC1FF, 0xC201, 0xC202, 0xC203, 0xC205, 0xC206, 
    0xC207, 0xC208, 0xC209, 0xC20A, 0xC20B, 0xC20E, 0xC210, 0xC212, 0xC213, 
    0xC214, 0xC215, 0xC216, 0xC217, 0xC21A, 0xC21B, 0xC21D, 0xC21E, 0xC221, 
    0xC222, 0xC223,  ERROR},
    { ERROR, 0xC224, 0xC225, 0xC226, 0xC227, 0xC22A, 0xC22C, 0xC22E, 0xC230, 
    0xC233, 0xC235, 0xC236, 0xC237, 0xC238, 0xC239, 0xC23A, 0xC23B, 0xC23C, 
    0xC23D, 0xC23E, 0xC23F, 0xC240, 0xC241, 0xC242, 0xC243, 0xC244, 0xC245, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC246, 0xC247, 0xC249, 
    0xC24A, 0xC24B, 0xC24C, 0xC24D, 0xC24E, 0xC24F, 0xC252, 0xC253, 0xC255, 
    0xC256, 0xC257, 0xC259, 0xC25A, 0xC25B, 0xC25C, 0xC25D, 0xC25E, 0xC25F, 
    0xC261, 0xC262, 0xC263, 0xC264, 0xC266,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC267, 0xC268, 0xC269, 0xC26A, 0xC26B, 0xC26E, 0xC26F, 
    0xC271, 0xC272, 0xC273, 0xC275, 0xC276, 0xC277, 0xC278, 0xC279, 0xC27A, 
    0xC27B, 0xC27E, 0xC280, 0xC282, 0xC283, 0xC284, 0xC285, 0xC286, 0xC287, 
    0xC28A, 0xC28B, 0xC28C, 0xC28D, 0xC28E, 0xC28F, 0xC291, 0xC292, 0xC293, 
    0xC294, 0xC295, 0xC296, 0xC297, 0xC299, 0xC29A, 0xC29C, 0xC29E, 0xC29F, 
    0xC2A0, 0xC2A1, 0xC2A2, 0xC2A3, 0xC2A6, 0xC2A7, 0xC2A9, 0xC2AA, 0xC2AB, 
    0xC2AE, 0xC2AF, 0xC2B0, 0xC2B1, 0xC2B2, 0xC2B3, 0xC2B6, 0xC2B8, 0xC2BA, 
    0xC2BB, 0xC2BC, 0xC2BD, 0xC2BE, 0xC2BF, 0xC2C0, 0xC2C1, 0xC2C2, 0xC2C3, 
    0xC2C4, 0xC2C5, 0xC2C6, 0xC2C7, 0xC2C8, 0xC2C9, 0xC2CA, 0xC2CB, 0xC2CC, 
    0xC2CD, 0xC2CE, 0xC2CF, 0xC2D0, 0xC2D1, 0xC2D2, 0xC2D3, 0xC2D4, 0xC2D5, 
    0xC2D6, 0xC2D7, 0xC2D8, 0xC2D9, 0xC2DA, 0xC2DB, 0xC2DE, 0xC2DF, 0xC2E1, 
    0xC2E2, 0xC2E5, 0xC2E6, 0xC2E7, 0xC2E8, 0xC2E9, 0xC2EA, 0xC2EE, 0xC2F0, 
    0xC2F2, 0xC2F3, 0xC2F4, 0xC2F5, 0xC2F7, 0xC2FA, 0xC2FD, 0xC2FE, 0xC2FF, 
    0xC301, 0xC302, 0xC303, 0xC304, 0xC305, 0xC306, 0xC307, 0xC30A, 0xC30B, 
    0xC30E, 0xC30F,  ERROR},
    { ERROR, 0xC310, 0xC311, 0xC312, 0xC316, 0xC317, 0xC319, 0xC31A, 0xC31B, 
    0xC31D, 0xC31E, 0xC31F, 0xC320, 0xC321, 0xC322, 0xC323, 0xC326, 0xC327, 
    0xC32A, 0xC32B, 0xC32C, 0xC32D, 0xC32E, 0xC32F, 0xC330, 0xC331, 0xC332, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC333, 0xC334, 0xC335, 
    0xC336, 0xC337, 0xC338, 0xC339, 0xC33A, 0xC33B, 0xC33C, 0xC33D, 0xC33E, 
    0xC33F, 0xC340, 0xC341, 0xC342, 0xC343, 0xC344, 0xC346, 0xC347, 0xC348, 
    0xC349, 0xC34A, 0xC34B, 0xC34C, 0xC34D,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC34E, 0xC34F, 0xC350, 0xC351, 0xC352, 0xC353, 0xC354, 
    0xC355, 0xC356, 0xC357, 0xC358, 0xC359, 0xC35A, 0xC35B, 0xC35C, 0xC35D, 
    0xC35E, 0xC35F, 0xC360, 0xC361, 0xC362, 0xC363, 0xC364, 0xC365, 0xC366, 
    0xC367, 0xC36A, 0xC36B, 0xC36D, 0xC36E, 0xC36F, 0xC371, 0xC373, 0xC374, 
    0xC375, 0xC376, 0xC377, 0xC37A, 0xC37B, 0xC37E, 0xC37F, 0xC380, 0xC381, 
    0xC382, 0xC383, 0xC385, 0xC386, 0xC387, 0xC389, 0xC38A, 0xC38B, 0xC38D, 
    0xC38E, 0xC38F, 0xC390, 0xC391, 0xC392, 0xC393, 0xC394, 0xC395, 0xC396, 
    0xC397, 0xC398, 0xC399, 0xC39A, 0xC39B, 0xC39C, 0xC39D, 0xC39E, 0xC39F, 
    0xC3A0, 0xC3A1, 0xC3A2, 0xC3A3, 0xC3A4, 0xC3A5, 0xC3A6, 0xC3A7, 0xC3A8, 
    0xC3A9, 0xC3AA, 0xC3AB, 0xC3AC, 0xC3AD, 0xC3AE, 0xC3AF, 0xC3B0, 0xC3B1, 
    0xC3B2, 0xC3B3, 0xC3B4, 0xC3B5, 0xC3B6, 0xC3B7, 0xC3B8, 0xC3B9, 0xC3BA, 
    0xC3BB, 0xC3BC, 0xC3BD, 0xC3BE, 0xC3BF, 0xC3C1, 0xC3C2, 0xC3C3, 0xC3C4, 
    0xC3C5, 0xC3C6, 0xC3C7, 0xC3C8, 0xC3C9, 0xC3CA, 0xC3CB, 0xC3CC, 0xC3CD, 
    0xC3CE, 0xC3CF, 0xC3D0, 0xC3D1, 0xC3D2, 0xC3D3, 0xC3D4, 0xC3D5, 0xC3D6, 
    0xC3D7, 0xC3DA,  ERROR},
    { ERROR, 0xC3DB, 0xC3DD, 0xC3DE, 0xC3E1, 0xC3E3, 0xC3E4, 0xC3E5, 0xC3E6, 
    0xC3E7, 0xC3EA, 0xC3EB, 0xC3EC, 0xC3EE, 0xC3EF, 0xC3F0, 0xC3F1, 0xC3F2, 
    0xC3F3, 0xC3F6, 0xC3F7, 0xC3F9, 0xC3FA, 0xC3FB, 0xC3FC, 0xC3FD, 0xC3FE, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC3FF, 0xC400, 0xC401, 
    0xC402, 0xC403, 0xC404, 0xC405, 0xC406, 0xC407, 0xC409, 0xC40A, 0xC40B, 
    0xC40C, 0xC40D, 0xC40E, 0xC40F, 0xC411, 0xC412, 0xC413, 0xC414, 0xC415, 
    0xC416, 0xC417, 0xC418, 0xC419, 0xC41A,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC41B, 0xC41C, 0xC41D, 0xC41E, 0xC41F, 0xC420, 0xC421, 
    0xC422, 0xC423, 0xC425, 0xC426, 0xC427, 0xC428, 0xC429, 0xC42A, 0xC42B, 
    0xC42D, 0xC42E, 0xC42F, 0xC431, 0xC432, 0xC433, 0xC435, 0xC436, 0xC437, 
    0xC438, 0xC439, 0xC43A, 0xC43B, 0xC43E, 0xC43F, 0xC440, 0xC441, 0xC442, 
    0xC443, 0xC444, 0xC445, 0xC446, 0xC447, 0xC449, 0xC44A, 0xC44B, 0xC44C, 
    0xC44D, 0xC44E, 0xC44F, 0xC450, 0xC451, 0xC452, 0xC453, 0xC454, 0xC455, 
    0xC456, 0xC457, 0xC458, 0xC459, 0xC45A, 0xC45B, 0xC45C, 0xC45D, 0xC45E, 
    0xC45F, 0xC460, 0xC461, 0xC462, 0xC463, 0xC466, 0xC467, 0xC469, 0xC46A, 
    0xC46B, 0xC46D, 0xC46E, 0xC46F, 0xC470, 0xC471, 0xC472, 0xC473, 0xC476, 
    0xC477, 0xC478, 0xC47A, 0xC47B, 0xC47C, 0xC47D, 0xC47E, 0xC47F, 0xC481, 
    0xC482, 0xC483, 0xC484, 0xC485, 0xC486, 0xC487, 0xC488, 0xC489, 0xC48A, 
    0xC48B, 0xC48C, 0xC48D, 0xC48E, 0xC48F, 0xC490, 0xC491, 0xC492, 0xC493, 
    0xC495, 0xC496, 0xC497, 0xC498, 0xC499, 0xC49A, 0xC49B, 0xC49D, 0xC49E, 
    0xC49F, 0xC4A0, 0xC4A1, 0xC4A2, 0xC4A3, 0xC4A4, 0xC4A5, 0xC4A6, 0xC4A7, 
    0xC4A8, 0xC4A9,  ERROR},
    { ERROR, 0xC4AA, 0xC4AB, 0xC4AC, 0xC4AD, 0xC4AE, 0xC4AF, 0xC4B0, 0xC4B1, 
    0xC4B2, 0xC4B3, 0xC4B4, 0xC4B5, 0xC4B6, 0xC4B7, 0xC4B9, 0xC4BA, 0xC4BB, 
    0xC4BD, 0xC4BE, 0xC4BF, 0xC4C0, 0xC4C1, 0xC4C2, 0xC4C3, 0xC4C4, 0xC4C5, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC4C6, 0xC4C7, 0xC4C8, 
    0xC4C9, 0xC4CA, 0xC4CB, 0xC4CC, 0xC4CD, 0xC4CE, 0xC4CF, 0xC4D0, 0xC4D1, 
    0xC4D2, 0xC4D3, 0xC4D4, 0xC4D5, 0xC4D6, 0xC4D7, 0xC4D8, 0xC4D9, 0xC4DA, 
    0xC4DB, 0xC4DC, 0xC4DD, 0xC4DE, 0xC4DF,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC4E0, 0xC4E1, 0xC4E2, 0xC4E3, 0xC4E4, 0xC4E5, 0xC4E6, 
    0xC4E7, 0xC4E8, 0xC4EA, 0xC4EB, 0xC4EC, 0xC4ED, 0xC4EE, 0xC4EF, 0xC4F2, 
    0xC4F3, 0xC4F5, 0xC4F6, 0xC4F7, 0xC4F9, 0xC4FB, 0xC4FC, 0xC4FD, 0xC4FE, 
    0xC502, 0xC503, 0xC504, 0xC505, 0xC506, 0xC507, 0xC508, 0xC509, 0xC50A, 
    0xC50B, 0xC50D, 0xC50E, 0xC50F, 0xC511, 0xC512, 0xC513, 0xC515, 0xC516, 
    0xC517, 0xC518, 0xC519, 0xC51A, 0xC51B, 0xC51D, 0xC51E, 0xC51F, 0xC520, 
    0xC521, 0xC522, 0xC523, 0xC524, 0xC525, 0xC526, 0xC527, 0xC52A, 0xC52B, 
    0xC52D, 0xC52E, 0xC52F, 0xC531, 0xC532, 0xC533, 0xC534, 0xC535, 0xC536, 
    0xC537, 0xC53A, 0xC53C, 0xC53E, 0xC53F, 0xC540, 0xC541, 0xC542, 0xC543, 
    0xC546, 0xC547, 0xC54B, 0xC54F, 0xC550, 0xC551, 0xC552, 0xC556, 0xC55A, 
    0xC55B, 0xC55C, 0xC55F, 0xC562, 0xC563, 0xC565, 0xC566, 0xC567, 0xC569, 
    0xC56A, 0xC56B, 0xC56C, 0xC56D, 0xC56E, 0xC56F, 0xC572, 0xC576, 0xC577, 
    0xC578, 0xC579, 0xC57A, 0xC57B, 0xC57E, 0xC57F, 0xC581, 0xC582, 0xC583, 
    0xC585, 0xC586, 0xC588, 0xC589, 0xC58A, 0xC58B, 0xC58E, 0xC590, 0xC592, 
    0xC593, 0xC594,  ERROR},
    { ERROR, 0xC596, 0xC599, 0xC59A, 0xC59B, 0xC59D, 0xC59E, 0xC59F, 0xC5A1, 
    0xC5A2, 0xC5A3, 0xC5A4, 0xC5A5, 0xC5A6, 0xC5A7, 0xC5A8, 0xC5AA, 0xC5AB, 
    0xC5AC, 0xC5AD, 0xC5AE, 0xC5AF, 0xC5B0, 0xC5B1, 0xC5B2, 0xC5B3, 0xC5B6, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC5B7, 0xC5BA, 0xC5BF, 
    0xC5C0, 0xC5C1, 0xC5C2, 0xC5C3, 0xC5CB, 0xC5CD, 0xC5CF, 0xC5D2, 0xC5D3, 
    0xC5D5, 0xC5D6, 0xC5D7, 0xC5D9, 0xC5DA, 0xC5DB, 0xC5DC, 0xC5DD, 0xC5DE, 
    0xC5DF, 0xC5E2, 0xC5E4, 0xC5E6, 0xC5E7,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC5E8, 0xC5E9, 0xC5EA, 0xC5EB, 0xC5EF, 0xC5F1, 0xC5F2, 
    0xC5F3, 0xC5F5, 0xC5F8, 0xC5F9, 0xC5FA, 0xC5FB, 0xC602, 0xC603, 0xC604, 
    0xC609, 0xC60A, 0xC60B, 0xC60D, 0xC60E, 0xC60F, 0xC611, 0xC612, 0xC613, 
    0xC614, 0xC615, 0xC616, 0xC617, 0xC61A, 0xC61D, 0xC61E, 0xC61F, 0xC620, 
    0xC621, 0xC622, 0xC623, 0xC626, 0xC627, 0xC629, 0xC62A, 0xC62B, 0xC62F, 
    0xC631, 0xC632, 0xC636, 0xC638, 0xC63A, 0xC63C, 0xC63D, 0xC63E, 0xC63F, 
    0xC642, 0xC643, 0xC645, 0xC646, 0xC647, 0xC649, 0xC64A, 0xC64B, 0xC64C, 
    0xC64D, 0xC64E, 0xC64F, 0xC652, 0xC656, 0xC657, 0xC658, 0xC659, 0xC65A, 
    0xC65B, 0xC65E, 0xC65F, 0xC661, 0xC662, 0xC663, 0xC664, 0xC665, 0xC666, 
    0xC667, 0xC668, 0xC669, 0xC66A, 0xC66B, 0xC66D, 0xC66E, 0xC670, 0xC672, 
    0xC673, 0xC674, 0xC675, 0xC676, 0xC677, 0xC67A, 0xC67B, 0xC67D, 0xC67E, 
    0xC67F, 0xC681, 0xC682, 0xC683, 0xC684, 0xC685, 0xC686, 0xC687, 0xC68A, 
    0xC68C, 0xC68E, 0xC68F, 0xC690, 0xC691, 0xC692, 0xC693, 0xC696, 0xC697, 
    0xC699, 0xC69A, 0xC69B, 0xC69D, 0xC69E, 0xC69F, 0xC6A0, 0xC6A1, 0xC6A2, 
    0xC6A3, 0xC6A6,  ERROR},
    { ERROR, 0xC6A8, 0xC6AA, 0xC6AB, 0xC6AC, 0xC6AD, 0xC6AE, 0xC6AF, 0xC6B2, 
    0xC6B3, 0xC6B5, 0xC6B6, 0xC6B7, 0xC6BB, 0xC6BC, 0xC6BD, 0xC6BE, 0xC6BF, 
    0xC6C2, 0xC6C4, 0xC6C6, 0xC6C7, 0xC6C8, 0xC6C9, 0xC6CA, 0xC6CB, 0xC6CE, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC6CF, 0xC6D1, 0xC6D2, 
    0xC6D3, 0xC6D5, 0xC6D6, 0xC6D7, 0xC6D8, 0xC6D9, 0xC6DA, 0xC6DB, 0xC6DE, 
    0xC6DF, 0xC6E2, 0xC6E3, 0xC6E4, 0xC6E5, 0xC6E6, 0xC6E7, 0xC6EA, 0xC6EB, 
    0xC6ED, 0xC6EE, 0xC6EF, 0xC6F1, 0xC6F2,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC6F3, 0xC6F4, 0xC6F5, 0xC6F6, 0xC6F7, 0xC6FA, 0xC6FB, 
    0xC6FC, 0xC6FE, 0xC6FF, 0xC700, 0xC701, 0xC702, 0xC703, 0xC706, 0xC707, 
    0xC709, 0xC70A, 0xC70B, 0xC70D, 0xC70E, 0xC70F, 0xC710, 0xC711, 0xC712, 
    0xC713, 0xC716, 0xC718, 0xC71A, 0xC71B, 0xC71C, 0xC71D, 0xC71E, 0xC71F, 
    0xC722, 0xC723, 0xC725, 0xC726, 0xC727, 0xC729, 0xC72A, 0xC72B, 0xC72C, 
    0xC72D, 0xC72E, 0xC72F, 0xC732, 0xC734, 0xC736, 0xC738, 0xC739, 0xC73A, 
    0xC73B, 0xC73E, 0xC73F, 0xC741, 0xC742, 0xC743, 0xC745, 0xC746, 0xC747, 
    0xC748, 0xC749, 0xC74B, 0xC74E, 0xC750, 0xC759, 0xC75A, 0xC75B, 0xC75D, 
    0xC75E, 0xC75F, 0xC761, 0xC762, 0xC763, 0xC764, 0xC765, 0xC766, 0xC767, 
    0xC769, 0xC76A, 0xC76C, 0xC76D, 0xC76E, 0xC76F, 0xC770, 0xC771, 0xC772, 
    0xC773, 0xC776, 0xC777, 0xC779, 0xC77A, 0xC77B, 0xC77F, 0xC780, 0xC781, 
    0xC782, 0xC786, 0xC78B, 0xC78C, 0xC78D, 0xC78F, 0xC792, 0xC793, 0xC795, 
    0xC799, 0xC79B, 0xC79C, 0xC79D, 0xC79E, 0xC79F, 0xC7A2, 0xC7A7, 0xC7A8, 
    0xC7A9, 0xC7AA, 0xC7AB, 0xC7AE, 0xC7AF, 0xC7B1, 0xC7B2, 0xC7B3, 0xC7B5, 
    0xC7B6, 0xC7B7,  ERROR},
    { ERROR, 0xC7B8, 0xC7B9, 0xC7BA, 0xC7BB, 0xC7BE, 0xC7C2, 0xC7C3, 0xC7C4, 
    0xC7C5, 0xC7C6, 0xC7C7, 0xC7CA, 0xC7CB, 0xC7CD, 0xC7CF, 0xC7D1, 0xC7D2, 
    0xC7D3, 0xC7D4, 0xC7D5, 0xC7D6, 0xC7D7, 0xC7D9, 0xC7DA, 0xC7DB, 0xC7DC, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC7DE, 0xC7DF, 0xC7E0, 
    0xC7E1, 0xC7E2, 0xC7E3, 0xC7E5, 0xC7E6, 0xC7E7, 0xC7E9, 0xC7EA, 0xC7EB, 
    0xC7ED, 0xC7EE, 0xC7EF, 0xC7F0, 0xC7F1, 0xC7F2, 0xC7F3, 0xC7F4, 0xC7F5, 
    0xC7F6, 0xC7F7, 0xC7F8, 0xC7F9, 0xC7FA,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC7FB, 0xC7FC, 0xC7FD, 0xC7FE, 0xC7FF, 0xC802, 0xC803, 
    0xC805, 0xC806, 0xC807, 0xC809, 0xC80B, 0xC80C, 0xC80D, 0xC80E, 0xC80F, 
    0xC812, 0xC814, 0xC817, 0xC818, 0xC819, 0xC81A, 0xC81B, 0xC81E, 0xC81F, 
    0xC821, 0xC822, 0xC823, 0xC825, 0xC826, 0xC827, 0xC828, 0xC829, 0xC82A, 
    0xC82B, 0xC82E, 0xC830, 0xC832, 0xC833, 0xC834, 0xC835, 0xC836, 0xC837, 
    0xC839, 0xC83A, 0xC83B, 0xC83D, 0xC83E, 0xC83F, 0xC841, 0xC842, 0xC843, 
    0xC844, 0xC845, 0xC846, 0xC847, 0xC84A, 0xC84B, 0xC84E, 0xC84F, 0xC850, 
    0xC851, 0xC852, 0xC853, 0xC855, 0xC856, 0xC857, 0xC858, 0xC859, 0xC85A, 
    0xC85B, 0xC85C, 0xC85D, 0xC85E, 0xC85F, 0xC860, 0xC861, 0xC862, 0xC863, 
    0xC864, 0xC865, 0xC866, 0xC867, 0xC868, 0xC869, 0xC86A, 0xC86B, 0xC86C, 
    0xC86D, 0xC86E, 0xC86F, 0xC872, 0xC873, 0xC875, 0xC876, 0xC877, 0xC879, 
    0xC87B, 0xC87C, 0xC87D, 0xC87E, 0xC87F, 0xC882, 0xC884, 0xC888, 0xC889, 
    0xC88A, 0xC88E, 0xC88F, 0xC890, 0xC891, 0xC892, 0xC893, 0xC895, 0xC896, 
    0xC897, 0xC898, 0xC899, 0xC89A, 0xC89B, 0xC89C, 0xC89E, 0xC8A0, 0xC8A2, 
    0xC8A3, 0xC8A4,  ERROR},
    { ERROR, 0xC8A5, 0xC8A6, 0xC8A7, 0xC8A9, 0xC8AA, 0xC8AB, 0xC8AC, 0xC8AD, 
    0xC8AE, 0xC8AF, 0xC8B0, 0xC8B1, 0xC8B2, 0xC8B3, 0xC8B4, 0xC8B5, 0xC8B6, 
    0xC8B7, 0xC8B8, 0xC8B9, 0xC8BA, 0xC8BB, 0xC8BE, 0xC8BF, 0xC8C0, 0xC8C1, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC8C2, 0xC8C3, 0xC8C5, 
    0xC8C6, 0xC8C7, 0xC8C9, 0xC8CA, 0xC8CB, 0xC8CD, 0xC8CE, 0xC8CF, 0xC8D0, 
    0xC8D1, 0xC8D2, 0xC8D3, 0xC8D6, 0xC8D8, 0xC8DA, 0xC8DB, 0xC8DC, 0xC8DD, 
    0xC8DE, 0xC8DF, 0xC8E2, 0xC8E3, 0xC8E5,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC8E6, 0xC8E7, 0xC8E8, 0xC8E9, 0xC8EA, 0xC8EB, 0xC8EC, 
    0xC8ED, 0xC8EE, 0xC8EF, 0xC8F0, 0xC8F1, 0xC8F2, 0xC8F3, 0xC8F4, 0xC8F6, 
    0xC8F7, 0xC8F8, 0xC8F9, 0xC8FA, 0xC8FB, 0xC8FE, 0xC8FF, 0xC901, 0xC902, 
    0xC903, 0xC907, 0xC908, 0xC909, 0xC90A, 0xC90B, 0xC90E, 0x3000, 0x3001, 
    0x3002, 0x00B7, 0x2025, 0x2026, 0x00A8, 0x3003, 0x00AD, 0x2015, 0x2225, 
    0xFF3C, 0x223C, 0x2018, 0x2019, 0x201C, 0x201D, 0x3014, 0x3015, 0x3008, 
    0x3009, 0x300A, 0x300B, 0x300C, 0x300D, 0x300E, 0x300F, 0x3010, 0x3011, 
    0x00B1, 0x00D7, 0x00F7, 0x2260, 0x2264, 0x2265, 0x221E, 0x2234, 0x00B0, 
    0x2032, 0x2033, 0x2103, 0x212B, 0xFFE0, 0xFFE1, 0xFFE5, 0x2642, 0x2640, 
    0x2220, 0x22A5, 0x2312, 0x2202, 0x2207, 0x2261, 0x2252, 0x00A7, 0x203B, 
    0x2606, 0x2605, 0x25CB, 0x25CF, 0x25CE, 0x25C7, 0x25C6, 0x25A1, 0x25A0, 
    0x25B3, 0x25B2, 0x25BD, 0x25BC, 0x2192, 0x2190, 0x2191, 0x2193, 0x2194, 
    0x3013, 0x226A, 0x226B, 0x221A, 0x223D, 0x221D, 0x2235, 0x222B, 0x222C, 
    0x2208, 0x220B, 0x2286, 0x2287, 0x2282, 0x2283, 0x222A, 0x2229, 0x2227, 
    0x2228, 0xFFE2,  ERROR},
    { ERROR, 0xC910, 0xC912, 0xC913, 0xC914, 0xC915, 0xC916, 0xC917, 0xC919, 
    0xC91A, 0xC91B, 0xC91C, 0xC91D, 0xC91E, 0xC91F, 0xC920, 0xC921, 0xC922, 
    0xC923, 0xC924, 0xC925, 0xC926, 0xC927, 0xC928, 0xC929, 0xC92A, 0xC92B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC92D, 0xC92E, 0xC92F, 
    0xC930, 0xC931, 0xC932, 0xC933, 0xC935, 0xC936, 0xC937, 0xC938, 0xC939, 
    0xC93A, 0xC93B, 0xC93C, 0xC93D, 0xC93E, 0xC93F, 0xC940, 0xC941, 0xC942, 
    0xC943, 0xC944, 0xC945, 0xC946, 0xC947,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC948, 0xC949, 0xC94A, 0xC94B, 0xC94C, 0xC94D, 0xC94E, 
    0xC94F, 0xC952, 0xC953, 0xC955, 0xC956, 0xC957, 0xC959, 0xC95A, 0xC95B, 
    0xC95C, 0xC95D, 0xC95E, 0xC95F, 0xC962, 0xC964, 0xC965, 0xC966, 0xC967, 
    0xC968, 0xC969, 0xC96A, 0xC96B, 0xC96D, 0xC96E, 0xC96F, 0x21D2, 0x21D4, 
    0x2200, 0x2203, 0x00B4, 0xFF5E, 0x02C7, 0x02D8, 0x02DD, 0x02DA, 0x02D9, 
    0x00B8, 0x02DB, 0x00A1, 0x00BF, 0x02D0, 0x222E, 0x2211, 0x220F, 0x00A4, 
    0x2109, 0x2030, 0x25C1, 0x25C0, 0x25B7, 0x25B6, 0x2664, 0x2660, 0x2661, 
    0x2665, 0x2667, 0x2663, 0x2299, 0x25C8, 0x25A3, 0x25D0, 0x25D1, 0x2592, 
    0x25A4, 0x25A5, 0x25A8, 0x25A7, 0x25A6, 0x25A9, 0x2668, 0x260F, 0x260E, 
    0x261C, 0x261E, 0x00B6, 0x2020, 0x2021, 0x2195, 0x2197, 0x2199, 0x2196, 
    0x2198, 0x266D, 0x2669, 0x266A, 0x266C, 0x327F, 0x321C, 0x2116, 0x33C7, 
    0x2122, 0x33C2, 0x33D8, 0x2121, 0x20AC, 0x00AE,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xC971, 0xC972, 0xC973, 0xC975, 0xC976, 0xC977, 0xC978, 0xC979, 
    0xC97A, 0xC97B, 0xC97D, 0xC97E, 0xC97F, 0xC980, 0xC981, 0xC982, 0xC983, 
    0xC984, 0xC985, 0xC986, 0xC987, 0xC98A, 0xC98B, 0xC98D, 0xC98E, 0xC98F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xC991, 0xC992, 0xC993, 
    0xC994, 0xC995, 0xC996, 0xC997, 0xC99A, 0xC99C, 0xC99E, 0xC99F, 0xC9A0, 
    0xC9A1, 0xC9A2, 0xC9A3, 0xC9A4, 0xC9A5, 0xC9A6, 0xC9A7, 0xC9A8, 0xC9A9, 
    0xC9AA, 0xC9AB, 0xC9AC, 0xC9AD, 0xC9AE,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xC9AF, 0xC9B0, 0xC9B1, 0xC9B2, 0xC9B3, 0xC9B4, 0xC9B5, 
    0xC9B6, 0xC9B7, 0xC9B8, 0xC9B9, 0xC9BA, 0xC9BB, 0xC9BC, 0xC9BD, 0xC9BE, 
    0xC9BF, 0xC9C2, 0xC9C3, 0xC9C5, 0xC9C6, 0xC9C9, 0xC9CB, 0xC9CC, 0xC9CD, 
    0xC9CE, 0xC9CF, 0xC9D2, 0xC9D4, 0xC9D7, 0xC9D8, 0xC9DB, 0xFF01, 0xFF02, 
    0xFF03, 0xFF04, 0xFF05, 0xFF06, 0xFF07, 0xFF08, 0xFF09, 0xFF0A, 0xFF0B, 
    0xFF0C, 0xFF0D, 0xFF0E, 0xFF0F, 0xFF10, 0xFF11, 0xFF12, 0xFF13, 0xFF14, 
    0xFF15, 0xFF16, 0xFF17, 0xFF18, 0xFF19, 0xFF1A, 0xFF1B, 0xFF1C, 0xFF1D, 
    0xFF1E, 0xFF1F, 0xFF20, 0xFF21, 0xFF22, 0xFF23, 0xFF24, 0xFF25, 0xFF26, 
    0xFF27, 0xFF28, 0xFF29, 0xFF2A, 0xFF2B, 0xFF2C, 0xFF2D, 0xFF2E, 0xFF2F, 
    0xFF30, 0xFF31, 0xFF32, 0xFF33, 0xFF34, 0xFF35, 0xFF36, 0xFF37, 0xFF38, 
    0xFF39, 0xFF3A, 0xFF3B, 0xFFE6, 0xFF3D, 0xFF3E, 0xFF3F, 0xFF40, 0xFF41, 
    0xFF42, 0xFF43, 0xFF44, 0xFF45, 0xFF46, 0xFF47, 0xFF48, 0xFF49, 0xFF4A, 
    0xFF4B, 0xFF4C, 0xFF4D, 0xFF4E, 0xFF4F, 0xFF50, 0xFF51, 0xFF52, 0xFF53, 
    0xFF54, 0xFF55, 0xFF56, 0xFF57, 0xFF58, 0xFF59, 0xFF5A, 0xFF5B, 0xFF5C, 
    0xFF5D, 0xFFE3,  ERROR},
    { ERROR, 0xC9DE, 0xC9DF, 0xC9E1, 0xC9E3, 0xC9E5, 0xC9E6, 0xC9E8, 0xC9E9, 
    0xC9EA, 0xC9EB, 0xC9EE, 0xC9F2, 0xC9F3, 0xC9F4, 0xC9F5, 0xC9F6, 0xC9F7, 
    0xC9FA, 0xC9FB, 0xC9FD, 0xC9FE, 0xC9FF, 0xCA01, 0xCA02, 0xCA03, 0xCA04, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCA05, 0xCA06, 0xCA07, 
    0xCA0A, 0xCA0E, 0xCA0F, 0xCA10, 0xCA11, 0xCA12, 0xCA13, 0xCA15, 0xCA16, 
    0xCA17, 0xCA19, 0xCA1A, 0xCA1B, 0xCA1C, 0xCA1D, 0xCA1E, 0xCA1F, 0xCA20, 
    0xCA21, 0xCA22, 0xCA23, 0xCA24, 0xCA25,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCA26, 0xCA27, 0xCA28, 0xCA2A, 0xCA2B, 0xCA2C, 0xCA2D, 
    0xCA2E, 0xCA2F, 0xCA30, 0xCA31, 0xCA32, 0xCA33, 0xCA34, 0xCA35, 0xCA36, 
    0xCA37, 0xCA38, 0xCA39, 0xCA3A, 0xCA3B, 0xCA3C, 0xCA3D, 0xCA3E, 0xCA3F, 
    0xCA40, 0xCA41, 0xCA42, 0xCA43, 0xCA44, 0xCA45, 0xCA46, 0x3131, 0x3132, 
    0x3133, 0x3134, 0x3135, 0x3136, 0x3137, 0x3138, 0x3139, 0x313A, 0x313B, 
    0x313C, 0x313D, 0x313E, 0x313F, 0x3140, 0x3141, 0x3142, 0x3143, 0x3144, 
    0x3145, 0x3146, 0x3147, 0x3148, 0x3149, 0x314A, 0x314B, 0x314C, 0x314D, 
    0x314E, 0x314F, 0x3150, 0x3151, 0x3152, 0x3153, 0x3154, 0x3155, 0x3156, 
    0x3157, 0x3158, 0x3159, 0x315A, 0x315B, 0x315C, 0x315D, 0x315E, 0x315F, 
    0x3160, 0x3161, 0x3162, 0x3163, 0x3164, 0x3165, 0x3166, 0x3167, 0x3168, 
    0x3169, 0x316A, 0x316B, 0x316C, 0x316D, 0x316E, 0x316F, 0x3170, 0x3171, 
    0x3172, 0x3173, 0x3174, 0x3175, 0x3176, 0x3177, 0x3178, 0x3179, 0x317A, 
    0x317B, 0x317C, 0x317D, 0x317E, 0x317F, 0x3180, 0x3181, 0x3182, 0x3183, 
    0x3184, 0x3185, 0x3186, 0x3187, 0x3188, 0x3189, 0x318A, 0x318B, 0x318C, 
    0x318D, 0x318E,  ERROR},
    { ERROR, 0xCA47, 0xCA48, 0xCA49, 0xCA4A, 0xCA4B, 0xCA4E, 0xCA4F, 0xCA51, 
    0xCA52, 0xCA53, 0xCA55, 0xCA56, 0xCA57, 0xCA58, 0xCA59, 0xCA5A, 0xCA5B, 
    0xCA5E, 0xCA62, 0xCA63, 0xCA64, 0xCA65, 0xCA66, 0xCA67, 0xCA69, 0xCA6A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCA6B, 0xCA6C, 0xCA6D, 
    0xCA6E, 0xCA6F, 0xCA70, 0xCA71, 0xCA72, 0xCA73, 0xCA74, 0xCA75, 0xCA76, 
    0xCA77, 0xCA78, 0xCA79, 0xCA7A, 0xCA7B, 0xCA7C, 0xCA7E, 0xCA7F, 0xCA80, 
    0xCA81, 0xCA82, 0xCA83, 0xCA85, 0xCA86,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCA87, 0xCA88, 0xCA89, 0xCA8A, 0xCA8B, 0xCA8C, 0xCA8D, 
    0xCA8E, 0xCA8F, 0xCA90, 0xCA91, 0xCA92, 0xCA93, 0xCA94, 0xCA95, 0xCA96, 
    0xCA97, 0xCA99, 0xCA9A, 0xCA9B, 0xCA9C, 0xCA9D, 0xCA9E, 0xCA9F, 0xCAA0, 
    0xCAA1, 0xCAA2, 0xCAA3, 0xCAA4, 0xCAA5, 0xCAA6, 0xCAA7, 0x2170, 0x2171, 
    0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177, 0x2178, 0x2179,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 
    0x2165, 0x2166, 0x2167, 0x2168, 0x2169,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 
    0x0397, 0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F, 
    0x03A0, 0x03A1, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, 0x03A8, 0x03A9, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x03B1, 
    0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, 0x03B8, 0x03B9, 0x03BA, 
    0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF, 0x03C0, 0x03C1, 0x03C3, 0x03C4, 
    0x03C5, 0x03C6, 0x03C7, 0x03C8, 0x03C9,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCAA8, 0xCAA9, 0xCAAA, 0xCAAB, 0xCAAC, 0xCAAD, 0xCAAE, 0xCAAF, 
    0xCAB0, 0xCAB1, 0xCAB2, 0xCAB3, 0xCAB4, 0xCAB5, 0xCAB6, 0xCAB7, 0xCAB8, 
    0xCAB9, 0xCABA, 0xCABB, 0xCABE, 0xCABF, 0xCAC1, 0xCAC2, 0xCAC3, 0xCAC5, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCAC6, 0xCAC7, 0xCAC8, 
    0xCAC9, 0xCACA, 0xCACB, 0xCACE, 0xCAD0, 0xCAD2, 0xCAD4, 0xCAD5, 0xCAD6, 
    0xCAD7, 0xCADA, 0xCADB, 0xCADC, 0xCADD, 0xCADE, 0xCADF, 0xCAE1, 0xCAE2, 
    0xCAE3, 0xCAE4, 0xCAE5, 0xCAE6, 0xCAE7,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCAE8, 0xCAE9, 0xCAEA, 0xCAEB, 0xCAED, 0xCAEE, 0xCAEF, 
    0xCAF0, 0xCAF1, 0xCAF2, 0xCAF3, 0xCAF5, 0xCAF6, 0xCAF7, 0xCAF8, 0xCAF9, 
    0xCAFA, 0xCAFB, 0xCAFC, 0xCAFD, 0xCAFE, 0xCAFF, 0xCB00, 0xCB01, 0xCB02, 
    0xCB03, 0xCB04, 0xCB05, 0xCB06, 0xCB07, 0xCB09, 0xCB0A, 0x2500, 0x2502, 
    0x250C, 0x2510, 0x2518, 0x2514, 0x251C, 0x252C, 0x2524, 0x2534, 0x253C, 
    0x2501, 0x2503, 0x250F, 0x2513, 0x251B, 0x2517, 0x2523, 0x2533, 0x252B, 
    0x253B, 0x254B, 0x2520, 0x252F, 0x2528, 0x2537, 0x253F, 0x251D, 0x2530, 
    0x2525, 0x2538, 0x2542, 0x2512, 0x2511, 0x251A, 0x2519, 0x2516, 0x2515, 
    0x250E, 0x250D, 0x251E, 0x251F, 0x2521, 0x2522, 0x2526, 0x2527, 0x2529, 
    0x252A, 0x252D, 0x252E, 0x2531, 0x2532, 0x2535, 0x2536, 0x2539, 0x253A, 
    0x253D, 0x253E, 0x2540, 0x2541, 0x2543, 0x2544, 0x2545, 0x2546, 0x2547, 
    0x2548, 0x2549, 0x254A,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCB0B, 0xCB0C, 0xCB0D, 0xCB0E, 0xCB0F, 0xCB11, 0xCB12, 0xCB13, 
    0xCB15, 0xCB16, 0xCB17, 0xCB19, 0xCB1A, 0xCB1B, 0xCB1C, 0xCB1D, 0xCB1E, 
    0xCB1F, 0xCB22, 0xCB23, 0xCB24, 0xCB25, 0xCB26, 0xCB27, 0xCB28, 0xCB29, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCB2A, 0xCB2B, 0xCB2C, 
    0xCB2D, 0xCB2E, 0xCB2F, 0xCB30, 0xCB31, 0xCB32, 0xCB33, 0xCB34, 0xCB35, 
    0xCB36, 0xCB37, 0xCB38, 0xCB39, 0xCB3A, 0xCB3B, 0xCB3C, 0xCB3D, 0xCB3E, 
    0xCB3F, 0xCB40, 0xCB42, 0xCB43, 0xCB44,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCB45, 0xCB46, 0xCB47, 0xCB4A, 0xCB4B, 0xCB4D, 0xCB4E, 
    0xCB4F, 0xCB51, 0xCB52, 0xCB53, 0xCB54, 0xCB55, 0xCB56, 0xCB57, 0xCB5A, 
    0xCB5B, 0xCB5C, 0xCB5E, 0xCB5F, 0xCB60, 0xCB61, 0xCB62, 0xCB63, 0xCB65, 
    0xCB66, 0xCB67, 0xCB68, 0xCB69, 0xCB6A, 0xCB6B, 0xCB6C, 0x3395, 0x3396, 
    0x3397, 0x2113, 0x3398, 0x33C4, 0x33A3, 0x33A4, 0x33A5, 0x33A6, 0x3399, 
    0x339A, 0x339B, 0x339C, 0x339D, 0x339E, 0x339F, 0x33A0, 0x33A1, 0x33A2, 
    0x33CA, 0x338D, 0x338E, 0x338F, 0x33CF, 0x3388, 0x3389, 0x33C8, 0x33A7, 
    0x33A8, 0x33B0, 0x33B1, 0x33B2, 0x33B3, 0x33B4, 0x33B5, 0x33B6, 0x33B7, 
    0x33B8, 0x33B9, 0x3380, 0x3381, 0x3382, 0x3383, 0x3384, 0x33BA, 0x33BB, 
    0x33BC, 0x33BD, 0x33BE, 0x33BF, 0x3390, 0x3391, 0x3392, 0x3393, 0x3394, 
    0x2126, 0x33C0, 0x33C1, 0x338A, 0x338B, 0x338C, 0x33D6, 0x33C5, 0x33AD, 
    0x33AE, 0x33AF, 0x33DB, 0x33A9, 0x33AA, 0x33AB, 0x33AC, 0x33DD, 0x33D0, 
    0x33D3, 0x33C3, 0x33C9, 0x33DC, 0x33C6,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCB6D, 0xCB6E, 0xCB6F, 0xCB70, 0xCB71, 0xCB72, 0xCB73, 0xCB74, 
    0xCB75, 0xCB76, 0xCB77, 0xCB7A, 0xCB7B, 0xCB7C, 0xCB7D, 0xCB7E, 0xCB7F, 
    0xCB80, 0xCB81, 0xCB82, 0xCB83, 0xCB84, 0xCB85, 0xCB86, 0xCB87, 0xCB88, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCB89, 0xCB8A, 0xCB8B, 
    0xCB8C, 0xCB8D, 0xCB8E, 0xCB8F, 0xCB90, 0xCB91, 0xCB92, 0xCB93, 0xCB94, 
    0xCB95, 0xCB96, 0xCB97, 0xCB98, 0xCB99, 0xCB9A, 0xCB9B, 0xCB9D, 0xCB9E, 
    0xCB9F, 0xCBA0, 0xCBA1, 0xCBA2, 0xCBA3,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCBA4, 0xCBA5, 0xCBA6, 0xCBA7, 0xCBA8, 0xCBA9, 0xCBAA, 
    0xCBAB, 0xCBAC, 0xCBAD, 0xCBAE, 0xCBAF, 0xCBB0, 0xCBB1, 0xCBB2, 0xCBB3, 
    0xCBB4, 0xCBB5, 0xCBB6, 0xCBB7, 0xCBB9, 0xCBBA, 0xCBBB, 0xCBBC, 0xCBBD, 
    0xCBBE, 0xCBBF, 0xCBC0, 0xCBC1, 0xCBC2, 0xCBC3, 0xCBC4, 0x00C6, 0x00D0, 
    0x00AA, 0x0126,  ERROR, 0x0132,  ERROR, 0x013F, 0x0141, 0x00D8, 0x0152, 
    0x00BA, 0x00DE, 0x0166, 0x014A,  ERROR, 0x3260, 0x3261, 0x3262, 0x3263, 
    0x3264, 0x3265, 0x3266, 0x3267, 0x3268, 0x3269, 0x326A, 0x326B, 0x326C, 
    0x326D, 0x326E, 0x326F, 0x3270, 0x3271, 0x3272, 0x3273, 0x3274, 0x3275, 
    0x3276, 0x3277, 0x3278, 0x3279, 0x327A, 0x327B, 0x24D0, 0x24D1, 0x24D2, 
    0x24D3, 0x24D4, 0x24D5, 0x24D6, 0x24D7, 0x24D8, 0x24D9, 0x24DA, 0x24DB, 
    0x24DC, 0x24DD, 0x24DE, 0x24DF, 0x24E0, 0x24E1, 0x24E2, 0x24E3, 0x24E4, 
    0x24E5, 0x24E6, 0x24E7, 0x24E8, 0x24E9, 0x2460, 0x2461, 0x2462, 0x2463, 
    0x2464, 0x2465, 0x2466, 0x2467, 0x2468, 0x2469, 0x246A, 0x246B, 0x246C, 
    0x246D, 0x246E, 0x00BD, 0x2153, 0x2154, 0x00BC, 0x00BE, 0x215B, 0x215C, 
    0x215D, 0x215E,  ERROR},
    { ERROR, 0xCBC5, 0xCBC6, 0xCBC7, 0xCBC8, 0xCBC9, 0xCBCA, 0xCBCB, 0xCBCC, 
    0xCBCD, 0xCBCE, 0xCBCF, 0xCBD0, 0xCBD1, 0xCBD2, 0xCBD3, 0xCBD5, 0xCBD6, 
    0xCBD7, 0xCBD8, 0xCBD9, 0xCBDA, 0xCBDB, 0xCBDC, 0xCBDD, 0xCBDE, 0xCBDF, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCBE0, 0xCBE1, 0xCBE2, 
    0xCBE3, 0xCBE5, 0xCBE6, 0xCBE8, 0xCBEA, 0xCBEB, 0xCBEC, 0xCBED, 0xCBEE, 
    0xCBEF, 0xCBF0, 0xCBF1, 0xCBF2, 0xCBF3, 0xCBF4, 0xCBF5, 0xCBF6, 0xCBF7, 
    0xCBF8, 0xCBF9, 0xCBFA, 0xCBFB, 0xCBFC,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCBFD, 0xCBFE, 0xCBFF, 0xCC00, 0xCC01, 0xCC02, 0xCC03, 
    0xCC04, 0xCC05, 0xCC06, 0xCC07, 0xCC08, 0xCC09, 0xCC0A, 0xCC0B, 0xCC0E, 
    0xCC0F, 0xCC11, 0xCC12, 0xCC13, 0xCC15, 0xCC16, 0xCC17, 0xCC18, 0xCC19, 
    0xCC1A, 0xCC1B, 0xCC1E, 0xCC1F, 0xCC20, 0xCC23, 0xCC24, 0x00E6, 0x0111, 
    0x00F0, 0x0127, 0x0131, 0x0133, 0x0138, 0x0140, 0x0142, 0x00F8, 0x0153, 
    0x00DF, 0x00FE, 0x0167, 0x014B, 0x0149, 0x3200, 0x3201, 0x3202, 0x3203, 
    0x3204, 0x3205, 0x3206, 0x3207, 0x3208, 0x3209, 0x320A, 0x320B, 0x320C, 
    0x320D, 0x320E, 0x320F, 0x3210, 0x3211, 0x3212, 0x3213, 0x3214, 0x3215, 
    0x3216, 0x3217, 0x3218, 0x3219, 0x321A, 0x321B, 0x249C, 0x249D, 0x249E, 
    0x249F, 0x24A0, 0x24A1, 0x24A2, 0x24A3, 0x24A4, 0x24A5, 0x24A6, 0x24A7, 
    0x24A8, 0x24A9, 0x24AA, 0x24AB, 0x24AC, 0x24AD, 0x24AE, 0x24AF, 0x24B0, 
    0x24B1, 0x24B2, 0x24B3, 0x24B4, 0x24B5, 0x2474, 0x2475, 0x2476, 0x2477, 
    0x2478, 0x2479, 0x247A, 0x247B, 0x247C, 0x247D, 0x247E, 0x247F, 0x2480, 
    0x2481, 0x2482, 0x00B9, 0x00B2, 0x00B3, 0x2074, 0x207F, 0x2081, 0x2082, 
    0x2083, 0x2084,  ERROR},
    { ERROR, 0xCC25, 0xCC26, 0xCC2A, 0xCC2B, 0xCC2D, 0xCC2F, 0xCC31, 0xCC32, 
    0xCC33, 0xCC34, 0xCC35, 0xCC36, 0xCC37, 0xCC3A, 0xCC3F, 0xCC40, 0xCC41, 
    0xCC42, 0xCC43, 0xCC46, 0xCC47, 0xCC49, 0xCC4A, 0xCC4B, 0xCC4D, 0xCC4E, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCC4F, 0xCC50, 0xCC51, 
    0xCC52, 0xCC53, 0xCC56, 0xCC5A, 0xCC5B, 0xCC5C, 0xCC5D, 0xCC5E, 0xCC5F, 
    0xCC61, 0xCC62, 0xCC63, 0xCC65, 0xCC67, 0xCC69, 0xCC6A, 0xCC6B, 0xCC6C, 
    0xCC6D, 0xCC6E, 0xCC6F, 0xCC71, 0xCC72,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCC73, 0xCC74, 0xCC76, 0xCC77, 0xCC78, 0xCC79, 0xCC7A, 
    0xCC7B, 0xCC7C, 0xCC7D, 0xCC7E, 0xCC7F, 0xCC80, 0xCC81, 0xCC82, 0xCC83, 
    0xCC84, 0xCC85, 0xCC86, 0xCC87, 0xCC88, 0xCC89, 0xCC8A, 0xCC8B, 0xCC8C, 
    0xCC8D, 0xCC8E, 0xCC8F, 0xCC90, 0xCC91, 0xCC92, 0xCC93, 0x3041, 0x3042, 
    0x3043, 0x3044, 0x3045, 0x3046, 0x3047, 0x3048, 0x3049, 0x304A, 0x304B, 
    0x304C, 0x304D, 0x304E, 0x304F, 0x3050, 0x3051, 0x3052, 0x3053, 0x3054, 
    0x3055, 0x3056, 0x3057, 0x3058, 0x3059, 0x305A, 0x305B, 0x305C, 0x305D, 
    0x305E, 0x305F, 0x3060, 0x3061, 0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 
    0x3067, 0x3068, 0x3069, 0x306A, 0x306B, 0x306C, 0x306D, 0x306E, 0x306F, 
    0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077, 0x3078, 
    0x3079, 0x307A, 0x307B, 0x307C, 0x307D, 0x307E, 0x307F, 0x3080, 0x3081, 
    0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087, 0x3088, 0x3089, 0x308A, 
    0x308B, 0x308C, 0x308D, 0x308E, 0x308F, 0x3090, 0x3091, 0x3092, 0x3093, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCC94, 0xCC95, 0xCC96, 0xCC97, 0xCC9A, 0xCC9B, 0xCC9D, 0xCC9E, 
    0xCC9F, 0xCCA1, 0xCCA2, 0xCCA3, 0xCCA4, 0xCCA5, 0xCCA6, 0xCCA7, 0xCCAA, 
    0xCCAE, 0xCCAF, 0xCCB0, 0xCCB1, 0xCCB2, 0xCCB3, 0xCCB6, 0xCCB7, 0xCCB9, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCCBA, 0xCCBB, 0xCCBD, 
    0xCCBE, 0xCCBF, 0xCCC0, 0xCCC1, 0xCCC2, 0xCCC3, 0xCCC6, 0xCCC8, 0xCCCA, 
    0xCCCB, 0xCCCC, 0xCCCD, 0xCCCE, 0xCCCF, 0xCCD1, 0xCCD2, 0xCCD3, 0xCCD5, 
    0xCCD6, 0xCCD7, 0xCCD8, 0xCCD9, 0xCCDA,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCCDB, 0xCCDC, 0xCCDD, 0xCCDE, 0xCCDF, 0xCCE0, 0xCCE1, 
    0xCCE2, 0xCCE3, 0xCCE5, 0xCCE6, 0xCCE7, 0xCCE8, 0xCCE9, 0xCCEA, 0xCCEB, 
    0xCCED, 0xCCEE, 0xCCEF, 0xCCF1, 0xCCF2, 0xCCF3, 0xCCF4, 0xCCF5, 0xCCF6, 
    0xCCF7, 0xCCF8, 0xCCF9, 0xCCFA, 0xCCFB, 0xCCFC, 0xCCFD, 0x30A1, 0x30A2, 
    0x30A3, 0x30A4, 0x30A5, 0x30A6, 0x30A7, 0x30A8, 0x30A9, 0x30AA, 0x30AB, 
    0x30AC, 0x30AD, 0x30AE, 0x30AF, 0x30B0, 0x30B1, 0x30B2, 0x30B3, 0x30B4, 
    0x30B5, 0x30B6, 0x30B7, 0x30B8, 0x30B9, 0x30BA, 0x30BB, 0x30BC, 0x30BD, 
    0x30BE, 0x30BF, 0x30C0, 0x30C1, 0x30C2, 0x30C3, 0x30C4, 0x30C5, 0x30C6, 
    0x30C7, 0x30C8, 0x30C9, 0x30CA, 0x30CB, 0x30CC, 0x30CD, 0x30CE, 0x30CF, 
    0x30D0, 0x30D1, 0x30D2, 0x30D3, 0x30D4, 0x30D5, 0x30D6, 0x30D7, 0x30D8, 
    0x30D9, 0x30DA, 0x30DB, 0x30DC, 0x30DD, 0x30DE, 0x30DF, 0x30E0, 0x30E1, 
    0x30E2, 0x30E3, 0x30E4, 0x30E5, 0x30E6, 0x30E7, 0x30E8, 0x30E9, 0x30EA, 
    0x30EB, 0x30EC, 0x30ED, 0x30EE, 0x30EF, 0x30F0, 0x30F1, 0x30F2, 0x30F3, 
    0x30F4, 0x30F5, 0x30F6,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCCFE, 0xCCFF, 0xCD00, 0xCD02, 0xCD03, 0xCD04, 0xCD05, 0xCD06, 
    0xCD07, 0xCD0A, 0xCD0B, 0xCD0D, 0xCD0E, 0xCD0F, 0xCD11, 0xCD12, 0xCD13, 
    0xCD14, 0xCD15, 0xCD16, 0xCD17, 0xCD1A, 0xCD1C, 0xCD1E, 0xCD1F, 0xCD20, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCD21, 0xCD22, 0xCD23, 
    0xCD25, 0xCD26, 0xCD27, 0xCD29, 0xCD2A, 0xCD2B, 0xCD2D, 0xCD2E, 0xCD2F, 
    0xCD30, 0xCD31, 0xCD32, 0xCD33, 0xCD34, 0xCD35, 0xCD36, 0xCD37, 0xCD38, 
    0xCD3A, 0xCD3B, 0xCD3C, 0xCD3D, 0xCD3E,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCD3F, 0xCD40, 0xCD41, 0xCD42, 0xCD43, 0xCD44, 0xCD45, 
    0xCD46, 0xCD47, 0xCD48, 0xCD49, 0xCD4A, 0xCD4B, 0xCD4C, 0xCD4D, 0xCD4E, 
    0xCD4F, 0xCD50, 0xCD51, 0xCD52, 0xCD53, 0xCD54, 0xCD55, 0xCD56, 0xCD57, 
    0xCD58, 0xCD59, 0xCD5A, 0xCD5B, 0xCD5D, 0xCD5E, 0xCD5F, 0x0410, 0x0411, 
    0x0412, 0x0413, 0x0414, 0x0415, 0x0401, 0x0416, 0x0417, 0x0418, 0x0419, 
    0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F, 0x0420, 0x0421, 0x0422, 
    0x0423, 0x0424, 0x0425, 0x0426, 0x0427, 0x0428, 0x0429, 0x042A, 0x042B, 
    0x042C, 0x042D, 0x042E, 0x042F,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451, 0x0436, 
    0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, 
    0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448, 
    0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCD61, 0xCD62, 0xCD63, 0xCD65, 0xCD66, 0xCD67, 0xCD68, 0xCD69, 
    0xCD6A, 0xCD6B, 0xCD6E, 0xCD70, 0xCD72, 0xCD73, 0xCD74, 0xCD75, 0xCD76, 
    0xCD77, 0xCD79, 0xCD7A, 0xCD7B, 0xCD7C, 0xCD7D, 0xCD7E, 0xCD7F, 0xCD80, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCD81, 0xCD82, 0xCD83, 
    0xCD84, 0xCD85, 0xCD86, 0xCD87, 0xCD89, 0xCD8A, 0xCD8B, 0xCD8C, 0xCD8D, 
    0xCD8E, 0xCD8F, 0xCD90, 0xCD91, 0xCD92, 0xCD93, 0xCD96, 0xCD97, 0xCD99, 
    0xCD9A, 0xCD9B, 0xCD9D, 0xCD9E, 0xCD9F,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCDA0, 0xCDA1, 0xCDA2, 0xCDA3, 0xCDA6, 0xCDA8, 0xCDAA, 
    0xCDAB, 0xCDAC, 0xCDAD, 0xCDAE, 0xCDAF, 0xCDB1, 0xCDB2, 0xCDB3, 0xCDB4, 
    0xCDB5, 0xCDB6, 0xCDB7, 0xCDB8, 0xCDB9, 0xCDBA, 0xCDBB, 0xCDBC, 0xCDBD, 
    0xCDBE, 0xCDBF, 0xCDC0, 0xCDC1, 0xCDC2, 0xCDC3, 0xCDC5,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCDC6, 0xCDC7, 0xCDC8, 0xCDC9, 0xCDCA, 0xCDCB, 0xCDCD, 0xCDCE, 
    0xCDCF, 0xCDD1, 0xCDD2, 0xCDD3, 0xCDD4, 0xCDD5, 0xCDD6, 0xCDD7, 0xCDD8, 
    0xCDD9, 0xCDDA, 0xCDDB, 0xCDDC, 0xCDDD, 0xCDDE, 0xCDDF, 0xCDE0, 0xCDE1, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCDE2, 0xCDE3, 0xCDE4, 
    0xCDE5, 0xCDE6, 0xCDE7, 0xCDE9, 0xCDEA, 0xCDEB, 0xCDED, 0xCDEE, 0xCDEF, 
    0xCDF1, 0xCDF2, 0xCDF3, 0xCDF4, 0xCDF5, 0xCDF6, 0xCDF7, 0xCDFA, 0xCDFC, 
    0xCDFE, 0xCDFF, 0xCE00, 0xCE01, 0xCE02,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCE03, 0xCE05, 0xCE06, 0xCE07, 0xCE09, 0xCE0A, 0xCE0B, 
    0xCE0D, 0xCE0E, 0xCE0F, 0xCE10, 0xCE11, 0xCE12, 0xCE13, 0xCE15, 0xCE16, 
    0xCE17, 0xCE18, 0xCE1A, 0xCE1B, 0xCE1C, 0xCE1D, 0xCE1E, 0xCE1F, 0xCE22, 
    0xCE23, 0xCE25, 0xCE26, 0xCE27, 0xCE29, 0xCE2A, 0xCE2B,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCE2C, 0xCE2D, 0xCE2E, 0xCE2F, 0xCE32, 0xCE34, 0xCE36, 0xCE37, 
    0xCE38, 0xCE39, 0xCE3A, 0xCE3B, 0xCE3C, 0xCE3D, 0xCE3E, 0xCE3F, 0xCE40, 
    0xCE41, 0xCE42, 0xCE43, 0xCE44, 0xCE45, 0xCE46, 0xCE47, 0xCE48, 0xCE49, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCE4A, 0xCE4B, 0xCE4C, 
    0xCE4D, 0xCE4E, 0xCE4F, 0xCE50, 0xCE51, 0xCE52, 0xCE53, 0xCE54, 0xCE55, 
    0xCE56, 0xCE57, 0xCE5A, 0xCE5B, 0xCE5D, 0xCE5E, 0xCE62, 0xCE63, 0xCE64, 
    0xCE65, 0xCE66, 0xCE67, 0xCE6A, 0xCE6C,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCE6E, 0xCE6F, 0xCE70, 0xCE71, 0xCE72, 0xCE73, 0xCE76, 
    0xCE77, 0xCE79, 0xCE7A, 0xCE7B, 0xCE7D, 0xCE7E, 0xCE7F, 0xCE80, 0xCE81, 
    0xCE82, 0xCE83, 0xCE86, 0xCE88, 0xCE8A, 0xCE8B, 0xCE8C, 0xCE8D, 0xCE8E, 
    0xCE8F, 0xCE92, 0xCE93, 0xCE95, 0xCE96, 0xCE97, 0xCE99,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR, 0xCE9A, 0xCE9B, 0xCE9C, 0xCE9D, 0xCE9E, 0xCE9F, 0xCEA2, 0xCEA6, 
    0xCEA7, 0xCEA8, 0xCEA9, 0xCEAA, 0xCEAB, 0xCEAE, 0xCEAF, 0xCEB0, 0xCEB1, 
    0xCEB2, 0xCEB3, 0xCEB4, 0xCEB5, 0xCEB6, 0xCEB7, 0xCEB8, 0xCEB9, 0xCEBA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCEBB, 0xCEBC, 0xCEBD, 
    0xCEBE, 0xCEBF, 0xCEC0, 0xCEC2, 0xCEC3, 0xCEC4, 0xCEC5, 0xCEC6, 0xCEC7, 
    0xCEC8, 0xCEC9, 0xCECA, 0xCECB, 0xCECC, 0xCECD, 0xCECE, 0xCECF, 0xCED0, 
    0xCED1, 0xCED2, 0xCED3, 0xCED4, 0xCED5,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCED6, 0xCED7, 0xCED8, 0xCED9, 0xCEDA, 0xCEDB, 0xCEDC, 
    0xCEDD, 0xCEDE, 0xCEDF, 0xCEE0, 0xCEE1, 0xCEE2, 0xCEE3, 0xCEE6, 0xCEE7, 
    0xCEE9, 0xCEEA, 0xCEED, 0xCEEE, 0xCEEF, 0xCEF0, 0xCEF1, 0xCEF2, 0xCEF3, 
    0xCEF6, 0xCEFA, 0xCEFB, 0xCEFC, 0xCEFD, 0xCEFE, 0xCEFF, 0xAC00, 0xAC01, 
    0xAC04, 0xAC07, 0xAC08, 0xAC09, 0xAC0A, 0xAC10, 0xAC11, 0xAC12, 0xAC13, 
    0xAC14, 0xAC15, 0xAC16, 0xAC17, 0xAC19, 0xAC1A, 0xAC1B, 0xAC1C, 0xAC1D, 
    0xAC20, 0xAC24, 0xAC2C, 0xAC2D, 0xAC2F, 0xAC30, 0xAC31, 0xAC38, 0xAC39, 
    0xAC3C, 0xAC40, 0xAC4B, 0xAC4D, 0xAC54, 0xAC58, 0xAC5C, 0xAC70, 0xAC71, 
    0xAC74, 0xAC77, 0xAC78, 0xAC7A, 0xAC80, 0xAC81, 0xAC83, 0xAC84, 0xAC85, 
    0xAC86, 0xAC89, 0xAC8A, 0xAC8B, 0xAC8C, 0xAC90, 0xAC94, 0xAC9C, 0xAC9D, 
    0xAC9F, 0xACA0, 0xACA1, 0xACA8, 0xACA9, 0xACAA, 0xACAC, 0xACAF, 0xACB0, 
    0xACB8, 0xACB9, 0xACBB, 0xACBC, 0xACBD, 0xACC1, 0xACC4, 0xACC8, 0xACCC, 
    0xACD5, 0xACD7, 0xACE0, 0xACE1, 0xACE4, 0xACE7, 0xACE8, 0xACEA, 0xACEC, 
    0xACEF, 0xACF0, 0xACF1, 0xACF3, 0xACF5, 0xACF6, 0xACFC, 0xACFD, 0xAD00, 
    0xAD04, 0xAD06,  ERROR},
    { ERROR, 0xCF02, 0xCF03, 0xCF05, 0xCF06, 0xCF07, 0xCF09, 0xCF0A, 0xCF0B, 
    0xCF0C, 0xCF0D, 0xCF0E, 0xCF0F, 0xCF12, 0xCF14, 0xCF16, 0xCF17, 0xCF18, 
    0xCF19, 0xCF1A, 0xCF1B, 0xCF1D, 0xCF1E, 0xCF1F, 0xCF21, 0xCF22, 0xCF23, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCF25, 0xCF26, 0xCF27, 
    0xCF28, 0xCF29, 0xCF2A, 0xCF2B, 0xCF2E, 0xCF32, 0xCF33, 0xCF34, 0xCF35, 
    0xCF36, 0xCF37, 0xCF39, 0xCF3A, 0xCF3B, 0xCF3C, 0xCF3D, 0xCF3E, 0xCF3F, 
    0xCF40, 0xCF41, 0xCF42, 0xCF43, 0xCF44,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCF45, 0xCF46, 0xCF47, 0xCF48, 0xCF49, 0xCF4A, 0xCF4B, 
    0xCF4C, 0xCF4D, 0xCF4E, 0xCF4F, 0xCF50, 0xCF51, 0xCF52, 0xCF53, 0xCF56, 
    0xCF57, 0xCF59, 0xCF5A, 0xCF5B, 0xCF5D, 0xCF5E, 0xCF5F, 0xCF60, 0xCF61, 
    0xCF62, 0xCF63, 0xCF66, 0xCF68, 0xCF6A, 0xCF6B, 0xCF6C, 0xAD0C, 0xAD0D, 
    0xAD0F, 0xAD11, 0xAD18, 0xAD1C, 0xAD20, 0xAD29, 0xAD2C, 0xAD2D, 0xAD34, 
    0xAD35, 0xAD38, 0xAD3C, 0xAD44, 0xAD45, 0xAD47, 0xAD49, 0xAD50, 0xAD54, 
    0xAD58, 0xAD61, 0xAD63, 0xAD6C, 0xAD6D, 0xAD70, 0xAD73, 0xAD74, 0xAD75, 
    0xAD76, 0xAD7B, 0xAD7C, 0xAD7D, 0xAD7F, 0xAD81, 0xAD82, 0xAD88, 0xAD89, 
    0xAD8C, 0xAD90, 0xAD9C, 0xAD9D, 0xADA4, 0xADB7, 0xADC0, 0xADC1, 0xADC4, 
    0xADC8, 0xADD0, 0xADD1, 0xADD3, 0xADDC, 0xADE0, 0xADE4, 0xADF8, 0xADF9, 
    0xADFC, 0xADFF, 0xAE00, 0xAE01, 0xAE08, 0xAE09, 0xAE0B, 0xAE0D, 0xAE14, 
    0xAE30, 0xAE31, 0xAE34, 0xAE37, 0xAE38, 0xAE3A, 0xAE40, 0xAE41, 0xAE43, 
    0xAE45, 0xAE46, 0xAE4A, 0xAE4C, 0xAE4D, 0xAE4E, 0xAE50, 0xAE54, 0xAE56, 
    0xAE5C, 0xAE5D, 0xAE5F, 0xAE60, 0xAE61, 0xAE65, 0xAE68, 0xAE69, 0xAE6C, 
    0xAE70, 0xAE78,  ERROR},
    { ERROR, 0xCF6D, 0xCF6E, 0xCF6F, 0xCF72, 0xCF73, 0xCF75, 0xCF76, 0xCF77, 
    0xCF79, 0xCF7A, 0xCF7B, 0xCF7C, 0xCF7D, 0xCF7E, 0xCF7F, 0xCF81, 0xCF82, 
    0xCF83, 0xCF84, 0xCF86, 0xCF87, 0xCF88, 0xCF89, 0xCF8A, 0xCF8B, 0xCF8D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCF8E, 0xCF8F, 0xCF90, 
    0xCF91, 0xCF92, 0xCF93, 0xCF94, 0xCF95, 0xCF96, 0xCF97, 0xCF98, 0xCF99, 
    0xCF9A, 0xCF9B, 0xCF9C, 0xCF9D, 0xCF9E, 0xCF9F, 0xCFA0, 0xCFA2, 0xCFA3, 
    0xCFA4, 0xCFA5, 0xCFA6, 0xCFA7, 0xCFA9,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xCFAA, 0xCFAB, 0xCFAC, 0xCFAD, 0xCFAE, 0xCFAF, 0xCFB1, 
    0xCFB2, 0xCFB3, 0xCFB4, 0xCFB5, 0xCFB6, 0xCFB7, 0xCFB8, 0xCFB9, 0xCFBA, 
    0xCFBB, 0xCFBC, 0xCFBD, 0xCFBE, 0xCFBF, 0xCFC0, 0xCFC1, 0xCFC2, 0xCFC3, 
    0xCFC5, 0xCFC6, 0xCFC7, 0xCFC8, 0xCFC9, 0xCFCA, 0xCFCB, 0xAE79, 0xAE7B, 
    0xAE7C, 0xAE7D, 0xAE84, 0xAE85, 0xAE8C, 0xAEBC, 0xAEBD, 0xAEBE, 0xAEC0, 
    0xAEC4, 0xAECC, 0xAECD, 0xAECF, 0xAED0, 0xAED1, 0xAED8, 0xAED9, 0xAEDC, 
    0xAEE8, 0xAEEB, 0xAEED, 0xAEF4, 0xAEF8, 0xAEFC, 0xAF07, 0xAF08, 0xAF0D, 
    0xAF10, 0xAF2C, 0xAF2D, 0xAF30, 0xAF32, 0xAF34, 0xAF3C, 0xAF3D, 0xAF3F, 
    0xAF41, 0xAF42, 0xAF43, 0xAF48, 0xAF49, 0xAF50, 0xAF5C, 0xAF5D, 0xAF64, 
    0xAF65, 0xAF79, 0xAF80, 0xAF84, 0xAF88, 0xAF90, 0xAF91, 0xAF95, 0xAF9C, 
    0xAFB8, 0xAFB9, 0xAFBC, 0xAFC0, 0xAFC7, 0xAFC8, 0xAFC9, 0xAFCB, 0xAFCD, 
    0xAFCE, 0xAFD4, 0xAFDC, 0xAFE8, 0xAFE9, 0xAFF0, 0xAFF1, 0xAFF4, 0xAFF8, 
    0xB000, 0xB001, 0xB004, 0xB00C, 0xB010, 0xB014, 0xB01C, 0xB01D, 0xB028, 
    0xB044, 0xB045, 0xB048, 0xB04A, 0xB04C, 0xB04E, 0xB053, 0xB054, 0xB055, 
    0xB057, 0xB059,  ERROR},
    { ERROR, 0xCFCC, 0xCFCD, 0xCFCE, 0xCFCF, 0xCFD0, 0xCFD1, 0xCFD2, 0xCFD3, 
    0xCFD4, 0xCFD5, 0xCFD6, 0xCFD7, 0xCFD8, 0xCFD9, 0xCFDA, 0xCFDB, 0xCFDC, 
    0xCFDD, 0xCFDE, 0xCFDF, 0xCFE2, 0xCFE3, 0xCFE5, 0xCFE6, 0xCFE7, 0xCFE9, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xCFEA, 0xCFEB, 0xCFEC, 
    0xCFED, 0xCFEE, 0xCFEF, 0xCFF2, 0xCFF4, 0xCFF6, 0xCFF7, 0xCFF8, 0xCFF9, 
    0xCFFA, 0xCFFB, 0xCFFD, 0xCFFE, 0xCFFF, 0xD001, 0xD002, 0xD003, 0xD005, 
    0xD006, 0xD007, 0xD008, 0xD009, 0xD00A,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD00B, 0xD00C, 0xD00D, 0xD00E, 0xD00F, 0xD010, 0xD012, 
    0xD013, 0xD014, 0xD015, 0xD016, 0xD017, 0xD019, 0xD01A, 0xD01B, 0xD01C, 
    0xD01D, 0xD01E, 0xD01F, 0xD020, 0xD021, 0xD022, 0xD023, 0xD024, 0xD025, 
    0xD026, 0xD027, 0xD028, 0xD029, 0xD02A, 0xD02B, 0xD02C, 0xB05D, 0xB07C, 
    0xB07D, 0xB080, 0xB084, 0xB08C, 0xB08D, 0xB08F, 0xB091, 0xB098, 0xB099, 
    0xB09A, 0xB09C, 0xB09F, 0xB0A0, 0xB0A1, 0xB0A2, 0xB0A8, 0xB0A9, 0xB0AB, 
    0xB0AC, 0xB0AD, 0xB0AE, 0xB0AF, 0xB0B1, 0xB0B3, 0xB0B4, 0xB0B5, 0xB0B8, 
    0xB0BC, 0xB0C4, 0xB0C5, 0xB0C7, 0xB0C8, 0xB0C9, 0xB0D0, 0xB0D1, 0xB0D4, 
    0xB0D8, 0xB0E0, 0xB0E5, 0xB108, 0xB109, 0xB10B, 0xB10C, 0xB110, 0xB112, 
    0xB113, 0xB118, 0xB119, 0xB11B, 0xB11C, 0xB11D, 0xB123, 0xB124, 0xB125, 
    0xB128, 0xB12C, 0xB134, 0xB135, 0xB137, 0xB138, 0xB139, 0xB140, 0xB141, 
    0xB144, 0xB148, 0xB150, 0xB151, 0xB154, 0xB155, 0xB158, 0xB15C, 0xB160, 
    0xB178, 0xB179, 0xB17C, 0xB180, 0xB182, 0xB188, 0xB189, 0xB18B, 0xB18D, 
    0xB192, 0xB193, 0xB194, 0xB198, 0xB19C, 0xB1A8, 0xB1CC, 0xB1D0, 0xB1D4, 
    0xB1DC, 0xB1DD,  ERROR},
    { ERROR, 0xD02E, 0xD02F, 0xD030, 0xD031, 0xD032, 0xD033, 0xD036, 0xD037, 
    0xD039, 0xD03A, 0xD03B, 0xD03D, 0xD03E, 0xD03F, 0xD040, 0xD041, 0xD042, 
    0xD043, 0xD046, 0xD048, 0xD04A, 0xD04B, 0xD04C, 0xD04D, 0xD04E, 0xD04F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD051, 0xD052, 0xD053, 
    0xD055, 0xD056, 0xD057, 0xD059, 0xD05A, 0xD05B, 0xD05C, 0xD05D, 0xD05E, 
    0xD05F, 0xD061, 0xD062, 0xD063, 0xD064, 0xD065, 0xD066, 0xD067, 0xD068, 
    0xD069, 0xD06A, 0xD06B, 0xD06E, 0xD06F,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD071, 0xD072, 0xD073, 0xD075, 0xD076, 0xD077, 0xD078, 
    0xD079, 0xD07A, 0xD07B, 0xD07E, 0xD07F, 0xD080, 0xD082, 0xD083, 0xD084, 
    0xD085, 0xD086, 0xD087, 0xD088, 0xD089, 0xD08A, 0xD08B, 0xD08C, 0xD08D, 
    0xD08E, 0xD08F, 0xD090, 0xD091, 0xD092, 0xD093, 0xD094, 0xB1DF, 0xB1E8, 
    0xB1E9, 0xB1EC, 0xB1F0, 0xB1F9, 0xB1FB, 0xB1FD, 0xB204, 0xB205, 0xB208, 
    0xB20B, 0xB20C, 0xB214, 0xB215, 0xB217, 0xB219, 0xB220, 0xB234, 0xB23C, 
    0xB258, 0xB25C, 0xB260, 0xB268, 0xB269, 0xB274, 0xB275, 0xB27C, 0xB284, 
    0xB285, 0xB289, 0xB290, 0xB291, 0xB294, 0xB298, 0xB299, 0xB29A, 0xB2A0, 
    0xB2A1, 0xB2A3, 0xB2A5, 0xB2A6, 0xB2AA, 0xB2AC, 0xB2B0, 0xB2B4, 0xB2C8, 
    0xB2C9, 0xB2CC, 0xB2D0, 0xB2D2, 0xB2D8, 0xB2D9, 0xB2DB, 0xB2DD, 0xB2E2, 
    0xB2E4, 0xB2E5, 0xB2E6, 0xB2E8, 0xB2EB, 0xB2EC, 0xB2ED, 0xB2EE, 0xB2EF, 
    0xB2F3, 0xB2F4, 0xB2F5, 0xB2F7, 0xB2F8, 0xB2F9, 0xB2FA, 0xB2FB, 0xB2FF, 
    0xB300, 0xB301, 0xB304, 0xB308, 0xB310, 0xB311, 0xB313, 0xB314, 0xB315, 
    0xB31C, 0xB354, 0xB355, 0xB356, 0xB358, 0xB35B, 0xB35C, 0xB35E, 0xB35F, 
    0xB364, 0xB365,  ERROR},
    { ERROR, 0xD095, 0xD096, 0xD097, 0xD098, 0xD099, 0xD09A, 0xD09B, 0xD09C, 
    0xD09D, 0xD09E, 0xD09F, 0xD0A0, 0xD0A1, 0xD0A2, 0xD0A3, 0xD0A6, 0xD0A7, 
    0xD0A9, 0xD0AA, 0xD0AB, 0xD0AD, 0xD0AE, 0xD0AF, 0xD0B0, 0xD0B1, 0xD0B2, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD0B3, 0xD0B6, 0xD0B8, 
    0xD0BA, 0xD0BB, 0xD0BC, 0xD0BD, 0xD0BE, 0xD0BF, 0xD0C2, 0xD0C3, 0xD0C5, 
    0xD0C6, 0xD0C7, 0xD0CA, 0xD0CB, 0xD0CC, 0xD0CD, 0xD0CE, 0xD0CF, 0xD0D2, 
    0xD0D6, 0xD0D7, 0xD0D8, 0xD0D9, 0xD0DA,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD0DB, 0xD0DE, 0xD0DF, 0xD0E1, 0xD0E2, 0xD0E3, 0xD0E5, 
    0xD0E6, 0xD0E7, 0xD0E8, 0xD0E9, 0xD0EA, 0xD0EB, 0xD0EE, 0xD0F2, 0xD0F3, 
    0xD0F4, 0xD0F5, 0xD0F6, 0xD0F7, 0xD0F9, 0xD0FA, 0xD0FB, 0xD0FC, 0xD0FD, 
    0xD0FE, 0xD0FF, 0xD100, 0xD101, 0xD102, 0xD103, 0xD104, 0xB367, 0xB369, 
    0xB36B, 0xB36E, 0xB370, 0xB371, 0xB374, 0xB378, 0xB380, 0xB381, 0xB383, 
    0xB384, 0xB385, 0xB38C, 0xB390, 0xB394, 0xB3A0, 0xB3A1, 0xB3A8, 0xB3AC, 
    0xB3C4, 0xB3C5, 0xB3C8, 0xB3CB, 0xB3CC, 0xB3CE, 0xB3D0, 0xB3D4, 0xB3D5, 
    0xB3D7, 0xB3D9, 0xB3DB, 0xB3DD, 0xB3E0, 0xB3E4, 0xB3E8, 0xB3FC, 0xB410, 
    0xB418, 0xB41C, 0xB420, 0xB428, 0xB429, 0xB42B, 0xB434, 0xB450, 0xB451, 
    0xB454, 0xB458, 0xB460, 0xB461, 0xB463, 0xB465, 0xB46C, 0xB480, 0xB488, 
    0xB49D, 0xB4A4, 0xB4A8, 0xB4AC, 0xB4B5, 0xB4B7, 0xB4B9, 0xB4C0, 0xB4C4, 
    0xB4C8, 0xB4D0, 0xB4D5, 0xB4DC, 0xB4DD, 0xB4E0, 0xB4E3, 0xB4E4, 0xB4E6, 
    0xB4EC, 0xB4ED, 0xB4EF, 0xB4F1, 0xB4F8, 0xB514, 0xB515, 0xB518, 0xB51B, 
    0xB51C, 0xB524, 0xB525, 0xB527, 0xB528, 0xB529, 0xB52A, 0xB530, 0xB531, 
    0xB534, 0xB538,  ERROR},
    { ERROR, 0xD105, 0xD106, 0xD107, 0xD108, 0xD109, 0xD10A, 0xD10B, 0xD10C, 
    0xD10E, 0xD10F, 0xD110, 0xD111, 0xD112, 0xD113, 0xD114, 0xD115, 0xD116, 
    0xD117, 0xD118, 0xD119, 0xD11A, 0xD11B, 0xD11C, 0xD11D, 0xD11E, 0xD11F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD120, 0xD121, 0xD122, 
    0xD123, 0xD124, 0xD125, 0xD126, 0xD127, 0xD128, 0xD129, 0xD12A, 0xD12B, 
    0xD12C, 0xD12D, 0xD12E, 0xD12F, 0xD132, 0xD133, 0xD135, 0xD136, 0xD137, 
    0xD139, 0xD13B, 0xD13C, 0xD13D, 0xD13E,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD13F, 0xD142, 0xD146, 0xD147, 0xD148, 0xD149, 0xD14A, 
    0xD14B, 0xD14E, 0xD14F, 0xD151, 0xD152, 0xD153, 0xD155, 0xD156, 0xD157, 
    0xD158, 0xD159, 0xD15A, 0xD15B, 0xD15E, 0xD160, 0xD162, 0xD163, 0xD164, 
    0xD165, 0xD166, 0xD167, 0xD169, 0xD16A, 0xD16B, 0xD16D, 0xB540, 0xB541, 
    0xB543, 0xB544, 0xB545, 0xB54B, 0xB54C, 0xB54D, 0xB550, 0xB554, 0xB55C, 
    0xB55D, 0xB55F, 0xB560, 0xB561, 0xB5A0, 0xB5A1, 0xB5A4, 0xB5A8, 0xB5AA, 
    0xB5AB, 0xB5B0, 0xB5B1, 0xB5B3, 0xB5B4, 0xB5B5, 0xB5BB, 0xB5BC, 0xB5BD, 
    0xB5C0, 0xB5C4, 0xB5CC, 0xB5CD, 0xB5CF, 0xB5D0, 0xB5D1, 0xB5D8, 0xB5EC, 
    0xB610, 0xB611, 0xB614, 0xB618, 0xB625, 0xB62C, 0xB634, 0xB648, 0xB664, 
    0xB668, 0xB69C, 0xB69D, 0xB6A0, 0xB6A4, 0xB6AB, 0xB6AC, 0xB6B1, 0xB6D4, 
    0xB6F0, 0xB6F4, 0xB6F8, 0xB700, 0xB701, 0xB705, 0xB728, 0xB729, 0xB72C, 
    0xB72F, 0xB730, 0xB738, 0xB739, 0xB73B, 0xB744, 0xB748, 0xB74C, 0xB754, 
    0xB755, 0xB760, 0xB764, 0xB768, 0xB770, 0xB771, 0xB773, 0xB775, 0xB77C, 
    0xB77D, 0xB780, 0xB784, 0xB78C, 0xB78D, 0xB78F, 0xB790, 0xB791, 0xB792, 
    0xB796, 0xB797,  ERROR},
    { ERROR, 0xD16E, 0xD16F, 0xD170, 0xD171, 0xD172, 0xD173, 0xD174, 0xD175, 
    0xD176, 0xD177, 0xD178, 0xD179, 0xD17A, 0xD17B, 0xD17D, 0xD17E, 0xD17F, 
    0xD180, 0xD181, 0xD182, 0xD183, 0xD185, 0xD186, 0xD187, 0xD189, 0xD18A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD18B, 0xD18C, 0xD18D, 
    0xD18E, 0xD18F, 0xD190, 0xD191, 0xD192, 0xD193, 0xD194, 0xD195, 0xD196, 
    0xD197, 0xD198, 0xD199, 0xD19A, 0xD19B, 0xD19C, 0xD19D, 0xD19E, 0xD19F, 
    0xD1A2, 0xD1A3, 0xD1A5, 0xD1A6, 0xD1A7,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD1A9, 0xD1AA, 0xD1AB, 0xD1AC, 0xD1AD, 0xD1AE, 0xD1AF, 
    0xD1B2, 0xD1B4, 0xD1B6, 0xD1B7, 0xD1B8, 0xD1B9, 0xD1BB, 0xD1BD, 0xD1BE, 
    0xD1BF, 0xD1C1, 0xD1C2, 0xD1C3, 0xD1C4, 0xD1C5, 0xD1C6, 0xD1C7, 0xD1C8, 
    0xD1C9, 0xD1CA, 0xD1CB, 0xD1CC, 0xD1CD, 0xD1CE, 0xD1CF, 0xB798, 0xB799, 
    0xB79C, 0xB7A0, 0xB7A8, 0xB7A9, 0xB7AB, 0xB7AC, 0xB7AD, 0xB7B4, 0xB7B5, 
    0xB7B8, 0xB7C7, 0xB7C9, 0xB7EC, 0xB7ED, 0xB7F0, 0xB7F4, 0xB7FC, 0xB7FD, 
    0xB7FF, 0xB800, 0xB801, 0xB807, 0xB808, 0xB809, 0xB80C, 0xB810, 0xB818, 
    0xB819, 0xB81B, 0xB81D, 0xB824, 0xB825, 0xB828, 0xB82C, 0xB834, 0xB835, 
    0xB837, 0xB838, 0xB839, 0xB840, 0xB844, 0xB851, 0xB853, 0xB85C, 0xB85D, 
    0xB860, 0xB864, 0xB86C, 0xB86D, 0xB86F, 0xB871, 0xB878, 0xB87C, 0xB88D, 
    0xB8A8, 0xB8B0, 0xB8B4, 0xB8B8, 0xB8C0, 0xB8C1, 0xB8C3, 0xB8C5, 0xB8CC, 
    0xB8D0, 0xB8D4, 0xB8DD, 0xB8DF, 0xB8E1, 0xB8E8, 0xB8E9, 0xB8EC, 0xB8F0, 
    0xB8F8, 0xB8F9, 0xB8FB, 0xB8FD, 0xB904, 0xB918, 0xB920, 0xB93C, 0xB93D, 
    0xB940, 0xB944, 0xB94C, 0xB94F, 0xB951, 0xB958, 0xB959, 0xB95C, 0xB960, 
    0xB968, 0xB969,  ERROR},
    { ERROR, 0xD1D0, 0xD1D1, 0xD1D2, 0xD1D3, 0xD1D4, 0xD1D5, 0xD1D6, 0xD1D7, 
    0xD1D9, 0xD1DA, 0xD1DB, 0xD1DC, 0xD1DD, 0xD1DE, 0xD1DF, 0xD1E0, 0xD1E1, 
    0xD1E2, 0xD1E3, 0xD1E4, 0xD1E5, 0xD1E6, 0xD1E7, 0xD1E8, 0xD1E9, 0xD1EA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD1EB, 0xD1EC, 0xD1ED, 
    0xD1EE, 0xD1EF, 0xD1F0, 0xD1F1, 0xD1F2, 0xD1F3, 0xD1F5, 0xD1F6, 0xD1F7, 
    0xD1F9, 0xD1FA, 0xD1FB, 0xD1FC, 0xD1FD, 0xD1FE, 0xD1FF, 0xD200, 0xD201, 
    0xD202, 0xD203, 0xD204, 0xD205, 0xD206,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD208, 0xD20A, 0xD20B, 0xD20C, 0xD20D, 0xD20E, 0xD20F, 
    0xD211, 0xD212, 0xD213, 0xD214, 0xD215, 0xD216, 0xD217, 0xD218, 0xD219, 
    0xD21A, 0xD21B, 0xD21C, 0xD21D, 0xD21E, 0xD21F, 0xD220, 0xD221, 0xD222, 
    0xD223, 0xD224, 0xD225, 0xD226, 0xD227, 0xD228, 0xD229, 0xB96B, 0xB96D, 
    0xB974, 0xB975, 0xB978, 0xB97C, 0xB984, 0xB985, 0xB987, 0xB989, 0xB98A, 
    0xB98D, 0xB98E, 0xB9AC, 0xB9AD, 0xB9B0, 0xB9B4, 0xB9BC, 0xB9BD, 0xB9BF, 
    0xB9C1, 0xB9C8, 0xB9C9, 0xB9CC, 0xB9CE, 0xB9CF, 0xB9D0, 0xB9D1, 0xB9D2, 
    0xB9D8, 0xB9D9, 0xB9DB, 0xB9DD, 0xB9DE, 0xB9E1, 0xB9E3, 0xB9E4, 0xB9E5, 
    0xB9E8, 0xB9EC, 0xB9F4, 0xB9F5, 0xB9F7, 0xB9F8, 0xB9F9, 0xB9FA, 0xBA00, 
    0xBA01, 0xBA08, 0xBA15, 0xBA38, 0xBA39, 0xBA3C, 0xBA40, 0xBA42, 0xBA48, 
    0xBA49, 0xBA4B, 0xBA4D, 0xBA4E, 0xBA53, 0xBA54, 0xBA55, 0xBA58, 0xBA5C, 
    0xBA64, 0xBA65, 0xBA67, 0xBA68, 0xBA69, 0xBA70, 0xBA71, 0xBA74, 0xBA78, 
    0xBA83, 0xBA84, 0xBA85, 0xBA87, 0xBA8C, 0xBAA8, 0xBAA9, 0xBAAB, 0xBAAC, 
    0xBAB0, 0xBAB2, 0xBAB8, 0xBAB9, 0xBABB, 0xBABD, 0xBAC4, 0xBAC8, 0xBAD8, 
    0xBAD9, 0xBAFC,  ERROR},
    { ERROR, 0xD22A, 0xD22B, 0xD22E, 0xD22F, 0xD231, 0xD232, 0xD233, 0xD235, 
    0xD236, 0xD237, 0xD238, 0xD239, 0xD23A, 0xD23B, 0xD23E, 0xD240, 0xD242, 
    0xD243, 0xD244, 0xD245, 0xD246, 0xD247, 0xD249, 0xD24A, 0xD24B, 0xD24C, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD24D, 0xD24E, 0xD24F, 
    0xD250, 0xD251, 0xD252, 0xD253, 0xD254, 0xD255, 0xD256, 0xD257, 0xD258, 
    0xD259, 0xD25A, 0xD25B, 0xD25D, 0xD25E, 0xD25F, 0xD260, 0xD261, 0xD262, 
    0xD263, 0xD265, 0xD266, 0xD267, 0xD268,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD269, 0xD26A, 0xD26B, 0xD26C, 0xD26D, 0xD26E, 0xD26F, 
    0xD270, 0xD271, 0xD272, 0xD273, 0xD274, 0xD275, 0xD276, 0xD277, 0xD278, 
    0xD279, 0xD27A, 0xD27B, 0xD27C, 0xD27D, 0xD27E, 0xD27F, 0xD282, 0xD283, 
    0xD285, 0xD286, 0xD287, 0xD289, 0xD28A, 0xD28B, 0xD28C, 0xBB00, 0xBB04, 
    0xBB0D, 0xBB0F, 0xBB11, 0xBB18, 0xBB1C, 0xBB20, 0xBB29, 0xBB2B, 0xBB34, 
    0xBB35, 0xBB36, 0xBB38, 0xBB3B, 0xBB3C, 0xBB3D, 0xBB3E, 0xBB44, 0xBB45, 
    0xBB47, 0xBB49, 0xBB4D, 0xBB4F, 0xBB50, 0xBB54, 0xBB58, 0xBB61, 0xBB63, 
    0xBB6C, 0xBB88, 0xBB8C, 0xBB90, 0xBBA4, 0xBBA8, 0xBBAC, 0xBBB4, 0xBBB7, 
    0xBBC0, 0xBBC4, 0xBBC8, 0xBBD0, 0xBBD3, 0xBBF8, 0xBBF9, 0xBBFC, 0xBBFF, 
    0xBC00, 0xBC02, 0xBC08, 0xBC09, 0xBC0B, 0xBC0C, 0xBC0D, 0xBC0F, 0xBC11, 
    0xBC14, 0xBC15, 0xBC16, 0xBC17, 0xBC18, 0xBC1B, 0xBC1C, 0xBC1D, 0xBC1E, 
    0xBC1F, 0xBC24, 0xBC25, 0xBC27, 0xBC29, 0xBC2D, 0xBC30, 0xBC31, 0xBC34, 
    0xBC38, 0xBC40, 0xBC41, 0xBC43, 0xBC44, 0xBC45, 0xBC49, 0xBC4C, 0xBC4D, 
    0xBC50, 0xBC5D, 0xBC84, 0xBC85, 0xBC88, 0xBC8B, 0xBC8C, 0xBC8E, 0xBC94, 
    0xBC95, 0xBC97,  ERROR},
    { ERROR, 0xD28D, 0xD28E, 0xD28F, 0xD292, 0xD293, 0xD294, 0xD296, 0xD297, 
    0xD298, 0xD299, 0xD29A, 0xD29B, 0xD29D, 0xD29E, 0xD29F, 0xD2A1, 0xD2A2, 
    0xD2A3, 0xD2A5, 0xD2A6, 0xD2A7, 0xD2A8, 0xD2A9, 0xD2AA, 0xD2AB, 0xD2AD, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD2AE, 0xD2AF, 0xD2B0, 
    0xD2B2, 0xD2B3, 0xD2B4, 0xD2B5, 0xD2B6, 0xD2B7, 0xD2BA, 0xD2BB, 0xD2BD, 
    0xD2BE, 0xD2C1, 0xD2C3, 0xD2C4, 0xD2C5, 0xD2C6, 0xD2C7, 0xD2CA, 0xD2CC, 
    0xD2CD, 0xD2CE, 0xD2CF, 0xD2D0, 0xD2D1,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD2D2, 0xD2D3, 0xD2D5, 0xD2D6, 0xD2D7, 0xD2D9, 0xD2DA, 
    0xD2DB, 0xD2DD, 0xD2DE, 0xD2DF, 0xD2E0, 0xD2E1, 0xD2E2, 0xD2E3, 0xD2E6, 
    0xD2E7, 0xD2E8, 0xD2E9, 0xD2EA, 0xD2EB, 0xD2EC, 0xD2ED, 0xD2EE, 0xD2EF, 
    0xD2F2, 0xD2F3, 0xD2F5, 0xD2F6, 0xD2F7, 0xD2F9, 0xD2FA, 0xBC99, 0xBC9A, 
    0xBCA0, 0xBCA1, 0xBCA4, 0xBCA7, 0xBCA8, 0xBCB0, 0xBCB1, 0xBCB3, 0xBCB4, 
    0xBCB5, 0xBCBC, 0xBCBD, 0xBCC0, 0xBCC4, 0xBCCD, 0xBCCF, 0xBCD0, 0xBCD1, 
    0xBCD5, 0xBCD8, 0xBCDC, 0xBCF4, 0xBCF5, 0xBCF6, 0xBCF8, 0xBCFC, 0xBD04, 
    0xBD05, 0xBD07, 0xBD09, 0xBD10, 0xBD14, 0xBD24, 0xBD2C, 0xBD40, 0xBD48, 
    0xBD49, 0xBD4C, 0xBD50, 0xBD58, 0xBD59, 0xBD64, 0xBD68, 0xBD80, 0xBD81, 
    0xBD84, 0xBD87, 0xBD88, 0xBD89, 0xBD8A, 0xBD90, 0xBD91, 0xBD93, 0xBD95, 
    0xBD99, 0xBD9A, 0xBD9C, 0xBDA4, 0xBDB0, 0xBDB8, 0xBDD4, 0xBDD5, 0xBDD8, 
    0xBDDC, 0xBDE9, 0xBDF0, 0xBDF4, 0xBDF8, 0xBE00, 0xBE03, 0xBE05, 0xBE0C, 
    0xBE0D, 0xBE10, 0xBE14, 0xBE1C, 0xBE1D, 0xBE1F, 0xBE44, 0xBE45, 0xBE48, 
    0xBE4C, 0xBE4E, 0xBE54, 0xBE55, 0xBE57, 0xBE59, 0xBE5A, 0xBE5B, 0xBE60, 
    0xBE61, 0xBE64,  ERROR},
    { ERROR, 0xD2FB, 0xD2FC, 0xD2FD, 0xD2FE, 0xD2FF, 0xD302, 0xD304, 0xD306, 
    0xD307, 0xD308, 0xD309, 0xD30A, 0xD30B, 0xD30F, 0xD311, 0xD312, 0xD313, 
    0xD315, 0xD317, 0xD318, 0xD319, 0xD31A, 0xD31B, 0xD31E, 0xD322, 0xD323, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD324, 0xD326, 0xD327, 
    0xD32A, 0xD32B, 0xD32D, 0xD32E, 0xD32F, 0xD331, 0xD332, 0xD333, 0xD334, 
    0xD335, 0xD336, 0xD337, 0xD33A, 0xD33E, 0xD33F, 0xD340, 0xD341, 0xD342, 
    0xD343, 0xD346, 0xD347, 0xD348, 0xD349,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD34A, 0xD34B, 0xD34C, 0xD34D, 0xD34E, 0xD34F, 0xD350, 
    0xD351, 0xD352, 0xD353, 0xD354, 0xD355, 0xD356, 0xD357, 0xD358, 0xD359, 
    0xD35A, 0xD35B, 0xD35C, 0xD35D, 0xD35E, 0xD35F, 0xD360, 0xD361, 0xD362, 
    0xD363, 0xD364, 0xD365, 0xD366, 0xD367, 0xD368, 0xD369, 0xBE68, 0xBE6A, 
    0xBE70, 0xBE71, 0xBE73, 0xBE74, 0xBE75, 0xBE7B, 0xBE7C, 0xBE7D, 0xBE80, 
    0xBE84, 0xBE8C, 0xBE8D, 0xBE8F, 0xBE90, 0xBE91, 0xBE98, 0xBE99, 0xBEA8, 
    0xBED0, 0xBED1, 0xBED4, 0xBED7, 0xBED8, 0xBEE0, 0xBEE3, 0xBEE4, 0xBEE5, 
    0xBEEC, 0xBF01, 0xBF08, 0xBF09, 0xBF18, 0xBF19, 0xBF1B, 0xBF1C, 0xBF1D, 
    0xBF40, 0xBF41, 0xBF44, 0xBF48, 0xBF50, 0xBF51, 0xBF55, 0xBF94, 0xBFB0, 
    0xBFC5, 0xBFCC, 0xBFCD, 0xBFD0, 0xBFD4, 0xBFDC, 0xBFDF, 0xBFE1, 0xC03C, 
    0xC051, 0xC058, 0xC05C, 0xC060, 0xC068, 0xC069, 0xC090, 0xC091, 0xC094, 
    0xC098, 0xC0A0, 0xC0A1, 0xC0A3, 0xC0A5, 0xC0AC, 0xC0AD, 0xC0AF, 0xC0B0, 
    0xC0B3, 0xC0B4, 0xC0B5, 0xC0B6, 0xC0BC, 0xC0BD, 0xC0BF, 0xC0C0, 0xC0C1, 
    0xC0C5, 0xC0C8, 0xC0C9, 0xC0CC, 0xC0D0, 0xC0D8, 0xC0D9, 0xC0DB, 0xC0DC, 
    0xC0DD, 0xC0E4,  ERROR},
    { ERROR, 0xD36A, 0xD36B, 0xD36C, 0xD36D, 0xD36E, 0xD36F, 0xD370, 0xD371, 
    0xD372, 0xD373, 0xD374, 0xD375, 0xD376, 0xD377, 0xD378, 0xD379, 0xD37A, 
    0xD37B, 0xD37E, 0xD37F, 0xD381, 0xD382, 0xD383, 0xD385, 0xD386, 0xD387, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD388, 0xD389, 0xD38A, 
    0xD38B, 0xD38E, 0xD392, 0xD393, 0xD394, 0xD395, 0xD396, 0xD397, 0xD39A, 
    0xD39B, 0xD39D, 0xD39E, 0xD39F, 0xD3A1, 0xD3A2, 0xD3A3, 0xD3A4, 0xD3A5, 
    0xD3A6, 0xD3A7, 0xD3AA, 0xD3AC, 0xD3AE,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD3AF, 0xD3B0, 0xD3B1, 0xD3B2, 0xD3B3, 0xD3B5, 0xD3B6, 
    0xD3B7, 0xD3B9, 0xD3BA, 0xD3BB, 0xD3BD, 0xD3BE, 0xD3BF, 0xD3C0, 0xD3C1, 
    0xD3C2, 0xD3C3, 0xD3C6, 0xD3C7, 0xD3CA, 0xD3CB, 0xD3CC, 0xD3CD, 0xD3CE, 
    0xD3CF, 0xD3D1, 0xD3D2, 0xD3D3, 0xD3D4, 0xD3D5, 0xD3D6, 0xC0E5, 0xC0E8, 
    0xC0EC, 0xC0F4, 0xC0F5, 0xC0F7, 0xC0F9, 0xC100, 0xC104, 0xC108, 0xC110, 
    0xC115, 0xC11C, 0xC11D, 0xC11E, 0xC11F, 0xC120, 0xC123, 0xC124, 0xC126, 
    0xC127, 0xC12C, 0xC12D, 0xC12F, 0xC130, 0xC131, 0xC136, 0xC138, 0xC139, 
    0xC13C, 0xC140, 0xC148, 0xC149, 0xC14B, 0xC14C, 0xC14D, 0xC154, 0xC155, 
    0xC158, 0xC15C, 0xC164, 0xC165, 0xC167, 0xC168, 0xC169, 0xC170, 0xC174, 
    0xC178, 0xC185, 0xC18C, 0xC18D, 0xC18E, 0xC190, 0xC194, 0xC196, 0xC19C, 
    0xC19D, 0xC19F, 0xC1A1, 0xC1A5, 0xC1A8, 0xC1A9, 0xC1AC, 0xC1B0, 0xC1BD, 
    0xC1C4, 0xC1C8, 0xC1CC, 0xC1D4, 0xC1D7, 0xC1D8, 0xC1E0, 0xC1E4, 0xC1E8, 
    0xC1F0, 0xC1F1, 0xC1F3, 0xC1FC, 0xC1FD, 0xC200, 0xC204, 0xC20C, 0xC20D, 
    0xC20F, 0xC211, 0xC218, 0xC219, 0xC21C, 0xC21F, 0xC220, 0xC228, 0xC229, 
    0xC22B, 0xC22D,  ERROR},
    { ERROR, 0xD3D7, 0xD3D9, 0xD3DA, 0xD3DB, 0xD3DC, 0xD3DD, 0xD3DE, 0xD3DF, 
    0xD3E0, 0xD3E2, 0xD3E4, 0xD3E5, 0xD3E6, 0xD3E7, 0xD3E8, 0xD3E9, 0xD3EA, 
    0xD3EB, 0xD3EE, 0xD3EF, 0xD3F1, 0xD3F2, 0xD3F3, 0xD3F5, 0xD3F6, 0xD3F7, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD3F8, 0xD3F9, 0xD3FA, 
    0xD3FB, 0xD3FE, 0xD400, 0xD402, 0xD403, 0xD404, 0xD405, 0xD406, 0xD407, 
    0xD409, 0xD40A, 0xD40B, 0xD40C, 0xD40D, 0xD40E, 0xD40F, 0xD410, 0xD411, 
    0xD412, 0xD413, 0xD414, 0xD415, 0xD416,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD417, 0xD418, 0xD419, 0xD41A, 0xD41B, 0xD41C, 0xD41E, 
    0xD41F, 0xD420, 0xD421, 0xD422, 0xD423, 0xD424, 0xD425, 0xD426, 0xD427, 
    0xD428, 0xD429, 0xD42A, 0xD42B, 0xD42C, 0xD42D, 0xD42E, 0xD42F, 0xD430, 
    0xD431, 0xD432, 0xD433, 0xD434, 0xD435, 0xD436, 0xD437, 0xC22F, 0xC231, 
    0xC232, 0xC234, 0xC248, 0xC250, 0xC251, 0xC254, 0xC258, 0xC260, 0xC265, 
    0xC26C, 0xC26D, 0xC270, 0xC274, 0xC27C, 0xC27D, 0xC27F, 0xC281, 0xC288, 
    0xC289, 0xC290, 0xC298, 0xC29B, 0xC29D, 0xC2A4, 0xC2A5, 0xC2A8, 0xC2AC, 
    0xC2AD, 0xC2B4, 0xC2B5, 0xC2B7, 0xC2B9, 0xC2DC, 0xC2DD, 0xC2E0, 0xC2E3, 
    0xC2E4, 0xC2EB, 0xC2EC, 0xC2ED, 0xC2EF, 0xC2F1, 0xC2F6, 0xC2F8, 0xC2F9, 
    0xC2FB, 0xC2FC, 0xC300, 0xC308, 0xC309, 0xC30C, 0xC30D, 0xC313, 0xC314, 
    0xC315, 0xC318, 0xC31C, 0xC324, 0xC325, 0xC328, 0xC329, 0xC345, 0xC368, 
    0xC369, 0xC36C, 0xC370, 0xC372, 0xC378, 0xC379, 0xC37C, 0xC37D, 0xC384, 
    0xC388, 0xC38C, 0xC3C0, 0xC3D8, 0xC3D9, 0xC3DC, 0xC3DF, 0xC3E0, 0xC3E2, 
    0xC3E8, 0xC3E9, 0xC3ED, 0xC3F4, 0xC3F5, 0xC3F8, 0xC408, 0xC410, 0xC424, 
    0xC42C, 0xC430,  ERROR},
    { ERROR, 0xD438, 0xD439, 0xD43A, 0xD43B, 0xD43C, 0xD43D, 0xD43E, 0xD43F, 
    0xD441, 0xD442, 0xD443, 0xD445, 0xD446, 0xD447, 0xD448, 0xD449, 0xD44A, 
    0xD44B, 0xD44C, 0xD44D, 0xD44E, 0xD44F, 0xD450, 0xD451, 0xD452, 0xD453, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD454, 0xD455, 0xD456, 
    0xD457, 0xD458, 0xD459, 0xD45A, 0xD45B, 0xD45D, 0xD45E, 0xD45F, 0xD461, 
    0xD462, 0xD463, 0xD465, 0xD466, 0xD467, 0xD468, 0xD469, 0xD46A, 0xD46B, 
    0xD46C, 0xD46E, 0xD470, 0xD471, 0xD472,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD473, 0xD474, 0xD475, 0xD476, 0xD477, 0xD47A, 0xD47B, 
    0xD47D, 0xD47E, 0xD481, 0xD483, 0xD484, 0xD485, 0xD486, 0xD487, 0xD48A, 
    0xD48C, 0xD48E, 0xD48F, 0xD490, 0xD491, 0xD492, 0xD493, 0xD495, 0xD496, 
    0xD497, 0xD498, 0xD499, 0xD49A, 0xD49B, 0xD49C, 0xD49D, 0xC434, 0xC43C, 
    0xC43D, 0xC448, 0xC464, 0xC465, 0xC468, 0xC46C, 0xC474, 0xC475, 0xC479, 
    0xC480, 0xC494, 0xC49C, 0xC4B8, 0xC4BC, 0xC4E9, 0xC4F0, 0xC4F1, 0xC4F4, 
    0xC4F8, 0xC4FA, 0xC4FF, 0xC500, 0xC501, 0xC50C, 0xC510, 0xC514, 0xC51C, 
    0xC528, 0xC529, 0xC52C, 0xC530, 0xC538, 0xC539, 0xC53B, 0xC53D, 0xC544, 
    0xC545, 0xC548, 0xC549, 0xC54A, 0xC54C, 0xC54D, 0xC54E, 0xC553, 0xC554, 
    0xC555, 0xC557, 0xC558, 0xC559, 0xC55D, 0xC55E, 0xC560, 0xC561, 0xC564, 
    0xC568, 0xC570, 0xC571, 0xC573, 0xC574, 0xC575, 0xC57C, 0xC57D, 0xC580, 
    0xC584, 0xC587, 0xC58C, 0xC58D, 0xC58F, 0xC591, 0xC595, 0xC597, 0xC598, 
    0xC59C, 0xC5A0, 0xC5A9, 0xC5B4, 0xC5B5, 0xC5B8, 0xC5B9, 0xC5BB, 0xC5BC, 
    0xC5BD, 0xC5BE, 0xC5C4, 0xC5C5, 0xC5C6, 0xC5C7, 0xC5C8, 0xC5C9, 0xC5CA, 
    0xC5CC, 0xC5CE,  ERROR},
    { ERROR, 0xD49E, 0xD49F, 0xD4A0, 0xD4A1, 0xD4A2, 0xD4A3, 0xD4A4, 0xD4A5, 
    0xD4A6, 0xD4A7, 0xD4A8, 0xD4AA, 0xD4AB, 0xD4AC, 0xD4AD, 0xD4AE, 0xD4AF, 
    0xD4B0, 0xD4B1, 0xD4B2, 0xD4B3, 0xD4B4, 0xD4B5, 0xD4B6, 0xD4B7, 0xD4B8, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD4B9, 0xD4BA, 0xD4BB, 
    0xD4BC, 0xD4BD, 0xD4BE, 0xD4BF, 0xD4C0, 0xD4C1, 0xD4C2, 0xD4C3, 0xD4C4, 
    0xD4C5, 0xD4C6, 0xD4C7, 0xD4C8, 0xD4C9, 0xD4CA, 0xD4CB, 0xD4CD, 0xD4CE, 
    0xD4CF, 0xD4D1, 0xD4D2, 0xD4D3, 0xD4D5,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD4D6, 0xD4D7, 0xD4D8, 0xD4D9, 0xD4DA, 0xD4DB, 0xD4DD, 
    0xD4DE, 0xD4E0, 0xD4E1, 0xD4E2, 0xD4E3, 0xD4E4, 0xD4E5, 0xD4E6, 0xD4E7, 
    0xD4E9, 0xD4EA, 0xD4EB, 0xD4ED, 0xD4EE, 0xD4EF, 0xD4F1, 0xD4F2, 0xD4F3, 
    0xD4F4, 0xD4F5, 0xD4F6, 0xD4F7, 0xD4F9, 0xD4FA, 0xD4FC, 0xC5D0, 0xC5D1, 
    0xC5D4, 0xC5D8, 0xC5E0, 0xC5E1, 0xC5E3, 0xC5E5, 0xC5EC, 0xC5ED, 0xC5EE, 
    0xC5F0, 0xC5F4, 0xC5F6, 0xC5F7, 0xC5FC, 0xC5FD, 0xC5FE, 0xC5FF, 0xC600, 
    0xC601, 0xC605, 0xC606, 0xC607, 0xC608, 0xC60C, 0xC610, 0xC618, 0xC619, 
    0xC61B, 0xC61C, 0xC624, 0xC625, 0xC628, 0xC62C, 0xC62D, 0xC62E, 0xC630, 
    0xC633, 0xC634, 0xC635, 0xC637, 0xC639, 0xC63B, 0xC640, 0xC641, 0xC644, 
    0xC648, 0xC650, 0xC651, 0xC653, 0xC654, 0xC655, 0xC65C, 0xC65D, 0xC660, 
    0xC66C, 0xC66F, 0xC671, 0xC678, 0xC679, 0xC67C, 0xC680, 0xC688, 0xC689, 
    0xC68B, 0xC68D, 0xC694, 0xC695, 0xC698, 0xC69C, 0xC6A4, 0xC6A5, 0xC6A7, 
    0xC6A9, 0xC6B0, 0xC6B1, 0xC6B4, 0xC6B8, 0xC6B9, 0xC6BA, 0xC6C0, 0xC6C1, 
    0xC6C3, 0xC6C5, 0xC6CC, 0xC6CD, 0xC6D0, 0xC6D4, 0xC6DC, 0xC6DD, 0xC6E0, 
    0xC6E1, 0xC6E8,  ERROR},
    { ERROR, 0xD4FE, 0xD4FF, 0xD500, 0xD501, 0xD502, 0xD503, 0xD505, 0xD506, 
    0xD507, 0xD509, 0xD50A, 0xD50B, 0xD50D, 0xD50E, 0xD50F, 0xD510, 0xD511, 
    0xD512, 0xD513, 0xD516, 0xD518, 0xD519, 0xD51A, 0xD51B, 0xD51C, 0xD51D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD51E, 0xD51F, 0xD520, 
    0xD521, 0xD522, 0xD523, 0xD524, 0xD525, 0xD526, 0xD527, 0xD528, 0xD529, 
    0xD52A, 0xD52B, 0xD52C, 0xD52D, 0xD52E, 0xD52F, 0xD530, 0xD531, 0xD532, 
    0xD533, 0xD534, 0xD535, 0xD536, 0xD537,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD538, 0xD539, 0xD53A, 0xD53B, 0xD53E, 0xD53F, 0xD541, 
    0xD542, 0xD543, 0xD545, 0xD546, 0xD547, 0xD548, 0xD549, 0xD54A, 0xD54B, 
    0xD54E, 0xD550, 0xD552, 0xD553, 0xD554, 0xD555, 0xD556, 0xD557, 0xD55A, 
    0xD55B, 0xD55D, 0xD55E, 0xD55F, 0xD561, 0xD562, 0xD563, 0xC6E9, 0xC6EC, 
    0xC6F0, 0xC6F8, 0xC6F9, 0xC6FD, 0xC704, 0xC705, 0xC708, 0xC70C, 0xC714, 
    0xC715, 0xC717, 0xC719, 0xC720, 0xC721, 0xC724, 0xC728, 0xC730, 0xC731, 
    0xC733, 0xC735, 0xC737, 0xC73C, 0xC73D, 0xC740, 0xC744, 0xC74A, 0xC74C, 
    0xC74D, 0xC74F, 0xC751, 0xC752, 0xC753, 0xC754, 0xC755, 0xC756, 0xC757, 
    0xC758, 0xC75C, 0xC760, 0xC768, 0xC76B, 0xC774, 0xC775, 0xC778, 0xC77C, 
    0xC77D, 0xC77E, 0xC783, 0xC784, 0xC785, 0xC787, 0xC788, 0xC789, 0xC78A, 
    0xC78E, 0xC790, 0xC791, 0xC794, 0xC796, 0xC797, 0xC798, 0xC79A, 0xC7A0, 
    0xC7A1, 0xC7A3, 0xC7A4, 0xC7A5, 0xC7A6, 0xC7AC, 0xC7AD, 0xC7B0, 0xC7B4, 
    0xC7BC, 0xC7BD, 0xC7BF, 0xC7C0, 0xC7C1, 0xC7C8, 0xC7C9, 0xC7CC, 0xC7CE, 
    0xC7D0, 0xC7D8, 0xC7DD, 0xC7E4, 0xC7E8, 0xC7EC, 0xC800, 0xC801, 0xC804, 
    0xC808, 0xC80A,  ERROR},
    { ERROR, 0xD564, 0xD566, 0xD567, 0xD56A, 0xD56C, 0xD56E, 0xD56F, 0xD570, 
    0xD571, 0xD572, 0xD573, 0xD576, 0xD577, 0xD579, 0xD57A, 0xD57B, 0xD57D, 
    0xD57E, 0xD57F, 0xD580, 0xD581, 0xD582, 0xD583, 0xD586, 0xD58A, 0xD58B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD58C, 0xD58D, 0xD58E, 
    0xD58F, 0xD591, 0xD592, 0xD593, 0xD594, 0xD595, 0xD596, 0xD597, 0xD598, 
    0xD599, 0xD59A, 0xD59B, 0xD59C, 0xD59D, 0xD59E, 0xD59F, 0xD5A0, 0xD5A1, 
    0xD5A2, 0xD5A3, 0xD5A4, 0xD5A6, 0xD5A7,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD5A8, 0xD5A9, 0xD5AA, 0xD5AB, 0xD5AC, 0xD5AD, 0xD5AE, 
    0xD5AF, 0xD5B0, 0xD5B1, 0xD5B2, 0xD5B3, 0xD5B4, 0xD5B5, 0xD5B6, 0xD5B7, 
    0xD5B8, 0xD5B9, 0xD5BA, 0xD5BB, 0xD5BC, 0xD5BD, 0xD5BE, 0xD5BF, 0xD5C0, 
    0xD5C1, 0xD5C2, 0xD5C3, 0xD5C4, 0xD5C5, 0xD5C6, 0xD5C7, 0xC810, 0xC811, 
    0xC813, 0xC815, 0xC816, 0xC81C, 0xC81D, 0xC820, 0xC824, 0xC82C, 0xC82D, 
    0xC82F, 0xC831, 0xC838, 0xC83C, 0xC840, 0xC848, 0xC849, 0xC84C, 0xC84D, 
    0xC854, 0xC870, 0xC871, 0xC874, 0xC878, 0xC87A, 0xC880, 0xC881, 0xC883, 
    0xC885, 0xC886, 0xC887, 0xC88B, 0xC88C, 0xC88D, 0xC894, 0xC89D, 0xC89F, 
    0xC8A1, 0xC8A8, 0xC8BC, 0xC8BD, 0xC8C4, 0xC8C8, 0xC8CC, 0xC8D4, 0xC8D5, 
    0xC8D7, 0xC8D9, 0xC8E0, 0xC8E1, 0xC8E4, 0xC8F5, 0xC8FC, 0xC8FD, 0xC900, 
    0xC904, 0xC905, 0xC906, 0xC90C, 0xC90D, 0xC90F, 0xC911, 0xC918, 0xC92C, 
    0xC934, 0xC950, 0xC951, 0xC954, 0xC958, 0xC960, 0xC961, 0xC963, 0xC96C, 
    0xC970, 0xC974, 0xC97C, 0xC988, 0xC989, 0xC98C, 0xC990, 0xC998, 0xC999, 
    0xC99B, 0xC99D, 0xC9C0, 0xC9C1, 0xC9C4, 0xC9C7, 0xC9C8, 0xC9CA, 0xC9D0, 
    0xC9D1, 0xC9D3,  ERROR},
    { ERROR, 0xD5CA, 0xD5CB, 0xD5CD, 0xD5CE, 0xD5CF, 0xD5D1, 0xD5D3, 0xD5D4, 
    0xD5D5, 0xD5D6, 0xD5D7, 0xD5DA, 0xD5DC, 0xD5DE, 0xD5DF, 0xD5E0, 0xD5E1, 
    0xD5E2, 0xD5E3, 0xD5E6, 0xD5E7, 0xD5E9, 0xD5EA, 0xD5EB, 0xD5ED, 0xD5EE, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD5EF, 0xD5F0, 0xD5F1, 
    0xD5F2, 0xD5F3, 0xD5F6, 0xD5F8, 0xD5FA, 0xD5FB, 0xD5FC, 0xD5FD, 0xD5FE, 
    0xD5FF, 0xD602, 0xD603, 0xD605, 0xD606, 0xD607, 0xD609, 0xD60A, 0xD60B, 
    0xD60C, 0xD60D, 0xD60E, 0xD60F, 0xD612,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD616, 0xD617, 0xD618, 0xD619, 0xD61A, 0xD61B, 0xD61D, 
    0xD61E, 0xD61F, 0xD621, 0xD622, 0xD623, 0xD625, 0xD626, 0xD627, 0xD628, 
    0xD629, 0xD62A, 0xD62B, 0xD62C, 0xD62E, 0xD62F, 0xD630, 0xD631, 0xD632, 
    0xD633, 0xD634, 0xD635, 0xD636, 0xD637, 0xD63A, 0xD63B, 0xC9D5, 0xC9D6, 
    0xC9D9, 0xC9DA, 0xC9DC, 0xC9DD, 0xC9E0, 0xC9E2, 0xC9E4, 0xC9E7, 0xC9EC, 
    0xC9ED, 0xC9EF, 0xC9F0, 0xC9F1, 0xC9F8, 0xC9F9, 0xC9FC, 0xCA00, 0xCA08, 
    0xCA09, 0xCA0B, 0xCA0C, 0xCA0D, 0xCA14, 0xCA18, 0xCA29, 0xCA4C, 0xCA4D, 
    0xCA50, 0xCA54, 0xCA5C, 0xCA5D, 0xCA5F, 0xCA60, 0xCA61, 0xCA68, 0xCA7D, 
    0xCA84, 0xCA98, 0xCABC, 0xCABD, 0xCAC0, 0xCAC4, 0xCACC, 0xCACD, 0xCACF, 
    0xCAD1, 0xCAD3, 0xCAD8, 0xCAD9, 0xCAE0, 0xCAEC, 0xCAF4, 0xCB08, 0xCB10, 
    0xCB14, 0xCB18, 0xCB20, 0xCB21, 0xCB41, 0xCB48, 0xCB49, 0xCB4C, 0xCB50, 
    0xCB58, 0xCB59, 0xCB5D, 0xCB64, 0xCB78, 0xCB79, 0xCB9C, 0xCBB8, 0xCBD4, 
    0xCBE4, 0xCBE7, 0xCBE9, 0xCC0C, 0xCC0D, 0xCC10, 0xCC14, 0xCC1C, 0xCC1D, 
    0xCC21, 0xCC22, 0xCC27, 0xCC28, 0xCC29, 0xCC2C, 0xCC2E, 0xCC30, 0xCC38, 
    0xCC39, 0xCC3B,  ERROR},
    { ERROR, 0xD63D, 0xD63E, 0xD63F, 0xD641, 0xD642, 0xD643, 0xD644, 0xD646, 
    0xD647, 0xD64A, 0xD64C, 0xD64E, 0xD64F, 0xD650, 0xD652, 0xD653, 0xD656, 
    0xD657, 0xD659, 0xD65A, 0xD65B, 0xD65D, 0xD65E, 0xD65F, 0xD660, 0xD661, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD662, 0xD663, 0xD664, 
    0xD665, 0xD666, 0xD668, 0xD66A, 0xD66B, 0xD66C, 0xD66D, 0xD66E, 0xD66F, 
    0xD672, 0xD673, 0xD675, 0xD676, 0xD677, 0xD678, 0xD679, 0xD67A, 0xD67B, 
    0xD67C, 0xD67D, 0xD67E, 0xD67F, 0xD680,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD681, 0xD682, 0xD684, 0xD686, 0xD687, 0xD688, 0xD689, 
    0xD68A, 0xD68B, 0xD68E, 0xD68F, 0xD691, 0xD692, 0xD693, 0xD695, 0xD696, 
    0xD697, 0xD698, 0xD699, 0xD69A, 0xD69B, 0xD69C, 0xD69E, 0xD6A0, 0xD6A2, 
    0xD6A3, 0xD6A4, 0xD6A5, 0xD6A6, 0xD6A7, 0xD6A9, 0xD6AA, 0xCC3C, 0xCC3D, 
    0xCC3E, 0xCC44, 0xCC45, 0xCC48, 0xCC4C, 0xCC54, 0xCC55, 0xCC57, 0xCC58, 
    0xCC59, 0xCC60, 0xCC64, 0xCC66, 0xCC68, 0xCC70, 0xCC75, 0xCC98, 0xCC99, 
    0xCC9C, 0xCCA0, 0xCCA8, 0xCCA9, 0xCCAB, 0xCCAC, 0xCCAD, 0xCCB4, 0xCCB5, 
    0xCCB8, 0xCCBC, 0xCCC4, 0xCCC5, 0xCCC7, 0xCCC9, 0xCCD0, 0xCCD4, 0xCCE4, 
    0xCCEC, 0xCCF0, 0xCD01, 0xCD08, 0xCD09, 0xCD0C, 0xCD10, 0xCD18, 0xCD19, 
    0xCD1B, 0xCD1D, 0xCD24, 0xCD28, 0xCD2C, 0xCD39, 0xCD5C, 0xCD60, 0xCD64, 
    0xCD6C, 0xCD6D, 0xCD6F, 0xCD71, 0xCD78, 0xCD88, 0xCD94, 0xCD95, 0xCD98, 
    0xCD9C, 0xCDA4, 0xCDA5, 0xCDA7, 0xCDA9, 0xCDB0, 0xCDC4, 0xCDCC, 0xCDD0, 
    0xCDE8, 0xCDEC, 0xCDF0, 0xCDF8, 0xCDF9, 0xCDFB, 0xCDFD, 0xCE04, 0xCE08, 
    0xCE0C, 0xCE14, 0xCE19, 0xCE20, 0xCE21, 0xCE24, 0xCE28, 0xCE30, 0xCE31, 
    0xCE33, 0xCE35,  ERROR},
    { ERROR, 0xD6AB, 0xD6AD, 0xD6AE, 0xD6AF, 0xD6B1, 0xD6B2, 0xD6B3, 0xD6B4, 
    0xD6B5, 0xD6B6, 0xD6B7, 0xD6B8, 0xD6BA, 0xD6BC, 0xD6BD, 0xD6BE, 0xD6BF, 
    0xD6C0, 0xD6C1, 0xD6C2, 0xD6C3, 0xD6C6, 0xD6C7, 0xD6C9, 0xD6CA, 0xD6CB, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD6CD, 0xD6CE, 0xD6CF, 
    0xD6D0, 0xD6D2, 0xD6D3, 0xD6D5, 0xD6D6, 0xD6D8, 0xD6DA, 0xD6DB, 0xD6DC, 
    0xD6DD, 0xD6DE, 0xD6DF, 0xD6E1, 0xD6E2, 0xD6E3, 0xD6E5, 0xD6E6, 0xD6E7, 
    0xD6E9, 0xD6EA, 0xD6EB, 0xD6EC, 0xD6ED,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD6EE, 0xD6EF, 0xD6F1, 0xD6F2, 0xD6F3, 0xD6F4, 0xD6F6, 
    0xD6F7, 0xD6F8, 0xD6F9, 0xD6FA, 0xD6FB, 0xD6FE, 0xD6FF, 0xD701, 0xD702, 
    0xD703, 0xD705, 0xD706, 0xD707, 0xD708, 0xD709, 0xD70A, 0xD70B, 0xD70C, 
    0xD70D, 0xD70E, 0xD70F, 0xD710, 0xD712, 0xD713, 0xD714, 0xCE58, 0xCE59, 
    0xCE5C, 0xCE5F, 0xCE60, 0xCE61, 0xCE68, 0xCE69, 0xCE6B, 0xCE6D, 0xCE74, 
    0xCE75, 0xCE78, 0xCE7C, 0xCE84, 0xCE85, 0xCE87, 0xCE89, 0xCE90, 0xCE91, 
    0xCE94, 0xCE98, 0xCEA0, 0xCEA1, 0xCEA3, 0xCEA4, 0xCEA5, 0xCEAC, 0xCEAD, 
    0xCEC1, 0xCEE4, 0xCEE5, 0xCEE8, 0xCEEB, 0xCEEC, 0xCEF4, 0xCEF5, 0xCEF7, 
    0xCEF8, 0xCEF9, 0xCF00, 0xCF01, 0xCF04, 0xCF08, 0xCF10, 0xCF11, 0xCF13, 
    0xCF15, 0xCF1C, 0xCF20, 0xCF24, 0xCF2C, 0xCF2D, 0xCF2F, 0xCF30, 0xCF31, 
    0xCF38, 0xCF54, 0xCF55, 0xCF58, 0xCF5C, 0xCF64, 0xCF65, 0xCF67, 0xCF69, 
    0xCF70, 0xCF71, 0xCF74, 0xCF78, 0xCF80, 0xCF85, 0xCF8C, 0xCFA1, 0xCFA8, 
    0xCFB0, 0xCFC4, 0xCFE0, 0xCFE1, 0xCFE4, 0xCFE8, 0xCFF0, 0xCFF1, 0xCFF3, 
    0xCFF5, 0xCFFC, 0xD000, 0xD004, 0xD011, 0xD018, 0xD02D, 0xD034, 0xD035, 
    0xD038, 0xD03C,  ERROR},
    { ERROR, 0xD715, 0xD716, 0xD717, 0xD71A, 0xD71B, 0xD71D, 0xD71E, 0xD71F, 
    0xD721, 0xD722, 0xD723, 0xD724, 0xD725, 0xD726, 0xD727, 0xD72A, 0xD72C, 
    0xD72E, 0xD72F, 0xD730, 0xD731, 0xD732, 0xD733, 0xD736, 0xD737, 0xD739, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD73A, 0xD73B, 0xD73D, 
    0xD73E, 0xD73F, 0xD740, 0xD741, 0xD742, 0xD743, 0xD745, 0xD746, 0xD748, 
    0xD74A, 0xD74B, 0xD74C, 0xD74D, 0xD74E, 0xD74F, 0xD752, 0xD753, 0xD755, 
    0xD75A, 0xD75B, 0xD75C, 0xD75D, 0xD75E,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR, 0xD75F, 0xD762, 0xD764, 0xD766, 0xD767, 0xD768, 0xD76A, 
    0xD76B, 0xD76D, 0xD76E, 0xD76F, 0xD771, 0xD772, 0xD773, 0xD775, 0xD776, 
    0xD777, 0xD778, 0xD779, 0xD77A, 0xD77B, 0xD77E, 0xD77F, 0xD780, 0xD782, 
    0xD783, 0xD784, 0xD785, 0xD786, 0xD787, 0xD78A, 0xD78B, 0xD044, 0xD045, 
    0xD047, 0xD049, 0xD050, 0xD054, 0xD058, 0xD060, 0xD06C, 0xD06D, 0xD070, 
    0xD074, 0xD07C, 0xD07D, 0xD081, 0xD0A4, 0xD0A5, 0xD0A8, 0xD0AC, 0xD0B4, 
    0xD0B5, 0xD0B7, 0xD0B9, 0xD0C0, 0xD0C1, 0xD0C4, 0xD0C8, 0xD0C9, 0xD0D0, 
    0xD0D1, 0xD0D3, 0xD0D4, 0xD0D5, 0xD0DC, 0xD0DD, 0xD0E0, 0xD0E4, 0xD0EC, 
    0xD0ED, 0xD0EF, 0xD0F0, 0xD0F1, 0xD0F8, 0xD10D, 0xD130, 0xD131, 0xD134, 
    0xD138, 0xD13A, 0xD140, 0xD141, 0xD143, 0xD144, 0xD145, 0xD14C, 0xD14D, 
    0xD150, 0xD154, 0xD15C, 0xD15D, 0xD15F, 0xD161, 0xD168, 0xD16C, 0xD17C, 
    0xD184, 0xD188, 0xD1A0, 0xD1A1, 0xD1A4, 0xD1A8, 0xD1B0, 0xD1B1, 0xD1B3, 
    0xD1B5, 0xD1BA, 0xD1BC, 0xD1C0, 0xD1D8, 0xD1F4, 0xD1F8, 0xD207, 0xD209, 
    0xD210, 0xD22C, 0xD22D, 0xD230, 0xD234, 0xD23C, 0xD23D, 0xD23F, 0xD241, 
    0xD248, 0xD25C,  ERROR},
    { ERROR, 0xD78D, 0xD78E, 0xD78F, 0xD791, 0xD792, 0xD793, 0xD794, 0xD795, 
    0xD796, 0xD797, 0xD79A, 0xD79C, 0xD79E, 0xD79F, 0xD7A0, 0xD7A1, 0xD7A2, 
    0xD7A3,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD264, 0xD280, 
    0xD281, 0xD284, 0xD288, 0xD290, 0xD291, 0xD295, 0xD29C, 0xD2A0, 0xD2A4, 
    0xD2AC, 0xD2B1, 0xD2B8, 0xD2B9, 0xD2BC, 0xD2BF, 0xD2C0, 0xD2C2, 0xD2C8, 
    0xD2C9, 0xD2CB, 0xD2D4, 0xD2D8, 0xD2DC, 0xD2E4, 0xD2E5, 0xD2F0, 0xD2F1, 
    0xD2F4, 0xD2F8, 0xD300, 0xD301, 0xD303, 0xD305, 0xD30C, 0xD30D, 0xD30E, 
    0xD310, 0xD314, 0xD316, 0xD31C, 0xD31D, 0xD31F, 0xD320, 0xD321, 0xD325, 
    0xD328, 0xD329, 0xD32C, 0xD330, 0xD338, 0xD339, 0xD33B, 0xD33C, 0xD33D, 
    0xD344, 0xD345, 0xD37C, 0xD37D, 0xD380, 0xD384, 0xD38C, 0xD38D, 0xD38F, 
    0xD390, 0xD391, 0xD398, 0xD399, 0xD39C, 0xD3A0, 0xD3A8, 0xD3A9, 0xD3AB, 
    0xD3AD, 0xD3B4, 0xD3B8, 0xD3BC, 0xD3C4, 0xD3C5, 0xD3C8, 0xD3C9, 0xD3D0, 
    0xD3D8, 0xD3E1, 0xD3E3, 0xD3EC, 0xD3ED, 0xD3F0, 0xD3F4, 0xD3FC, 0xD3FD, 
    0xD3FF, 0xD401,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD408, 0xD41D, 
    0xD440, 0xD444, 0xD45C, 0xD460, 0xD464, 0xD46D, 0xD46F, 0xD478, 0xD479, 
    0xD47C, 0xD47F, 0xD480, 0xD482, 0xD488, 0xD489, 0xD48B, 0xD48D, 0xD494, 
    0xD4A9, 0xD4CC, 0xD4D0, 0xD4D4, 0xD4DC, 0xD4DF, 0xD4E8, 0xD4EC, 0xD4F0, 
    0xD4F8, 0xD4FB, 0xD4FD, 0xD504, 0xD508, 0xD50C, 0xD514, 0xD515, 0xD517, 
    0xD53C, 0xD53D, 0xD540, 0xD544, 0xD54C, 0xD54D, 0xD54F, 0xD551, 0xD558, 
    0xD559, 0xD55C, 0xD560, 0xD565, 0xD568, 0xD569, 0xD56B, 0xD56D, 0xD574, 
    0xD575, 0xD578, 0xD57C, 0xD584, 0xD585, 0xD587, 0xD588, 0xD589, 0xD590, 
    0xD5A5, 0xD5C8, 0xD5C9, 0xD5CC, 0xD5D0, 0xD5D2, 0xD5D8, 0xD5D9, 0xD5DB, 
    0xD5DD, 0xD5E4, 0xD5E5, 0xD5E8, 0xD5EC, 0xD5F4, 0xD5F5, 0xD5F7, 0xD5F9, 
    0xD600, 0xD601, 0xD604, 0xD608, 0xD610, 0xD611, 0xD613, 0xD614, 0xD615, 
    0xD61C, 0xD620,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xD624, 0xD62D, 
    0xD638, 0xD639, 0xD63C, 0xD640, 0xD645, 0xD648, 0xD649, 0xD64B, 0xD64D, 
    0xD651, 0xD654, 0xD655, 0xD658, 0xD65C, 0xD667, 0xD669, 0xD670, 0xD671, 
    0xD674, 0xD683, 0xD685, 0xD68C, 0xD68D, 0xD690, 0xD694, 0xD69D, 0xD69F, 
    0xD6A1, 0xD6A8, 0xD6AC, 0xD6B0, 0xD6B9, 0xD6BB, 0xD6C4, 0xD6C5, 0xD6C8, 
    0xD6CC, 0xD6D1, 0xD6D4, 0xD6D7, 0xD6D9, 0xD6E0, 0xD6E4, 0xD6E8, 0xD6F0, 
    0xD6F5, 0xD6FC, 0xD6FD, 0xD700, 0xD704, 0xD711, 0xD718, 0xD719, 0xD71C, 
    0xD720, 0xD728, 0xD729, 0xD72B, 0xD72D, 0xD734, 0xD735, 0xD738, 0xD73C, 
    0xD744, 0xD747, 0xD749, 0xD750, 0xD751, 0xD754, 0xD756, 0xD757, 0xD758, 
    0xD759, 0xD760, 0xD761, 0xD763, 0xD765, 0xD769, 0xD76C, 0xD770, 0xD774, 
    0xD77C, 0xD77D, 0xD781, 0xD788, 0xD789, 0xD78C, 0xD790, 0xD798, 0xD799, 
    0xD79B, 0xD79D,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x4F3D, 0x4F73, 
    0x5047, 0x50F9, 0x52A0, 0x53EF, 0x5475, 0x54E5, 0x5609, 0x5AC1, 0x5BB6, 
    0x6687, 0x67B6, 0x67B7, 0x67EF, 0x6B4C, 0x73C2, 0x75C2, 0x7A3C, 0x82DB, 
    0x8304, 0x8857, 0x8888, 0x8A36, 0x8CC8, 0x8DCF, 0x8EFB, 0x8FE6, 0x99D5, 
    0x523B, 0x5374, 0x5404, 0x606A, 0x6164, 0x6BBC, 0x73CF, 0x811A, 0x89BA, 
    0x89D2, 0x95A3, 0x4F83, 0x520A, 0x58BE, 0x5978, 0x59E6, 0x5E72, 0x5E79, 
    0x61C7, 0x63C0, 0x6746, 0x67EC, 0x687F, 0x6F97, 0x764E, 0x770B, 0x78F5, 
    0x7A08, 0x7AFF, 0x7C21, 0x809D, 0x826E, 0x8271, 0x8AEB, 0x9593, 0x4E6B, 
    0x559D, 0x66F7, 0x6E34, 0x78A3, 0x7AED, 0x845B, 0x8910, 0x874E, 0x97A8, 
    0x52D8, 0x574E, 0x582A, 0x5D4C, 0x611F, 0x61BE, 0x6221, 0x6562, 0x67D1, 
    0x6A44, 0x6E1B, 0x7518, 0x75B3, 0x76E3, 0x77B0, 0x7D3A, 0x90AF, 0x9451, 
    0x9452, 0x9F95,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5323, 0x5CAC, 
    0x7532, 0x80DB, 0x9240, 0x9598, 0x525B, 0x5808, 0x59DC, 0x5CA1, 0x5D17, 
    0x5EB7, 0x5F3A, 0x5F4A, 0x6177, 0x6C5F, 0x757A, 0x7586, 0x7CE0, 0x7D73, 
    0x7DB1, 0x7F8C, 0x8154, 0x8221, 0x8591, 0x8941, 0x8B1B, 0x92FC, 0x964D, 
    0x9C47, 0x4ECB, 0x4EF7, 0x500B, 0x51F1, 0x584F, 0x6137, 0x613E, 0x6168, 
    0x6539, 0x69EA, 0x6F11, 0x75A5, 0x7686, 0x76D6, 0x7B87, 0x82A5, 0x84CB, 
    0xF900, 0x93A7, 0x958B, 0x5580, 0x5BA2, 0x5751, 0xF901, 0x7CB3, 0x7FB9, 
    0x91B5, 0x5028, 0x53BB, 0x5C45, 0x5DE8, 0x62D2, 0x636E, 0x64DA, 0x64E7, 
    0x6E20, 0x70AC, 0x795B, 0x8DDD, 0x8E1E, 0xF902, 0x907D, 0x9245, 0x92F8, 
    0x4E7E, 0x4EF6, 0x5065, 0x5DFE, 0x5EFA, 0x6106, 0x6957, 0x8171, 0x8654, 
    0x8E47, 0x9375, 0x9A2B, 0x4E5E, 0x5091, 0x6770, 0x6840, 0x5109, 0x528D, 
    0x5292, 0x6AA2,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x77BC, 0x9210, 
    0x9ED4, 0x52AB, 0x602F, 0x8FF2, 0x5048, 0x61A9, 0x63ED, 0x64CA, 0x683C, 
    0x6A84, 0x6FC0, 0x8188, 0x89A1, 0x9694, 0x5805, 0x727D, 0x72AC, 0x7504, 
    0x7D79, 0x7E6D, 0x80A9, 0x898B, 0x8B74, 0x9063, 0x9D51, 0x6289, 0x6C7A, 
    0x6F54, 0x7D50, 0x7F3A, 0x8A23, 0x517C, 0x614A, 0x7B9D, 0x8B19, 0x9257, 
    0x938C, 0x4EAC, 0x4FD3, 0x501E, 0x50BE, 0x5106, 0x52C1, 0x52CD, 0x537F, 
    0x5770, 0x5883, 0x5E9A, 0x5F91, 0x6176, 0x61AC, 0x64CE, 0x656C, 0x666F, 
    0x66BB, 0x66F4, 0x6897, 0x6D87, 0x7085, 0x70F1, 0x749F, 0x74A5, 0x74CA, 
    0x75D9, 0x786C, 0x78EC, 0x7ADF, 0x7AF6, 0x7D45, 0x7D93, 0x8015, 0x803F, 
    0x811B, 0x8396, 0x8B66, 0x8F15, 0x9015, 0x93E1, 0x9803, 0x9838, 0x9A5A, 
    0x9BE8, 0x4FC2, 0x5553, 0x583A, 0x5951, 0x5B63, 0x5C46, 0x60B8, 0x6212, 
    0x6842, 0x68B0,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x68E8, 0x6EAA, 
    0x754C, 0x7678, 0x78CE, 0x7A3D, 0x7CFB, 0x7E6B, 0x7E7C, 0x8A08, 0x8AA1, 
    0x8C3F, 0x968E, 0x9DC4, 0x53E4, 0x53E9, 0x544A, 0x5471, 0x56FA, 0x59D1, 
    0x5B64, 0x5C3B, 0x5EAB, 0x62F7, 0x6537, 0x6545, 0x6572, 0x66A0, 0x67AF, 
    0x69C1, 0x6CBD, 0x75FC, 0x7690, 0x777E, 0x7A3F, 0x7F94, 0x8003, 0x80A1, 
    0x818F, 0x82E6, 0x82FD, 0x83F0, 0x85C1, 0x8831, 0x88B4, 0x8AA5, 0xF903, 
    0x8F9C, 0x932E, 0x96C7, 0x9867, 0x9AD8, 0x9F13, 0x54ED, 0x659B, 0x66F2, 
    0x688F, 0x7A40, 0x8C37, 0x9D60, 0x56F0, 0x5764, 0x5D11, 0x6606, 0x68B1, 
    0x68CD, 0x6EFE, 0x7428, 0x889E, 0x9BE4, 0x6C68, 0xF904, 0x9AA8, 0x4F9B, 
    0x516C, 0x5171, 0x529F, 0x5B54, 0x5DE5, 0x6050, 0x606D, 0x62F1, 0x63A7, 
    0x653B, 0x73D9, 0x7A7A, 0x86A3, 0x8CA2, 0x978F, 0x4E32, 0x5BE1, 0x6208, 
    0x679C, 0x74DC,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x79D1, 0x83D3, 
    0x8A87, 0x8AB2, 0x8DE8, 0x904E, 0x934B, 0x9846, 0x5ED3, 0x69E8, 0x85FF, 
    0x90ED, 0xF905, 0x51A0, 0x5B98, 0x5BEC, 0x6163, 0x68FA, 0x6B3E, 0x704C, 
    0x742F, 0x74D8, 0x7BA1, 0x7F50, 0x83C5, 0x89C0, 0x8CAB, 0x95DC, 0x9928, 
    0x522E, 0x605D, 0x62EC, 0x9002, 0x4F8A, 0x5149, 0x5321, 0x58D9, 0x5EE3, 
    0x66E0, 0x6D38, 0x709A, 0x72C2, 0x73D6, 0x7B50, 0x80F1, 0x945B, 0x5366, 
    0x639B, 0x7F6B, 0x4E56, 0x5080, 0x584A, 0x58DE, 0x602A, 0x6127, 0x62D0, 
    0x69D0, 0x9B41, 0x5B8F, 0x7D18, 0x80B1, 0x8F5F, 0x4EA4, 0x50D1, 0x54AC, 
    0x55AC, 0x5B0C, 0x5DA0, 0x5DE7, 0x652A, 0x654E, 0x6821, 0x6A4B, 0x72E1, 
    0x768E, 0x77EF, 0x7D5E, 0x7FF9, 0x81A0, 0x854E, 0x86DF, 0x8F03, 0x8F4E, 
    0x90CA, 0x9903, 0x9A55, 0x9BAB, 0x4E18, 0x4E45, 0x4E5D, 0x4EC7, 0x4FF1, 
    0x5177, 0x52FE,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5340, 0x53E3, 
    0x53E5, 0x548E, 0x5614, 0x5775, 0x57A2, 0x5BC7, 0x5D87, 0x5ED0, 0x61FC, 
    0x62D8, 0x6551, 0x67B8, 0x67E9, 0x69CB, 0x6B50, 0x6BC6, 0x6BEC, 0x6C42, 
    0x6E9D, 0x7078, 0x72D7, 0x7396, 0x7403, 0x77BF, 0x77E9, 0x7A76, 0x7D7F, 
    0x8009, 0x81FC, 0x8205, 0x820A, 0x82DF, 0x8862, 0x8B33, 0x8CFC, 0x8EC0, 
    0x9011, 0x90B1, 0x9264, 0x92B6, 0x99D2, 0x9A45, 0x9CE9, 0x9DD7, 0x9F9C, 
    0x570B, 0x5C40, 0x83CA, 0x97A0, 0x97AB, 0x9EB4, 0x541B, 0x7A98, 0x7FA4, 
    0x88D9, 0x8ECD, 0x90E1, 0x5800, 0x5C48, 0x6398, 0x7A9F, 0x5BAE, 0x5F13, 
    0x7A79, 0x7AAE, 0x828E, 0x8EAC, 0x5026, 0x5238, 0x52F8, 0x5377, 0x5708, 
    0x62F3, 0x6372, 0x6B0A, 0x6DC3, 0x7737, 0x53A5, 0x7357, 0x8568, 0x8E76, 
    0x95D5, 0x673A, 0x6AC3, 0x6F70, 0x8A6D, 0x8ECC, 0x994B, 0xF906, 0x6677, 
    0x6B78, 0x8CB4,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9B3C, 0xF907, 
    0x53EB, 0x572D, 0x594E, 0x63C6, 0x69FB, 0x73EA, 0x7845, 0x7ABA, 0x7AC5, 
    0x7CFE, 0x8475, 0x898F, 0x8D73, 0x9035, 0x95A8, 0x52FB, 0x5747, 0x7547, 
    0x7B60, 0x83CC, 0x921E, 0xF908, 0x6A58, 0x514B, 0x524B, 0x5287, 0x621F, 
    0x68D8, 0x6975, 0x9699, 0x50C5, 0x52A4, 0x52E4, 0x61C3, 0x65A4, 0x6839, 
    0x69FF, 0x747E, 0x7B4B, 0x82B9, 0x83EB, 0x89B2, 0x8B39, 0x8FD1, 0x9949, 
    0xF909, 0x4ECA, 0x5997, 0x64D2, 0x6611, 0x6A8E, 0x7434, 0x7981, 0x79BD, 
    0x82A9, 0x887E, 0x887F, 0x895F, 0xF90A, 0x9326, 0x4F0B, 0x53CA, 0x6025, 
    0x6271, 0x6C72, 0x7D1A, 0x7D66, 0x4E98, 0x5162, 0x77DC, 0x80AF, 0x4F01, 
    0x4F0E, 0x5176, 0x5180, 0x55DC, 0x5668, 0x573B, 0x57FA, 0x57FC, 0x5914, 
    0x5947, 0x5993, 0x5BC4, 0x5C90, 0x5D0E, 0x5DF1, 0x5E7E, 0x5FCC, 0x6280, 
    0x65D7, 0x65E3,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x671E, 0x671F, 
    0x675E, 0x68CB, 0x68C4, 0x6A5F, 0x6B3A, 0x6C23, 0x6C7D, 0x6C82, 0x6DC7, 
    0x7398, 0x7426, 0x742A, 0x7482, 0x74A3, 0x7578, 0x757F, 0x7881, 0x78EF, 
    0x7941, 0x7947, 0x7948, 0x797A, 0x7B95, 0x7D00, 0x7DBA, 0x7F88, 0x8006, 
    0x802D, 0x808C, 0x8A18, 0x8B4F, 0x8C48, 0x8D77, 0x9321, 0x9324, 0x98E2, 
    0x9951, 0x9A0E, 0x9A0F, 0x9A65, 0x9E92, 0x7DCA, 0x4F76, 0x5409, 0x62EE, 
    0x6854, 0x91D1, 0x55AB, 0x513A, 0xF90B, 0xF90C, 0x5A1C, 0x61E6, 0xF90D, 
    0x62CF, 0x62FF, 0xF90E, 0xF90F, 0xF910, 0xF911, 0xF912, 0xF913, 0x90A3, 
    0xF914, 0xF915, 0xF916, 0xF917, 0xF918, 0x8AFE, 0xF919, 0xF91A, 0xF91B, 
    0xF91C, 0x6696, 0xF91D, 0x7156, 0xF91E, 0xF91F, 0x96E3, 0xF920, 0x634F, 
    0x637A, 0x5357, 0xF921, 0x678F, 0x6960, 0x6E73, 0xF922, 0x7537, 0xF923, 
    0xF924, 0xF925,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7D0D, 0xF926, 
    0xF927, 0x8872, 0x56CA, 0x5A18, 0xF928, 0xF929, 0xF92A, 0xF92B, 0xF92C, 
    0x4E43, 0xF92D, 0x5167, 0x5948, 0x67F0, 0x8010, 0xF92E, 0x5973, 0x5E74, 
    0x649A, 0x79CA, 0x5FF5, 0x606C, 0x62C8, 0x637B, 0x5BE7, 0x5BD7, 0x52AA, 
    0xF92F, 0x5974, 0x5F29, 0x6012, 0xF930, 0xF931, 0xF932, 0x7459, 0xF933, 
    0xF934, 0xF935, 0xF936, 0xF937, 0xF938, 0x99D1, 0xF939, 0xF93A, 0xF93B, 
    0xF93C, 0xF93D, 0xF93E, 0xF93F, 0xF940, 0xF941, 0xF942, 0xF943, 0x6FC3, 
    0xF944, 0xF945, 0x81BF, 0x8FB2, 0x60F1, 0xF946, 0xF947, 0x8166, 0xF948, 
    0xF949, 0x5C3F, 0xF94A, 0xF94B, 0xF94C, 0xF94D, 0xF94E, 0xF94F, 0xF950, 
    0xF951, 0x5AE9, 0x8A25, 0x677B, 0x7D10, 0xF952, 0xF953, 0xF954, 0xF955, 
    0xF956, 0xF957, 0x80FD, 0xF958, 0xF959, 0x5C3C, 0x6CE5, 0x533F, 0x6EBA, 
    0x591A, 0x8336,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x4E39, 0x4EB6, 
    0x4F46, 0x55AE, 0x5718, 0x58C7, 0x5F56, 0x65B7, 0x65E6, 0x6A80, 0x6BB5, 
    0x6E4D, 0x77ED, 0x7AEF, 0x7C1E, 0x7DDE, 0x86CB, 0x8892, 0x9132, 0x935B, 
    0x64BB, 0x6FBE, 0x737A, 0x75B8, 0x9054, 0x5556, 0x574D, 0x61BA, 0x64D4, 
    0x66C7, 0x6DE1, 0x6E5B, 0x6F6D, 0x6FB9, 0x75F0, 0x8043, 0x81BD, 0x8541, 
    0x8983, 0x8AC7, 0x8B5A, 0x931F, 0x6C93, 0x7553, 0x7B54, 0x8E0F, 0x905D, 
    0x5510, 0x5802, 0x5858, 0x5E62, 0x6207, 0x649E, 0x68E0, 0x7576, 0x7CD6, 
    0x87B3, 0x9EE8, 0x4EE3, 0x5788, 0x576E, 0x5927, 0x5C0D, 0x5CB1, 0x5E36, 
    0x5F85, 0x6234, 0x64E1, 0x73B3, 0x81FA, 0x888B, 0x8CB8, 0x968A, 0x9EDB, 
    0x5B85, 0x5FB7, 0x60B3, 0x5012, 0x5200, 0x5230, 0x5716, 0x5835, 0x5857, 
    0x5C0E, 0x5C60, 0x5CF6, 0x5D8B, 0x5EA6, 0x5F92, 0x60BC, 0x6311, 0x6389, 
    0x6417, 0x6843,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x68F9, 0x6AC2, 
    0x6DD8, 0x6E21, 0x6ED4, 0x6FE4, 0x71FE, 0x76DC, 0x7779, 0x79B1, 0x7A3B, 
    0x8404, 0x89A9, 0x8CED, 0x8DF3, 0x8E48, 0x9003, 0x9014, 0x9053, 0x90FD, 
    0x934D, 0x9676, 0x97DC, 0x6BD2, 0x7006, 0x7258, 0x72A2, 0x7368, 0x7763, 
    0x79BF, 0x7BE4, 0x7E9B, 0x8B80, 0x58A9, 0x60C7, 0x6566, 0x65FD, 0x66BE, 
    0x6C8C, 0x711E, 0x71C9, 0x8C5A, 0x9813, 0x4E6D, 0x7A81, 0x4EDD, 0x51AC, 
    0x51CD, 0x52D5, 0x540C, 0x61A7, 0x6771, 0x6850, 0x68DF, 0x6D1E, 0x6F7C, 
    0x75BC, 0x77B3, 0x7AE5, 0x80F4, 0x8463, 0x9285, 0x515C, 0x6597, 0x675C, 
    0x6793, 0x75D8, 0x7AC7, 0x8373, 0xF95A, 0x8C46, 0x9017, 0x982D, 0x5C6F, 
    0x81C0, 0x829A, 0x9041, 0x906F, 0x920D, 0x5F97, 0x5D9D, 0x6A59, 0x71C8, 
    0x767B, 0x7B49, 0x85E4, 0x8B04, 0x9127, 0x9A30, 0x5587, 0x61F6, 0xF95B, 
    0x7669, 0x7F85,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x863F, 0x87BA, 
    0x88F8, 0x908F, 0xF95C, 0x6D1B, 0x70D9, 0x73DE, 0x7D61, 0x843D, 0xF95D, 
    0x916A, 0x99F1, 0xF95E, 0x4E82, 0x5375, 0x6B04, 0x6B12, 0x703E, 0x721B, 
    0x862D, 0x9E1E, 0x524C, 0x8FA3, 0x5D50, 0x64E5, 0x652C, 0x6B16, 0x6FEB, 
    0x7C43, 0x7E9C, 0x85CD, 0x8964, 0x89BD, 0x62C9, 0x81D8, 0x881F, 0x5ECA, 
    0x6717, 0x6D6A, 0x72FC, 0x7405, 0x746F, 0x8782, 0x90DE, 0x4F86, 0x5D0D, 
    0x5FA0, 0x840A, 0x51B7, 0x63A0, 0x7565, 0x4EAE, 0x5006, 0x5169, 0x51C9, 
    0x6881, 0x6A11, 0x7CAE, 0x7CB1, 0x7CE7, 0x826F, 0x8AD2, 0x8F1B, 0x91CF, 
    0x4FB6, 0x5137, 0x52F5, 0x5442, 0x5EEC, 0x616E, 0x623E, 0x65C5, 0x6ADA, 
    0x6FFE, 0x792A, 0x85DC, 0x8823, 0x95AD, 0x9A62, 0x9A6A, 0x9E97, 0x9ECE, 
    0x529B, 0x66C6, 0x6B77, 0x701D, 0x792B, 0x8F62, 0x9742, 0x6190, 0x6200, 
    0x6523, 0x6F23,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7149, 0x7489, 
    0x7DF4, 0x806F, 0x84EE, 0x8F26, 0x9023, 0x934A, 0x51BD, 0x5217, 0x52A3, 
    0x6D0C, 0x70C8, 0x88C2, 0x5EC9, 0x6582, 0x6BAE, 0x6FC2, 0x7C3E, 0x7375, 
    0x4EE4, 0x4F36, 0x56F9, 0xF95F, 0x5CBA, 0x5DBA, 0x601C, 0x73B2, 0x7B2D, 
    0x7F9A, 0x7FCE, 0x8046, 0x901E, 0x9234, 0x96F6, 0x9748, 0x9818, 0x9F61, 
    0x4F8B, 0x6FA7, 0x79AE, 0x91B4, 0x96B7, 0x52DE, 0xF960, 0x6488, 0x64C4, 
    0x6AD3, 0x6F5E, 0x7018, 0x7210, 0x76E7, 0x8001, 0x8606, 0x865C, 0x8DEF, 
    0x8F05, 0x9732, 0x9B6F, 0x9DFA, 0x9E75, 0x788C, 0x797F, 0x7DA0, 0x83C9, 
    0x9304, 0x9E7F, 0x9E93, 0x8AD6, 0x58DF, 0x5F04, 0x6727, 0x7027, 0x74CF, 
    0x7C60, 0x807E, 0x5121, 0x7028, 0x7262, 0x78CA, 0x8CC2, 0x8CDA, 0x8CF4, 
    0x96F7, 0x4E86, 0x50DA, 0x5BEE, 0x5ED6, 0x6599, 0x71CE, 0x7642, 0x77AD, 
    0x804A, 0x84FC,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x907C, 0x9B27, 
    0x9F8D, 0x58D8, 0x5A41, 0x5C62, 0x6A13, 0x6DDA, 0x6F0F, 0x763B, 0x7D2F, 
    0x7E37, 0x851E, 0x8938, 0x93E4, 0x964B, 0x5289, 0x65D2, 0x67F3, 0x69B4, 
    0x6D41, 0x6E9C, 0x700F, 0x7409, 0x7460, 0x7559, 0x7624, 0x786B, 0x8B2C, 
    0x985E, 0x516D, 0x622E, 0x9678, 0x4F96, 0x502B, 0x5D19, 0x6DEA, 0x7DB8, 
    0x8F2A, 0x5F8B, 0x6144, 0x6817, 0xF961, 0x9686, 0x52D2, 0x808B, 0x51DC, 
    0x51CC, 0x695E, 0x7A1C, 0x7DBE, 0x83F1, 0x9675, 0x4FDA, 0x5229, 0x5398, 
    0x540F, 0x550E, 0x5C65, 0x60A7, 0x674E, 0x68A8, 0x6D6C, 0x7281, 0x72F8, 
    0x7406, 0x7483, 0xF962, 0x75E2, 0x7C6C, 0x7F79, 0x7FB8, 0x8389, 0x88CF, 
    0x88E1, 0x91CC, 0x91D0, 0x96E2, 0x9BC9, 0x541D, 0x6F7E, 0x71D0, 0x7498, 
    0x85FA, 0x8EAA, 0x96A3, 0x9C57, 0x9E9F, 0x6797, 0x6DCB, 0x7433, 0x81E8, 
    0x9716, 0x782C,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7ACB, 0x7B20, 
    0x7C92, 0x6469, 0x746A, 0x75F2, 0x78BC, 0x78E8, 0x99AC, 0x9B54, 0x9EBB, 
    0x5BDE, 0x5E55, 0x6F20, 0x819C, 0x83AB, 0x9088, 0x4E07, 0x534D, 0x5A29, 
    0x5DD2, 0x5F4E, 0x6162, 0x633D, 0x6669, 0x66FC, 0x6EFF, 0x6F2B, 0x7063, 
    0x779E, 0x842C, 0x8513, 0x883B, 0x8F13, 0x9945, 0x9C3B, 0x551C, 0x62B9, 
    0x672B, 0x6CAB, 0x8309, 0x896A, 0x977A, 0x4EA1, 0x5984, 0x5FD8, 0x5FD9, 
    0x671B, 0x7DB2, 0x7F54, 0x8292, 0x832B, 0x83BD, 0x8F1E, 0x9099, 0x57CB, 
    0x59B9, 0x5A92, 0x5BD0, 0x6627, 0x679A, 0x6885, 0x6BCF, 0x7164, 0x7F75, 
    0x8CB7, 0x8CE3, 0x9081, 0x9B45, 0x8108, 0x8C8A, 0x964C, 0x9A40, 0x9EA5, 
    0x5B5F, 0x6C13, 0x731B, 0x76F2, 0x76DF, 0x840C, 0x51AA, 0x8993, 0x514D, 
    0x5195, 0x52C9, 0x68C9, 0x6C94, 0x7704, 0x7720, 0x7DBF, 0x7DEC, 0x9762, 
    0x9EB5, 0x6EC5,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8511, 0x51A5, 
    0x540D, 0x547D, 0x660E, 0x669D, 0x6927, 0x6E9F, 0x76BF, 0x7791, 0x8317, 
    0x84C2, 0x879F, 0x9169, 0x9298, 0x9CF4, 0x8882, 0x4FAE, 0x5192, 0x52DF, 
    0x59C6, 0x5E3D, 0x6155, 0x6478, 0x6479, 0x66AE, 0x67D0, 0x6A21, 0x6BCD, 
    0x6BDB, 0x725F, 0x7261, 0x7441, 0x7738, 0x77DB, 0x8017, 0x82BC, 0x8305, 
    0x8B00, 0x8B28, 0x8C8C, 0x6728, 0x6C90, 0x7267, 0x76EE, 0x7766, 0x7A46, 
    0x9DA9, 0x6B7F, 0x6C92, 0x5922, 0x6726, 0x8499, 0x536F, 0x5893, 0x5999, 
    0x5EDF, 0x63CF, 0x6634, 0x6773, 0x6E3A, 0x732B, 0x7AD7, 0x82D7, 0x9328, 
    0x52D9, 0x5DEB, 0x61AE, 0x61CB, 0x620A, 0x62C7, 0x64AB, 0x65E0, 0x6959, 
    0x6B66, 0x6BCB, 0x7121, 0x73F7, 0x755D, 0x7E46, 0x821E, 0x8302, 0x856A, 
    0x8AA3, 0x8CBF, 0x9727, 0x9D61, 0x58A8, 0x9ED8, 0x5011, 0x520E, 0x543B, 
    0x554F, 0x6587,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6C76, 0x7D0A, 
    0x7D0B, 0x805E, 0x868A, 0x9580, 0x96EF, 0x52FF, 0x6C95, 0x7269, 0x5473, 
    0x5A9A, 0x5C3E, 0x5D4B, 0x5F4C, 0x5FAE, 0x672A, 0x68B6, 0x6963, 0x6E3C, 
    0x6E44, 0x7709, 0x7C73, 0x7F8E, 0x8587, 0x8B0E, 0x8FF7, 0x9761, 0x9EF4, 
    0x5CB7, 0x60B6, 0x610D, 0x61AB, 0x654F, 0x65FB, 0x65FC, 0x6C11, 0x6CEF, 
    0x739F, 0x73C9, 0x7DE1, 0x9594, 0x5BC6, 0x871C, 0x8B10, 0x525D, 0x535A, 
    0x62CD, 0x640F, 0x64B2, 0x6734, 0x6A38, 0x6CCA, 0x73C0, 0x749E, 0x7B94, 
    0x7C95, 0x7E1B, 0x818A, 0x8236, 0x8584, 0x8FEB, 0x96F9, 0x99C1, 0x4F34, 
    0x534A, 0x53CD, 0x53DB, 0x62CC, 0x642C, 0x6500, 0x6591, 0x69C3, 0x6CEE, 
    0x6F58, 0x73ED, 0x7554, 0x7622, 0x76E4, 0x76FC, 0x78D0, 0x78FB, 0x792C, 
    0x7D46, 0x822C, 0x87E0, 0x8FD4, 0x9812, 0x98EF, 0x52C3, 0x62D4, 0x64A5, 
    0x6E24, 0x6F51,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x767C, 0x8DCB, 
    0x91B1, 0x9262, 0x9AEE, 0x9B43, 0x5023, 0x508D, 0x574A, 0x59A8, 0x5C28, 
    0x5E47, 0x5F77, 0x623F, 0x653E, 0x65B9, 0x65C1, 0x6609, 0x678B, 0x699C, 
    0x6EC2, 0x78C5, 0x7D21, 0x80AA, 0x8180, 0x822B, 0x82B3, 0x84A1, 0x868C, 
    0x8A2A, 0x8B17, 0x90A6, 0x9632, 0x9F90, 0x500D, 0x4FF3, 0xF963, 0x57F9, 
    0x5F98, 0x62DC, 0x6392, 0x676F, 0x6E43, 0x7119, 0x76C3, 0x80CC, 0x80DA, 
    0x88F4, 0x88F5, 0x8919, 0x8CE0, 0x8F29, 0x914D, 0x966A, 0x4F2F, 0x4F70, 
    0x5E1B, 0x67CF, 0x6822, 0x767D, 0x767E, 0x9B44, 0x5E61, 0x6A0A, 0x7169, 
    0x71D4, 0x756A, 0xF964, 0x7E41, 0x8543, 0x85E9, 0x98DC, 0x4F10, 0x7B4F, 
    0x7F70, 0x95A5, 0x51E1, 0x5E06, 0x68B5, 0x6C3E, 0x6C4E, 0x6CDB, 0x72AF, 
    0x7BC4, 0x8303, 0x6CD5, 0x743A, 0x50FB, 0x5288, 0x58C1, 0x64D8, 0x6A97, 
    0x74A7, 0x7656,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x78A7, 0x8617, 
    0x95E2, 0x9739, 0xF965, 0x535E, 0x5F01, 0x8B8A, 0x8FA8, 0x8FAF, 0x908A, 
    0x5225, 0x77A5, 0x9C49, 0x9F08, 0x4E19, 0x5002, 0x5175, 0x5C5B, 0x5E77, 
    0x661E, 0x663A, 0x67C4, 0x68C5, 0x70B3, 0x7501, 0x75C5, 0x79C9, 0x7ADD, 
    0x8F27, 0x9920, 0x9A08, 0x4FDD, 0x5821, 0x5831, 0x5BF6, 0x666E, 0x6B65, 
    0x6D11, 0x6E7A, 0x6F7D, 0x73E4, 0x752B, 0x83E9, 0x88DC, 0x8913, 0x8B5C, 
    0x8F14, 0x4F0F, 0x50D5, 0x5310, 0x535C, 0x5B93, 0x5FA9, 0x670D, 0x798F, 
    0x8179, 0x832F, 0x8514, 0x8907, 0x8986, 0x8F39, 0x8F3B, 0x99A5, 0x9C12, 
    0x672C, 0x4E76, 0x4FF8, 0x5949, 0x5C01, 0x5CEF, 0x5CF0, 0x6367, 0x68D2, 
    0x70FD, 0x71A2, 0x742B, 0x7E2B, 0x84EC, 0x8702, 0x9022, 0x92D2, 0x9CF3, 
    0x4E0D, 0x4ED8, 0x4FEF, 0x5085, 0x5256, 0x526F, 0x5426, 0x5490, 0x57E0, 
    0x592B, 0x5A66,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5B5A, 0x5B75, 
    0x5BCC, 0x5E9C, 0xF966, 0x6276, 0x6577, 0x65A7, 0x6D6E, 0x6EA5, 0x7236, 
    0x7B26, 0x7C3F, 0x7F36, 0x8150, 0x8151, 0x819A, 0x8240, 0x8299, 0x83A9, 
    0x8A03, 0x8CA0, 0x8CE6, 0x8CFB, 0x8D74, 0x8DBA, 0x90E8, 0x91DC, 0x961C, 
    0x9644, 0x99D9, 0x9CE7, 0x5317, 0x5206, 0x5429, 0x5674, 0x58B3, 0x5954, 
    0x596E, 0x5FFF, 0x61A4, 0x626E, 0x6610, 0x6C7E, 0x711A, 0x76C6, 0x7C89, 
    0x7CDE, 0x7D1B, 0x82AC, 0x8CC1, 0x96F0, 0xF967, 0x4F5B, 0x5F17, 0x5F7F, 
    0x62C2, 0x5D29, 0x670B, 0x68DA, 0x787C, 0x7E43, 0x9D6C, 0x4E15, 0x5099, 
    0x5315, 0x532A, 0x5351, 0x5983, 0x5A62, 0x5E87, 0x60B2, 0x618A, 0x6249, 
    0x6279, 0x6590, 0x6787, 0x69A7, 0x6BD4, 0x6BD6, 0x6BD7, 0x6BD8, 0x6CB8, 
    0xF968, 0x7435, 0x75FA, 0x7812, 0x7891, 0x79D5, 0x79D8, 0x7C83, 0x7DCB, 
    0x7FE1, 0x80A5,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x813E, 0x81C2, 
    0x83F2, 0x871A, 0x88E8, 0x8AB9, 0x8B6C, 0x8CBB, 0x9119, 0x975E, 0x98DB, 
    0x9F3B, 0x56AC, 0x5B2A, 0x5F6C, 0x658C, 0x6AB3, 0x6BAF, 0x6D5C, 0x6FF1, 
    0x7015, 0x725D, 0x73AD, 0x8CA7, 0x8CD3, 0x983B, 0x6191, 0x6C37, 0x8058, 
    0x9A01, 0x4E4D, 0x4E8B, 0x4E9B, 0x4ED5, 0x4F3A, 0x4F3C, 0x4F7F, 0x4FDF, 
    0x50FF, 0x53F2, 0x53F8, 0x5506, 0x55E3, 0x56DB, 0x58EB, 0x5962, 0x5A11, 
    0x5BEB, 0x5BFA, 0x5C04, 0x5DF3, 0x5E2B, 0x5F99, 0x601D, 0x6368, 0x659C, 
    0x65AF, 0x67F6, 0x67FB, 0x68AD, 0x6B7B, 0x6C99, 0x6CD7, 0x6E23, 0x7009, 
    0x7345, 0x7802, 0x793E, 0x7940, 0x7960, 0x79C1, 0x7BE9, 0x7D17, 0x7D72, 
    0x8086, 0x820D, 0x838E, 0x84D1, 0x86C7, 0x88DF, 0x8A50, 0x8A5E, 0x8B1D, 
    0x8CDC, 0x8D66, 0x8FAD, 0x90AA, 0x98FC, 0x99DF, 0x9E9D, 0x524A, 0xF969, 
    0x6714, 0xF96A,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5098, 0x522A, 
    0x5C71, 0x6563, 0x6C55, 0x73CA, 0x7523, 0x759D, 0x7B97, 0x849C, 0x9178, 
    0x9730, 0x4E77, 0x6492, 0x6BBA, 0x715E, 0x85A9, 0x4E09, 0xF96B, 0x6749, 
    0x68EE, 0x6E17, 0x829F, 0x8518, 0x886B, 0x63F7, 0x6F81, 0x9212, 0x98AF, 
    0x4E0A, 0x50B7, 0x50CF, 0x511F, 0x5546, 0x55AA, 0x5617, 0x5B40, 0x5C19, 
    0x5CE0, 0x5E38, 0x5E8A, 0x5EA0, 0x5EC2, 0x60F3, 0x6851, 0x6A61, 0x6E58, 
    0x723D, 0x7240, 0x72C0, 0x76F8, 0x7965, 0x7BB1, 0x7FD4, 0x88F3, 0x89F4, 
    0x8A73, 0x8C61, 0x8CDE, 0x971C, 0x585E, 0x74BD, 0x8CFD, 0x55C7, 0xF96C, 
    0x7A61, 0x7D22, 0x8272, 0x7272, 0x751F, 0x7525, 0xF96D, 0x7B19, 0x5885, 
    0x58FB, 0x5DBC, 0x5E8F, 0x5EB6, 0x5F90, 0x6055, 0x6292, 0x637F, 0x654D, 
    0x6691, 0x66D9, 0x66F8, 0x6816, 0x68F2, 0x7280, 0x745E, 0x7B6E, 0x7D6E, 
    0x7DD6, 0x7F72,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x80E5, 0x8212, 
    0x85AF, 0x897F, 0x8A93, 0x901D, 0x92E4, 0x9ECD, 0x9F20, 0x5915, 0x596D, 
    0x5E2D, 0x60DC, 0x6614, 0x6673, 0x6790, 0x6C50, 0x6DC5, 0x6F5F, 0x77F3, 
    0x78A9, 0x84C6, 0x91CB, 0x932B, 0x4ED9, 0x50CA, 0x5148, 0x5584, 0x5B0B, 
    0x5BA3, 0x6247, 0x657E, 0x65CB, 0x6E32, 0x717D, 0x7401, 0x7444, 0x7487, 
    0x74BF, 0x766C, 0x79AA, 0x7DDA, 0x7E55, 0x7FA8, 0x817A, 0x81B3, 0x8239, 
    0x861A, 0x87EC, 0x8A75, 0x8DE3, 0x9078, 0x9291, 0x9425, 0x994D, 0x9BAE, 
    0x5368, 0x5C51, 0x6954, 0x6CC4, 0x6D29, 0x6E2B, 0x820C, 0x859B, 0x893B, 
    0x8A2D, 0x8AAA, 0x96EA, 0x9F67, 0x5261, 0x66B9, 0x6BB2, 0x7E96, 0x87FE, 
    0x8D0D, 0x9583, 0x965D, 0x651D, 0x6D89, 0x71EE, 0xF96E, 0x57CE, 0x59D3, 
    0x5BAC, 0x6027, 0x60FA, 0x6210, 0x661F, 0x665F, 0x7329, 0x73F9, 0x76DB, 
    0x7701, 0x7B6C,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8056, 0x8072, 
    0x8165, 0x8AA0, 0x9192, 0x4E16, 0x52E2, 0x6B72, 0x6D17, 0x7A05, 0x7B39, 
    0x7D30, 0xF96F, 0x8CB0, 0x53EC, 0x562F, 0x5851, 0x5BB5, 0x5C0F, 0x5C11, 
    0x5DE2, 0x6240, 0x6383, 0x6414, 0x662D, 0x68B3, 0x6CBC, 0x6D88, 0x6EAF, 
    0x701F, 0x70A4, 0x71D2, 0x7526, 0x758F, 0x758E, 0x7619, 0x7B11, 0x7BE0, 
    0x7C2B, 0x7D20, 0x7D39, 0x852C, 0x856D, 0x8607, 0x8A34, 0x900D, 0x9061, 
    0x90B5, 0x92B7, 0x97F6, 0x9A37, 0x4FD7, 0x5C6C, 0x675F, 0x6D91, 0x7C9F, 
    0x7E8C, 0x8B16, 0x8D16, 0x901F, 0x5B6B, 0x5DFD, 0x640D, 0x84C0, 0x905C, 
    0x98E1, 0x7387, 0x5B8B, 0x609A, 0x677E, 0x6DDE, 0x8A1F, 0x8AA6, 0x9001, 
    0x980C, 0x5237, 0xF970, 0x7051, 0x788E, 0x9396, 0x8870, 0x91D7, 0x4FEE, 
    0x53D7, 0x55FD, 0x56DA, 0x5782, 0x58FD, 0x5AC2, 0x5B88, 0x5CAB, 0x5CC0, 
    0x5E25, 0x6101,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x620D, 0x624B, 
    0x6388, 0x641C, 0x6536, 0x6578, 0x6A39, 0x6B8A, 0x6C34, 0x6D19, 0x6F31, 
    0x71E7, 0x72E9, 0x7378, 0x7407, 0x74B2, 0x7626, 0x7761, 0x79C0, 0x7A57, 
    0x7AEA, 0x7CB9, 0x7D8F, 0x7DAC, 0x7E61, 0x7F9E, 0x8129, 0x8331, 0x8490, 
    0x84DA, 0x85EA, 0x8896, 0x8AB0, 0x8B90, 0x8F38, 0x9042, 0x9083, 0x916C, 
    0x9296, 0x92B9, 0x968B, 0x96A7, 0x96A8, 0x96D6, 0x9700, 0x9808, 0x9996, 
    0x9AD3, 0x9B1A, 0x53D4, 0x587E, 0x5919, 0x5B70, 0x5BBF, 0x6DD1, 0x6F5A, 
    0x719F, 0x7421, 0x74B9, 0x8085, 0x83FD, 0x5DE1, 0x5F87, 0x5FAA, 0x6042, 
    0x65EC, 0x6812, 0x696F, 0x6A53, 0x6B89, 0x6D35, 0x6DF3, 0x73E3, 0x76FE, 
    0x77AC, 0x7B4D, 0x7D14, 0x8123, 0x821C, 0x8340, 0x84F4, 0x8563, 0x8A62, 
    0x8AC4, 0x9187, 0x931E, 0x9806, 0x99B4, 0x620C, 0x8853, 0x8FF0, 0x9265, 
    0x5D07, 0x5D27,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5D69, 0x745F, 
    0x819D, 0x8768, 0x6FD5, 0x62FE, 0x7FD2, 0x8936, 0x8972, 0x4E1E, 0x4E58, 
    0x50E7, 0x52DD, 0x5347, 0x627F, 0x6607, 0x7E69, 0x8805, 0x965E, 0x4F8D, 
    0x5319, 0x5636, 0x59CB, 0x5AA4, 0x5C38, 0x5C4E, 0x5C4D, 0x5E02, 0x5F11, 
    0x6043, 0x65BD, 0x662F, 0x6642, 0x67BE, 0x67F4, 0x731C, 0x77E2, 0x793A, 
    0x7FC5, 0x8494, 0x84CD, 0x8996, 0x8A66, 0x8A69, 0x8AE1, 0x8C55, 0x8C7A, 
    0x57F4, 0x5BD4, 0x5F0F, 0x606F, 0x62ED, 0x690D, 0x6B96, 0x6E5C, 0x7184, 
    0x7BD2, 0x8755, 0x8B58, 0x8EFE, 0x98DF, 0x98FE, 0x4F38, 0x4F81, 0x4FE1, 
    0x547B, 0x5A20, 0x5BB8, 0x613C, 0x65B0, 0x6668, 0x71FC, 0x7533, 0x795E, 
    0x7D33, 0x814E, 0x81E3, 0x8398, 0x85AA, 0x85CE, 0x8703, 0x8A0A, 0x8EAB, 
    0x8F9B, 0xF971, 0x8FC5, 0x5931, 0x5BA4, 0x5BE6, 0x6089, 0x5BE9, 0x5C0B, 
    0x5FC3, 0x6C81,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xF972, 0x6DF1, 
    0x700B, 0x751A, 0x82AF, 0x8AF6, 0x4EC0, 0x5341, 0xF973, 0x96D9, 0x6C0F, 
    0x4E9E, 0x4FC4, 0x5152, 0x555E, 0x5A25, 0x5CE8, 0x6211, 0x7259, 0x82BD, 
    0x83AA, 0x86FE, 0x8859, 0x8A1D, 0x963F, 0x96C5, 0x9913, 0x9D09, 0x9D5D, 
    0x580A, 0x5CB3, 0x5DBD, 0x5E44, 0x60E1, 0x6115, 0x63E1, 0x6A02, 0x6E25, 
    0x9102, 0x9354, 0x984E, 0x9C10, 0x9F77, 0x5B89, 0x5CB8, 0x6309, 0x664F, 
    0x6848, 0x773C, 0x96C1, 0x978D, 0x9854, 0x9B9F, 0x65A1, 0x8B01, 0x8ECB, 
    0x95BC, 0x5535, 0x5CA9, 0x5DD6, 0x5EB5, 0x6697, 0x764C, 0x83F4, 0x95C7, 
    0x58D3, 0x62BC, 0x72CE, 0x9D28, 0x4EF0, 0x592E, 0x600F, 0x663B, 0x6B83, 
    0x79E7, 0x9D26, 0x5393, 0x54C0, 0x57C3, 0x5D16, 0x611B, 0x66D6, 0x6DAF, 
    0x788D, 0x827E, 0x9698, 0x9744, 0x5384, 0x627C, 0x6396, 0x6DB2, 0x7E0A, 
    0x814B, 0x984D,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6AFB, 0x7F4C, 
    0x9DAF, 0x9E1A, 0x4E5F, 0x503B, 0x51B6, 0x591C, 0x60F9, 0x63F6, 0x6930, 
    0x723A, 0x8036, 0xF974, 0x91CE, 0x5F31, 0xF975, 0xF976, 0x7D04, 0x82E5, 
    0x846F, 0x84BB, 0x85E5, 0x8E8D, 0xF977, 0x4F6F, 0xF978, 0xF979, 0x58E4, 
    0x5B43, 0x6059, 0x63DA, 0x6518, 0x656D, 0x6698, 0xF97A, 0x694A, 0x6A23, 
    0x6D0B, 0x7001, 0x716C, 0x75D2, 0x760D, 0x79B3, 0x7A70, 0xF97B, 0x7F8A, 
    0xF97C, 0x8944, 0xF97D, 0x8B93, 0x91C0, 0x967D, 0xF97E, 0x990A, 0x5704, 
    0x5FA1, 0x65BC, 0x6F01, 0x7600, 0x79A6, 0x8A9E, 0x99AD, 0x9B5A, 0x9F6C, 
    0x5104, 0x61B6, 0x6291, 0x6A8D, 0x81C6, 0x5043, 0x5830, 0x5F66, 0x7109, 
    0x8A00, 0x8AFA, 0x5B7C, 0x8616, 0x4FFA, 0x513C, 0x56B4, 0x5944, 0x63A9, 
    0x6DF9, 0x5DAA, 0x696D, 0x5186, 0x4E88, 0x4F59, 0xF97F, 0xF980, 0xF981, 
    0x5982, 0xF982,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xF983, 0x6B5F, 
    0x6C5D, 0xF984, 0x74B5, 0x7916, 0xF985, 0x8207, 0x8245, 0x8339, 0x8F3F, 
    0x8F5D, 0xF986, 0x9918, 0xF987, 0xF988, 0xF989, 0x4EA6, 0xF98A, 0x57DF, 
    0x5F79, 0x6613, 0xF98B, 0xF98C, 0x75AB, 0x7E79, 0x8B6F, 0xF98D, 0x9006, 
    0x9A5B, 0x56A5, 0x5827, 0x59F8, 0x5A1F, 0x5BB4, 0xF98E, 0x5EF6, 0xF98F, 
    0xF990, 0x6350, 0x633B, 0xF991, 0x693D, 0x6C87, 0x6CBF, 0x6D8E, 0x6D93, 
    0x6DF5, 0x6F14, 0xF992, 0x70DF, 0x7136, 0x7159, 0xF993, 0x71C3, 0x71D5, 
    0xF994, 0x784F, 0x786F, 0xF995, 0x7B75, 0x7DE3, 0xF996, 0x7E2F, 0xF997, 
    0x884D, 0x8EDF, 0xF998, 0xF999, 0xF99A, 0x925B, 0xF99B, 0x9CF6, 0xF99C, 
    0xF99D, 0xF99E, 0x6085, 0x6D85, 0xF99F, 0x71B1, 0xF9A0, 0xF9A1, 0x95B1, 
    0x53AD, 0xF9A2, 0xF9A3, 0xF9A4, 0x67D3, 0xF9A5, 0x708E, 0x7130, 0x7430, 
    0x8276, 0x82D2,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xF9A6, 0x95BB, 
    0x9AE5, 0x9E7D, 0x66C4, 0xF9A7, 0x71C1, 0x8449, 0xF9A8, 0xF9A9, 0x584B, 
    0xF9AA, 0xF9AB, 0x5DB8, 0x5F71, 0xF9AC, 0x6620, 0x668E, 0x6979, 0x69AE, 
    0x6C38, 0x6CF3, 0x6E36, 0x6F41, 0x6FDA, 0x701B, 0x702F, 0x7150, 0x71DF, 
    0x7370, 0xF9AD, 0x745B, 0xF9AE, 0x74D4, 0x76C8, 0x7A4E, 0x7E93, 0xF9AF, 
    0xF9B0, 0x82F1, 0x8A60, 0x8FCE, 0xF9B1, 0x9348, 0xF9B2, 0x9719, 0xF9B3, 
    0xF9B4, 0x4E42, 0x502A, 0xF9B5, 0x5208, 0x53E1, 0x66F3, 0x6C6D, 0x6FCA, 
    0x730A, 0x777F, 0x7A62, 0x82AE, 0x85DD, 0x8602, 0xF9B6, 0x88D4, 0x8A63, 
    0x8B7D, 0x8C6B, 0xF9B7, 0x92B3, 0xF9B8, 0x9713, 0x9810, 0x4E94, 0x4F0D, 
    0x4FC9, 0x50B2, 0x5348, 0x543E, 0x5433, 0x55DA, 0x5862, 0x58BA, 0x5967, 
    0x5A1B, 0x5BE4, 0x609F, 0xF9B9, 0x61CA, 0x6556, 0x65FF, 0x6664, 0x68A7, 
    0x6C5A, 0x6FB3,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x70CF, 0x71AC, 
    0x7352, 0x7B7D, 0x8708, 0x8AA4, 0x9C32, 0x9F07, 0x5C4B, 0x6C83, 0x7344, 
    0x7389, 0x923A, 0x6EAB, 0x7465, 0x761F, 0x7A69, 0x7E15, 0x860A, 0x5140, 
    0x58C5, 0x64C1, 0x74EE, 0x7515, 0x7670, 0x7FC1, 0x9095, 0x96CD, 0x9954, 
    0x6E26, 0x74E6, 0x7AA9, 0x7AAA, 0x81E5, 0x86D9, 0x8778, 0x8A1B, 0x5A49, 
    0x5B8C, 0x5B9B, 0x68A1, 0x6900, 0x6D63, 0x73A9, 0x7413, 0x742C, 0x7897, 
    0x7DE9, 0x7FEB, 0x8118, 0x8155, 0x839E, 0x8C4C, 0x962E, 0x9811, 0x66F0, 
    0x5F80, 0x65FA, 0x6789, 0x6C6A, 0x738B, 0x502D, 0x5A03, 0x6B6A, 0x77EE, 
    0x5916, 0x5D6C, 0x5DCD, 0x7325, 0x754F, 0xF9BA, 0xF9BB, 0x50E5, 0x51F9, 
    0x582F, 0x592D, 0x5996, 0x59DA, 0x5BE5, 0xF9BC, 0xF9BD, 0x5DA2, 0x62D7, 
    0x6416, 0x6493, 0x64FE, 0xF9BE, 0x66DC, 0xF9BF, 0x6A48, 0xF9C0, 0x71FF, 
    0x7464, 0xF9C1,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7A88, 0x7AAF, 
    0x7E47, 0x7E5E, 0x8000, 0x8170, 0xF9C2, 0x87EF, 0x8981, 0x8B20, 0x9059, 
    0xF9C3, 0x9080, 0x9952, 0x617E, 0x6B32, 0x6D74, 0x7E1F, 0x8925, 0x8FB1, 
    0x4FD1, 0x50AD, 0x5197, 0x52C7, 0x57C7, 0x5889, 0x5BB9, 0x5EB8, 0x6142, 
    0x6995, 0x6D8C, 0x6E67, 0x6EB6, 0x7194, 0x7462, 0x7528, 0x752C, 0x8073, 
    0x8338, 0x84C9, 0x8E0A, 0x9394, 0x93DE, 0xF9C4, 0x4E8E, 0x4F51, 0x5076, 
    0x512A, 0x53C8, 0x53CB, 0x53F3, 0x5B87, 0x5BD3, 0x5C24, 0x611A, 0x6182, 
    0x65F4, 0x725B, 0x7397, 0x7440, 0x76C2, 0x7950, 0x7991, 0x79B9, 0x7D06, 
    0x7FBD, 0x828B, 0x85D5, 0x865E, 0x8FC2, 0x9047, 0x90F5, 0x91EA, 0x9685, 
    0x96E8, 0x96E9, 0x52D6, 0x5F67, 0x65ED, 0x6631, 0x682F, 0x715C, 0x7A36, 
    0x90C1, 0x980A, 0x4E91, 0xF9C5, 0x6A52, 0x6B9E, 0x6F90, 0x7189, 0x8018, 
    0x82B8, 0x8553,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x904B, 0x9695, 
    0x96F2, 0x97FB, 0x851A, 0x9B31, 0x4E90, 0x718A, 0x96C4, 0x5143, 0x539F, 
    0x54E1, 0x5713, 0x5712, 0x57A3, 0x5A9B, 0x5AC4, 0x5BC3, 0x6028, 0x613F, 
    0x63F4, 0x6C85, 0x6D39, 0x6E72, 0x6E90, 0x7230, 0x733F, 0x7457, 0x82D1, 
    0x8881, 0x8F45, 0x9060, 0xF9C6, 0x9662, 0x9858, 0x9D1B, 0x6708, 0x8D8A, 
    0x925E, 0x4F4D, 0x5049, 0x50DE, 0x5371, 0x570D, 0x59D4, 0x5A01, 0x5C09, 
    0x6170, 0x6690, 0x6E2D, 0x7232, 0x744B, 0x7DEF, 0x80C3, 0x840E, 0x8466, 
    0x853F, 0x875F, 0x885B, 0x8918, 0x8B02, 0x9055, 0x97CB, 0x9B4F, 0x4E73, 
    0x4F91, 0x5112, 0x516A, 0xF9C7, 0x552F, 0x55A9, 0x5B7A, 0x5BA5, 0x5E7C, 
    0x5E7D, 0x5EBE, 0x60A0, 0x60DF, 0x6108, 0x6109, 0x63C4, 0x6538, 0x6709, 
    0xF9C8, 0x67D4, 0x67DA, 0xF9C9, 0x6961, 0x6962, 0x6CB9, 0x6D27, 0xF9CA, 
    0x6E38, 0xF9CB,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6FE1, 0x7336, 
    0x7337, 0xF9CC, 0x745C, 0x7531, 0xF9CD, 0x7652, 0xF9CE, 0xF9CF, 0x7DAD, 
    0x81FE, 0x8438, 0x88D5, 0x8A98, 0x8ADB, 0x8AED, 0x8E30, 0x8E42, 0x904A, 
    0x903E, 0x907A, 0x9149, 0x91C9, 0x936E, 0xF9D0, 0xF9D1, 0x5809, 0xF9D2, 
    0x6BD3, 0x8089, 0x80B2, 0xF9D3, 0xF9D4, 0x5141, 0x596B, 0x5C39, 0xF9D5, 
    0xF9D6, 0x6F64, 0x73A7, 0x80E4, 0x8D07, 0xF9D7, 0x9217, 0x958F, 0xF9D8, 
    0xF9D9, 0xF9DA, 0xF9DB, 0x807F, 0x620E, 0x701C, 0x7D68, 0x878D, 0xF9DC, 
    0x57A0, 0x6069, 0x6147, 0x6BB7, 0x8ABE, 0x9280, 0x96B1, 0x4E59, 0x541F, 
    0x6DEB, 0x852D, 0x9670, 0x97F3, 0x98EE, 0x63D6, 0x6CE3, 0x9091, 0x51DD, 
    0x61C9, 0x81BA, 0x9DF9, 0x4F9D, 0x501A, 0x5100, 0x5B9C, 0x610F, 0x61FF, 
    0x64EC, 0x6905, 0x6BC5, 0x7591, 0x77E3, 0x7FA9, 0x8264, 0x858F, 0x87FB, 
    0x8863, 0x8ABC,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8B70, 0x91AB, 
    0x4E8C, 0x4EE5, 0x4F0A, 0xF9DD, 0xF9DE, 0x5937, 0x59E8, 0xF9DF, 0x5DF2, 
    0x5F1B, 0x5F5B, 0x6021, 0xF9E0, 0xF9E1, 0xF9E2, 0xF9E3, 0x723E, 0x73E5, 
    0xF9E4, 0x7570, 0x75CD, 0xF9E5, 0x79FB, 0xF9E6, 0x800C, 0x8033, 0x8084, 
    0x82E1, 0x8351, 0xF9E7, 0xF9E8, 0x8CBD, 0x8CB3, 0x9087, 0xF9E9, 0xF9EA, 
    0x98F4, 0x990C, 0xF9EB, 0xF9EC, 0x7037, 0x76CA, 0x7FCA, 0x7FCC, 0x7FFC, 
    0x8B1A, 0x4EBA, 0x4EC1, 0x5203, 0x5370, 0xF9ED, 0x54BD, 0x56E0, 0x59FB, 
    0x5BC5, 0x5F15, 0x5FCD, 0x6E6E, 0xF9EE, 0xF9EF, 0x7D6A, 0x8335, 0xF9F0, 
    0x8693, 0x8A8D, 0xF9F1, 0x976D, 0x9777, 0xF9F2, 0xF9F3, 0x4E00, 0x4F5A, 
    0x4F7E, 0x58F9, 0x65E5, 0x6EA2, 0x9038, 0x93B0, 0x99B9, 0x4EFB, 0x58EC, 
    0x598A, 0x59D9, 0x6041, 0xF9F4, 0xF9F5, 0x7A14, 0xF9F6, 0x834F, 0x8CC3, 
    0x5165, 0x5344,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xF9F7, 0xF9F8, 
    0xF9F9, 0x4ECD, 0x5269, 0x5B55, 0x82BF, 0x4ED4, 0x523A, 0x54A8, 0x59C9, 
    0x59FF, 0x5B50, 0x5B57, 0x5B5C, 0x6063, 0x6148, 0x6ECB, 0x7099, 0x716E, 
    0x7386, 0x74F7, 0x75B5, 0x78C1, 0x7D2B, 0x8005, 0x81EA, 0x8328, 0x8517, 
    0x85C9, 0x8AEE, 0x8CC7, 0x96CC, 0x4F5C, 0x52FA, 0x56BC, 0x65AB, 0x6628, 
    0x707C, 0x70B8, 0x7235, 0x7DBD, 0x828D, 0x914C, 0x96C0, 0x9D72, 0x5B71, 
    0x68E7, 0x6B98, 0x6F7A, 0x76DE, 0x5C91, 0x66AB, 0x6F5B, 0x7BB4, 0x7C2A, 
    0x8836, 0x96DC, 0x4E08, 0x4ED7, 0x5320, 0x5834, 0x58BB, 0x58EF, 0x596C, 
    0x5C07, 0x5E33, 0x5E84, 0x5F35, 0x638C, 0x66B2, 0x6756, 0x6A1F, 0x6AA3, 
    0x6B0C, 0x6F3F, 0x7246, 0xF9FA, 0x7350, 0x748B, 0x7AE0, 0x7CA7, 0x8178, 
    0x81DF, 0x81E7, 0x838A, 0x846C, 0x8523, 0x8594, 0x85CF, 0x88DD, 0x8D13, 
    0x91AC, 0x9577,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x969C, 0x518D, 
    0x54C9, 0x5728, 0x5BB0, 0x624D, 0x6750, 0x683D, 0x6893, 0x6E3D, 0x6ED3, 
    0x707D, 0x7E21, 0x88C1, 0x8CA1, 0x8F09, 0x9F4B, 0x9F4E, 0x722D, 0x7B8F, 
    0x8ACD, 0x931A, 0x4F47, 0x4F4E, 0x5132, 0x5480, 0x59D0, 0x5E95, 0x62B5, 
    0x6775, 0x696E, 0x6A17, 0x6CAE, 0x6E1A, 0x72D9, 0x732A, 0x75BD, 0x7BB8, 
    0x7D35, 0x82E7, 0x83F9, 0x8457, 0x85F7, 0x8A5B, 0x8CAF, 0x8E87, 0x9019, 
    0x90B8, 0x96CE, 0x9F5F, 0x52E3, 0x540A, 0x5AE1, 0x5BC2, 0x6458, 0x6575, 
    0x6EF4, 0x72C4, 0xF9FB, 0x7684, 0x7A4D, 0x7B1B, 0x7C4D, 0x7E3E, 0x7FDF, 
    0x837B, 0x8B2B, 0x8CCA, 0x8D64, 0x8DE1, 0x8E5F, 0x8FEA, 0x8FF9, 0x9069, 
    0x93D1, 0x4F43, 0x4F7A, 0x50B3, 0x5168, 0x5178, 0x524D, 0x526A, 0x5861, 
    0x587C, 0x5960, 0x5C08, 0x5C55, 0x5EDB, 0x609B, 0x6230, 0x6813, 0x6BBF, 
    0x6C08, 0x6FB1,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x714E, 0x7420, 
    0x7530, 0x7538, 0x7551, 0x7672, 0x7B4C, 0x7B8B, 0x7BAD, 0x7BC6, 0x7E8F, 
    0x8A6E, 0x8F3E, 0x8F49, 0x923F, 0x9293, 0x9322, 0x942B, 0x96FB, 0x985A, 
    0x986B, 0x991E, 0x5207, 0x622A, 0x6298, 0x6D59, 0x7664, 0x7ACA, 0x7BC0, 
    0x7D76, 0x5360, 0x5CBE, 0x5E97, 0x6F38, 0x70B9, 0x7C98, 0x9711, 0x9B8E, 
    0x9EDE, 0x63A5, 0x647A, 0x8776, 0x4E01, 0x4E95, 0x4EAD, 0x505C, 0x5075, 
    0x5448, 0x59C3, 0x5B9A, 0x5E40, 0x5EAD, 0x5EF7, 0x5F81, 0x60C5, 0x633A, 
    0x653F, 0x6574, 0x65CC, 0x6676, 0x6678, 0x67FE, 0x6968, 0x6A89, 0x6B63, 
    0x6C40, 0x6DC0, 0x6DE8, 0x6E1F, 0x6E5E, 0x701E, 0x70A1, 0x738E, 0x73FD, 
    0x753A, 0x775B, 0x7887, 0x798E, 0x7A0B, 0x7A7D, 0x7CBE, 0x7D8E, 0x8247, 
    0x8A02, 0x8AEA, 0x8C9E, 0x912D, 0x914A, 0x91D8, 0x9266, 0x92CC, 0x9320, 
    0x9706, 0x9756,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x975C, 0x9802, 
    0x9F0E, 0x5236, 0x5291, 0x557C, 0x5824, 0x5E1D, 0x5F1F, 0x608C, 0x63D0, 
    0x68AF, 0x6FDF, 0x796D, 0x7B2C, 0x81CD, 0x85BA, 0x88FD, 0x8AF8, 0x8E44, 
    0x918D, 0x9664, 0x969B, 0x973D, 0x984C, 0x9F4A, 0x4FCE, 0x5146, 0x51CB, 
    0x52A9, 0x5632, 0x5F14, 0x5F6B, 0x63AA, 0x64CD, 0x65E9, 0x6641, 0x66FA, 
    0x66F9, 0x671D, 0x689D, 0x68D7, 0x69FD, 0x6F15, 0x6F6E, 0x7167, 0x71E5, 
    0x722A, 0x74AA, 0x773A, 0x7956, 0x795A, 0x79DF, 0x7A20, 0x7A95, 0x7C97, 
    0x7CDF, 0x7D44, 0x7E70, 0x8087, 0x85FB, 0x86A4, 0x8A54, 0x8ABF, 0x8D99, 
    0x8E81, 0x9020, 0x906D, 0x91E3, 0x963B, 0x96D5, 0x9CE5, 0x65CF, 0x7C07, 
    0x8DB3, 0x93C3, 0x5B58, 0x5C0A, 0x5352, 0x62D9, 0x731D, 0x5027, 0x5B97, 
    0x5F9E, 0x60B0, 0x616B, 0x68D5, 0x6DD9, 0x742E, 0x7A2E, 0x7D42, 0x7D9C, 
    0x7E31, 0x816B,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8E2A, 0x8E35, 
    0x937E, 0x9418, 0x4F50, 0x5750, 0x5DE6, 0x5EA7, 0x632B, 0x7F6A, 0x4E3B, 
    0x4F4F, 0x4F8F, 0x505A, 0x59DD, 0x80C4, 0x546A, 0x5468, 0x55FE, 0x594F, 
    0x5B99, 0x5DDE, 0x5EDA, 0x665D, 0x6731, 0x67F1, 0x682A, 0x6CE8, 0x6D32, 
    0x6E4A, 0x6F8D, 0x70B7, 0x73E0, 0x7587, 0x7C4C, 0x7D02, 0x7D2C, 0x7DA2, 
    0x821F, 0x86DB, 0x8A3B, 0x8A85, 0x8D70, 0x8E8A, 0x8F33, 0x9031, 0x914E, 
    0x9152, 0x9444, 0x99D0, 0x7AF9, 0x7CA5, 0x4FCA, 0x5101, 0x51C6, 0x57C8, 
    0x5BEF, 0x5CFB, 0x6659, 0x6A3D, 0x6D5A, 0x6E96, 0x6FEC, 0x710C, 0x756F, 
    0x7AE3, 0x8822, 0x9021, 0x9075, 0x96CB, 0x99FF, 0x8301, 0x4E2D, 0x4EF2, 
    0x8846, 0x91CD, 0x537D, 0x6ADB, 0x696B, 0x6C41, 0x847A, 0x589E, 0x618E, 
    0x66FE, 0x62EF, 0x70DD, 0x7511, 0x75C7, 0x7E52, 0x84B8, 0x8B49, 0x8D08, 
    0x4E4B, 0x53EA,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x54AB, 0x5730, 
    0x5740, 0x5FD7, 0x6301, 0x6307, 0x646F, 0x652F, 0x65E8, 0x667A, 0x679D, 
    0x67B3, 0x6B62, 0x6C60, 0x6C9A, 0x6F2C, 0x77E5, 0x7825, 0x7949, 0x7957, 
    0x7D19, 0x80A2, 0x8102, 0x81F3, 0x829D, 0x82B7, 0x8718, 0x8A8C, 0xF9FC, 
    0x8D04, 0x8DBE, 0x9072, 0x76F4, 0x7A19, 0x7A37, 0x7E54, 0x8077, 0x5507, 
    0x55D4, 0x5875, 0x632F, 0x6422, 0x6649, 0x664B, 0x686D, 0x699B, 0x6B84, 
    0x6D25, 0x6EB1, 0x73CD, 0x7468, 0x74A1, 0x755B, 0x75B9, 0x76E1, 0x771E, 
    0x778B, 0x79E6, 0x7E09, 0x7E1D, 0x81FB, 0x852F, 0x8897, 0x8A3A, 0x8CD1, 
    0x8EEB, 0x8FB0, 0x9032, 0x93AD, 0x9663, 0x9673, 0x9707, 0x4F84, 0x53F1, 
    0x59EA, 0x5AC9, 0x5E19, 0x684E, 0x74C6, 0x75BE, 0x79E9, 0x7A92, 0x81A3, 
    0x86ED, 0x8CEA, 0x8DCC, 0x8FED, 0x659F, 0x6715, 0xF9FD, 0x57F7, 0x6F57, 
    0x7DDD, 0x8F2F,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x93F6, 0x96C6, 
    0x5FB5, 0x61F2, 0x6F84, 0x4E14, 0x4F98, 0x501F, 0x53C9, 0x55DF, 0x5D6F, 
    0x5DEE, 0x6B21, 0x6B64, 0x78CB, 0x7B9A, 0xF9FE, 0x8E49, 0x8ECA, 0x906E, 
    0x6349, 0x643E, 0x7740, 0x7A84, 0x932F, 0x947F, 0x9F6A, 0x64B0, 0x6FAF, 
    0x71E6, 0x74A8, 0x74DA, 0x7AC4, 0x7C12, 0x7E82, 0x7CB2, 0x7E98, 0x8B9A, 
    0x8D0A, 0x947D, 0x9910, 0x994C, 0x5239, 0x5BDF, 0x64E6, 0x672D, 0x7D2E, 
    0x50ED, 0x53C3, 0x5879, 0x6158, 0x6159, 0x61FA, 0x65AC, 0x7AD9, 0x8B92, 
    0x8B96, 0x5009, 0x5021, 0x5275, 0x5531, 0x5A3C, 0x5EE0, 0x5F70, 0x6134, 
    0x655E, 0x660C, 0x6636, 0x66A2, 0x69CD, 0x6EC4, 0x6F32, 0x7316, 0x7621, 
    0x7A93, 0x8139, 0x8259, 0x83D6, 0x84BC, 0x50B5, 0x57F0, 0x5BC0, 0x5BE8, 
    0x5F69, 0x63A1, 0x7826, 0x7DB5, 0x83DC, 0x8521, 0x91C7, 0x91F5, 0x518A, 
    0x67F5, 0x7B56,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8CAC, 0x51C4, 
    0x59BB, 0x60BD, 0x8655, 0x501C, 0xF9FF, 0x5254, 0x5C3A, 0x617D, 0x621A, 
    0x62D3, 0x64F2, 0x65A5, 0x6ECC, 0x7620, 0x810A, 0x8E60, 0x965F, 0x96BB, 
    0x4EDF, 0x5343, 0x5598, 0x5929, 0x5DDD, 0x64C5, 0x6CC9, 0x6DFA, 0x7394, 
    0x7A7F, 0x821B, 0x85A6, 0x8CE4, 0x8E10, 0x9077, 0x91E7, 0x95E1, 0x9621, 
    0x97C6, 0x51F8, 0x54F2, 0x5586, 0x5FB9, 0x64A4, 0x6F88, 0x7DB4, 0x8F1F, 
    0x8F4D, 0x9435, 0x50C9, 0x5C16, 0x6CBE, 0x6DFB, 0x751B, 0x77BB, 0x7C3D, 
    0x7C64, 0x8A79, 0x8AC2, 0x581E, 0x59BE, 0x5E16, 0x6377, 0x7252, 0x758A, 
    0x776B, 0x8ADC, 0x8CBC, 0x8F12, 0x5EF3, 0x6674, 0x6DF8, 0x807D, 0x83C1, 
    0x8ACB, 0x9751, 0x9BD6, 0xFA00, 0x5243, 0x66FF, 0x6D95, 0x6EEF, 0x7DE0, 
    0x8AE6, 0x902E, 0x905E, 0x9AD4, 0x521D, 0x527F, 0x54E8, 0x6194, 0x6284, 
    0x62DB, 0x68A2,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6912, 0x695A, 
    0x6A35, 0x7092, 0x7126, 0x785D, 0x7901, 0x790E, 0x79D2, 0x7A0D, 0x8096, 
    0x8278, 0x82D5, 0x8349, 0x8549, 0x8C82, 0x8D85, 0x9162, 0x918B, 0x91AE, 
    0x4FC3, 0x56D1, 0x71ED, 0x77D7, 0x8700, 0x89F8, 0x5BF8, 0x5FD6, 0x6751, 
    0x90A8, 0x53E2, 0x585A, 0x5BF5, 0x60A4, 0x6181, 0x6460, 0x7E3D, 0x8070, 
    0x8525, 0x9283, 0x64AE, 0x50AC, 0x5D14, 0x6700, 0x589C, 0x62BD, 0x63A8, 
    0x690E, 0x6978, 0x6A1E, 0x6E6B, 0x76BA, 0x79CB, 0x82BB, 0x8429, 0x8ACF, 
    0x8DA8, 0x8FFD, 0x9112, 0x914B, 0x919C, 0x9310, 0x9318, 0x939A, 0x96DB, 
    0x9A36, 0x9C0D, 0x4E11, 0x755C, 0x795D, 0x7AFA, 0x7B51, 0x7BC9, 0x7E2E, 
    0x84C4, 0x8E59, 0x8E74, 0x8EF8, 0x9010, 0x6625, 0x693F, 0x7443, 0x51FA, 
    0x672E, 0x9EDC, 0x5145, 0x5FE0, 0x6C96, 0x87F2, 0x885D, 0x8877, 0x60B4, 
    0x81B5, 0x8403,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8D05, 0x53D6, 
    0x5439, 0x5634, 0x5A36, 0x5C31, 0x708A, 0x7FE0, 0x805A, 0x8106, 0x81ED, 
    0x8DA3, 0x9189, 0x9A5F, 0x9DF2, 0x5074, 0x4EC4, 0x53A0, 0x60FB, 0x6E2C, 
    0x5C64, 0x4F88, 0x5024, 0x55E4, 0x5CD9, 0x5E5F, 0x6065, 0x6894, 0x6CBB, 
    0x6DC4, 0x71BE, 0x75D4, 0x75F4, 0x7661, 0x7A1A, 0x7A49, 0x7DC7, 0x7DFB, 
    0x7F6E, 0x81F4, 0x86A9, 0x8F1C, 0x96C9, 0x99B3, 0x9F52, 0x5247, 0x52C5, 
    0x98ED, 0x89AA, 0x4E03, 0x67D2, 0x6F06, 0x4FB5, 0x5BE2, 0x6795, 0x6C88, 
    0x6D78, 0x741B, 0x7827, 0x91DD, 0x937C, 0x87C4, 0x79E4, 0x7A31, 0x5FEB, 
    0x4ED6, 0x54A4, 0x553E, 0x58AE, 0x59A5, 0x60F0, 0x6253, 0x62D6, 0x6736, 
    0x6955, 0x8235, 0x9640, 0x99B1, 0x99DD, 0x502C, 0x5353, 0x5544, 0x577C, 
    0xFA01, 0x6258, 0xFA02, 0x64E2, 0x666B, 0x67DD, 0x6FC1, 0x6FEF, 0x7422, 
    0x7438, 0x8A17,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9438, 0x5451, 
    0x5606, 0x5766, 0x5F48, 0x619A, 0x6B4E, 0x7058, 0x70AD, 0x7DBB, 0x8A95, 
    0x596A, 0x812B, 0x63A2, 0x7708, 0x803D, 0x8CAA, 0x5854, 0x642D, 0x69BB, 
    0x5B95, 0x5E11, 0x6E6F, 0xFA03, 0x8569, 0x514C, 0x53F0, 0x592A, 0x6020, 
    0x614B, 0x6B86, 0x6C70, 0x6CF0, 0x7B1E, 0x80CE, 0x82D4, 0x8DC6, 0x90B0, 
    0x98B1, 0xFA04, 0x64C7, 0x6FA4, 0x6491, 0x6504, 0x514E, 0x5410, 0x571F, 
    0x8A0E, 0x615F, 0x6876, 0xFA05, 0x75DB, 0x7B52, 0x7D71, 0x901A, 0x5806, 
    0x69CC, 0x817F, 0x892A, 0x9000, 0x9839, 0x5078, 0x5957, 0x59AC, 0x6295, 
    0x900F, 0x9B2A, 0x615D, 0x7279, 0x95D6, 0x5761, 0x5A46, 0x5DF4, 0x628A, 
    0x64AD, 0x64FA, 0x6777, 0x6CE2, 0x6D3E, 0x722C, 0x7436, 0x7834, 0x7F77, 
    0x82AD, 0x8DDB, 0x9817, 0x5224, 0x5742, 0x677F, 0x7248, 0x74E3, 0x8CA9, 
    0x8FA6, 0x9211,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x962A, 0x516B, 
    0x53ED, 0x634C, 0x4F69, 0x5504, 0x6096, 0x6557, 0x6C9B, 0x6D7F, 0x724C, 
    0x72FD, 0x7A17, 0x8987, 0x8C9D, 0x5F6D, 0x6F8E, 0x70F9, 0x81A8, 0x610E, 
    0x4FBF, 0x504F, 0x6241, 0x7247, 0x7BC7, 0x7DE8, 0x7FE9, 0x904D, 0x97AD, 
    0x9A19, 0x8CB6, 0x576A, 0x5E73, 0x67B0, 0x840D, 0x8A55, 0x5420, 0x5B16, 
    0x5E63, 0x5EE2, 0x5F0A, 0x6583, 0x80BA, 0x853D, 0x9589, 0x965B, 0x4F48, 
    0x5305, 0x530D, 0x530F, 0x5486, 0x54FA, 0x5703, 0x5E03, 0x6016, 0x629B, 
    0x62B1, 0x6355, 0xFA06, 0x6CE1, 0x6D66, 0x75B1, 0x7832, 0x80DE, 0x812F, 
    0x82DE, 0x8461, 0x84B2, 0x888D, 0x8912, 0x900B, 0x92EA, 0x98FD, 0x9B91, 
    0x5E45, 0x66B4, 0x66DD, 0x7011, 0x7206, 0xFA07, 0x4FF5, 0x527D, 0x5F6A, 
    0x6153, 0x6753, 0x6A19, 0x6F02, 0x74E2, 0x7968, 0x8868, 0x8C79, 0x98C7, 
    0x98C4, 0x9A43,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x54C1, 0x7A1F, 
    0x6953, 0x8AF7, 0x8C4A, 0x98A8, 0x99AE, 0x5F7C, 0x62AB, 0x75B2, 0x76AE, 
    0x88AB, 0x907F, 0x9642, 0x5339, 0x5F3C, 0x5FC5, 0x6CCC, 0x73CC, 0x7562, 
    0x758B, 0x7B46, 0x82FE, 0x999D, 0x4E4F, 0x903C, 0x4E0B, 0x4F55, 0x53A6, 
    0x590F, 0x5EC8, 0x6630, 0x6CB3, 0x7455, 0x8377, 0x8766, 0x8CC0, 0x9050, 
    0x971E, 0x9C15, 0x58D1, 0x5B78, 0x8650, 0x8B14, 0x9DB4, 0x5BD2, 0x6068, 
    0x608D, 0x65F1, 0x6C57, 0x6F22, 0x6FA3, 0x701A, 0x7F55, 0x7FF0, 0x9591, 
    0x9592, 0x9650, 0x97D3, 0x5272, 0x8F44, 0x51FD, 0x542B, 0x54B8, 0x5563, 
    0x558A, 0x6ABB, 0x6DB5, 0x7DD8, 0x8266, 0x929C, 0x9677, 0x9E79, 0x5408, 
    0x54C8, 0x76D2, 0x86E4, 0x95A4, 0x95D4, 0x965C, 0x4EA2, 0x4F09, 0x59EE, 
    0x5AE6, 0x5DF7, 0x6052, 0x6297, 0x676D, 0x6841, 0x6C86, 0x6E2F, 0x7F38, 
    0x809B, 0x822A,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xFA08, 0xFA09, 
    0x9805, 0x4EA5, 0x5055, 0x54B3, 0x5793, 0x595A, 0x5B69, 0x5BB3, 0x61C8, 
    0x6977, 0x6D77, 0x7023, 0x87F9, 0x89E3, 0x8A72, 0x8AE7, 0x9082, 0x99ED, 
    0x9AB8, 0x52BE, 0x6838, 0x5016, 0x5E78, 0x674F, 0x8347, 0x884C, 0x4EAB, 
    0x5411, 0x56AE, 0x73E6, 0x9115, 0x97FF, 0x9909, 0x9957, 0x9999, 0x5653, 
    0x589F, 0x865B, 0x8A31, 0x61B2, 0x6AF6, 0x737B, 0x8ED2, 0x6B47, 0x96AA, 
    0x9A57, 0x5955, 0x7200, 0x8D6B, 0x9769, 0x4FD4, 0x5CF4, 0x5F26, 0x61F8, 
    0x665B, 0x6CEB, 0x70AB, 0x7384, 0x73B9, 0x73FE, 0x7729, 0x774D, 0x7D43, 
    0x7D62, 0x7E23, 0x8237, 0x8852, 0xFA0A, 0x8CE2, 0x9249, 0x986F, 0x5B51, 
    0x7A74, 0x8840, 0x9801, 0x5ACC, 0x4FE0, 0x5354, 0x593E, 0x5CFD, 0x633E, 
    0x6D79, 0x72F9, 0x8105, 0x8107, 0x83A2, 0x92CF, 0x9830, 0x4EA8, 0x5144, 
    0x5211, 0x578B,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5F62, 0x6CC2, 
    0x6ECE, 0x7005, 0x7050, 0x70AF, 0x7192, 0x73E9, 0x7469, 0x834A, 0x87A2, 
    0x8861, 0x9008, 0x90A2, 0x93A3, 0x99A8, 0x516E, 0x5F57, 0x60E0, 0x6167, 
    0x66B3, 0x8559, 0x8E4A, 0x91AF, 0x978B, 0x4E4E, 0x4E92, 0x547C, 0x58D5, 
    0x58FA, 0x597D, 0x5CB5, 0x5F27, 0x6236, 0x6248, 0x660A, 0x6667, 0x6BEB, 
    0x6D69, 0x6DCF, 0x6E56, 0x6EF8, 0x6F94, 0x6FE0, 0x6FE9, 0x705D, 0x72D0, 
    0x7425, 0x745A, 0x74E0, 0x7693, 0x795C, 0x7CCA, 0x7E1E, 0x80E1, 0x82A6, 
    0x846B, 0x84BF, 0x864E, 0x865F, 0x8774, 0x8B77, 0x8C6A, 0x93AC, 0x9800, 
    0x9865, 0x60D1, 0x6216, 0x9177, 0x5A5A, 0x660F, 0x6DF7, 0x6E3E, 0x743F, 
    0x9B42, 0x5FFD, 0x60DA, 0x7B0F, 0x54C4, 0x5F18, 0x6C5E, 0x6CD3, 0x6D2A, 
    0x70D8, 0x7D05, 0x8679, 0x8A0C, 0x9D3B, 0x5316, 0x548C, 0x5B05, 0x6A3A, 
    0x706B, 0x7575,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x798D, 0x79BE, 
    0x82B1, 0x83EF, 0x8A71, 0x8B41, 0x8CA8, 0x9774, 0xFA0B, 0x64F4, 0x652B, 
    0x78BA, 0x78BB, 0x7A6B, 0x4E38, 0x559A, 0x5950, 0x5BA6, 0x5E7B, 0x60A3, 
    0x63DB, 0x6B61, 0x6665, 0x6853, 0x6E19, 0x7165, 0x74B0, 0x7D08, 0x9084, 
    0x9A69, 0x9C25, 0x6D3B, 0x6ED1, 0x733E, 0x8C41, 0x95CA, 0x51F0, 0x5E4C, 
    0x5FA8, 0x604D, 0x60F6, 0x6130, 0x614C, 0x6643, 0x6644, 0x69A5, 0x6CC1, 
    0x6E5F, 0x6EC9, 0x6F62, 0x714C, 0x749C, 0x7687, 0x7BC1, 0x7C27, 0x8352, 
    0x8757, 0x9051, 0x968D, 0x9EC3, 0x532F, 0x56DE, 0x5EFB, 0x5F8A, 0x6062, 
    0x6094, 0x61F7, 0x6666, 0x6703, 0x6A9C, 0x6DEE, 0x6FAE, 0x7070, 0x736A, 
    0x7E6A, 0x81BE, 0x8334, 0x86D4, 0x8AA8, 0x8CC4, 0x5283, 0x7372, 0x5B96, 
    0x6A6B, 0x9404, 0x54EE, 0x5686, 0x5B5D, 0x6548, 0x6585, 0x66C9, 0x689F, 
    0x6D8D, 0x6DC6,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x723B, 0x80B4, 
    0x9175, 0x9A4D, 0x4FAF, 0x5019, 0x539A, 0x540E, 0x543C, 0x5589, 0x55C5, 
    0x5E3F, 0x5F8C, 0x673D, 0x7166, 0x73DD, 0x9005, 0x52DB, 0x52F3, 0x5864, 
    0x58CE, 0x7104, 0x718F, 0x71FB, 0x85B0, 0x8A13, 0x6688, 0x85A8, 0x55A7, 
    0x6684, 0x714A, 0x8431, 0x5349, 0x5599, 0x6BC1, 0x5F59, 0x5FBD, 0x63EE, 
    0x6689, 0x7147, 0x8AF1, 0x8F1D, 0x9EBE, 0x4F11, 0x643A, 0x70CB, 0x7566, 
    0x8667, 0x6064, 0x8B4E, 0x9DF8, 0x5147, 0x51F6, 0x5308, 0x6D36, 0x80F8, 
    0x9ED1, 0x6615, 0x6B23, 0x7098, 0x75D5, 0x5403, 0x5C79, 0x7D07, 0x8A16, 
    0x6B20, 0x6B3D, 0x6B46, 0x5438, 0x6070, 0x6D3D, 0x7FD5, 0x8208, 0x50D6, 
    0x51DE, 0x559C, 0x566B, 0x56CD, 0x59EC, 0x5B09, 0x5E0C, 0x6199, 0x6198, 
    0x6231, 0x665E, 0x66E6, 0x7199, 0x71B9, 0x71BA, 0x72A7, 0x79A7, 0x7A00, 
    0x7FB2, 0x8A70,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR},
};

/*
 * Mapping from Unicode back to KS X 1001.
 * 
 * This inverse is done the same way the SBCS ones are done: I
 * provide a list of KS X 1001 positions, sorted into ascending
 * order of their Unicode values. Then we can binary-search through
 * this list, referring to the above table for each comparison, to
 * find the coordinates for a given code point.
 * 
 * Generated by running the following sh+Perl over KSX1001.TXT:

cat CP949.TXT | \
perl -ne '$a{$3}=[hex$1,hex$2] if /^0x(\S\S)(\S\S)\s+(0x\S+)\s/;' \
      -e 'END {$o="    ";for $k (sort keys %a) { ($r,$c) = @{$a{$k}}; ' \
      -e '$o .= $C; $C = ", "; (print "$o\n"), $o="    " if length $o > 70;' \
      -e '$o .= sprintf "{%d,%d}", $r-128,$c-64; } print "$o\n"; }'

 */

static const struct { unsigned char r, c; } cp949_backward[] = {
    {34,110}, {34,116}, {33,151}, {33,103}, {40,99}, {33,105}, {34,167}, 
    {33,134}, {33,126}, {41,183}, {41,184}, {34,101}, {34,146}, {33,100}, 
    {34,108}, {41,182}, {40,108}, {40,185}, {40,182}, {40,186}, {34,111}, 
    {40,97}, {40,98}, {33,127}, {40,106}, {40,109}, {41,108}, {41,97}, 
    {41,99}, {33,128}, {41,106}, {41,109}, {41,98}, {40,100}, {41,100}, 
    {41,101}, {40,102}, {41,102}, {41,103}, {40,104}, {41,104}, {40,105}, 
    {41,105}, {41,112}, {40,111}, {41,111}, {40,107}, {41,107}, {40,110}, 
    {41,110}, {34,103}, {34,112}, {34,104}, {34,107}, {34,106}, {34,109}, 
    {34,105}, {37,129}, {37,130}, {37,131}, {37,132}, {37,133}, {37,134}, 
    {37,135}, {37,136}, {37,137}, {37,138}, {37,139}, {37,140}, {37,141}, 
    {37,142}, {37,143}, {37,144}, {37,145}, {37,146}, {37,147}, {37,148}, 
    {37,149}, {37,150}, {37,151}, {37,152}, {37,161}, {37,162}, {37,163}, 
    {37,164}, {37,165}, {37,166}, {37,167}, {37,168}, {37,169}, {37,170}, 
    {37,171}, {37,172}, {37,173}, {37,174}, {37,175}, {37,176}, {37,177}, 
    {37,178}, {37,179}, {37,180}, {37,181}, {37,182}, {37,183}, {37,184}, 
    {44,103}, {44,97}, {44,98}, {44,99}, {44,100}, {44,101}, {44,102}, 
    {44,104}, {44,105}, {44,106}, {44,107}, {44,108}, {44,109}, {44,110}, 
    {44,111}, {44,112}, {44,113}, {44,114}, {44,115}, {44,116}, {44,117}, 
    {44,118}, {44,119}, {44,120}, {44,121}, {44,122}, {44,123}, {44,124}, 
    {44,125}, {44,126}, {44,127}, {44,128}, {44,129}, {44,145}, {44,146}, 
    {44,147}, {44,148}, {44,149}, {44,150}, {44,152}, {44,153}, {44,154}, 
    {44,155}, {44,156}, {44,157}, {44,158}, {44,159}, {44,160}, {44,161}, 
    {44,162}, {44,163}, {44,164}, {44,165}, {44,166}, {44,167}, {44,168}, 
    {44,169}, {44,170}, {44,171}, {44,172}, {44,173}, {44,174}, {44,175}, 
    {44,176}, {44,177}, {44,151}, {33,106}, {33,110}, {33,111}, {33,112}, 
    {33,113}, {34,147}, {34,148}, {33,101}, {33,102}, {34,118}, {33,135}, 
    {33,136}, {33,152}, {41,185}, {41,186}, {41,187}, {41,188}, {41,189}, 
    {41,190}, {34,166}, {33,137}, {34,117}, {39,100}, {34,160}, {34,165}, 
    {34,162}, {39,153}, {33,138}, {40,183}, {40,184}, {40,187}, {40,188}, 
    {40,189}, {40,190}, {37,112}, {37,113}, {37,114}, {37,115}, {37,116}, 
    {37,117}, {37,118}, {37,119}, {37,120}, {37,121}, {37,97}, {37,98}, 
    {37,99}, {37,100}, {37,101}, {37,102}, {37,103}, {37,104}, {37,105}, 
    {37,106}, {33,167}, {33,168}, {33,166}, {33,169}, {33,170}, {34,149}, 
    {34,152}, {34,150}, {34,153}, {34,151}, {34,97}, {34,98}, {34,99}, 
    {33,147}, {34,100}, {33,148}, {33,180}, {33,181}, {34,115}, {34,114}, 
    {33,174}, {33,176}, {33,132}, {33,144}, {33,107}, {33,188}, {33,189}, 
    {33,187}, {33,186}, {33,178}, {33,179}, {34,113}, {33,133}, {33,177}, 
    {33,109}, {33,175}, {33,150}, {33,129}, {33,149}, {33,130}, {33,131}, 
    {33,172}, {33,173}, {33,184}, {33,185}, {33,182}, {33,183}, {34,129}, 
    {33,145}, {33,146}, {40,167}, {40,168}, {40,169}, {40,170}, {40,171}, 
    {40,172}, {40,173}, {40,174}, {40,175}, {40,176}, {40,177}, {40,178}, 
    {40,179}, {40,180}, {40,181}, {41,167}, {41,168}, {41,169}, {41,170}, 
    {41,171}, {41,172}, {41,173}, {41,174}, {41,175}, {41,176}, {41,177}, 
    {41,178}, {41,179}, {41,180}, {41,181}, {41,141}, {41,142}, {41,143}, 
    {41,144}, {41,145}, {41,146}, {41,147}, {41,148}, {41,149}, {41,150}, 
    {41,151}, {41,152}, {41,153}, {41,154}, {41,155}, {41,156}, {41,157}, 
    {41,158}, {41,159}, {41,160}, {41,161}, {41,162}, {41,163}, {41,164}, 
    {41,165}, {41,166}, {40,141}, {40,142}, {40,143}, {40,144}, {40,145}, 
    {40,146}, {40,147}, {40,148}, {40,149}, {40,150}, {40,151}, {40,152}, 
    {40,153}, {40,154}, {40,155}, {40,156}, {40,157}, {40,158}, {40,159}, 
    {40,160}, {40,161}, {40,162}, {40,163}, {40,164}, {40,165}, {40,166}, 
    {38,97}, {38,108}, {38,98}, {38,109}, {38,99}, {38,136}, {38,135}, 
    {38,110}, {38,100}, {38,130}, {38,129}, {38,111}, {38,102}, {38,134}, 
    {38,133}, {38,113}, {38,101}, {38,132}, {38,131}, {38,112}, {38,103}, 
    {38,124}, {38,137}, {38,138}, {38,119}, {38,139}, {38,140}, {38,114}, 
    {38,105}, {38,126}, {38,141}, {38,142}, {38,121}, {38,143}, {38,144}, 
    {38,116}, {38,104}, {38,145}, {38,146}, {38,120}, {38,125}, {38,147}, 
    {38,148}, {38,115}, {38,106}, {38,149}, {38,150}, {38,122}, {38,127}, 
    {38,151}, {38,152}, {38,117}, {38,107}, {38,153}, {38,154}, {38,123}, 
    {38,155}, {38,156}, {38,128}, {38,157}, {38,158}, {38,159}, {38,160}, 
    {38,161}, {38,162}, {38,163}, {38,164}, {38,118}, {34,134}, {33,161}, 
    {33,160}, {34,131}, {34,135}, {34,136}, {34,139}, {34,138}, {34,137}, 
    {34,140}, {33,163}, {33,162}, {34,122}, {34,121}, {33,165}, {33,164}, 
    {34,120}, {34,119}, {33,159}, {33,158}, {34,130}, {33,155}, {33,157}, 
    {33,156}, {34,132}, {34,133}, {33,154}, {33,153}, {34,143}, {34,142}, 
    {34,144}, {34,145}, {33,143}, {33,142}, {34,124}, {34,125}, {34,128}, 
    {34,123}, {34,126}, {34,127}, {34,141}, {34,155}, {34,156}, {34,157}, 
    {34,154}, {33,97}, {33,98}, {33,99}, {33,104}, {33,116}, {33,117}, 
    {33,118}, {33,119}, {33,120}, {33,121}, {33,122}, {33,123}, {33,124}, 
    {33,125}, {33,171}, {33,114}, {33,115}, {42,97}, {42,98}, {42,99}, 
    {42,100}, {42,101}, {42,102}, {42,103}, {42,104}, {42,105}, {42,106}, 
    {42,107}, {42,108}, {42,109}, {42,110}, {42,111}, {42,112}, {42,113}, 
    {42,114}, {42,115}, {42,116}, {42,117}, {42,118}, {42,119}, {42,120}, 
    {42,121}, {42,122}, {42,123}, {42,124}, {42,125}, {42,126}, {42,127}, 
    {42,128}, {42,129}, {42,130}, {42,131}, {42,132}, {42,133}, {42,134}, 
    {42,135}, {42,136}, {42,137}, {42,138}, {42,139}, {42,140}, {42,141}, 
    {42,142}, {42,143}, {42,144}, {42,145}, {42,146}, {42,147}, {42,148}, 
    {42,149}, {42,150}, {42,151}, {42,152}, {42,153}, {42,154}, {42,155}, 
    {42,156}, {42,157}, {42,158}, {42,159}, {42,160}, {42,161}, {42,162}, 
    {42,163}, {42,164}, {42,165}, {42,166}, {42,167}, {42,168}, {42,169}, 
    {42,170}, {42,171}, {42,172}, {42,173}, {42,174}, {42,175}, {42,176}, 
    {42,177}, {42,178}, {42,179}, {43,97}, {43,98}, {43,99}, {43,100}, 
    {43,101}, {43,102}, {43,103}, {43,104}, {43,105}, {43,106}, {43,107}, 
    {43,108}, {43,109}, {43,110}, {43,111}, {43,112}, {43,113}, {43,114}, 
    {43,115}, {43,116}, {43,117}, {43,118}, {43,119}, {43,120}, {43,121}, 
    {43,122}, {43,123}, {43,124}, {43,125}, {43,126}, {43,127}, {43,128}, 
    {43,129}, {43,130}, {43,131}, {43,132}, {43,133}, {43,134}, {43,135}, 
    {43,136}, {43,137}, {43,138}, {43,139}, {43,140}, {43,141}, {43,142}, 
    {43,143}, {43,144}, {43,145}, {43,146}, {43,147}, {43,148}, {43,149}, 
    {43,150}, {43,151}, {43,152}, {43,153}, {43,154}, {43,155}, {43,156}, 
    {43,157}, {43,158}, {43,159}, {43,160}, {43,161}, {43,162}, {43,163}, 
    {43,164}, {43,165}, {43,166}, {43,167}, {43,168}, {43,169}, {43,170}, 
    {43,171}, {43,172}, {43,173}, {43,174}, {43,175}, {43,176}, {43,177}, 
    {43,178}, {43,179}, {43,180}, {43,181}, {43,182}, {36,97}, {36,98}, 
    {36,99}, {36,100}, {36,101}, {36,102}, {36,103}, {36,104}, {36,105}, 
    {36,106}, {36,107}, {36,108}, {36,109}, {36,110}, {36,111}, {36,112}, 
    {36,113}, {36,114}, {36,115}, {36,116}, {36,117}, {36,118}, {36,119}, 
    {36,120}, {36,121}, {36,122}, {36,123}, {36,124}, {36,125}, {36,126}, 
    {36,127}, {36,128}, {36,129}, {36,130}, {36,131}, {36,132}, {36,133}, 
    {36,134}, {36,135}, {36,136}, {36,137}, {36,138}, {36,139}, {36,140}, 
    {36,141}, {36,142}, {36,143}, {36,144}, {36,145}, {36,146}, {36,147}, 
    {36,148}, {36,149}, {36,150}, {36,151}, {36,152}, {36,153}, {36,154}, 
    {36,155}, {36,156}, {36,157}, {36,158}, {36,159}, {36,160}, {36,161}, 
    {36,162}, {36,163}, {36,164}, {36,165}, {36,166}, {36,167}, {36,168}, 
    {36,169}, {36,170}, {36,171}, {36,172}, {36,173}, {36,174}, {36,175}, 
    {36,176}, {36,177}, {36,178}, {36,179}, {36,180}, {36,181}, {36,182}, 
    {36,183}, {36,184}, {36,185}, {36,186}, {36,187}, {36,188}, {36,189}, 
    {36,190}, {41,113}, {41,114}, {41,115}, {41,116}, {41,117}, {41,118}, 
    {41,119}, {41,120}, {41,121}, {41,122}, {41,123}, {41,124}, {41,125}, 
    {41,126}, {41,127}, {41,128}, {41,129}, {41,130}, {41,131}, {41,132}, 
    {41,133}, {41,134}, {41,135}, {41,136}, {41,137}, {41,138}, {41,139}, 
    {41,140}, {34,159}, {40,113}, {40,114}, {40,115}, {40,116}, {40,117}, 
    {40,118}, {40,119}, {40,120}, {40,121}, {40,122}, {40,123}, {40,124}, 
    {40,125}, {40,126}, {40,127}, {40,128}, {40,129}, {40,130}, {40,131}, 
    {40,132}, {40,133}, {40,134}, {40,135}, {40,136}, {40,137}, {40,138}, 
    {40,139}, {40,140}, {34,158}, {39,137}, {39,138}, {39,139}, {39,140}, 
    {39,141}, {39,122}, {39,123}, {39,156}, {39,157}, {39,158}, {39,118}, 
    {39,119}, {39,120}, {39,148}, {39,149}, {39,150}, {39,151}, {39,152}, 
    {39,97}, {39,98}, {39,99}, {39,101}, {39,107}, {39,108}, {39,109}, 
    {39,110}, {39,111}, {39,112}, {39,113}, {39,114}, {39,115}, {39,116}, 
    {39,103}, {39,104}, {39,105}, {39,106}, {39,125}, {39,126}, {39,165}, 
    {39,166}, {39,167}, {39,168}, {39,161}, {39,162}, {39,163}, {39,127}, 
    {39,128}, {39,129}, {39,130}, {39,131}, {39,132}, {39,133}, {39,134}, 
    {39,135}, {39,136}, {39,142}, {39,143}, {39,144}, {39,145}, {39,146}, 
    {39,147}, {39,154}, {39,155}, {34,163}, {39,172}, {39,102}, {39,160}, 
    {39,175}, {34,161}, {39,124}, {39,173}, {39,117}, {39,121}, {39,170}, 
    {39,171}, {39,159}, {34,164}, {39,164}, {39,174}, {39,169}, {108,169}, 
    {111,139}, {118,146}, {88,114}, {109,155}, {95,114}, {95,126}, {121,123}, 
    {92,180}, {117,164}, {115,102}, {93,160}, {97,102}, {78,184}, {92,112}, 
    {99,106}, {113,169}, {77,186}, {124,111}, {83,97}, {113,107}, {103,145}, 
    {82,108}, {78,185}, {113,189}, {94,127}, {123,122}, {121,121}, {78,146}, 
    {99,107}, {107,160}, {78,186}, {75,183}, {101,101}, {74,161}, {84,140}, 
    {106,161}, {92,163}, {95,109}, {75,171}, {85,111}, {86,181}, {101,184}, 
    {94,128}, {108,99}, {105,141}, {106,103}, {105,182}, {123,123}, {103,169}, 
    {111,140}, {80,166}, {94,129}, {100,108}, {88,140}, {121,177}, {78,159}, 
    {122,100}, {102,114}, {122,187}, {122,125}, {76,136}, {111,141}, {85,149}, 
    {83,98}, {108,145}, {100,103}, {108,146}, {118,113}, {78,187}, {80,145}, 
    {75,127}, {109,100}, {109,104}, {94,130}, {118,162}, {109,156}, {92,181}, 
    {96,121}, {84,142}, {116,117}, {83,155}, {86,117}, {108,100}, {100,166}, 
    {113,170}, {75,172}, {75,128}, {108,178}, {80,170}, {121,178}, {108,101}, 
    {80,159}, {103,170}, {80,171}, {92,145}, {91,169}, {125,140}, {91,151}, 
    {90,161}, {86,118}, {99,159}, {94,131}, {94,132}, {74,97}, {110,172}, 
    {83,99}, {110,119}, {120,143}, {106,136}, {110,120}, {113,108}, {113,101}, 
    {105,142}, {121,124}, {101,185}, {108,170}, {93,150}, {109,130}, {120,101}, 
    {101,122}, {91,152}, {74,98}, {81,141}, {110,173}, {108,171}, {94,133}, 
    {99,160}, {74,137}, {114,169}, {85,142}, {118,118}, {78,130}, {86,135}, 
    {99,116}, {113,109}, {106,162}, {87,130}, {115,103}, {77,170}, {107,174}, 
    {89,114}, {125,101}, {118,149}, {85,162}, {120,117}, {76,181}, {117,117}, 
    {100,109}, {103,171}, {113,149}, {112,123}, {105,117}, {76,137}, {122,149}, 
    {97,148}, {87,150}, {92,129}, {94,134}, {122,175}, {99,161}, {97,179}, 
    {92,182}, {78,188}, {91,132}, {120,177}, {92,164}, {101,175}, {92,113}, 
    {85,150}, {115,154}, {75,129}, {91,131}, {89,186}, {83,174}, {122,120}, 
    {125,102}, {107,175}, {116,102}, {76,138}, {115,104}, {115,155}, {91,103}, 
    {118,119}, {79,166}, {112,178}, {75,154}, {103,146}, {87,131}, {118,176}, 
    {104,158}, {101,102}, {101,167}, {74,99}, {76,103}, {106,137}, {120,118}, 
    {122,101}, {113,110}, {111,142}, {75,173}, {118,112}, {111,143}, {105,143}, 
    {119,158}, {78,147}, {92,183}, {91,104}, {75,184}, {95,97}, {93,161}, 
    {117,138}, {105,118}, {103,172}, {110,174}, {115,176}, {95,127}, {76,139}, 
    {80,129}, {116,146}, {96,122}, {95,128}, {78,160}, {92,146}, {125,170}, 
    {86,182}, {106,138}, {104,169}, {99,108}, {115,144}, {74,100}, {91,184}, 
    {94,135}, {107,176}, {113,150}, {101,162}, {76,140}, {75,187}, {106,163}, 
    {95,129}, {86,173}, {105,144}, {110,121}, {85,163}, {81,147}, {101,176}, 
    {104,116}, {107,131}, {106,106}, {122,188}, {117,182}, {112,124}, {125,148}, 
    {96,123}, {78,131}, {80,122}, {119,122}, {88,179}, {119,141}, {100,110}, 
    {84,159}, {80,167}, {108,189}, {82,110}, {110,175}, {85,151}, {106,164}, 
    {120,98}, {77,171}, {87,127}, {123,113}, {77,172}, {92,114}, {80,172}, 
    {78,189}, {110,176}, {76,130}, {80,173}, {101,183}, {115,188}, {110,98}, 
    {89,115}, {88,180}, {105,119}, {78,110}, {89,98}, {88,177}, {84,143}, 
    {101,103}, {85,146}, {86,105}, {116,98}, {113,151}, {85,152}, {112,125}, 
    {87,144}, {84,144}, {87,143}, {107,170}, {125,171}, {91,173}, {124,133}, 
    {75,130}, {125,149}, {116,136}, {104,170}, {117,179}, {121,158}, {83,175}, 
    {108,147}, {93,130}, {111,119}, {103,148}, {74,138}, {89,187}, {122,189}, 
    {86,106}, {116,184}, {119,183}, {92,108}, {87,151}, {95,98}, {78,126}, 
    {83,176}, {112,100}, {97,172}, {79,167}, {115,139}, {109,105}, {74,126}, 
    {116,175}, {118,142}, {94,187}, {80,123}, {85,119}, {110,177}, {116,104}, 
    {92,184}, {75,103}, {90,142}, {96,166}, {109,101}, {110,178}, {92,185}, 
    {121,156}, {115,156}, {120,178}, {116,185}, {124,177}, {80,124}, {91,185}, 
    {87,113}, {75,188}, {112,101}, {75,189}, {85,180}, {77,173}, {74,101}, 
    {86,107}, {80,130}, {112,126}, {82,125}, {76,100}, {122,118}, {76,141}, 
    {90,186}, {118,143}, {105,120}, {88,181}, {76,142}, {87,141}, {84,145}, 
    {105,173}, {74,171}, {89,162}, {125,114}, {99,109}, {86,140}, {89,116}, 
    {97,103}, {110,147}, {80,131}, {125,115}, {85,164}, {79,168}, {109,131}, 
    {80,114}, {78,190}, {90,104}, {120,144}, {125,150}, {120,145}, {120,146}, 
    {92,147}, {93,162}, {123,185}, {93,129}, {99,117}, {109,157}, {78,132}, 
    {75,97}, {93,163}, {124,157}, {121,111}, {82,187}, {79,97}, {100,104}, 
    {116,118}, {108,190}, {99,110}, {103,173}, {125,129}, {90,162}, {88,115}, 
    {93,164}, {112,175}, {118,177}, {122,176}, {81,181}, {90,143}, {92,148}, 
    {92,102}, {111,127}, {78,143}, {96,153}, {89,150}, {108,148}, {106,139}, 
    {74,127}, {85,112}, {79,169}, {113,173}, {76,143}, {100,184}, {100,173}, 
    {87,152}, {125,103}, {106,107}, {118,114}, {79,176}, {121,125}, {102,180}, 
    {75,155}, {115,145}, {105,145}, {115,105}, {80,160}, {105,146}, {90,163}, 
    {98,146}, {118,98}, {97,180}, {90,164}, {103,149}, {117,127}, {79,98}, 
    {77,111}, {79,99}, {77,112}, {113,190}, {80,99}, {97,111}, {120,99}, 
    {74,102}, {119,123}, {114,170}, {94,136}, {105,147}, {94,137}, {125,158}, 
    {74,128}, {121,170}, {81,142}, {110,148}, {84,146}, {89,99}, {125,104}, 
    {87,153}, {119,142}, {122,126}, {79,150}, {87,176}, {107,161}, {120,133}, 
    {92,186}, {93,131}, {121,159}, {103,175}, {125,165}, {118,99}, {89,188}, 
    {125,105}, {103,174}, {85,165}, {111,144}, {77,113}, {119,98}, {113,114}, 
    {113,113}, {77,114}, {90,107}, {74,103}, {99,162}, {123,124}, {89,100}, 
    {110,122}, {120,147}, {123,186}, {79,100}, {92,187}, {118,163}, {109,106}, 
    {114,97}, {78,161}, {122,102}, {121,160}, {108,150}, {100,174}, {121,97}, 
    {123,175}, {121,171}, {110,99}, {106,108}, {74,104}, {116,186}, {77,150}, 
    {124,182}, {116,137}, {120,148}, {120,102}, {94,138}, {114,134}, {87,154}, 
    {83,144}, {88,133}, {106,166}, {115,157}, {100,154}, {118,164}, {118,178}, 
    {95,130}, {89,189}, {76,182}, {83,122}, {100,111}, {121,161}, {112,102}, 
    {75,147}, {96,124}, {116,138}, {84,186}, {125,106}, {121,162}, {116,119}, 
    {125,130}, {124,112}, {125,172}, {74,162}, {125,125}, {106,167}, {95,131}, 
    {81,146}, {78,162}, {83,100}, {125,107}, {95,160}, {114,135}, {103,176}, 
    {80,174}, {115,106}, {94,139}, {118,120}, {97,181}, {113,115}, {119,99}, 
    {74,105}, {79,101}, {95,132}, {97,112}, {112,127}, {118,100}, {99,118}, 
    {122,134}, {80,175}, {125,173}, {93,132}, {124,183}, {102,127}, {94,109}, 
    {122,127}, {101,177}, {109,132}, {82,101}, {125,174}, {117,118}, {97,182}, 
    {94,140}, {124,158}, {108,151}, {77,157}, {86,119}, {77,115}, {120,149}, 
    {101,152}, {79,170}, {79,144}, {106,140}, {106,110}, {106,109}, {83,177}, 
    {83,101}, {119,143}, {110,100}, {80,100}, {114,98}, {80,176}, {114,99}, 
    {119,184}, {80,115}, {91,105}, {83,123}, {74,172}, {113,102}, {75,149}, 
    {119,167}, {77,158}, {119,100}, {120,128}, {83,157}, {76,144}, {79,102}, 
    {118,179}, {97,183}, {83,156}, {122,190}, {122,103}, {107,153}, {79,103}, 
    {106,111}, {100,175}, {105,121}, {113,152}, {88,152}, {96,178}, {102,116}, 
    {92,188}, {115,177}, {99,144}, {114,187}, {91,134}, {80,177}, {80,178}, 
    {79,156}, {83,145}, {76,113}, {119,152}, {75,104}, {107,124}, {100,126}, 
    {116,156}, {92,130}, {112,103}, {102,128}, {74,173}, {104,171}, {101,168}, 
    {92,131}, {109,158}, {83,178}, {76,183}, {78,148}, {103,107}, {75,131}, 
    {97,113}, {119,114}, {83,179}, {83,146}, {117,128}, {95,157}, {110,179}, 
    {103,177}, {125,116}, {114,136}, {115,146}, {110,180}, {98,147}, {76,145}, 
    {95,170}, {105,122}, {89,151}, {117,141}, {113,178}, {122,135}, {89,184}, 
    {84,130}, {118,165}, {93,133}, {103,178}, {109,159}, {74,139}, {91,186}, 
    {104,117}, {83,102}, {125,117}, {121,137}, {100,162}, {123,125}, {87,100}, 
    {78,133}, {78,149}, {86,166}, {101,125}, {94,141}, {108,179}, {109,160}, 
    {108,172}, {123,126}, {95,171}, {97,184}, {121,126}, {80,179}, {96,106}, 
    {104,162}, {98,148}, {82,189}, {101,104}, {89,147}, {83,158}, {116,120}, 
    {119,124}, {92,189}, {104,172}, {100,167}, {99,183}, {108,104}, {122,177}, 
    {101,178}, {80,180}, {82,111}, {92,165}, {80,101}, {113,116}, {124,113}, 
    {76,184}, {93,134}, {122,145}, {119,159}, {122,104}, {110,181}, {94,142}, 
    {103,179}, {119,108}, {107,132}, {109,161}, {96,107}, {93,135}, {82,115}, 
    {82,127}, {74,140}, {123,127}, {101,189}, {93,165}, {88,141}, {108,180}, 
    {80,181}, {104,173}, {80,146}, {89,152}, {118,166}, {91,106}, {119,160}, 
    {88,153}, {116,99}, {116,157}, {111,145}, {89,117}, {109,107}, {99,119}, 
    {110,123}, {77,116}, {96,179}, {106,141}, {108,181}, {104,174}, {75,105}, 
    {113,111}, {74,141}, {108,105}, {114,171}, {125,175}, {121,179}, {102,129}, 
    {108,152}, {109,108}, {106,142}, {104,159}, {94,143}, {82,102}, {103,180}, 
    {81,150}, {102,130}, {99,163}, {100,112}, {88,116}, {118,101}, {115,158}, 
    {87,101}, {119,168}, {104,134}, {123,166}, {93,166}, {92,190}, {88,154}, 
    {90,108}, {106,112}, {99,120}, {74,106}, {97,185}, {106,113}, {114,172}, 
    {122,174}, {110,149}, {121,180}, {82,172}, {123,187}, {125,176}, {96,125}, 
    {78,163}, {120,134}, {94,110}, {95,133}, {101,126}, {109,109}, {122,170}, 
    {77,174}, {109,102}, {109,110}, {112,173}, {93,97}, {109,111}, {124,184}, 
    {88,171}, {76,185}, {77,117}, {122,105}, {97,157}, {98,149}, {109,143}, 
    {93,98}, {121,138}, {106,168}, {101,173}, {83,171}, {105,148}, {97,186}, 
    {100,140}, {97,164}, {104,135}, {78,155}, {92,149}, {119,117}, {124,179}, 
    {112,179}, {78,111}, {113,117}, {111,146}, {104,136}, {107,177}, {75,148}, 
    {96,126}, {99,184}, {106,169}, {124,114}, {96,180}, {79,160}, {110,101}, 
    {122,106}, {102,131}, {97,114}, {74,107}, {99,164}, {105,123}, {98,150}, 
    {115,178}, {110,150}, {106,114}, {80,182}, {108,153}, {90,139}, {79,104}, 
    {93,99}, {88,155}, {121,142}, {105,149}, {99,145}, {82,124}, {88,108}, 
    {115,140}, {77,187}, {118,150}, {103,181}, {104,175}, {99,185}, {82,123}, 
    {115,179}, {99,187}, {94,144}, {78,112}, {86,183}, {113,153}, {117,129}, 
    {92,132}, {117,123}, {94,145}, {92,166}, {94,146}, {109,162}, {110,182}, 
    {106,143}, {112,174}, {99,188}, {83,159}, {83,180}, {97,115}, {97,116}, 
    {116,147}, {95,134}, {105,150}, {91,107}, {118,102}, {99,121}, {107,133}, 
    {116,105}, {77,118}, {82,185}, {90,109}, {82,163}, {79,145}, {75,156}, 
    {76,186}, {79,157}, {104,105}, {99,123}, {99,122}, {96,154}, {110,183}, 
    {92,115}, {83,181}, {87,102}, {118,117}, {87,155}, {97,149}, {84,170}, 
    {95,99}, {125,159}, {80,183}, {109,148}, {75,106}, {100,155}, {97,187}, 
    {75,98}, {83,160}, {100,127}, {123,128}, {90,126}, {100,141}, {86,121}, 
    {111,128}, {97,188}, {118,121}, {95,135}, {100,113}, {92,167}, {92,168}, 
    {122,150}, {83,182}, {113,154}, {122,178}, {98,189}, {85,143}, {80,184}, 
    {77,159}, {117,139}, {100,176}, {75,107}, {87,132}, {98,190}, {93,154}, 
    {90,110}, {74,174}, {85,121}, {99,97}, {104,163}, {115,107}, {79,105}, 
    {83,183}, {84,177}, {78,164}, {104,178}, {101,181}, {103,110}, {86,122}, 
    {95,172}, {100,128}, {104,164}, {88,117}, {100,156}, {116,121}, {113,118}, 
    {98,158}, {97,117}, {77,175}, {113,103}, {78,165}, {75,157}, {89,163}, 
    {115,108}, {80,185}, {108,107}, {94,147}, {119,169}, {121,181}, {97,158}, 
    {75,174}, {99,124}, {120,150}, {91,174}, {125,177}, {119,118}, {116,158}, 
    {114,173}, {91,153}, {112,104}, {97,189}, {94,148}, {96,108}, {109,163}, 
    {83,161}, {95,136}, {89,118}, {125,108}, {111,147}, {100,129}, {120,171}, 
    {91,108}, {124,134}, {88,109}, {118,122}, {91,159}, {83,147}, {120,135}, 
    {74,142}, {120,129}, {82,116}, {92,116}, {122,121}, {74,143}, {124,115}, 
    {106,170}, {106,171}, {80,186}, {109,164}, {93,167}, {95,137}, {95,173}, 
    {110,124}, {111,129}, {76,146}, {93,100}, {95,138}, {83,184}, {113,104}, 
    {77,119}, {111,148}, {100,157}, {95,174}, {75,108}, {105,124}, {106,172}, 
    {95,139}, {121,127}, {86,111}, {85,134}, {79,106}, {78,105}, {86,184}, 
    {113,119}, {110,184}, {89,153}, {115,159}, {120,136}, {78,134}, {85,166}, 
    {116,166}, {102,133}, {111,149}, {75,175}, {124,159}, {92,103}, {86,167}, 
    {120,137}, {99,146}, {99,125}, {79,161}, {112,128}, {108,154}, {93,151}, 
    {123,176}, {108,108}, {112,105}, {122,151}, {123,129}, {82,128}, {101,112}, 
    {109,165}, {75,109}, {121,112}, {119,101}, {75,110}, {90,111}, {88,118}, 
    {83,103}, {123,114}, {125,132}, {108,109}, {123,97}, {101,169}, {105,174}, 
    {115,180}, {120,179}, {112,129}, {94,111}, {120,112}, {115,160}, {103,111}, 
    {91,109}, {102,117}, {121,104}, {93,152}, {104,153}, {111,150}, {83,162}, 
    {98,159}, {124,160}, {87,136}, {125,109}, {95,175}, {76,147}, {83,185}, 
    {84,176}, {91,135}, {94,149}, {112,180}, {85,144}, {101,153}, {124,135}, 
    {92,150}, {98,160}, {90,112}, {115,99}, {83,172}, {116,139}, {125,133}, 
    {99,189}, {121,113}, {80,187}, {108,155}, {117,124}, {114,100}, {88,142}, 
    {88,143}, {117,183}, {118,161}, {82,119}, {123,172}, {93,136}, {100,168}, 
    {82,129}, {120,151}, {86,123}, {94,150}, {119,125}, {108,110}, {80,161}, 
    {96,181}, {106,115}, {78,150}, {76,101}, {108,182}, {98,161}, {99,126}, 
    {124,136}, {77,176}, {121,182}, {95,176}, {101,127}, {78,127}, {124,161}, 
    {109,112}, {125,145}, {118,123}, {121,143}, {107,154}, {74,129}, {82,120}, 
    {77,177}, {99,147}, {125,166}, {102,173}, {99,186}, {112,106}, {121,144}, 
    {124,162}, {120,103}, {97,165}, {110,185}, {103,182}, {106,173}, {124,116}, 
    {117,130}, {87,156}, {112,181}, {93,168}, {83,173}, {117,188}, {90,127}, 
    {76,187}, {83,186}, {116,100}, {111,151}, {84,131}, {123,163}, {123,173}, 
    {96,109}, {106,174}, {123,115}, {100,130}, {118,167}, {82,157}, {95,140}, 
    {124,137}, {101,105}, {96,182}, {118,115}, {97,190}, {75,176}, {106,175}, 
    {106,176}, {90,128}, {120,116}, {107,178}, {100,131}, {105,151}, {100,177}, 
    {74,175}, {78,151}, {124,138}, {115,161}, {75,132}, {99,165}, {75,133}, 
    {106,116}, {105,125}, {87,137}, {107,155}, {109,113}, {76,131}, {119,126}, 
    {124,139}, {120,180}, {89,119}, {115,147}, {115,148}, {119,164}, {119,145}, 
    {88,119}, {78,113}, {74,130}, {123,116}, {75,134}, {112,182}, {85,167}, 
    {106,144}, {76,148}, {75,111}, {116,106}, {105,111}, {117,131}, {105,152}, 
    {93,169}, {113,179}, {85,187}, {94,123}, {116,187}, {125,179}, {125,178}, 
    {119,102}, {93,137}, {84,147}, {76,104}, {90,129}, {76,149}, {89,164}, 
    {122,138}, {101,163}, {83,124}, {74,176}, {80,132}, {74,144}, {122,107}, 
    {107,171}, {103,184}, {89,165}, {81,151}, {115,100}, {84,187}, {124,163}, 
    {122,152}, {115,149}, {79,107}, {107,179}, {85,188}, {83,148}, {77,188}, 
    {89,166}, {98,185}, {98,97}, {107,148}, {96,183}, {100,114}, {76,188}, 
    {123,164}, {116,107}, {80,125}, {74,177}, {111,120}, {87,128}, {110,186}, 
    {125,180}, {83,163}, {123,130}, {85,168}, {91,110}, {97,118}, {120,119}, 
    {96,127}, {123,131}, {93,170}, {98,98}, {110,102}, {118,168}, {118,181}, 
    {93,138}, {80,162}, {93,102}, {93,171}, {100,185}, {99,111}, {80,188}, 
    {116,188}, {76,124}, {119,170}, {101,164}, {95,177}, {119,161}, {121,183}, 
    {111,121}, {120,152}, {121,105}, {120,153}, {110,125}, {88,134}, {100,163}, 
    {117,142}, {93,153}, {89,167}, {82,121}, {85,131}, {90,165}, {90,144}, 
    {81,153}, {78,152}, {75,158}, {116,108}, {90,187}, {118,169}, {104,179}, 
    {79,108}, {112,176}, {116,189}, {91,136}, {78,128}, {99,148}, {81,143}, 
    {113,181}, {77,178}, {79,171}, {77,120}, {99,102}, {81,154}, {114,101}, 
    {114,102}, {100,142}, {83,187}, {113,105}, {114,137}, {111,152}, {102,137}, 
    {88,120}, {122,179}, {115,117}, {120,100}, {81,179}, {102,136}, {120,154}, 
    {92,169}, {94,151}, {75,159}, {79,172}, {116,159}, {81,180}, {82,122}, 
    {95,178}, {97,119}, {98,99}, {83,188}, {109,166}, {91,137}, {100,186}, 
    {79,158}, {78,144}, {85,147}, {115,181}, {119,110}, {111,136}, {77,179}, 
    {117,143}, {101,179}, {112,130}, {74,145}, {106,177}, {80,102}, {89,154}, 
    {112,107}, {107,167}, {101,128}, {124,117}, {100,132}, {76,105}, {125,134}, 
    {106,117}, {101,106}, {95,122}, {97,159}, {90,145}, {97,120}, {104,180}, 
    {83,189}, {98,100}, {114,138}, {90,166}, {119,115}, {125,141}, {115,118}, 
    {110,151}, {117,132}, {88,100}, {114,103}, {89,120}, {89,121}, {111,137}, 
    {86,142}, {119,139}, {95,110}, {104,181}, {82,117}, {83,149}, {116,140}, 
    {90,188}, {89,168}, {119,171}, {117,137}, {115,124}, {90,146}, {83,117}, 
    {104,118}, {86,143}, {116,122}, {119,137}, {76,106}, {112,131}, {76,150}, 
    {80,147}, {83,125}, {91,187}, {75,160}, {83,164}, {118,183}, {85,122}, 
    {115,141}, {75,161}, {107,180}, {116,109}, {124,106}, {119,172}, {104,182}, 
    {90,167}, {119,140}, {101,129}, {96,174}, {85,189}, {78,166}, {124,107}, 
    {85,123}, {114,104}, {98,101}, {77,121}, {106,178}, {75,135}, {77,180}, 
    {91,111}, {111,153}, {77,122}, {124,185}, {95,179}, {78,167}, {90,130}, 
    {79,109}, {103,185}, {120,104}, {115,162}, {74,178}, {95,100}, {84,132}, 
    {76,151}, {101,130}, {77,123}, {111,154}, {110,152}, {93,103}, {98,102}, 
    {96,128}, {86,112}, {120,138}, {124,186}, {89,190}, {94,112}, {93,172}, 
    {90,168}, {84,160}, {86,185}, {77,151}, {94,152}, {114,184}, {100,150}, 
    {80,133}, {116,110}, {93,104}, {109,133}, {115,150}, {94,153}, {99,166}, 
    {83,104}, {91,112}, {101,154}, {99,127}, {91,113}, {85,169}, {96,129}, 
    {111,155}, {112,169}, {87,114}, {80,189}, {89,169}, {80,190}, {108,173}, 
    {83,105}, {114,105}, {112,132}, {98,162}, {105,175}, {121,145}, {105,153}, 
    {104,154}, {90,131}, {90,132}, {84,133}, {103,186}, {77,160}, {99,112}, 
    {91,114}, {123,132}, {115,163}, {89,101}, {123,167}, {93,139}, {80,148}, 
    {102,118}, {96,110}, {125,154}, {92,117}, {96,184}, {103,113}, {117,176}, 
    {88,156}, {109,134}, {97,121}, {99,128}, {121,128}, {105,176}, {89,155}, 
    {115,164}, {92,118}, {100,169}, {112,133}, {99,129}, {124,140}, {124,141}, 
    {114,139}, {114,140}, {100,143}, {113,155}, {122,153}, {113,120}, {125,181}, 
    {96,185}, {103,187}, {124,119}, {124,164}, {123,133}, {99,167}, {88,121}, 
    {118,184}, {92,133}, {76,152}, {96,111}, {116,167}, {111,156}, {79,188}, 
    {111,157}, {114,106}, {125,126}, {74,108}, {125,123}, {125,135}, {103,114}, 
    {106,145}, {95,180}, {81,172}, {100,158}, {101,131}, {89,102}, {77,124}, 
    {115,165}, {109,149}, {89,122}, {109,167}, {123,117}, {120,172}, {96,167}, 
    {76,153}, {84,134}, {103,101}, {85,181}, {83,126}, {124,187}, {100,178}, 
    {95,181}, {104,184}, {120,173}, {78,135}, {125,182}, {104,152}, {77,152}, 
    {103,150}, {76,154}, {74,163}, {95,182}, {112,135}, {112,134}, {88,122}, 
    {113,180}, {116,176}, {117,140}, {124,165}, {106,133}, {106,179}, {93,155}, 
    {92,151}, {94,189}, {114,185}, {85,135}, {88,144}, {112,136}, {81,97}, 
    {81,98}, {89,148}, {86,168}, {89,138}, {90,113}, {88,135}, {92,162}, 
    {115,142}, {117,180}, {113,121}, {90,147}, {118,170}, {79,181}, {125,110}, 
    {74,146}, {95,116}, {87,157}, {122,122}, {110,103}, {117,125}, {120,181}, 
    {109,168}, {84,161}, {81,99}, {97,150}, {121,184}, {91,138}, {75,185}, 
    {84,148}, {89,156}, {110,126}, {119,173}, {82,174}, {97,166}, {119,185}, 
    {93,173}, {104,155}, {91,115}, {81,183}, {96,112}, {84,162}, {118,151}, 
    {87,185}, {88,157}, {77,189}, {114,107}, {77,125}, {120,130}, {114,108}, 
    {74,109}, {74,110}, {79,110}, {99,130}, {92,119}, {91,154}, {89,123}, 
    {74,179}, {118,147}, {102,184}, {106,181}, {106,182}, {118,185}, {79,111}, 
    {74,147}, {74,111}, {82,112}, {113,122}, {87,115}, {99,131}, {115,189}, 
    {94,154}, {94,155}, {111,158}, {98,163}, {110,187}, {95,183}, {87,138}, 
    {78,168}, {91,155}, {113,123}, {105,177}, {122,119}, {80,134}, {76,107}, 
    {110,104}, {75,186}, {121,185}, {76,189}, {83,190}, {100,144}, {114,174}, 
    {84,149}, {95,141}, {124,120}, {81,144}, {114,141}, {119,146}, {74,148}, 
    {85,153}, {88,158}, {77,153}, {110,105}, {118,124}, {76,155}, {112,137}, 
    {124,188}, {104,137}, {116,190}, {103,188}, {87,158}, {94,156}, {112,108}, 
    {76,190}, {77,161}, {97,122}, {91,175}, {90,114}, {81,101}, {92,120}, 
    {88,182}, {81,100}, {77,162}, {92,170}, {112,183}, {112,138}, {80,126}, 
    {93,156}, {84,150}, {83,150}, {109,144}, {77,97}, {95,117}, {95,184}, 
    {84,97}, {78,114}, {104,138}, {107,181}, {99,149}, {117,144}, {117,97}, 
    {89,103}, {101,107}, {102,139}, {117,177}, {101,133}, {121,99}, {96,155}, 
    {118,171}, {75,177}, {89,170}, {117,98}, {87,145}, {81,184}, {106,184}, 
    {106,185}, {90,115}, {111,159}, {113,175}, {101,182}, {110,127}, {98,164}, 
    {80,127}, {122,108}, {117,145}, {103,115}, {105,126}, {114,142}, {91,116}, 
    {124,142}, {93,174}, {103,116}, {87,116}, {119,116}, {77,126}, {90,169}, 
    {79,112}, {119,153}, {115,166}, {78,153}, {78,106}, {75,136}, {80,103}, 
    {112,139}, {80,135}, {100,133}, {91,160}, {85,154}, {87,103}, {110,128}, 
    {120,182}, {117,146}, {109,169}, {89,124}, {101,134}, {117,99}, {90,148}, 
    {98,103}, {123,188}, {113,156}, {74,180}, {104,186}, {78,169}, {105,184}, 
    {98,165}, {80,121}, {84,178}, {81,102}, {95,142}, {124,180}, {83,106}, 
    {76,108}, {111,160}, {101,165}, {80,149}, {91,188}, {124,166}, {75,190}, 
    {109,170}, {94,113}, {121,163}, {84,98}, {79,182}, {86,144}, {85,170}, 
    {113,174}, {122,139}, {101,97}, {85,113}, {79,173}, {109,171}, {85,114}, 
    {85,124}, {125,162}, {115,109}, {125,155}, {105,112}, {81,103}, {125,163}, 
    {78,115}, {125,164}, {122,142}, {74,112}, {119,103}, {79,113}, {102,98}, 
    {124,118}, {114,109}, {111,161}, {115,110}, {92,134}, {89,171}, {104,160}, 
    {97,104}, {85,182}, {79,189}, {94,157}, {89,145}, {100,170}, {114,143}, 
    {119,127}, {98,166}, {98,104}, {99,150}, {109,145}, {105,185}, {86,113}, 
    {94,114}, {96,168}, {83,107}, {107,156}, {95,111}, {74,131}, {110,188}, 
    {125,131}, {107,182}, {79,114}, {89,172}, {89,125}, {88,159}, {84,120}, 
    {107,126}, {93,175}, {93,176}, {93,177}, {93,178}, {89,126}, {123,134}, 
    {79,115}, {110,189}, {100,107}, {90,133}, {88,172}, {81,104}, {98,105}, 
    {94,124}, {103,117}, {91,176}, {111,162}, {113,176}, {79,116}, {91,177}, 
    {96,113}, {95,101}, {121,146}, {103,189}, {102,99}, {123,177}, {75,112}, 
    {114,110}, {77,167}, {104,156}, {103,151}, {119,128}, {80,163}, {90,97}, 
    {76,125}, {81,105}, {93,140}, {99,190}, {81,106}, {104,106}, {106,118}, 
    {121,186}, {102,140}, {118,152}, {84,135}, {89,139}, {89,146}, {83,139}, 
    {88,183}, {90,105}, {117,184}, {94,158}, {114,111}, {120,105}, {88,136}, 
    {110,129}, {121,129}, {93,179}, {106,186}, {118,125}, {97,123}, {77,127}, 
    {116,148}, {102,141}, {124,143}, {123,98}, {96,156}, {116,123}, {90,149}, 
    {121,114}, {123,178}, {91,182}, {94,159}, {91,178}, {120,156}, {119,174}, 
    {107,168}, {82,186}, {113,124}, {122,154}, {90,170}, {90,134}, {119,129}, 
    {103,118}, {101,135}, {86,108}, {92,135}, {97,105}, {98,106}, {85,102}, 
    {84,151}, {114,144}, {106,187}, {96,157}, {123,179}, {113,125}, {98,167}, 
    {125,151}, {78,136}, {106,119}, {124,128}, {125,167}, {119,175}, {87,117}, 
    {111,122}, {113,157}, {94,115}, {104,139}, {120,157}, {123,135}, {85,136}, 
    {87,159}, {93,105}, {105,113}, {122,109}, {118,153}, {122,180}, {120,106}, 
    {102,174}, {76,156}, {97,124}, {96,175}, {105,127}, {124,189}, {102,142}, 
    {97,151}, {102,143}, {116,177}, {100,179}, {100,187}, {121,164}, {111,163}, 
    {79,174}, {118,126}, {96,114}, {124,190}, {81,107}, {87,186}, {123,136}, 
    {98,151}, {84,99}, {112,184}, {87,104}, {97,167}, {83,127}, {111,164}, 
    {87,133}, {107,162}, {124,167}, {100,98}, {98,168}, {102,144}, {123,168}, 
    {116,168}, {101,180}, {116,124}, {116,149}, {95,118}, {124,121}, {110,130}, 
    {74,181}, {111,165}, {75,162}, {84,100}, {94,160}, {90,189}, {100,134}, 
    {104,126}, {96,158}, {118,116}, {106,146}, {121,187}, {96,130}, {74,164}, 
    {103,119}, {106,189}, {89,157}, {90,116}, {110,106}, {123,169}, {91,139}, 
    {90,117}, {113,126}, {83,108}, {123,137}, {95,143}, {83,128}, {99,151}, 
    {111,166}, {124,144}, {105,128}, {117,147}, {108,156}, {119,119}, {106,120}, 
    {81,185}, {92,136}, {106,121}, {113,158}, {87,118}, {79,117}, {89,104}, 
    {108,174}, {93,106}, {77,98}, {104,110}, {97,125}, {114,145}, {105,129}, 
    {82,188}, {91,117}, {115,167}, {88,190}, {124,145}, {109,114}, {116,111}, 
    {123,99}, {124,129}, {110,107}, {84,101}, {116,178}, {110,153}, {123,138}, 
    {77,163}, {88,123}, {101,155}, {120,183}, {118,148}, {87,105}, {75,137}, 
    {102,145}, {112,140}, {88,110}, {121,147}, {85,190}, {88,124}, {114,112}, 
    {98,107}, {115,168}, {111,130}, {109,172}, {103,120}, {90,190}, {76,126}, 
    {114,188}, {90,171}, {98,152}, {109,150}, {86,145}, {96,115}, {124,146}, 
    {107,136}, {83,129}, {112,141}, {79,183}, {109,146}, {84,152}, {92,137}, 
    {87,177}, {95,123}, {115,101}, {116,141}, {113,127}, {120,113}, {105,186}, 
    {123,139}, {74,149}, {121,148}, {119,138}, {86,136}, {124,168}, {115,125}, 
    {110,190}, {103,190}, {83,130}, {83,118}, {76,109}, {118,186}, {86,114}, 
    {82,152}, {103,152}, {99,101}, {103,121}, {112,109}, {123,140}, {107,97}, 
    {84,102}, {123,141}, {85,125}, {113,159}, {118,187}, {94,116}, {85,171}, 
    {101,136}, {123,100}, {84,121}, {94,161}, {100,99}, {87,119}, {120,174}, 
    {94,117}, {86,146}, {121,149}, {103,122}, {107,149}, {85,183}, {111,167}, 
    {97,126}, {122,110}, {86,169}, {86,174}, {103,123}, {108,139}, {85,115}, 
    {78,116}, {123,101}, {97,174}, {119,104}, {123,142}, {88,125}, {123,189}, 
    {124,169}, {79,118}, {109,135}, {110,108}, {76,157}, {118,103}, {102,186}, 
    {117,100}, {125,156}, {109,115}, {78,137}, {111,168}, {97,127}, {122,155}, 
    {75,163}, {119,105}, {123,102}, {92,121}, {113,128}, {109,136}, {111,131}, 
    {86,109}, {125,142}, {104,97}, {123,180}, {85,103}, {113,182}, {102,147}, 
    {76,158}, {120,114}, {92,171}, {125,118}, {101,170}, {113,160}, {91,140}, 
    {93,141}, {84,136}, {89,173}, {117,101}, {102,187}, {102,148}, {125,136}, 
    {86,97}, {125,127}, {124,147}, {111,97}, {103,124}, {81,174}, {102,149}, 
    {105,178}, {95,112}, {88,160}, {124,122}, {125,111}, {112,142}, {91,161}, 
    {101,137}, {109,116}, {96,131}, {99,152}, {105,187}, {106,104}, {125,119}, 
    {123,103}, {105,130}, {125,183}, {98,153}, {92,172}, {104,98}, {102,176}, 
    {125,184}, {125,185}, {118,127}, {103,103}, {102,151}, {84,179}, {84,137}, 
    {86,186}, {87,178}, {97,128}, {91,162}, {102,152}, {103,125}, {112,143}, 
    {115,126}, {98,108}, {117,119}, {96,176}, {125,120}, {99,168}, {84,103}, 
    {104,188}, {122,146}, {120,175}, {86,147}, {85,116}, {112,144}, {119,176}, 
    {110,115}, {106,122}, {106,147}, {109,137}, {93,107}, {101,108}, {125,97}, 
    {95,144}, {108,115}, {95,145}, {109,173}, {120,120}, {119,186}, {120,107}, 
    {116,160}, {84,122}, {100,115}, {105,154}, {94,118}, {89,127}, {89,128}, 
    {86,175}, {89,140}, {90,106}, {95,165}, {119,165}, {76,114}, {95,185}, 
    {87,160}, {84,123}, {125,186}, {76,115}, {91,179}, {95,146}, {78,138}, 
    {110,154}, {100,164}, {123,143}, {79,119}, {110,131}, {78,170}, {98,109}, 
    {87,161}, {122,181}, {85,137}, {120,108}, {103,153}, {115,169}, {88,173}, 
    {99,132}, {112,177}, {104,165}, {96,186}, {110,132}, {89,158}, {107,98}, 
    {107,99}, {124,130}, {106,123}, {104,107}, {94,162}, {109,175}, {104,99}, 
    {79,177}, {84,124}, {124,170}, {103,126}, {124,178}, {86,116}, {98,110}, 
    {83,119}, {122,140}, {122,156}, {109,117}, {97,163}, {104,108}, {104,157}, 
    {111,169}, {116,125}, {79,120}, {105,155}, {81,108}, {90,135}, {107,137}, 
    {104,140}, {94,119}, {86,124}, {83,165}, {122,157}, {90,150}, {74,113}, 
    {90,136}, {95,102}, {121,115}, {114,146}, {74,132}, {78,139}, {77,181}, 
    {125,112}, {85,104}, {113,129}, {98,169}, {92,138}, {108,116}, {122,128}, 
    {123,104}, {80,104}, {90,172}, {89,174}, {96,187}, {111,170}, {122,158}, 
    {96,132}, {79,121}, {85,138}, {87,162}, {98,111}, {87,120}, {104,141}, 
    {118,154}, {111,98}, {98,154}, {118,188}, {123,144}, {81,109}, {77,164}, 
    {81,110}, {92,173}, {104,142}, {112,185}, {78,117}, {102,188}, {87,187}, 
    {80,150}, {93,181}, {119,177}, {118,189}, {91,183}, {123,170}, {105,156}, 
    {89,129}, {117,178}, {96,133}, {106,148}, {121,130}, {106,124}, {82,133}, 
    {123,145}, {103,128}, {107,101}, {95,186}, {99,98}, {87,121}, {105,131}, 
    {104,189}, {104,111}, {114,147}, {123,105}, {88,101}, {85,139}, {80,136}, 
    {81,111}, {87,163}, {96,134}, {86,98}, {109,176}, {87,179}, {124,148}, 
    {90,151}, {76,159}, {114,148}, {81,112}, {76,160}, {91,189}, {115,127}, 
    {112,145}, {124,123}, {98,112}, {102,101}, {98,155}, {95,158}, {96,135}, 
    {114,175}, {76,161}, {86,170}, {103,130}, {78,118}, {115,128}, {77,190}, 
    {123,146}, {120,184}, {119,187}, {104,127}, {104,119}, {109,118}, {92,122}, 
    {76,116}, {113,183}, {104,120}, {74,182}, {100,100}, {116,150}, {95,166}, 
    {95,103}, {95,167}, {97,129}, {105,132}, {92,139}, {105,133}, {111,99}, 
    {107,102}, {75,99}, {99,169}, {81,187}, {111,100}, {111,171}, {80,116}, 
    {77,99}, {104,166}, {111,101}, {83,140}, {90,173}, {87,122}, {114,149}, 
    {117,165}, {89,175}, {121,116}, {85,148}, {125,143}, {91,163}, {113,161}, 
    {108,118}, {123,190}, {83,151}, {81,113}, {75,113}, {81,114}, {75,114}, 
    {113,130}, {116,161}, {121,117}, {97,131}, {97,130}, {107,183}, {95,104}, 
    {75,138}, {102,121}, {120,158}, {121,106}, {74,183}, {109,119}, {83,120}, 
    {114,150}, {84,153}, {110,133}, {114,176}, {74,114}, {92,123}, {113,184}, 
    {108,119}, {101,138}, {118,128}, {125,157}, {84,163}, {76,162}, {119,148}, 
    {87,165}, {83,131}, {88,102}, {118,129}, {93,182}, {77,128}, {101,156}, 
    {101,139}, {97,132}, {104,112}, {116,112}, {115,170}, {90,174}, {87,123}, 
    {98,113}, {87,106}, {86,187}, {100,159}, {74,150}, {107,104}, {91,190}, 
    {118,130}, {111,123}, {84,189}, {96,136}, {104,121}, {111,102}, {77,100}, 
    {84,180}, {91,97}, {91,156}, {91,157}, {110,156}, {75,139}, {124,149}, 
    {78,171}, {77,129}, {123,147}, {121,107}, {117,148}, {89,105}, {105,157}, 
    {91,141}, {93,142}, {103,131}, {108,140}, {121,172}, {75,140}, {96,188}, 
    {84,104}, {109,147}, {88,175}, {114,151}, {74,184}, {90,175}, {86,148}, 
    {89,141}, {88,174}, {114,129}, {95,147}, {90,176}, {98,170}, {96,189}, 
    {88,184}, {119,111}, {90,118}, {74,151}, {114,152}, {88,185}, {122,159}, 
    {79,175}, {89,130}, {112,146}, {100,145}, {115,119}, {122,160}, {111,172}, 
    {98,114}, {84,125}, {89,142}, {116,162}, {84,105}, {77,130}, {103,154}, 
    {114,153}, {89,106}, {88,126}, {92,109}, {98,171}, {86,188}, {74,185}, 
    {84,154}, {116,151}, {76,97}, {79,122}, {117,120}, {89,131}, {80,168}, 
    {99,133}, {107,184}, {114,113}, {79,123}, {83,109}, {104,161}, {78,172}, 
    {96,116}, {94,163}, {93,183}, {114,114}, {115,182}, {118,155}, {87,190}, 
    {120,159}, {119,178}, {80,105}, {102,154}, {117,102}, {87,124}, {76,163}, 
    {102,155}, {93,157}, {81,115}, {111,173}, {86,158}, {100,180}, {97,175}, 
    {93,184}, {104,143}, {74,165}, {92,97}, {96,117}, {124,108}, {124,109}, 
    {88,103}, {109,120}, {91,118}, {86,176}, {115,111}, {77,101}, {90,177}, 
    {88,104}, {76,164}, {81,116}, {74,152}, {90,178}, {117,103}, {117,104}, 
    {102,102}, {85,172}, {85,184}, {90,179}, {99,134}, {94,164}, {94,165}, 
    {81,117}, {81,118}, {81,119}, {114,115}, {105,158}, {112,147}, {114,116}, 
    {112,148}, {75,164}, {123,148}, {117,166}, {99,170}, {94,166}, {95,148}, 
    {120,185}, {112,110}, {81,120}, {86,159}, {80,151}, {124,97}, {111,174}, 
    {92,152}, {105,159}, {101,157}, {125,187}, {96,137}, {86,137}, {84,106}, 
    {101,140}, {105,160}, {80,152}, {124,98}, {84,126}, {98,115}, {94,167}, 
    {92,124}, {82,118}, {117,149}, {78,97}, {117,105}, {93,185}, {93,186}, 
    {112,149}, {118,159}, {114,154}, {100,171}, {114,177}, {108,121}, {125,188}, 
    {97,106}, {74,153}, {111,175}, {117,106}, {108,185}, {120,109}, {114,130}, 
    {118,131}, {87,146}, {121,98}, {112,150}, {112,186}, {118,160}, {105,179}, 
    {114,131}, {84,107}, {74,115}, {77,102}, {77,131}, {77,154}, {89,143}, 
    {118,132}, {110,157}, {103,132}, {98,116}, {95,162}, {103,155}, {104,113}, 
    {124,110}, {101,141}, {122,171}, {79,124}, {79,162}, {77,182}, {111,176}, 
    {116,126}, {84,141}, {115,120}, {105,97}, {114,178}, {115,171}, {112,151}, 
    {79,151}, {79,159}, {104,128}, {104,129}, {79,163}, {105,98}, {80,106}, 
    {115,129}, {80,107}, {84,164}, {111,124}, {88,97}, {89,159}, {115,151}, 
    {92,125}, {76,165}, {109,177}, {113,162}, {84,155}, {98,117}, {74,166}, 
    {83,110}, {76,166}, {113,147}, {117,167}, {74,154}, {123,174}, {97,133}, 
    {95,169}, {110,158}, {119,130}, {88,98}, {93,108}, {112,111}, {86,125}, 
    {97,107}, {121,118}, {84,181}, {80,137}, {111,103}, {98,172}, {91,170}, 
    {78,140}, {117,168}, {119,149}, {83,141}, {115,190}, {80,117}, {96,190}, 
    {95,187}, {102,157}, {104,100}, {75,141}, {111,104}, {110,116}, {90,152}, 
    {81,121}, {95,105}, {115,112}, {76,132}, {78,119}, {111,105}, {95,149}, 
    {109,151}, {110,134}, {111,125}, {124,150}, {91,180}, {111,106}, {120,121}, 
    {117,169}, {99,153}, {97,134}, {84,127}, {94,168}, {112,170}, {115,130}, 
    {83,111}, {74,155}, {124,151}, {109,152}, {97,135}, {116,152}, {86,115}, 
    {93,109}, {85,126}, {113,131}, {110,159}, {86,171}, {116,153}, {87,166}, 
    {90,119}, {93,187}, {93,143}, {88,99}, {90,153}, {112,152}, {111,132}, 
    {97,152}, {113,148}, {109,178}, {85,155}, {85,156}, {115,132}, {75,151}, 
    {98,118}, {111,177}, {123,149}, {83,152}, {93,144}, {112,153}, {75,115}, 
    {85,157}, {77,103}, {80,108}, {81,122}, {113,132}, {101,115}, {123,181}, 
    {105,161}, {125,160}, {124,124}, {90,98}, {90,99}, {82,97}, {82,175}, 
    {98,173}, {94,169}, {78,156}, {114,117}, {80,164}, {93,145}, {97,136}, 
    {91,119}, {95,163}, {109,121}, {113,133}, {115,143}, {87,107}, {97,108}, 
    {99,171}, {110,135}, {97,137}, {74,186}, {112,187}, {122,161}, {112,154}, 
    {76,167}, {90,180}, {76,127}, {78,173}, {85,105}, {122,162}, {80,165}, 
    {107,150}, {108,159}, {95,188}, {119,150}, {94,170}, {75,116}, {111,126}, 
    {76,117}, {79,125}, {111,178}, {98,119}, {76,168}, {112,188}, {86,160}, 
    {113,134}, {98,120}, {107,107}, {75,117}, {88,145}, {116,142}, {115,183}, 
    {87,134}, {81,123}, {119,106}, {109,138}, {87,147}, {88,186}, {118,133}, 
    {81,140}, {93,188}, {95,189}, {121,165}, {96,138}, {114,189}, {83,112}, 
    {116,179}, {90,137}, {102,158}, {120,122}, {104,144}, {88,187}, {106,149}, 
    {86,99}, {118,134}, {114,155}, {100,188}, {104,114}, {90,154}, {114,156}, 
    {123,150}, {105,114}, {110,109}, {122,163}, {92,174}, {117,170}, {102,160}, 
    {112,189}, {87,108}, {117,133}, {110,160}, {91,165}, {93,158}, {89,176}, 
    {105,99}, {113,185}, {114,132}, {96,139}, {105,100}, {98,121}, {99,113}, 
    {124,171}, {77,104}, {76,118}, {112,155}, {102,122}, {77,105}, {115,131}, 
    {97,153}, {111,107}, {103,133}, {96,169}, {115,133}, {84,128}, {85,127}, 
    {93,110}, {121,188}, {76,128}, {101,98}, {78,120}, {88,146}, {121,150}, 
    {113,106}, {78,145}, {118,135}, {91,171}, {95,190}, {88,161}, {119,179}, 
    {87,167}, {84,190}, {81,124}, {101,143}, {75,118}, {90,120}, {77,132}, 
    {86,126}, {98,122}, {79,152}, {96,140}, {107,185}, {125,189}, {87,168}, 
    {75,152}, {105,162}, {104,122}, {99,135}, {108,141}, {108,142}, {86,127}, 
    {99,103}, {95,150}, {125,168}, {110,161}, {118,104}, {93,189}, {120,123}, 
    {104,145}, {121,151}, {78,174}, {108,143}, {105,101}, {86,149}, {77,133}, 
    {109,122}, {81,125}, {79,126}, {108,123}, {82,113}, {76,169}, {89,132}, 
    {105,188}, {81,126}, {108,124}, {101,109}, {119,112}, {76,170}, {83,132}, 
    {86,128}, {86,189}, {97,97}, {94,125}, {118,105}, {90,100}, {86,100}, 
    {117,134}, {97,98}, {105,134}, {114,133}, {116,169}, {86,172}, {107,147}, 
    {108,125}, {98,156}, {94,171}, {112,156}, {107,127}, {87,142}, {81,127}, 
    {117,107}, {121,189}, {74,156}, {77,134}, {114,118}, {93,190}, {76,119}, 
    {91,120}, {80,169}, {78,157}, {107,128}, {125,98}, {120,139}, {106,150}, 
    {113,112}, {91,142}, {119,131}, {91,143}, {75,100}, {120,160}, {123,151}, 
    {107,138}, {96,97}, {78,141}, {84,156}, {125,152}, {82,182}, {114,119}, 
    {122,182}, {118,106}, {122,183}, {88,166}, {116,113}, {104,146}, {74,133}, 
    {76,171}, {98,174}, {98,123}, {119,109}, {120,161}, {115,172}, {94,97}, 
    {100,189}, {99,172}, {93,111}, {93,112}, {75,119}, {104,147}, {97,99}, 
    {82,160}, {112,190}, {105,102}, {75,178}, {109,179}, {92,153}, {96,141}, 
    {119,154}, {91,121}, {76,110}, {90,155}, {77,135}, {93,113}, {88,111}, 
    {99,99}, {78,175}, {114,179}, {120,115}, {96,142}, {117,189}, {107,172}, 
    {83,133}, {124,172}, {82,155}, {84,171}, {94,98}, {101,166}, {112,112}, 
    {85,132}, {109,180}, {99,173}, {104,130}, {109,181}, {87,188}, {109,123}, 
    {118,107}, {114,120}, {118,136}, {83,166}, {114,157}, {79,127}, {107,108}, 
    {79,128}, {102,104}, {125,169}, {79,129}, {96,159}, {94,172}, {96,98}, 
    {116,127}, {98,175}, {89,177}, {113,135}, {75,120}, {121,190}, {91,122}, 
    {90,181}, {118,172}, {90,156}, {122,164}, {96,143}, {93,114}, {102,105}, 
    {111,179}, {115,173}, {107,186}, {121,166}, {74,157}, {85,158}, {74,158}, 
    {95,164}, {102,189}, {117,108}, {100,181}, {105,163}, {109,139}, {79,164}, 
    {88,147}, {93,115}, {84,172}, {114,121}, {95,119}, {75,142}, {123,152}, 
    {80,153}, {93,146}, {119,180}, {103,156}, {100,101}, {124,99}, {91,123}, 
    {114,122}, {105,189}, {80,138}, {117,150}, {89,133}, {100,116}, {109,103}, 
    {106,125}, {102,190}, {119,132}, {117,109}, {89,160}, {74,116}, {120,162}, 
    {79,130}, {108,126}, {101,116}, {77,136}, {110,136}, {103,136}, {77,137}, 
    {121,119}, {113,168}, {89,178}, {91,181}, {74,117}, {89,134}, {88,137}, 
    {89,107}, {109,124}, {88,148}, {92,154}, {98,124}, {124,173}, {108,160}, 
    {82,190}, {105,135}, {102,106}, {98,176}, {122,123}, {117,110}, {123,106}, 
    {108,187}, {108,127}, {124,152}, {84,165}, {121,131}, {110,162}, {87,169}, 
    {109,182}, {94,173}, {76,172}, {99,174}, {104,148}, {122,184}, {93,116}, 
    {100,117}, {88,112}, {88,149}, {116,170}, {78,121}, {86,161}, {79,146}, 
    {80,118}, {78,98}, {115,174}, {115,184}, {92,140}, {80,139}, {124,100}, 
    {77,138}, {87,148}, {94,99}, {100,160}, {110,137}, {98,157}, {117,190}, 
    {84,108}, {85,145}, {88,176}, {120,131}, {106,151}, {117,151}, {88,127}, 
    {125,128}, {107,109}, {85,106}, {103,104}, {110,138}, {74,167}, {120,163}, 
    {84,157}, {106,152}, {123,153}, {109,183}, {101,117}, {80,109}, {113,177}, 
    {98,125}, {99,136}, {89,149}, {95,106}, {91,124}, {120,164}, {113,186}, 
    {101,118}, {115,175}, {123,154}, {97,160}, {89,108}, {117,171}, {96,118}, 
    {105,136}, {75,143}, {99,137}, {94,174}, {98,126}, {92,175}, {86,101}, 
    {98,177}, {86,190}, {89,97}, {88,128}, {92,155}, {109,125}, {95,120}, 
    {106,101}, {87,109}, {115,185}, {109,184}, {117,135}, {97,138}, {107,163}, 
    {114,158}, {120,140}, {106,153}, {83,134}, {91,166}, {117,111}, {78,176}, 
    {105,190}, {123,118}, {98,178}, {79,178}, {119,121}, {89,179}, {97,139}, 
    {90,157}, {90,121}, {107,187}, {75,121}, {109,185}, {96,160}, {116,128}, 
    {125,124}, {95,113}, {99,175}, {96,99}, {125,121}, {112,113}, {77,139}, 
    {109,126}, {85,128}, {99,176}, {109,186}, {105,164}, {85,173}, {103,157}, 
    {84,182}, {101,119}, {91,167}, {98,127}, {110,139}, {87,180}, {112,157}, 
    {78,107}, {103,158}, {86,150}, {97,140}, {104,115}, {101,174}, {92,98}, 
    {96,144}, {85,117}, {85,97}, {123,155}, {121,139}, {75,179}, {116,101}, 
    {122,136}, {86,151}, {105,165}, {123,156}, {125,144}, {123,182}, {90,101}, 
    {91,125}, {108,162}, {77,183}, {112,158}, {118,137}, {94,175}, {83,113}, 
    {124,174}, {104,131}, {113,136}, {78,177}, {121,173}, {114,180}, {100,118}, 
    {117,121}, {92,176}, {99,177}, {104,101}, {114,123}, {94,100}, {90,140}, 
    {74,169}, {99,154}, {124,153}, {106,154}, {121,132}, {99,100}, {123,157}, 
    {111,138}, {104,132}, {85,140}, {107,151}, {89,109}, {123,107}, {83,153}, 
    {85,98}, {118,158}, {90,182}, {96,145}, {105,104}, {117,185}, {122,111}, 
    {107,188}, {96,170}, {99,114}, {85,133}, {113,163}, {85,174}, {77,140}, 
    {109,153}, {88,129}, {122,172}, {113,171}, {122,124}, {102,162}, {122,165}, 
    {98,186}, {74,118}, {100,119}, {106,155}, {117,186}, {123,108}, {79,131}, 
    {107,189}, {120,186}, {95,121}, {97,177}, {82,100}, {117,187}, {80,154}, 
    {80,155}, {106,126}, {89,113}, {74,119}, {83,167}, {120,165}, {83,114}, 
    {98,128}, {114,159}, {77,165}, {121,108}, {77,141}, {110,110}, {86,110}, 
    {87,170}, {103,160}, {107,110}, {79,153}, {92,141}, {109,187}, {94,176}, 
    {87,171}, {94,101}, {95,151}, {91,144}, {91,145}, {85,99}, {112,114}, 
    {92,156}, {74,168}, {120,166}, {92,142}, {106,156}, {91,146}, {105,115}, 
    {119,155}, {99,104}, {87,110}, {96,161}, {75,122}, {101,145}, {80,156}, 
    {85,129}, {88,138}, {99,105}, {96,100}, {105,105}, {83,135}, {92,157}, 
    {120,110}, {76,120}, {80,110}, {88,178}, {99,138}, {76,111}, {84,109}, 
    {118,145}, {80,140}, {74,134}, {85,130}, {78,122}, {74,135}, {122,112}, 
    {95,152}, {117,122}, {101,171}, {111,180}, {93,117}, {77,106}, {99,178}, 
    {123,183}, {119,144}, {125,122}, {125,161}, {118,190}, {81,128}, {104,133}, 
    {100,120}, {97,168}, {76,129}, {82,173}, {91,126}, {96,162}, {122,137}, 
    {97,141}, {74,120}, {114,160}, {113,137}, {94,177}, {112,159}, {120,132}, 
    {110,140}, {94,178}, {103,137}, {98,179}, {103,161}, {99,139}, {99,140}, 
    {79,184}, {111,108}, {125,190}, {124,101}, {122,113}, {95,153}, {96,146}, 
    {116,154}, {113,138}, {78,99}, {114,124}, {108,163}, {96,101}, {119,107}, 
    {107,111}, {101,158}, {97,100}, {77,107}, {89,180}, {104,102}, {77,142}, 
    {97,169}, {124,175}, {96,163}, {98,129}, {78,100}, {94,102}, {107,190}, 
    {107,157}, {112,160}, {116,155}, {98,180}, {83,136}, {116,171}, {110,117}, 
    {117,152}, {85,159}, {86,165}, {107,112}, {116,163}, {99,141}, {116,180}, 
    {122,114}, {111,181}, {74,159}, {107,113}, {109,127}, {125,137}, {100,102}, 
    {121,100}, {112,115}, {101,172}, {81,167}, {89,135}, {100,151}, {106,157}, 
    {84,183}, {90,122}, {90,141}, {121,140}, {97,154}, {91,127}, {76,133}, 
    {108,144}, {75,123}, {94,179}, {105,106}, {89,136}, {110,163}, {87,125}, 
    {79,132}, {80,141}, {124,102}, {113,187}, {125,146}, {81,129}, {99,155}, 
    {83,137}, {92,143}, {76,173}, {94,103}, {102,123}, {108,97}, {76,121}, 
    {123,158}, {103,162}, {84,129}, {92,104}, {98,130}, {115,152}, {101,147}, 
    {115,153}, {115,134}, {77,155}, {77,108}, {124,131}, {84,167}, {81,130}, 
    {121,101}, {104,149}, {99,142}, {84,138}, {95,154}, {123,159}, {103,163}, 
    {120,187}, {99,143}, {117,112}, {88,167}, {89,137}, {120,111}, {111,182}, 
    {93,118}, {110,111}, {77,184}, {94,120}, {124,103}, {119,188}, {119,113}, 
    {78,123}, {116,97}, {110,141}, {97,110}, {108,131}, {79,190}, {120,127}, 
    {88,162}, {83,168}, {94,104}, {116,164}, {108,130}, {89,181}, {121,133}, 
    {93,147}, {86,177}, {108,188}, {124,176}, {109,128}, {74,121}, {110,164}, 
    {114,161}, {94,121}, {86,178}, {94,180}, {95,155}, {91,147}, {122,167}, 
    {88,163}, {116,129}, {93,119}, {114,181}, {84,110}, {86,179}, {93,120}, 
    {79,133}, {95,159}, {114,126}, {118,97}, {107,139}, {113,188}, {115,135}, 
    {96,171}, {109,188}, {97,155}, {110,165}, {94,181}, {122,147}, {113,139}, 
    {80,111}, {93,121}, {81,131}, {117,113}, {106,134}, {112,161}, {118,108}, 
    {117,153}, {112,171}, {93,122}, {114,127}, {119,133}, {91,98}, {114,182}, 
    {74,122}, {119,181}, {75,165}, {110,166}, {96,147}, {78,101}, {86,152}, 
    {84,111}, {105,137}, {83,142}, {116,130}, {75,166}, {113,97}, {107,114}, 
    {113,98}, {107,115}, {112,116}, {75,180}, {84,112}, {115,114}, {123,119}, 
    {117,172}, {110,167}, {116,114}, {117,173}, {79,179}, {112,162}, {110,142}, 
    {113,140}, {101,120}, {87,181}, {99,179}, {79,165}, {79,134}, {115,115}, 
    {100,152}, {79,185}, {79,154}, {122,141}, {102,163}, {114,162}, {117,174}, 
    {74,123}, {99,156}, {78,178}, {86,153}, {110,112}, {116,165}, {88,130}, 
    {92,144}, {76,174}, {85,160}, {118,138}, {125,138}, {88,150}, {116,143}, 
    {86,102}, {92,126}, {91,148}, {87,135}, {114,190}, {113,141}, {98,131}, 
    {92,158}, {92,159}, {111,109}, {102,107}, {121,157}, {106,127}, {111,110}, 
    {116,144}, {78,179}, {102,108}, {78,158}, {85,185}, {99,180}, {77,144}, 
    {85,120}, {119,189}, {92,105}, {94,182}, {92,106}, {114,163}, {105,116}, 
    {82,156}, {105,166}, {99,182}, {103,138}, {80,142}, {90,183}, {74,124}, 
    {110,168}, {90,158}, {114,183}, {98,187}, {76,102}, {90,123}, {110,169}, 
    {117,154}, {119,156}, {97,170}, {78,129}, {84,113}, {125,113}, {102,125}, 
    {123,109}, {120,167}, {97,142}, {119,162}, {117,175}, {79,135}, {84,114}, 
    {76,175}, {84,168}, {110,143}, {119,151}, {96,102}, {86,129}, {97,156}, 
    {112,163}, {113,164}, {92,177}, {86,103}, {116,181}, {113,142}, {114,164}, 
    {80,112}, {108,175}, {121,122}, {107,117}, {84,173}, {98,132}, {105,167}, 
    {107,116}, {106,97}, {120,124}, {78,102}, {121,134}, {124,154}, {84,115}, 
    {83,121}, {106,158}, {105,107}, {97,161}, {83,143}, {116,182}, {106,128}, 
    {97,143}, {76,122}, {110,170}, {112,164}, {115,116}, {84,174}, {114,128}, 
    {113,165}, {116,131}, {96,148}, {107,118}, {87,97}, {75,168}, {121,109}, 
    {105,109}, {88,164}, {122,115}, {98,133}, {124,125}, {108,132}, {88,113}, 
    {92,107}, {85,100}, {107,169}, {104,123}, {88,151}, {123,110}, {81,161}, 
    {91,128}, {117,126}, {94,183}, {74,187}, {119,134}, {79,136}, {97,144}, 
    {110,144}, {105,180}, {78,180}, {85,141}, {79,155}, {93,123}, {78,108}, 
    {105,168}, {84,116}, {100,135}, {117,155}, {122,129}, {94,105}, {84,184}, 
    {111,183}, {83,115}, {107,119}, {111,184}, {117,156}, {109,140}, {91,149}, 
    {113,143}, {113,144}, {117,114}, {89,110}, {85,108}, {98,134}, {125,99}, 
    {123,165}, {95,107}, {98,181}, {118,109}, {117,115}, {112,117}, {97,101}, 
    {117,157}, {108,98}, {109,189}, {117,116}, {123,120}, {91,99}, {86,138}, 
    {75,153}, {101,148}, {115,186}, {107,120}, {96,119}, {87,172}, {113,172}, 
    {101,111}, {85,161}, {87,173}, {81,145}, {97,178}, {111,185}, {93,124}, 
    {118,156}, {112,165}, {116,132}, {105,169}, {115,187}, {84,175}, {76,98}, 
    {119,190}, {95,124}, {107,141}, {80,119}, {86,130}, {104,109}, {111,111}, 
    {75,101}, {75,169}, {122,168}, {76,134}, {102,167}, {106,135}, {91,100}, 
    {79,137}, {98,188}, {111,186}, {107,158}, {117,136}, {84,158}, {96,149}, 
    {111,112}, {98,135}, {89,111}, {121,167}, {103,165}, {79,138}, {97,145}, 
    {98,136}, {111,187}, {122,185}, {92,178}, {96,103}, {120,168}, {75,170}, 
    {75,124}, {86,162}, {117,158}, {117,159}, {110,118}, {98,182}, {83,138}, 
    {111,188}, {81,132}, {111,113}, {81,133}, {80,158}, {89,161}, {96,120}, 
    {77,145}, {115,121}, {103,140}, {86,104}, {78,103}, {84,117}, {100,136}, 
    {83,116}, {107,121}, {75,181}, {118,157}, {113,99}, {76,135}, {105,138}, 
    {97,176}, {117,160}, {123,111}, {75,145}, {123,160}, {114,165}, {108,176}, 
    {112,172}, {110,171}, {105,139}, {76,176}, {87,111}, {115,97}, {124,181}, 
    {113,100}, {96,150}, {111,114}, {116,145}, {119,97}, {113,145}, {74,188}, 
    {74,189}, {78,142}, {115,136}, {115,122}, {109,190}, {90,102}, {96,172}, 
    {120,141}, {75,146}, {107,142}, {121,152}, {121,153}, {74,160}, {90,138}, 
    {75,102}, {74,136}, {121,174}, {91,172}, {80,113}, {85,175}, {102,179}, 
    {103,98}, {100,153}, {100,161}, {124,132}, {121,175}, {79,180}, {119,166}, 
    {78,124}, {116,133}, {92,99}, {93,125}, {116,134}, {120,97}, {104,150}, 
    {91,129}, {112,166}, {100,121}, {118,173}, {121,110}, {93,126}, {87,112}, 
    {88,168}, {75,125}, {121,154}, {120,142}, {121,176}, {96,173}, {99,115}, 
    {116,115}, {106,130}, {114,166}, {112,118}, {91,150}, {107,164}, {114,167}, 
    {87,149}, {84,118}, {121,168}, {87,129}, {101,149}, {105,170}, {87,140}, 
    {83,169}, {98,137}, {124,155}, {77,109}, {76,112}, {106,98}, {100,182}, 
    {80,128}, {112,119}, {110,97}, {87,182}, {98,138}, {98,139}, {122,143}, 
    {107,159}, {86,139}, {116,116}, {109,141}, {100,146}, {106,105}, {100,122}, 
    {115,98}, {77,146}, {118,139}, {113,166}, {109,129}, {104,124}, {110,145}, 
    {112,167}, {98,140}, {100,106}, {117,161}, {109,154}, {87,174}, {81,177}, 
    {105,171}, {105,172}, {96,164}, {90,103}, {93,148}, {106,99}, {86,131}, 
    {86,180}, {90,159}, {111,115}, {98,141}, {111,189}, {114,168}, {111,133}, 
    {103,167}, {87,189}, {103,142}, {95,156}, {121,135}, {89,182}, {95,108}, 
    {86,154}, {92,100}, {112,120}, {85,186}, {100,183}, {86,132}, {116,172}, 
    {111,190}, {112,97}, {94,106}, {90,124}, {88,188}, {122,148}, {108,165}, 
    {124,104}, {108,166}, {88,139}, {123,121}, {100,147}, {77,185}, {79,147}, 
    {74,170}, {79,148}, {120,125}, {116,135}, {106,159}, {121,155}, {84,119}, 
    {107,165}, {97,146}, {106,100}, {122,130}, {123,161}, {122,173}, {112,98}, 
    {76,177}, {122,99}, {98,183}, {98,142}, {105,181}, {97,171}, {103,168}, 
    {104,151}, {90,184}, {84,139}, {119,182}, {86,133}, {84,169}, {122,186}, 
    {76,178}, {119,157}, {94,122}, {78,104}, {112,121}, {100,190}, {100,137}, 
    {100,148}, {106,131}, {111,116}, {87,126}, {123,162}, {77,147}, {111,117}, 
    {122,169}, {121,102}, {95,125}, {119,135}, {120,189}, {120,188}, {94,107}, 
    {91,168}, {99,157}, {97,162}, {81,134}, {118,144}, {107,166}, {90,185}, 
    {108,135}, {94,184}, {120,169}, {99,158}, {78,181}, {122,131}, {101,151}, 
    {108,136}, {115,137}, {100,123}, {102,110}, {111,118}, {92,127}, {78,125}, 
    {88,131}, {80,143}, {79,186}, {115,138}, {96,151}, {81,135}, {105,110}, 
    {104,125}, {122,132}, {98,143}, {122,133}, {121,120}, {92,160}, {123,112}, 
    {88,105}, {101,159}, {121,103}, {118,174}, {118,140}, {98,184}, {108,177}, 
    {90,160}, {113,146}, {82,140}, {79,139}, {74,125}, {93,127}, {118,175}, 
    {94,185}, {122,116}, {85,109}, {113,167}, {94,126}, {92,128}, {81,136}, 
    {81,137}, {120,126}, {75,182}, {84,185}, {117,162}, {97,147}, {88,169}, 
    {120,190}, {79,140}, {125,100}, {78,182}, {122,144}, {76,179}, {102,126}, 
    {118,110}, {85,176}, {81,138}, {124,126}, {85,177}, {77,169}, {122,117}, 
    {98,144}, {116,183}, {77,148}, {103,99}, {91,101}, {98,145}, {87,98}, 
    {119,163}, {106,102}, {80,97}, {78,154}, {123,171}, {91,102}, {91,158}, 
    {88,165}, {106,160}, {88,106}, {101,160}, {86,155}, {111,134}, {120,170}, 
    {100,149}, {78,183}, {96,152}, {87,175}, {116,173}, {77,166}, {76,180}, 
    {117,163}, {100,138}, {92,161}, {121,136}, {124,127}, {104,103}, {88,132}, 
    {75,126}, {92,110}, {87,183}, {112,168}, {93,128}, {79,141}, {92,179}, 
    {89,112}, {102,169}, {100,124}, {106,132}, {100,172}, {100,165}, {123,184}, 
    {76,123}, {100,125}, {77,156}, {89,183}, {93,159}, {109,142}, {89,144}, 
    {101,99}, {121,141}, {77,110}, {79,142}, {118,111}, {125,147}, {107,173}, 
    {86,156}, {101,100}, {85,118}, {86,157}, {121,169}, {103,100}, {86,163}, 
    {81,139}, {86,164}, {85,178}, {94,186}, {87,184}, {88,170}, {79,149}, 
    {88,189}, {88,107}, {125,139}, {124,156}, {96,104}, {85,179}, {125,153}, 
    {76,99}, {89,185}, {83,170}, {117,181}, {111,135}, {83,154}, {90,125}, 
    {104,104}, {92,111}, {112,99}, {77,149}, {96,105}, {94,108}, {112,122}, 
    {110,113}, {110,114}, {118,141}, {110,146}, {86,134}, {96,165}, {115,123}, 
    {101,161}, {100,139}, {87,99}, {91,130}, {74,190}, {79,143}, {48,97}, 
    {48,98}, {1,1}, {1,2}, {48,99}, {1,3}, {1,4}, {48,100}, {48,101}, {48,102}, 
    {48,103}, {1,5}, {1,6}, {1,7}, {1,8}, {1,9}, {48,104}, {48,105}, {48,106}, 
    {48,107}, {48,108}, {48,109}, {48,110}, {48,111}, {1,10}, {48,112}, 
    {48,113}, {48,114}, {48,115}, {48,116}, {1,11}, {1,12}, {48,117}, {1,13}, 
    {1,14}, {1,15}, {48,118}, {1,16}, {1,17}, {1,18}, {1,19}, {1,20}, {1,21}, 
    {1,22}, {48,119}, {48,120}, {1,23}, {48,121}, {48,122}, {48,123}, {1,24}, 
    {1,25}, {1,26}, {1,33}, {1,34}, {1,35}, {48,124}, {48,125}, {1,36}, 
    {1,37}, {48,126}, {1,38}, {1,39}, {1,40}, {48,127}, {1,41}, {1,42}, 
    {1,43}, {1,44}, {1,45}, {1,46}, {1,47}, {1,48}, {1,49}, {1,50}, {48,128}, 
    {1,51}, {48,129}, {1,52}, {1,53}, {1,54}, {1,55}, {1,56}, {1,57}, {48,130}, 
    {1,58}, {1,65}, {1,66}, {48,131}, {1,67}, {1,68}, {1,69}, {48,132}, 
    {1,70}, {1,71}, {1,72}, {1,73}, {1,74}, {1,75}, {1,76}, {1,77}, {1,78}, 
    {1,79}, {1,80}, {1,81}, {1,82}, {1,83}, {1,84}, {1,85}, {1,86}, {1,87}, 
    {1,88}, {48,133}, {48,134}, {1,89}, {1,90}, {48,135}, {1,91}, {1,92}, 
    {48,136}, {48,137}, {1,93}, {48,138}, {1,94}, {1,95}, {1,96}, {1,97}, 
    {1,98}, {48,139}, {48,140}, {1,99}, {48,141}, {48,142}, {48,143}, {48,144}, 
    {1,100}, {1,101}, {48,145}, {48,146}, {48,147}, {48,148}, {1,102}, 
    {1,103}, {1,104}, {48,149}, {1,105}, {1,106}, {1,107}, {48,150}, {1,108}, 
    {1,109}, {1,110}, {1,111}, {1,112}, {1,113}, {1,114}, {48,151}, {48,152}, 
    {1,115}, {48,153}, {48,154}, {48,155}, {1,116}, {1,117}, {1,118}, {1,119}, 
    {1,120}, {1,121}, {48,156}, {48,157}, {48,158}, {1,122}, {48,159}, 
    {1,123}, {1,124}, {48,160}, {48,161}, {1,125}, {1,126}, {1,127}, {1,128}, 
    {1,129}, {1,130}, {1,131}, {48,162}, {48,163}, {1,132}, {48,164}, {48,165}, 
    {48,166}, {1,133}, {1,134}, {1,135}, {48,167}, {1,136}, {1,137}, {48,168}, 
    {1,138}, {1,139}, {1,140}, {48,169}, {1,141}, {1,142}, {1,143}, {48,170}, 
    {1,144}, {1,145}, {1,146}, {1,147}, {1,148}, {1,149}, {1,150}, {1,151}, 
    {48,171}, {1,152}, {48,172}, {1,153}, {1,154}, {1,155}, {1,156}, {1,157}, 
    {1,158}, {1,159}, {1,160}, {48,173}, {48,174}, {1,161}, {1,162}, {48,175}, 
    {1,163}, {1,164}, {48,176}, {48,177}, {1,165}, {48,178}, {1,166}, {48,179}, 
    {1,167}, {1,168}, {48,180}, {48,181}, {48,182}, {1,169}, {48,183}, 
    {1,170}, {48,184}, {48,185}, {1,171}, {1,172}, {1,173}, {1,174}, {1,175}, 
    {48,186}, {48,187}, {1,176}, {1,177}, {48,188}, {1,178}, {1,179}, {1,180}, 
    {48,189}, {1,181}, {48,190}, {1,182}, {1,183}, {1,184}, {1,185}, {1,186}, 
    {49,97}, {49,98}, {1,187}, {49,99}, {1,188}, {49,100}, {1,189}, {1,190}, 
    {2,1}, {2,2}, {2,3}, {2,4}, {49,101}, {2,5}, {2,6}, {2,7}, {49,102}, 
    {2,8}, {2,9}, {2,10}, {49,103}, {2,11}, {2,12}, {2,13}, {2,14}, {2,15}, 
    {2,16}, {2,17}, {2,18}, {49,104}, {2,19}, {2,20}, {49,105}, {49,106}, 
    {2,21}, {2,22}, {2,23}, {2,24}, {2,25}, {2,26}, {49,107}, {49,108}, 
    {2,33}, {2,34}, {49,109}, {2,35}, {2,36}, {2,37}, {49,110}, {2,38}, 
    {2,39}, {2,40}, {2,41}, {2,42}, {2,43}, {2,44}, {49,111}, {49,112}, 
    {2,45}, {49,113}, {2,46}, {49,114}, {2,47}, {2,48}, {2,49}, {2,50}, 
    {2,51}, {2,52}, {49,115}, {2,53}, {2,54}, {2,55}, {49,116}, {2,56}, 
    {2,57}, {2,58}, {49,117}, {2,65}, {2,66}, {2,67}, {2,68}, {2,69}, {2,70}, 
    {2,71}, {2,72}, {49,118}, {2,73}, {49,119}, {2,74}, {2,75}, {2,76}, 
    {2,77}, {2,78}, {2,79}, {2,80}, {2,81}, {49,120}, {49,121}, {2,82}, 
    {2,83}, {49,122}, {2,84}, {2,85}, {49,123}, {49,124}, {49,125}, {49,126}, 
    {2,86}, {2,87}, {2,88}, {2,89}, {49,127}, {49,128}, {49,129}, {2,90}, 
    {49,130}, {2,91}, {49,131}, {49,132}, {2,92}, {2,93}, {2,94}, {2,95}, 
    {2,96}, {49,133}, {49,134}, {2,97}, {2,98}, {49,135}, {2,99}, {2,100}, 
    {2,101}, {49,136}, {2,102}, {2,103}, {2,104}, {2,105}, {2,106}, {2,107}, 
    {2,108}, {2,109}, {2,110}, {2,111}, {2,112}, {49,137}, {49,138}, {2,113}, 
    {2,114}, {2,115}, {2,116}, {2,117}, {2,118}, {49,139}, {2,119}, {2,120}, 
    {2,121}, {2,122}, {2,123}, {2,124}, {2,125}, {2,126}, {2,127}, {2,128}, 
    {2,129}, {2,130}, {2,131}, {2,132}, {2,133}, {2,134}, {2,135}, {2,136}, 
    {49,140}, {2,137}, {2,138}, {2,139}, {2,140}, {2,141}, {2,142}, {2,143}, 
    {2,144}, {49,141}, {49,142}, {2,145}, {2,146}, {49,143}, {2,147}, {2,148}, 
    {2,149}, {49,144}, {2,150}, {2,151}, {2,152}, {2,153}, {2,154}, {2,155}, 
    {2,156}, {49,145}, {49,146}, {2,157}, {49,147}, {2,158}, {2,159}, {2,160}, 
    {2,161}, {2,162}, {2,163}, {2,164}, {2,165}, {49,148}, {2,166}, {2,167}, 
    {2,168}, {49,149}, {2,169}, {2,170}, {2,171}, {49,150}, {2,172}, {2,173}, 
    {2,174}, {2,175}, {2,176}, {2,177}, {2,178}, {2,179}, {2,180}, {2,181}, 
    {2,182}, {2,183}, {2,184}, {2,185}, {2,186}, {2,187}, {2,188}, {2,189}, 
    {2,190}, {49,151}, {49,152}, {3,1}, {3,2}, {49,153}, {3,3}, {3,4}, 
    {49,154}, {49,155}, {49,156}, {3,5}, {3,6}, {3,7}, {3,8}, {3,9}, {3,10}, 
    {49,157}, {49,158}, {3,11}, {49,159}, {3,12}, {49,160}, {3,13}, {3,14}, 
    {3,15}, {3,16}, {3,17}, {3,18}, {49,161}, {3,19}, {3,20}, {3,21}, {3,22}, 
    {3,23}, {3,24}, {3,25}, {3,26}, {3,33}, {3,34}, {3,35}, {3,36}, {3,37}, 
    {3,38}, {3,39}, {3,40}, {3,41}, {3,42}, {3,43}, {3,44}, {3,45}, {3,46}, 
    {3,47}, {3,48}, {3,49}, {3,50}, {3,51}, {49,162}, {49,163}, {3,52}, 
    {3,53}, {49,164}, {3,54}, {3,55}, {49,165}, {49,166}, {3,56}, {49,167}, 
    {3,57}, {3,58}, {3,65}, {3,66}, {3,67}, {49,168}, {49,169}, {3,68}, 
    {49,170}, {3,69}, {49,171}, {49,172}, {3,70}, {3,71}, {3,72}, {49,173}, 
    {3,73}, {49,174}, {49,175}, {49,176}, {3,74}, {49,177}, {3,75}, {3,76}, 
    {3,77}, {49,178}, {3,78}, {49,179}, {3,79}, {3,80}, {3,81}, {3,82}, 
    {3,83}, {49,180}, {49,181}, {3,84}, {49,182}, {49,183}, {49,184}, {3,85}, 
    {3,86}, {3,87}, {49,185}, {3,88}, {3,89}, {49,186}, {49,187}, {3,90}, 
    {3,91}, {49,188}, {3,92}, {3,93}, {3,94}, {49,189}, {3,95}, {3,96}, 
    {3,97}, {3,98}, {3,99}, {3,100}, {3,101}, {49,190}, {50,97}, {3,102}, 
    {50,98}, {50,99}, {50,100}, {3,103}, {3,104}, {3,105}, {3,106}, {3,107}, 
    {3,108}, {50,101}, {50,102}, {3,109}, {3,110}, {3,111}, {3,112}, {3,113}, 
    {3,114}, {50,103}, {3,115}, {3,116}, {3,117}, {3,118}, {3,119}, {3,120}, 
    {3,121}, {3,122}, {3,123}, {3,124}, {3,125}, {3,126}, {3,127}, {3,128}, 
    {3,129}, {3,130}, {3,131}, {3,132}, {3,133}, {3,134}, {3,135}, {3,136}, 
    {3,137}, {3,138}, {3,139}, {3,140}, {3,141}, {3,142}, {3,143}, {3,144}, 
    {3,145}, {3,146}, {3,147}, {3,148}, {3,149}, {3,150}, {3,151}, {3,152}, 
    {3,153}, {3,154}, {3,155}, {3,156}, {3,157}, {3,158}, {3,159}, {3,160}, 
    {3,161}, {50,104}, {50,105}, {50,106}, {3,162}, {50,107}, {3,163}, 
    {3,164}, {3,165}, {50,108}, {3,166}, {3,167}, {3,168}, {3,169}, {3,170}, 
    {3,171}, {3,172}, {50,109}, {50,110}, {3,173}, {50,111}, {50,112}, 
    {50,113}, {3,174}, {3,175}, {3,176}, {3,177}, {3,178}, {3,179}, {50,114}, 
    {50,115}, {3,180}, {3,181}, {50,116}, {3,182}, {3,183}, {3,184}, {3,185}, 
    {3,186}, {3,187}, {3,188}, {3,189}, {3,190}, {4,1}, {4,2}, {50,117}, 
    {4,3}, {4,4}, {50,118}, {4,5}, {50,119}, {4,6}, {4,7}, {4,8}, {4,9}, 
    {4,10}, {4,11}, {50,120}, {4,12}, {4,13}, {4,14}, {50,121}, {4,15}, 
    {4,16}, {4,17}, {50,122}, {4,18}, {4,19}, {4,20}, {4,21}, {4,22}, {4,23}, 
    {4,24}, {4,25}, {4,26}, {4,33}, {50,123}, {50,124}, {4,34}, {4,35}, 
    {4,36}, {4,37}, {50,125}, {4,38}, {4,39}, {50,126}, {4,40}, {4,41}, 
    {4,42}, {4,43}, {4,44}, {4,45}, {4,46}, {4,47}, {4,48}, {4,49}, {4,50}, 
    {4,51}, {4,52}, {4,53}, {4,54}, {4,55}, {4,56}, {4,57}, {4,58}, {4,65}, 
    {4,66}, {4,67}, {4,68}, {4,69}, {4,70}, {4,71}, {4,72}, {50,127}, {50,128}, 
    {4,73}, {4,74}, {50,129}, {4,75}, {50,130}, {4,76}, {50,131}, {4,77}, 
    {4,78}, {4,79}, {4,80}, {4,81}, {4,82}, {4,83}, {50,132}, {50,133}, 
    {4,84}, {50,134}, {4,85}, {50,135}, {50,136}, {50,137}, {4,86}, {4,87}, 
    {4,88}, {4,89}, {50,138}, {50,139}, {4,90}, {4,91}, {4,92}, {4,93}, 
    {4,94}, {4,95}, {50,140}, {4,96}, {4,97}, {4,98}, {4,99}, {4,100}, 
    {4,101}, {4,102}, {4,103}, {4,104}, {4,105}, {4,106}, {50,141}, {50,142}, 
    {4,107}, {4,108}, {4,109}, {4,110}, {4,111}, {4,112}, {50,143}, {50,144}, 
    {4,113}, {4,114}, {4,115}, {4,116}, {4,117}, {4,118}, {4,119}, {4,120}, 
    {4,121}, {4,122}, {4,123}, {4,124}, {4,125}, {4,126}, {4,127}, {4,128}, 
    {4,129}, {4,130}, {4,131}, {50,145}, {4,132}, {4,133}, {4,134}, {4,135}, 
    {4,136}, {4,137}, {50,146}, {4,138}, {4,139}, {4,140}, {50,147}, {4,141}, 
    {4,142}, {4,143}, {50,148}, {4,144}, {4,145}, {4,146}, {4,147}, {4,148}, 
    {4,149}, {4,150}, {50,149}, {50,150}, {4,151}, {4,152}, {4,153}, {50,151}, 
    {4,154}, {4,155}, {4,156}, {4,157}, {4,158}, {4,159}, {50,152}, {4,160}, 
    {4,161}, {4,162}, {4,163}, {4,164}, {4,165}, {4,166}, {4,167}, {4,168}, 
    {4,169}, {4,170}, {4,171}, {4,172}, {4,173}, {4,174}, {4,175}, {4,176}, 
    {4,177}, {4,178}, {4,179}, {4,180}, {4,181}, {4,182}, {4,183}, {4,184}, 
    {4,185}, {4,186}, {50,153}, {50,154}, {4,187}, {4,188}, {50,155}, {4,189}, 
    {4,190}, {5,1}, {50,156}, {5,2}, {5,3}, {5,4}, {5,5}, {5,6}, {5,7}, 
    {50,157}, {50,158}, {50,159}, {5,8}, {50,160}, {5,9}, {50,161}, {50,162}, 
    {5,10}, {5,11}, {5,12}, {5,13}, {5,14}, {50,163}, {5,15}, {5,16}, {5,17}, 
    {5,18}, {5,19}, {5,20}, {5,21}, {50,164}, {5,22}, {5,23}, {5,24}, {5,25}, 
    {5,26}, {5,33}, {5,34}, {5,35}, {5,36}, {5,37}, {5,38}, {50,165}, {50,166}, 
    {5,39}, {5,40}, {5,41}, {5,42}, {5,43}, {5,44}, {50,167}, {50,168}, 
    {5,45}, {5,46}, {50,169}, {5,47}, {5,48}, {5,49}, {50,170}, {5,50}, 
    {5,51}, {5,52}, {5,53}, {5,54}, {5,55}, {5,56}, {50,171}, {50,172}, 
    {5,57}, {5,58}, {50,173}, {5,65}, {5,66}, {5,67}, {5,68}, {5,69}, {5,70}, 
    {5,71}, {50,174}, {5,72}, {5,73}, {5,74}, {50,175}, {5,75}, {5,76}, 
    {5,77}, {50,176}, {5,78}, {5,79}, {5,80}, {5,81}, {5,82}, {5,83}, {5,84}, 
    {50,177}, {50,178}, {5,85}, {5,86}, {5,87}, {5,88}, {5,89}, {5,90}, 
    {5,91}, {5,92}, {5,93}, {5,94}, {50,179}, {5,95}, {5,96}, {5,97}, {5,98}, 
    {5,99}, {5,100}, {5,101}, {5,102}, {5,103}, {5,104}, {5,105}, {5,106}, 
    {5,107}, {5,108}, {5,109}, {5,110}, {5,111}, {5,112}, {5,113}, {5,114}, 
    {5,115}, {5,116}, {5,117}, {5,118}, {5,119}, {5,120}, {5,121}, {50,180}, 
    {50,181}, {5,122}, {5,123}, {50,182}, {5,124}, {50,183}, {5,125}, {50,184}, 
    {5,126}, {50,185}, {5,127}, {5,128}, {5,129}, {5,130}, {50,186}, {50,187}, 
    {50,188}, {5,131}, {50,189}, {5,132}, {50,190}, {5,133}, {5,134}, {5,135}, 
    {51,97}, {5,136}, {5,137}, {5,138}, {5,139}, {5,140}, {5,141}, {5,142}, 
    {5,143}, {5,144}, {5,145}, {5,146}, {5,147}, {5,148}, {5,149}, {5,150}, 
    {5,151}, {5,152}, {5,153}, {5,154}, {5,155}, {5,156}, {5,157}, {5,158}, 
    {5,159}, {5,160}, {5,161}, {5,162}, {5,163}, {5,164}, {5,165}, {51,98}, 
    {51,99}, {5,166}, {5,167}, {51,100}, {5,168}, {5,169}, {5,170}, {51,101}, 
    {5,171}, {5,172}, {5,173}, {5,174}, {5,175}, {5,176}, {5,177}, {51,102}, 
    {51,103}, {5,178}, {51,104}, {5,179}, {51,105}, {5,180}, {5,181}, {5,182}, 
    {5,183}, {5,184}, {5,185}, {51,106}, {51,107}, {51,108}, {5,186}, {51,109}, 
    {5,187}, {5,188}, {51,110}, {51,111}, {51,112}, {51,113}, {5,189}, 
    {5,190}, {6,1}, {6,2}, {6,3}, {51,114}, {51,115}, {6,4}, {51,116}, 
    {51,117}, {51,118}, {51,119}, {51,120}, {6,5}, {51,121}, {6,6}, {51,122}, 
    {51,123}, {51,124}, {6,7}, {6,8}, {51,125}, {6,9}, {6,10}, {6,11}, 
    {51,126}, {6,12}, {6,13}, {6,14}, {6,15}, {6,16}, {6,17}, {6,18}, {51,127}, 
    {51,128}, {6,19}, {51,129}, {51,130}, {51,131}, {6,20}, {6,21}, {6,22}, 
    {6,23}, {6,24}, {6,25}, {51,132}, {51,133}, {6,26}, {6,33}, {51,134}, 
    {6,34}, {6,35}, {6,36}, {51,135}, {6,37}, {6,38}, {6,39}, {6,40}, {6,41}, 
    {6,42}, {6,43}, {51,136}, {6,44}, {6,45}, {6,46}, {6,47}, {51,137}, 
    {6,48}, {6,49}, {6,50}, {6,51}, {6,52}, {6,53}, {6,54}, {6,55}, {6,56}, 
    {6,57}, {6,58}, {6,65}, {6,66}, {6,67}, {6,68}, {6,69}, {6,70}, {6,71}, 
    {6,72}, {6,73}, {6,74}, {6,75}, {6,76}, {6,77}, {6,78}, {6,79}, {6,80}, 
    {6,81}, {6,82}, {6,83}, {6,84}, {6,85}, {6,86}, {6,87}, {51,138}, {51,139}, 
    {6,88}, {51,140}, {51,141}, {6,89}, {6,90}, {6,91}, {51,142}, {6,92}, 
    {51,143}, {51,144}, {6,93}, {6,94}, {6,95}, {6,96}, {51,145}, {51,146}, 
    {6,97}, {51,147}, {51,148}, {51,149}, {6,98}, {6,99}, {6,100}, {6,101}, 
    {6,102}, {51,150}, {51,151}, {51,152}, {6,103}, {6,104}, {51,153}, 
    {6,105}, {6,106}, {6,107}, {51,154}, {6,108}, {6,109}, {6,110}, {6,111}, 
    {6,112}, {6,113}, {6,114}, {51,155}, {51,156}, {6,115}, {51,157}, {51,158}, 
    {51,159}, {6,116}, {6,117}, {6,118}, {6,119}, {6,120}, {6,121}, {51,160}, 
    {51,161}, {6,122}, {6,123}, {51,162}, {6,124}, {6,125}, {6,126}, {51,163}, 
    {6,127}, {6,128}, {6,129}, {6,130}, {6,131}, {6,132}, {6,133}, {51,164}, 
    {51,165}, {6,134}, {6,135}, {51,166}, {51,167}, {6,136}, {6,137}, {51,168}, 
    {6,138}, {6,139}, {6,140}, {51,169}, {6,141}, {6,142}, {6,143}, {51,170}, 
    {6,144}, {6,145}, {6,146}, {6,147}, {6,148}, {6,149}, {6,150}, {6,151}, 
    {6,152}, {6,153}, {6,154}, {6,155}, {6,156}, {6,157}, {6,158}, {6,159}, 
    {6,160}, {6,161}, {6,162}, {6,163}, {6,164}, {6,165}, {6,166}, {51,171}, 
    {51,172}, {6,167}, {6,168}, {51,173}, {6,169}, {6,170}, {6,171}, {51,174}, 
    {6,172}, {51,175}, {6,173}, {6,174}, {6,175}, {6,176}, {6,177}, {51,176}, 
    {51,177}, {6,178}, {51,178}, {6,179}, {51,179}, {6,180}, {6,181}, {6,182}, 
    {6,183}, {51,180}, {51,181}, {51,182}, {6,184}, {6,185}, {6,186}, {51,183}, 
    {6,187}, {6,188}, {6,189}, {51,184}, {6,190}, {7,1}, {7,2}, {7,3}, 
    {7,4}, {7,5}, {7,6}, {7,7}, {7,8}, {7,9}, {7,10}, {51,185}, {7,11}, 
    {7,12}, {7,13}, {7,14}, {7,15}, {7,16}, {7,17}, {7,18}, {7,19}, {7,20}, 
    {7,21}, {7,22}, {7,23}, {7,24}, {7,25}, {7,26}, {7,33}, {7,34}, {7,35}, 
    {7,36}, {7,37}, {7,38}, {7,39}, {7,40}, {7,41}, {7,42}, {7,43}, {7,44}, 
    {7,45}, {7,46}, {7,47}, {7,48}, {7,49}, {7,50}, {7,51}, {51,186}, {7,52}, 
    {7,53}, {7,54}, {51,187}, {7,55}, {7,56}, {7,57}, {51,188}, {7,58}, 
    {7,65}, {7,66}, {7,67}, {7,68}, {7,69}, {7,70}, {51,189}, {51,190}, 
    {7,71}, {52,97}, {7,72}, {7,73}, {7,74}, {7,75}, {7,76}, {7,77}, {7,78}, 
    {7,79}, {52,98}, {52,99}, {7,80}, {7,81}, {52,100}, {7,82}, {7,83}, 
    {7,84}, {52,101}, {7,85}, {7,86}, {7,87}, {7,88}, {7,89}, {7,90}, {7,91}, 
    {7,92}, {52,102}, {7,93}, {52,103}, {7,94}, {52,104}, {7,95}, {7,96}, 
    {7,97}, {7,98}, {7,99}, {7,100}, {52,105}, {52,106}, {7,101}, {7,102}, 
    {52,107}, {7,103}, {7,104}, {52,108}, {52,109}, {7,105}, {7,106}, {7,107}, 
    {7,108}, {7,109}, {7,110}, {7,111}, {52,110}, {52,111}, {7,112}, {52,112}, 
    {7,113}, {52,113}, {7,114}, {7,115}, {7,116}, {7,117}, {7,118}, {7,119}, 
    {52,114}, {7,120}, {7,121}, {7,122}, {7,123}, {7,124}, {7,125}, {7,126}, 
    {7,127}, {7,128}, {7,129}, {7,130}, {7,131}, {7,132}, {7,133}, {7,134}, 
    {7,135}, {7,136}, {7,137}, {7,138}, {52,115}, {7,139}, {7,140}, {7,141}, 
    {7,142}, {7,143}, {7,144}, {7,145}, {52,116}, {7,146}, {7,147}, {7,148}, 
    {7,149}, {7,150}, {7,151}, {7,152}, {7,153}, {7,154}, {7,155}, {7,156}, 
    {7,157}, {7,158}, {7,159}, {7,160}, {7,161}, {7,162}, {7,163}, {7,164}, 
    {7,165}, {7,166}, {7,167}, {7,168}, {7,169}, {7,170}, {7,171}, {7,172}, 
    {52,117}, {7,173}, {7,174}, {7,175}, {52,118}, {7,176}, {7,177}, {7,178}, 
    {52,119}, {7,179}, {7,180}, {7,181}, {7,182}, {7,183}, {7,184}, {7,185}, 
    {52,120}, {52,121}, {7,186}, {7,187}, {7,188}, {7,189}, {7,190}, {8,1}, 
    {8,2}, {8,3}, {8,4}, {8,5}, {52,122}, {52,123}, {8,6}, {8,7}, {8,8}, 
    {8,9}, {8,10}, {8,11}, {52,124}, {8,12}, {8,13}, {8,14}, {8,15}, {8,16}, 
    {8,17}, {8,18}, {52,125}, {52,126}, {8,19}, {8,20}, {8,21}, {52,127}, 
    {8,22}, {8,23}, {8,24}, {8,25}, {8,26}, {8,33}, {52,128}, {52,129}, 
    {8,34}, {8,35}, {52,130}, {8,36}, {8,37}, {8,38}, {52,131}, {52,132}, 
    {52,133}, {8,39}, {8,40}, {8,41}, {8,42}, {8,43}, {52,134}, {52,135}, 
    {8,44}, {52,136}, {8,45}, {52,137}, {52,138}, {8,46}, {8,47}, {8,48}, 
    {52,139}, {8,49}, {52,140}, {8,50}, {8,51}, {8,52}, {52,141}, {8,53}, 
    {8,54}, {8,55}, {52,142}, {8,56}, {8,57}, {8,58}, {8,65}, {8,66}, {8,67}, 
    {8,68}, {8,69}, {8,70}, {8,71}, {8,72}, {8,73}, {8,74}, {8,75}, {8,76}, 
    {8,77}, {8,78}, {8,79}, {8,80}, {52,143}, {52,144}, {8,81}, {8,82}, 
    {52,145}, {8,83}, {8,84}, {8,85}, {52,146}, {8,86}, {52,147}, {8,87}, 
    {8,88}, {8,89}, {8,90}, {8,91}, {52,148}, {52,149}, {8,92}, {52,150}, 
    {8,93}, {52,151}, {8,94}, {8,95}, {8,96}, {8,97}, {52,152}, {8,98}, 
    {52,153}, {52,154}, {52,155}, {8,99}, {52,156}, {8,100}, {8,101}, {52,157}, 
    {52,158}, {52,159}, {52,160}, {52,161}, {8,102}, {8,103}, {8,104}, 
    {52,162}, {52,163}, {52,164}, {8,105}, {52,165}, {52,166}, {52,167}, 
    {52,168}, {52,169}, {8,106}, {8,107}, {8,108}, {52,170}, {52,171}, 
    {52,172}, {8,109}, {8,110}, {52,173}, {8,111}, {8,112}, {8,113}, {52,174}, 
    {8,114}, {8,115}, {8,116}, {8,117}, {8,118}, {8,119}, {8,120}, {52,175}, 
    {52,176}, {8,121}, {52,177}, {52,178}, {52,179}, {8,122}, {8,123}, 
    {8,124}, {8,125}, {8,126}, {8,127}, {52,180}, {8,128}, {8,129}, {8,130}, 
    {8,131}, {8,132}, {8,133}, {8,134}, {8,135}, {8,136}, {8,137}, {8,138}, 
    {8,139}, {8,140}, {8,141}, {8,142}, {8,143}, {8,144}, {8,145}, {8,146}, 
    {8,147}, {8,148}, {8,149}, {8,150}, {8,151}, {8,152}, {8,153}, {8,154}, 
    {8,155}, {8,156}, {8,157}, {8,158}, {8,159}, {8,160}, {8,161}, {8,162}, 
    {8,163}, {8,164}, {8,165}, {8,166}, {8,167}, {8,168}, {8,169}, {8,170}, 
    {8,171}, {8,172}, {8,173}, {8,174}, {8,175}, {8,176}, {8,177}, {8,178}, 
    {8,179}, {8,180}, {8,181}, {8,182}, {52,181}, {52,182}, {52,183}, {8,183}, 
    {52,184}, {8,184}, {8,185}, {52,185}, {52,186}, {8,186}, {52,187}, 
    {52,188}, {8,187}, {8,188}, {8,189}, {8,190}, {52,189}, {52,190}, {9,1}, 
    {53,97}, {9,2}, {53,98}, {9,3}, {53,99}, {9,4}, {9,5}, {53,100}, {9,6}, 
    {53,101}, {53,102}, {9,7}, {9,8}, {53,103}, {9,9}, {9,10}, {9,11}, 
    {53,104}, {9,12}, {9,13}, {9,14}, {9,15}, {9,16}, {9,17}, {9,18}, {53,105}, 
    {53,106}, {9,19}, {53,107}, {53,108}, {53,109}, {9,20}, {9,21}, {9,22}, 
    {9,23}, {9,24}, {9,25}, {53,110}, {9,26}, {9,33}, {9,34}, {53,111}, 
    {9,35}, {9,36}, {9,37}, {53,112}, {9,38}, {9,39}, {9,40}, {9,41}, {9,42}, 
    {9,43}, {9,44}, {9,45}, {9,46}, {9,47}, {9,48}, {53,113}, {53,114}, 
    {9,49}, {9,50}, {9,51}, {9,52}, {9,53}, {9,54}, {53,115}, {9,55}, {9,56}, 
    {9,57}, {53,116}, {9,58}, {9,65}, {9,66}, {9,67}, {9,68}, {9,69}, {9,70}, 
    {9,71}, {9,72}, {9,73}, {9,74}, {9,75}, {9,76}, {9,77}, {9,78}, {9,79}, 
    {9,80}, {9,81}, {9,82}, {9,83}, {9,84}, {9,85}, {9,86}, {53,117}, {53,118}, 
    {9,87}, {9,88}, {53,119}, {9,89}, {9,90}, {53,120}, {53,121}, {9,91}, 
    {53,122}, {9,92}, {53,123}, {9,93}, {9,94}, {9,95}, {53,124}, {53,125}, 
    {9,96}, {53,126}, {9,97}, {53,127}, {9,98}, {53,128}, {9,99}, {53,129}, 
    {9,100}, {9,101}, {53,130}, {9,102}, {9,103}, {9,104}, {53,131}, {9,105}, 
    {9,106}, {9,107}, {53,132}, {9,108}, {9,109}, {9,110}, {9,111}, {9,112}, 
    {9,113}, {9,114}, {9,115}, {9,116}, {9,117}, {9,118}, {9,119}, {9,120}, 
    {9,121}, {9,122}, {9,123}, {9,124}, {9,125}, {9,126}, {53,133}, {9,127}, 
    {9,128}, {9,129}, {9,130}, {9,131}, {9,132}, {9,133}, {9,134}, {9,135}, 
    {9,136}, {9,137}, {9,138}, {9,139}, {9,140}, {9,141}, {9,142}, {9,143}, 
    {9,144}, {9,145}, {53,134}, {9,146}, {9,147}, {9,148}, {9,149}, {9,150}, 
    {9,151}, {9,152}, {53,135}, {9,153}, {9,154}, {9,155}, {53,136}, {9,156}, 
    {9,157}, {9,158}, {53,137}, {9,159}, {9,160}, {9,161}, {9,162}, {9,163}, 
    {9,164}, {9,165}, {53,138}, {53,139}, {9,166}, {53,140}, {9,167}, {9,168}, 
    {9,169}, {9,170}, {9,171}, {9,172}, {9,173}, {9,174}, {53,141}, {9,175}, 
    {9,176}, {9,177}, {9,178}, {9,179}, {9,180}, {9,181}, {9,182}, {9,183}, 
    {9,184}, {9,185}, {9,186}, {9,187}, {9,188}, {9,189}, {9,190}, {10,1}, 
    {10,2}, {10,3}, {10,4}, {10,5}, {10,6}, {10,7}, {10,8}, {10,9}, {10,10}, 
    {10,11}, {53,142}, {53,143}, {10,12}, {10,13}, {53,144}, {10,14}, {10,15}, 
    {10,16}, {53,145}, {10,17}, {10,18}, {10,19}, {10,20}, {10,21}, {10,22}, 
    {10,23}, {53,146}, {53,147}, {10,24}, {53,148}, {10,25}, {53,149}, 
    {10,26}, {10,33}, {10,34}, {10,35}, {10,36}, {10,37}, {53,150}, {10,38}, 
    {10,39}, {10,40}, {10,41}, {10,42}, {10,43}, {10,44}, {10,45}, {10,46}, 
    {10,47}, {10,48}, {10,49}, {10,50}, {10,51}, {10,52}, {10,53}, {10,54}, 
    {10,55}, {10,56}, {53,151}, {10,57}, {10,58}, {10,65}, {10,66}, {10,67}, 
    {10,68}, {10,69}, {53,152}, {10,70}, {10,71}, {10,72}, {10,73}, {10,74}, 
    {10,75}, {10,76}, {10,77}, {10,78}, {10,79}, {10,80}, {10,81}, {10,82}, 
    {10,83}, {10,84}, {10,85}, {10,86}, {10,87}, {10,88}, {10,89}, {53,153}, 
    {10,90}, {10,91}, {10,92}, {10,93}, {10,94}, {10,95}, {53,154}, {10,96}, 
    {10,97}, {10,98}, {53,155}, {10,99}, {10,100}, {10,101}, {53,156}, 
    {10,102}, {10,103}, {10,104}, {10,105}, {10,106}, {10,107}, {10,108}, 
    {10,109}, {53,157}, {10,110}, {53,158}, {10,111}, {53,159}, {10,112}, 
    {10,113}, {10,114}, {10,115}, {10,116}, {10,117}, {53,160}, {10,118}, 
    {10,119}, {10,120}, {53,161}, {10,121}, {10,122}, {10,123}, {53,162}, 
    {10,124}, {10,125}, {10,126}, {10,127}, {10,128}, {10,129}, {10,130}, 
    {53,163}, {10,131}, {10,132}, {10,133}, {10,134}, {53,164}, {10,135}, 
    {10,136}, {10,137}, {10,138}, {10,139}, {10,140}, {53,165}, {53,166}, 
    {10,141}, {10,142}, {53,167}, {10,143}, {10,144}, {53,168}, {53,169}, 
    {10,145}, {53,170}, {10,146}, {10,147}, {10,148}, {10,149}, {10,150}, 
    {53,171}, {53,172}, {10,151}, {53,173}, {10,152}, {53,174}, {10,153}, 
    {10,154}, {10,155}, {10,156}, {10,157}, {10,158}, {53,175}, {10,159}, 
    {10,160}, {10,161}, {10,162}, {10,163}, {10,164}, {10,165}, {10,166}, 
    {10,167}, {10,168}, {10,169}, {10,170}, {10,171}, {10,172}, {10,173}, 
    {10,174}, {10,175}, {10,176}, {10,177}, {10,178}, {10,179}, {10,180}, 
    {10,181}, {10,182}, {10,183}, {10,184}, {10,185}, {53,176}, {53,177}, 
    {10,186}, {10,187}, {53,178}, {10,188}, {10,189}, {53,179}, {53,180}, 
    {10,190}, {11,1}, {11,2}, {11,3}, {11,4}, {11,5}, {11,6}, {53,181}, 
    {53,182}, {11,7}, {53,183}, {53,184}, {53,185}, {53,186}, {11,8}, {11,9}, 
    {11,10}, {11,11}, {11,12}, {53,187}, {53,188}, {11,13}, {11,14}, {53,189}, 
    {11,15}, {11,16}, {11,17}, {53,190}, {11,18}, {11,19}, {11,20}, {11,21}, 
    {11,22}, {11,23}, {11,24}, {54,97}, {54,98}, {11,25}, {54,99}, {54,100}, 
    {54,101}, {11,26}, {11,33}, {11,34}, {11,35}, {11,36}, {54,102}, {54,103}, 
    {54,104}, {11,37}, {11,38}, {54,105}, {11,39}, {11,40}, {11,41}, {54,106}, 
    {11,42}, {11,43}, {11,44}, {11,45}, {11,46}, {11,47}, {11,48}, {54,107}, 
    {54,108}, {11,49}, {54,109}, {54,110}, {54,111}, {11,50}, {11,51}, 
    {11,52}, {11,53}, {11,54}, {11,55}, {11,56}, {11,57}, {11,58}, {11,65}, 
    {11,66}, {11,67}, {11,68}, {11,69}, {11,70}, {11,71}, {11,72}, {11,73}, 
    {11,74}, {11,75}, {11,76}, {11,77}, {11,78}, {11,79}, {11,80}, {11,81}, 
    {11,82}, {11,83}, {11,84}, {11,85}, {11,86}, {11,87}, {11,88}, {11,89}, 
    {11,90}, {11,91}, {11,92}, {11,93}, {11,94}, {11,95}, {11,96}, {11,97}, 
    {11,98}, {11,99}, {11,100}, {11,101}, {11,102}, {11,103}, {11,104}, 
    {11,105}, {11,106}, {11,107}, {11,108}, {11,109}, {11,110}, {11,111}, 
    {11,112}, {11,113}, {11,114}, {11,115}, {11,116}, {11,117}, {54,112}, 
    {54,113}, {11,118}, {11,119}, {54,114}, {11,120}, {11,121}, {11,122}, 
    {54,115}, {11,123}, {54,116}, {54,117}, {11,124}, {11,125}, {11,126}, 
    {11,127}, {54,118}, {54,119}, {11,128}, {54,120}, {54,121}, {54,122}, 
    {11,129}, {11,130}, {11,131}, {11,132}, {11,133}, {54,123}, {54,124}, 
    {54,125}, {11,134}, {11,135}, {54,126}, {11,136}, {11,137}, {11,138}, 
    {54,127}, {11,139}, {11,140}, {11,141}, {11,142}, {11,143}, {11,144}, 
    {11,145}, {54,128}, {54,129}, {11,146}, {54,130}, {54,131}, {54,132}, 
    {11,147}, {11,148}, {11,149}, {11,150}, {11,151}, {11,152}, {54,133}, 
    {11,153}, {11,154}, {11,155}, {11,156}, {11,157}, {11,158}, {11,159}, 
    {11,160}, {11,161}, {11,162}, {11,163}, {11,164}, {11,165}, {11,166}, 
    {11,167}, {11,168}, {11,169}, {11,170}, {11,171}, {54,134}, {11,172}, 
    {11,173}, {11,174}, {11,175}, {11,176}, {11,177}, {11,178}, {11,179}, 
    {11,180}, {11,181}, {11,182}, {11,183}, {11,184}, {11,185}, {11,186}, 
    {11,187}, {11,188}, {11,189}, {11,190}, {12,1}, {12,2}, {12,3}, {12,4}, 
    {12,5}, {12,6}, {12,7}, {12,8}, {12,9}, {12,10}, {12,11}, {12,12}, 
    {12,13}, {12,14}, {12,15}, {12,16}, {54,135}, {54,136}, {12,17}, {12,18}, 
    {54,137}, {12,19}, {12,20}, {12,21}, {54,138}, {12,22}, {12,23}, {12,24}, 
    {12,25}, {12,26}, {12,33}, {12,34}, {12,35}, {12,36}, {12,37}, {12,38}, 
    {12,39}, {54,139}, {12,40}, {12,41}, {12,42}, {12,43}, {12,44}, {12,45}, 
    {54,140}, {12,46}, {12,47}, {12,48}, {12,49}, {12,50}, {12,51}, {12,52}, 
    {54,141}, {12,53}, {12,54}, {12,55}, {12,56}, {12,57}, {12,58}, {12,65}, 
    {12,66}, {12,67}, {12,68}, {12,69}, {12,70}, {12,71}, {12,72}, {12,73}, 
    {12,74}, {12,75}, {12,76}, {12,77}, {54,142}, {12,78}, {12,79}, {12,80}, 
    {12,81}, {12,82}, {12,83}, {12,84}, {12,85}, {12,86}, {12,87}, {12,88}, 
    {12,89}, {12,90}, {12,91}, {12,92}, {12,93}, {12,94}, {12,95}, {12,96}, 
    {12,97}, {12,98}, {12,99}, {12,100}, {12,101}, {12,102}, {12,103}, 
    {12,104}, {54,143}, {12,105}, {12,106}, {12,107}, {54,144}, {12,108}, 
    {12,109}, {12,110}, {12,111}, {12,112}, {12,113}, {12,114}, {12,115}, 
    {12,116}, {12,117}, {12,118}, {12,119}, {12,120}, {12,121}, {12,122}, 
    {12,123}, {12,124}, {12,125}, {12,126}, {12,127}, {12,128}, {12,129}, 
    {12,130}, {12,131}, {12,132}, {12,133}, {12,134}, {12,135}, {12,136}, 
    {12,137}, {12,138}, {12,139}, {12,140}, {12,141}, {12,142}, {12,143}, 
    {12,144}, {12,145}, {12,146}, {12,147}, {12,148}, {12,149}, {12,150}, 
    {12,151}, {12,152}, {12,153}, {12,154}, {12,155}, {12,156}, {12,157}, 
    {12,158}, {54,145}, {54,146}, {12,159}, {12,160}, {54,147}, {12,161}, 
    {12,162}, {12,163}, {54,148}, {12,164}, {12,165}, {12,166}, {12,167}, 
    {12,168}, {12,169}, {54,149}, {54,150}, {12,170}, {12,171}, {12,172}, 
    {12,173}, {54,151}, {12,174}, {12,175}, {12,176}, {12,177}, {12,178}, 
    {12,179}, {12,180}, {12,181}, {12,182}, {12,183}, {12,184}, {12,185}, 
    {12,186}, {12,187}, {12,188}, {12,189}, {12,190}, {13,1}, {13,2}, {13,3}, 
    {13,4}, {13,5}, {13,6}, {13,7}, {13,8}, {13,9}, {13,10}, {13,11}, {13,12}, 
    {13,13}, {13,14}, {13,15}, {13,16}, {13,17}, {54,152}, {13,18}, {13,19}, 
    {13,20}, {13,21}, {13,22}, {13,23}, {13,24}, {13,25}, {13,26}, {13,33}, 
    {13,34}, {13,35}, {13,36}, {13,37}, {13,38}, {13,39}, {13,40}, {13,41}, 
    {13,42}, {13,43}, {13,44}, {13,45}, {13,46}, {13,47}, {13,48}, {13,49}, 
    {13,50}, {54,153}, {13,51}, {13,52}, {13,53}, {54,154}, {13,54}, {13,55}, 
    {13,56}, {54,155}, {13,57}, {13,58}, {13,65}, {13,66}, {13,67}, {13,68}, 
    {13,69}, {54,156}, {54,157}, {13,70}, {13,71}, {13,72}, {54,158}, {13,73}, 
    {13,74}, {13,75}, {13,76}, {13,77}, {13,78}, {13,79}, {13,80}, {13,81}, 
    {13,82}, {13,83}, {13,84}, {13,85}, {13,86}, {13,87}, {13,88}, {13,89}, 
    {13,90}, {13,91}, {13,92}, {13,93}, {13,94}, {13,95}, {13,96}, {13,97}, 
    {13,98}, {13,99}, {13,100}, {13,101}, {13,102}, {13,103}, {13,104}, 
    {13,105}, {13,106}, {54,159}, {54,160}, {13,107}, {13,108}, {54,161}, 
    {13,109}, {13,110}, {54,162}, {54,163}, {13,111}, {13,112}, {13,113}, 
    {13,114}, {13,115}, {13,116}, {13,117}, {54,164}, {54,165}, {13,118}, 
    {54,166}, {13,119}, {13,120}, {13,121}, {13,122}, {13,123}, {13,124}, 
    {13,125}, {13,126}, {54,167}, {13,127}, {13,128}, {13,129}, {54,168}, 
    {13,130}, {13,131}, {13,132}, {54,169}, {13,133}, {13,134}, {13,135}, 
    {13,136}, {13,137}, {13,138}, {13,139}, {54,170}, {54,171}, {13,140}, 
    {13,141}, {13,142}, {13,143}, {13,144}, {13,145}, {13,146}, {13,147}, 
    {13,148}, {13,149}, {54,172}, {13,150}, {13,151}, {13,152}, {54,173}, 
    {13,153}, {13,154}, {13,155}, {54,174}, {13,156}, {13,157}, {13,158}, 
    {13,159}, {13,160}, {13,161}, {13,162}, {54,175}, {54,176}, {13,163}, 
    {54,177}, {13,164}, {54,178}, {13,165}, {13,166}, {13,167}, {13,168}, 
    {13,169}, {13,170}, {54,179}, {54,180}, {13,171}, {13,172}, {54,181}, 
    {13,173}, {13,174}, {13,175}, {54,182}, {13,176}, {13,177}, {13,178}, 
    {13,179}, {13,180}, {13,181}, {13,182}, {54,183}, {54,184}, {13,183}, 
    {54,185}, {54,186}, {54,187}, {54,188}, {13,184}, {13,185}, {13,186}, 
    {54,189}, {54,190}, {55,97}, {55,98}, {13,187}, {13,188}, {55,99}, 
    {13,189}, {13,190}, {14,1}, {55,100}, {14,2}, {14,3}, {14,4}, {14,5}, 
    {14,6}, {14,7}, {14,8}, {55,101}, {55,102}, {14,9}, {55,103}, {55,104}, 
    {55,105}, {14,10}, {14,11}, {14,12}, {14,13}, {14,14}, {14,15}, {55,106}, 
    {55,107}, {14,16}, {14,17}, {55,108}, {14,18}, {14,19}, {14,20}, {14,21}, 
    {14,22}, {14,23}, {14,24}, {14,25}, {14,26}, {14,33}, {14,34}, {14,35}, 
    {14,36}, {14,37}, {55,109}, {14,38}, {55,110}, {14,39}, {14,40}, {14,41}, 
    {14,42}, {14,43}, {14,44}, {14,45}, {14,46}, {14,47}, {14,48}, {14,49}, 
    {14,50}, {14,51}, {14,52}, {14,53}, {14,54}, {14,55}, {14,56}, {14,57}, 
    {14,58}, {14,65}, {14,66}, {14,67}, {14,68}, {14,69}, {14,70}, {14,71}, 
    {14,72}, {14,73}, {14,74}, {14,75}, {14,76}, {14,77}, {14,78}, {55,111}, 
    {55,112}, {14,79}, {14,80}, {55,113}, {14,81}, {14,82}, {14,83}, {55,114}, 
    {14,84}, {14,85}, {14,86}, {14,87}, {14,88}, {14,89}, {14,90}, {55,115}, 
    {55,116}, {14,91}, {55,117}, {55,118}, {55,119}, {14,92}, {14,93}, 
    {14,94}, {14,95}, {14,96}, {55,120}, {55,121}, {55,122}, {14,97}, {14,98}, 
    {55,123}, {14,99}, {14,100}, {14,101}, {55,124}, {14,102}, {14,103}, 
    {14,104}, {14,105}, {14,106}, {14,107}, {14,108}, {55,125}, {55,126}, 
    {14,109}, {55,127}, {14,110}, {55,128}, {14,111}, {14,112}, {14,113}, 
    {14,114}, {14,115}, {14,116}, {55,129}, {55,130}, {14,117}, {14,118}, 
    {55,131}, {14,119}, {14,120}, {14,121}, {55,132}, {14,122}, {14,123}, 
    {14,124}, {14,125}, {14,126}, {14,127}, {14,128}, {55,133}, {55,134}, 
    {14,129}, {55,135}, {55,136}, {55,137}, {14,130}, {14,131}, {14,132}, 
    {14,133}, {14,134}, {14,135}, {55,138}, {14,136}, {14,137}, {14,138}, 
    {55,139}, {14,139}, {14,140}, {14,141}, {14,142}, {14,143}, {14,144}, 
    {14,145}, {14,146}, {14,147}, {14,148}, {14,149}, {14,150}, {55,140}, 
    {14,151}, {55,141}, {14,152}, {14,153}, {14,154}, {14,155}, {14,156}, 
    {14,157}, {14,158}, {14,159}, {55,142}, {55,143}, {14,160}, {14,161}, 
    {55,144}, {14,162}, {14,163}, {14,164}, {55,145}, {14,165}, {14,166}, 
    {14,167}, {14,168}, {14,169}, {14,170}, {14,171}, {55,146}, {55,147}, 
    {14,172}, {55,148}, {14,173}, {55,149}, {14,174}, {14,175}, {14,176}, 
    {14,177}, {14,178}, {14,179}, {55,150}, {14,180}, {14,181}, {14,182}, 
    {55,151}, {14,183}, {14,184}, {14,185}, {14,186}, {14,187}, {14,188}, 
    {14,189}, {14,190}, {15,1}, {15,2}, {15,3}, {15,4}, {15,5}, {15,6}, 
    {15,7}, {15,8}, {55,152}, {15,9}, {15,10}, {15,11}, {15,12}, {15,13}, 
    {15,14}, {15,15}, {15,16}, {15,17}, {15,18}, {15,19}, {15,20}, {15,21}, 
    {15,22}, {15,23}, {15,24}, {15,25}, {15,26}, {15,33}, {15,34}, {15,35}, 
    {15,36}, {15,37}, {15,38}, {15,39}, {15,40}, {55,153}, {15,41}, {15,42}, 
    {15,43}, {15,44}, {15,45}, {15,46}, {15,47}, {55,154}, {15,48}, {15,49}, 
    {15,50}, {55,155}, {15,51}, {15,52}, {15,53}, {55,156}, {15,54}, {15,55}, 
    {15,56}, {15,57}, {15,58}, {15,65}, {15,66}, {55,157}, {55,158}, {15,67}, 
    {55,159}, {15,68}, {55,160}, {15,69}, {15,70}, {15,71}, {15,72}, {15,73}, 
    {15,74}, {55,161}, {15,75}, {15,76}, {15,77}, {55,162}, {15,78}, {15,79}, 
    {15,80}, {55,163}, {15,81}, {15,82}, {15,83}, {15,84}, {15,85}, {15,86}, 
    {15,87}, {15,88}, {55,164}, {15,89}, {55,165}, {15,90}, {55,166}, {15,91}, 
    {15,92}, {15,93}, {15,94}, {15,95}, {15,96}, {55,167}, {55,168}, {15,97}, 
    {15,98}, {55,169}, {15,99}, {15,100}, {15,101}, {55,170}, {15,102}, 
    {15,103}, {15,104}, {15,105}, {15,106}, {15,107}, {15,108}, {55,171}, 
    {55,172}, {15,109}, {55,173}, {15,110}, {55,174}, {15,111}, {15,112}, 
    {15,113}, {15,114}, {15,115}, {15,116}, {55,175}, {15,117}, {15,118}, 
    {15,119}, {15,120}, {15,121}, {15,122}, {15,123}, {15,124}, {15,125}, 
    {15,126}, {15,127}, {15,128}, {15,129}, {15,130}, {15,131}, {15,132}, 
    {15,133}, {15,134}, {15,135}, {55,176}, {15,136}, {15,137}, {15,138}, 
    {15,139}, {15,140}, {15,141}, {15,142}, {55,177}, {15,143}, {15,144}, 
    {15,145}, {15,146}, {15,147}, {15,148}, {15,149}, {15,150}, {15,151}, 
    {15,152}, {15,153}, {15,154}, {15,155}, {15,156}, {15,157}, {15,158}, 
    {15,159}, {15,160}, {15,161}, {15,162}, {15,163}, {15,164}, {15,165}, 
    {15,166}, {15,167}, {15,168}, {15,169}, {55,178}, {55,179}, {15,170}, 
    {15,171}, {55,180}, {15,172}, {15,173}, {15,174}, {55,181}, {15,175}, 
    {15,176}, {15,177}, {15,178}, {15,179}, {15,180}, {15,181}, {55,182}, 
    {15,182}, {15,183}, {55,183}, {15,184}, {55,184}, {15,185}, {15,186}, 
    {15,187}, {15,188}, {15,189}, {15,190}, {55,185}, {55,186}, {16,1}, 
    {16,2}, {55,187}, {16,3}, {16,4}, {16,5}, {55,188}, {16,6}, {16,7}, 
    {16,8}, {16,9}, {16,10}, {16,11}, {16,12}, {55,189}, {55,190}, {16,13}, 
    {56,97}, {16,14}, {56,98}, {16,15}, {16,16}, {16,17}, {16,18}, {16,19}, 
    {16,20}, {56,99}, {56,100}, {16,21}, {16,22}, {56,101}, {16,23}, {16,24}, 
    {16,25}, {56,102}, {16,26}, {16,33}, {16,34}, {16,35}, {16,36}, {16,37}, 
    {16,38}, {56,103}, {56,104}, {16,39}, {56,105}, {16,40}, {56,106}, 
    {56,107}, {16,41}, {16,42}, {56,108}, {56,109}, {16,43}, {16,44}, {16,45}, 
    {16,46}, {16,47}, {16,48}, {16,49}, {16,50}, {16,51}, {16,52}, {16,53}, 
    {16,54}, {16,55}, {16,56}, {16,57}, {16,58}, {16,65}, {16,66}, {16,67}, 
    {16,68}, {16,69}, {16,70}, {16,71}, {16,72}, {16,73}, {16,74}, {16,75}, 
    {16,76}, {16,77}, {56,110}, {56,111}, {16,78}, {16,79}, {56,112}, {16,80}, 
    {16,81}, {16,82}, {56,113}, {16,83}, {16,84}, {16,85}, {16,86}, {16,87}, 
    {16,88}, {16,89}, {56,114}, {56,115}, {16,90}, {56,116}, {16,91}, {56,117}, 
    {16,92}, {16,93}, {16,94}, {16,95}, {16,96}, {16,97}, {56,118}, {56,119}, 
    {16,98}, {16,99}, {56,120}, {16,100}, {56,121}, {56,122}, {56,123}, 
    {56,124}, {56,125}, {16,101}, {16,102}, {16,103}, {16,104}, {16,105}, 
    {56,126}, {56,127}, {16,106}, {56,128}, {16,107}, {56,129}, {56,130}, 
    {16,108}, {16,109}, {56,131}, {16,110}, {56,132}, {56,133}, {56,134}, 
    {16,111}, {16,112}, {56,135}, {16,113}, {16,114}, {16,115}, {56,136}, 
    {16,116}, {16,117}, {16,118}, {16,119}, {16,120}, {16,121}, {16,122}, 
    {56,137}, {56,138}, {16,123}, {56,139}, {56,140}, {56,141}, {56,142}, 
    {16,124}, {16,125}, {16,126}, {16,127}, {16,128}, {56,143}, {56,144}, 
    {16,129}, {16,130}, {16,131}, {16,132}, {16,133}, {16,134}, {56,145}, 
    {16,135}, {16,136}, {16,137}, {16,138}, {16,139}, {16,140}, {16,141}, 
    {16,142}, {16,143}, {16,144}, {16,145}, {16,146}, {56,146}, {16,147}, 
    {16,148}, {16,149}, {16,150}, {16,151}, {16,152}, {16,153}, {16,154}, 
    {16,155}, {16,156}, {16,157}, {16,158}, {16,159}, {16,160}, {16,161}, 
    {16,162}, {16,163}, {16,164}, {16,165}, {16,166}, {16,167}, {16,168}, 
    {16,169}, {16,170}, {16,171}, {16,172}, {16,173}, {16,174}, {16,175}, 
    {16,176}, {16,177}, {16,178}, {16,179}, {16,180}, {56,147}, {56,148}, 
    {16,181}, {16,182}, {56,149}, {16,183}, {16,184}, {16,185}, {56,150}, 
    {16,186}, {56,151}, {16,187}, {16,188}, {16,189}, {16,190}, {17,1}, 
    {56,152}, {56,153}, {17,2}, {56,154}, {17,3}, {56,155}, {56,156}, {17,4}, 
    {17,5}, {17,6}, {17,7}, {56,157}, {56,158}, {56,159}, {17,8}, {17,9}, 
    {56,160}, {17,10}, {17,11}, {17,12}, {56,161}, {17,13}, {17,14}, {17,15}, 
    {17,16}, {17,17}, {17,18}, {17,19}, {56,162}, {56,163}, {17,20}, {56,164}, 
    {56,165}, {56,166}, {17,21}, {17,22}, {17,23}, {17,24}, {17,25}, {17,26}, 
    {56,167}, {56,168}, {17,33}, {17,34}, {56,169}, {17,35}, {17,36}, {17,37}, 
    {56,170}, {17,38}, {17,39}, {17,40}, {17,41}, {17,42}, {17,43}, {17,44}, 
    {17,45}, {17,46}, {17,47}, {56,171}, {56,172}, {56,173}, {17,48}, {56,174}, 
    {17,49}, {17,50}, {17,51}, {17,52}, {56,175}, {17,53}, {17,54}, {17,55}, 
    {17,56}, {17,57}, {17,58}, {17,65}, {17,66}, {17,67}, {17,68}, {17,69}, 
    {17,70}, {17,71}, {17,72}, {17,73}, {17,74}, {17,75}, {17,76}, {17,77}, 
    {17,78}, {17,79}, {17,80}, {17,81}, {17,82}, {17,83}, {17,84}, {17,85}, 
    {56,176}, {56,177}, {17,86}, {56,178}, {56,179}, {17,87}, {17,88}, 
    {17,89}, {56,180}, {17,90}, {56,181}, {17,91}, {17,92}, {17,93}, {17,94}, 
    {17,95}, {56,182}, {56,183}, {17,96}, {56,184}, {17,97}, {56,185}, 
    {17,98}, {17,99}, {17,100}, {17,101}, {17,102}, {17,103}, {56,186}, 
    {17,104}, {17,105}, {17,106}, {56,187}, {17,107}, {17,108}, {17,109}, 
    {17,110}, {17,111}, {17,112}, {17,113}, {17,114}, {17,115}, {17,116}, 
    {17,117}, {17,118}, {17,119}, {17,120}, {17,121}, {56,188}, {56,189}, 
    {17,122}, {17,123}, {17,124}, {17,125}, {17,126}, {17,127}, {17,128}, 
    {17,129}, {17,130}, {17,131}, {17,132}, {17,133}, {17,134}, {17,135}, 
    {17,136}, {17,137}, {17,138}, {17,139}, {17,140}, {17,141}, {17,142}, 
    {17,143}, {17,144}, {17,145}, {17,146}, {17,147}, {17,148}, {17,149}, 
    {17,150}, {17,151}, {17,152}, {17,153}, {17,154}, {17,155}, {56,190}, 
    {17,156}, {17,157}, {17,158}, {57,97}, {17,159}, {17,160}, {17,161}, 
    {57,98}, {17,162}, {17,163}, {17,164}, {17,165}, {17,166}, {17,167}, 
    {17,168}, {17,169}, {57,99}, {17,170}, {57,100}, {17,171}, {57,101}, 
    {17,172}, {17,173}, {17,174}, {17,175}, {17,176}, {17,177}, {57,102}, 
    {17,178}, {17,179}, {17,180}, {57,103}, {17,181}, {17,182}, {17,183}, 
    {57,104}, {17,184}, {17,185}, {17,186}, {17,187}, {17,188}, {17,189}, 
    {17,190}, {18,1}, {57,105}, {18,2}, {57,106}, {18,3}, {18,4}, {18,5}, 
    {18,6}, {18,7}, {18,8}, {18,9}, {18,10}, {57,107}, {57,108}, {57,109}, 
    {18,11}, {57,110}, {18,12}, {18,13}, {57,111}, {57,112}, {57,113}, 
    {57,114}, {18,14}, {18,15}, {18,16}, {18,17}, {18,18}, {57,115}, {57,116}, 
    {18,19}, {57,117}, {18,20}, {57,118}, {18,21}, {18,22}, {18,23}, {57,119}, 
    {18,24}, {57,120}, {57,121}, {18,25}, {18,26}, {18,33}, {57,122}, {18,34}, 
    {18,35}, {18,36}, {57,123}, {18,37}, {18,38}, {18,39}, {18,40}, {18,41}, 
    {18,42}, {18,43}, {18,44}, {57,124}, {18,45}, {57,125}, {18,46}, {18,47}, 
    {18,48}, {18,49}, {18,50}, {18,51}, {18,52}, {18,53}, {57,126}, {18,54}, 
    {18,55}, {18,56}, {18,57}, {18,58}, {18,65}, {18,66}, {18,67}, {18,68}, 
    {18,69}, {18,70}, {18,71}, {18,72}, {18,73}, {18,74}, {18,75}, {18,76}, 
    {18,77}, {18,78}, {18,79}, {18,80}, {18,81}, {18,82}, {18,83}, {18,84}, 
    {18,85}, {18,86}, {57,127}, {18,87}, {18,88}, {18,89}, {57,128}, {18,90}, 
    {18,91}, {18,92}, {57,129}, {18,93}, {18,94}, {18,95}, {18,96}, {18,97}, 
    {18,98}, {18,99}, {18,100}, {18,101}, {18,102}, {18,103}, {18,104}, 
    {18,105}, {18,106}, {18,107}, {18,108}, {18,109}, {18,110}, {18,111}, 
    {57,130}, {18,112}, {18,113}, {18,114}, {57,131}, {18,115}, {18,116}, 
    {18,117}, {57,132}, {18,118}, {18,119}, {18,120}, {18,121}, {18,122}, 
    {18,123}, {18,124}, {57,133}, {18,125}, {18,126}, {57,134}, {18,127}, 
    {18,128}, {18,129}, {18,130}, {18,131}, {18,132}, {18,133}, {18,134}, 
    {57,135}, {18,135}, {18,136}, {18,137}, {57,136}, {18,138}, {18,139}, 
    {18,140}, {57,137}, {18,141}, {18,142}, {18,143}, {18,144}, {18,145}, 
    {18,146}, {18,147}, {57,138}, {18,148}, {18,149}, {57,139}, {18,150}, 
    {18,151}, {18,152}, {18,153}, {18,154}, {18,155}, {18,156}, {18,157}, 
    {18,158}, {18,159}, {18,160}, {18,161}, {18,162}, {18,163}, {18,164}, 
    {18,165}, {18,166}, {18,167}, {18,168}, {18,169}, {18,170}, {18,171}, 
    {18,172}, {18,173}, {18,174}, {18,175}, {18,176}, {18,177}, {18,178}, 
    {18,179}, {18,180}, {18,181}, {18,182}, {18,183}, {18,184}, {18,185}, 
    {57,140}, {57,141}, {18,186}, {18,187}, {57,142}, {18,188}, {18,189}, 
    {57,143}, {57,144}, {18,190}, {57,145}, {19,1}, {19,2}, {19,3}, {19,4}, 
    {19,5}, {57,146}, {57,147}, {19,6}, {57,148}, {57,149}, {57,150}, {19,7}, 
    {57,151}, {19,8}, {57,152}, {19,9}, {19,10}, {57,153}, {57,154}, {57,155}, 
    {57,156}, {57,157}, {19,11}, {19,12}, {57,158}, {57,159}, {57,160}, 
    {57,161}, {57,162}, {19,13}, {19,14}, {19,15}, {19,16}, {57,163}, {57,164}, 
    {19,17}, {57,165}, {19,18}, {57,166}, {19,19}, {19,20}, {19,21}, {57,167}, 
    {19,22}, {19,23}, {57,168}, {57,169}, {19,24}, {19,25}, {57,170}, {19,26}, 
    {19,33}, {19,34}, {57,171}, {19,35}, {19,36}, {19,37}, {19,38}, {19,39}, 
    {19,40}, {19,41}, {57,172}, {57,173}, {19,42}, {57,174}, {57,175}, 
    {57,176}, {19,43}, {19,44}, {19,45}, {57,177}, {19,46}, {19,47}, {57,178}, 
    {57,179}, {19,48}, {19,49}, {57,180}, {19,50}, {19,51}, {19,52}, {19,53}, 
    {19,54}, {19,55}, {19,56}, {19,57}, {19,58}, {19,65}, {19,66}, {19,67}, 
    {57,181}, {19,68}, {19,69}, {19,70}, {19,71}, {19,72}, {19,73}, {19,74}, 
    {19,75}, {19,76}, {19,77}, {19,78}, {19,79}, {19,80}, {19,81}, {19,82}, 
    {19,83}, {19,84}, {19,85}, {19,86}, {19,87}, {19,88}, {19,89}, {19,90}, 
    {19,91}, {19,92}, {19,93}, {19,94}, {19,95}, {19,96}, {19,97}, {19,98}, 
    {19,99}, {19,100}, {19,101}, {19,102}, {19,103}, {19,104}, {19,105}, 
    {57,182}, {57,183}, {19,106}, {19,107}, {57,184}, {19,108}, {19,109}, 
    {57,185}, {57,186}, {19,110}, {57,187}, {19,111}, {19,112}, {19,113}, 
    {19,114}, {19,115}, {57,188}, {57,189}, {19,116}, {57,190}, {19,117}, 
    {58,97}, {58,98}, {19,118}, {19,119}, {19,120}, {19,121}, {19,122}, 
    {58,99}, {58,100}, {19,123}, {19,124}, {58,101}, {19,125}, {19,126}, 
    {58,102}, {58,103}, {19,127}, {19,128}, {19,129}, {19,130}, {19,131}, 
    {19,132}, {19,133}, {58,104}, {58,105}, {19,134}, {58,106}, {58,107}, 
    {58,108}, {19,135}, {19,136}, {19,137}, {19,138}, {19,139}, {19,140}, 
    {58,109}, {58,110}, {19,141}, {19,142}, {58,111}, {19,143}, {19,144}, 
    {19,145}, {58,112}, {19,146}, {19,147}, {19,148}, {19,149}, {19,150}, 
    {19,151}, {19,152}, {19,153}, {58,113}, {19,154}, {58,114}, {58,115}, 
    {58,116}, {19,155}, {19,156}, {19,157}, {58,117}, {19,158}, {19,159}, 
    {58,118}, {19,160}, {19,161}, {19,162}, {58,119}, {19,163}, {19,164}, 
    {19,165}, {19,166}, {19,167}, {19,168}, {19,169}, {19,170}, {19,171}, 
    {19,172}, {19,173}, {19,174}, {19,175}, {19,176}, {19,177}, {19,178}, 
    {19,179}, {19,180}, {19,181}, {19,182}, {19,183}, {19,184}, {19,185}, 
    {58,120}, {58,121}, {58,122}, {19,186}, {58,123}, {19,187}, {19,188}, 
    {19,189}, {58,124}, {19,190}, {20,1}, {20,2}, {20,3}, {20,4}, {20,5}, 
    {20,6}, {58,125}, {58,126}, {20,7}, {58,127}, {20,8}, {58,128}, {20,9}, 
    {20,10}, {20,11}, {20,12}, {20,13}, {20,14}, {58,129}, {20,15}, {20,16}, 
    {20,17}, {58,130}, {20,18}, {20,19}, {20,20}, {20,21}, {20,22}, {20,23}, 
    {20,24}, {20,25}, {20,26}, {20,33}, {20,34}, {20,35}, {20,36}, {20,37}, 
    {20,38}, {58,131}, {20,39}, {20,40}, {20,41}, {20,42}, {20,43}, {20,44}, 
    {20,45}, {58,132}, {20,46}, {20,47}, {20,48}, {20,49}, {20,50}, {20,51}, 
    {20,52}, {20,53}, {20,54}, {20,55}, {20,56}, {20,57}, {20,58}, {20,65}, 
    {20,66}, {20,67}, {20,68}, {20,69}, {20,70}, {58,133}, {20,71}, {20,72}, 
    {20,73}, {20,74}, {20,75}, {20,76}, {20,77}, {58,134}, {58,135}, {20,78}, 
    {20,79}, {58,136}, {20,80}, {20,81}, {20,82}, {58,137}, {20,83}, {20,84}, 
    {20,85}, {20,86}, {20,87}, {20,88}, {20,89}, {58,138}, {58,139}, {20,90}, 
    {20,91}, {20,92}, {20,93}, {20,94}, {20,95}, {20,96}, {20,97}, {20,98}, 
    {20,99}, {58,140}, {20,100}, {20,101}, {20,102}, {58,141}, {20,103}, 
    {20,104}, {20,105}, {20,106}, {20,107}, {20,108}, {20,109}, {20,110}, 
    {20,111}, {20,112}, {20,113}, {20,114}, {20,115}, {20,116}, {20,117}, 
    {20,118}, {20,119}, {20,120}, {20,121}, {20,122}, {20,123}, {20,124}, 
    {20,125}, {58,142}, {58,143}, {20,126}, {20,127}, {58,144}, {20,128}, 
    {20,129}, {58,145}, {58,146}, {58,147}, {58,148}, {20,130}, {20,131}, 
    {20,132}, {20,133}, {20,134}, {58,149}, {58,150}, {20,135}, {58,151}, 
    {20,136}, {58,152}, {20,137}, {20,138}, {20,139}, {58,153}, {58,154}, 
    {20,140}, {58,155}, {20,141}, {20,142}, {20,143}, {20,144}, {20,145}, 
    {20,146}, {20,147}, {58,156}, {20,148}, {20,149}, {20,150}, {20,151}, 
    {20,152}, {20,153}, {20,154}, {20,155}, {20,156}, {20,157}, {20,158}, 
    {58,157}, {20,159}, {20,160}, {20,161}, {20,162}, {20,163}, {20,164}, 
    {20,165}, {58,158}, {20,166}, {20,167}, {20,168}, {20,169}, {20,170}, 
    {20,171}, {20,172}, {20,173}, {20,174}, {20,175}, {20,176}, {20,177}, 
    {20,178}, {20,179}, {20,180}, {20,181}, {20,182}, {20,183}, {20,184}, 
    {20,185}, {20,186}, {20,187}, {20,188}, {20,189}, {20,190}, {21,1}, 
    {21,2}, {58,159}, {58,160}, {21,3}, {21,4}, {58,161}, {21,5}, {21,6}, 
    {21,7}, {58,162}, {21,8}, {21,9}, {21,10}, {21,11}, {21,12}, {21,13}, 
    {21,14}, {21,15}, {21,16}, {21,17}, {21,18}, {21,19}, {58,163}, {21,20}, 
    {21,21}, {21,22}, {21,23}, {21,24}, {21,25}, {58,164}, {21,26}, {21,33}, 
    {21,34}, {58,165}, {21,35}, {21,36}, {21,37}, {58,166}, {21,38}, {21,39}, 
    {21,40}, {21,41}, {21,42}, {21,43}, {21,44}, {58,167}, {21,45}, {21,46}, 
    {58,168}, {21,47}, {58,169}, {21,48}, {21,49}, {21,50}, {21,51}, {21,52}, 
    {21,53}, {58,170}, {58,171}, {21,54}, {21,55}, {58,172}, {21,56}, {21,57}, 
    {21,58}, {58,173}, {21,65}, {21,66}, {21,67}, {21,68}, {21,69}, {21,70}, 
    {21,71}, {58,174}, {58,175}, {21,72}, {58,176}, {21,73}, {21,74}, {21,75}, 
    {21,76}, {21,77}, {21,78}, {21,79}, {21,80}, {21,81}, {21,82}, {21,83}, 
    {21,84}, {21,85}, {21,86}, {21,87}, {21,88}, {21,89}, {21,90}, {21,91}, 
    {21,92}, {21,93}, {21,94}, {21,95}, {21,96}, {21,97}, {21,98}, {21,99}, 
    {21,100}, {21,101}, {21,102}, {21,103}, {21,104}, {21,105}, {21,106}, 
    {21,107}, {21,108}, {58,177}, {58,178}, {21,109}, {21,110}, {58,179}, 
    {21,111}, {21,112}, {21,113}, {58,180}, {21,114}, {58,181}, {21,115}, 
    {21,116}, {21,117}, {21,118}, {21,119}, {58,182}, {58,183}, {21,120}, 
    {58,184}, {21,121}, {58,185}, {58,186}, {58,187}, {21,122}, {21,123}, 
    {21,124}, {21,125}, {58,188}, {58,189}, {21,126}, {21,127}, {58,190}, 
    {21,128}, {21,129}, {21,130}, {59,97}, {21,131}, {59,98}, {21,132}, 
    {21,133}, {21,134}, {21,135}, {21,136}, {59,99}, {59,100}, {21,137}, 
    {59,101}, {59,102}, {59,103}, {21,138}, {21,139}, {21,140}, {21,141}, 
    {21,142}, {59,104}, {59,105}, {59,106}, {21,143}, {21,144}, {59,107}, 
    {21,145}, {21,146}, {21,147}, {59,108}, {21,148}, {21,149}, {21,150}, 
    {21,151}, {21,152}, {21,153}, {21,154}, {59,109}, {59,110}, {21,155}, 
    {59,111}, {59,112}, {59,113}, {21,156}, {21,157}, {21,158}, {21,159}, 
    {21,160}, {21,161}, {59,114}, {59,115}, {21,162}, {21,163}, {21,164}, 
    {21,165}, {21,166}, {21,167}, {21,168}, {21,169}, {21,170}, {21,171}, 
    {21,172}, {21,173}, {21,174}, {21,175}, {59,116}, {21,176}, {21,177}, 
    {21,178}, {21,179}, {21,180}, {21,181}, {21,182}, {21,183}, {21,184}, 
    {21,185}, {21,186}, {21,187}, {21,188}, {21,189}, {21,190}, {22,1}, 
    {22,2}, {22,3}, {22,4}, {22,5}, {22,6}, {22,7}, {22,8}, {22,9}, {22,10}, 
    {22,11}, {22,12}, {22,13}, {22,14}, {22,15}, {22,16}, {22,17}, {22,18}, 
    {22,19}, {22,20}, {22,21}, {22,22}, {22,23}, {22,24}, {59,117}, {59,118}, 
    {22,25}, {22,26}, {59,119}, {22,33}, {22,34}, {59,120}, {59,121}, {22,35}, 
    {22,36}, {22,37}, {22,38}, {22,39}, {22,40}, {22,41}, {59,122}, {22,42}, 
    {22,43}, {59,123}, {59,124}, {59,125}, {22,44}, {22,45}, {22,46}, {22,47}, 
    {22,48}, {22,49}, {59,126}, {22,50}, {22,51}, {22,52}, {22,53}, {22,54}, 
    {22,55}, {22,56}, {22,57}, {22,58}, {22,65}, {22,66}, {22,67}, {22,68}, 
    {22,69}, {22,70}, {22,71}, {22,72}, {22,73}, {22,74}, {22,75}, {59,127}, 
    {22,76}, {22,77}, {22,78}, {22,79}, {22,80}, {22,81}, {59,128}, {59,129}, 
    {22,82}, {22,83}, {22,84}, {22,85}, {22,86}, {22,87}, {22,88}, {22,89}, 
    {22,90}, {22,91}, {22,92}, {22,93}, {22,94}, {22,95}, {59,130}, {59,131}, 
    {22,96}, {59,132}, {59,133}, {59,134}, {22,97}, {22,98}, {22,99}, {22,100}, 
    {22,101}, {22,102}, {22,103}, {22,104}, {22,105}, {22,106}, {22,107}, 
    {22,108}, {22,109}, {22,110}, {22,111}, {22,112}, {22,113}, {22,114}, 
    {22,115}, {22,116}, {22,117}, {22,118}, {22,119}, {22,120}, {22,121}, 
    {22,122}, {22,123}, {22,124}, {22,125}, {22,126}, {22,127}, {22,128}, 
    {22,129}, {22,130}, {59,135}, {59,136}, {22,131}, {22,132}, {59,137}, 
    {22,133}, {22,134}, {22,135}, {59,138}, {22,136}, {22,137}, {22,138}, 
    {22,139}, {22,140}, {22,141}, {22,142}, {59,139}, {59,140}, {22,143}, 
    {22,144}, {22,145}, {59,141}, {22,146}, {22,147}, {22,148}, {22,149}, 
    {22,150}, {22,151}, {22,152}, {22,153}, {22,154}, {22,155}, {22,156}, 
    {22,157}, {22,158}, {22,159}, {22,160}, {22,161}, {22,162}, {22,163}, 
    {22,164}, {22,165}, {22,166}, {22,167}, {22,168}, {22,169}, {22,170}, 
    {22,171}, {22,172}, {22,173}, {22,174}, {22,175}, {22,176}, {22,177}, 
    {22,178}, {22,179}, {22,180}, {22,181}, {22,182}, {22,183}, {22,184}, 
    {22,185}, {22,186}, {22,187}, {22,188}, {22,189}, {22,190}, {23,1}, 
    {23,2}, {23,3}, {23,4}, {23,5}, {23,6}, {23,7}, {23,8}, {23,9}, {23,10}, 
    {23,11}, {23,12}, {23,13}, {23,14}, {23,15}, {23,16}, {23,17}, {59,142}, 
    {23,18}, {23,19}, {23,20}, {23,21}, {23,22}, {23,23}, {23,24}, {23,25}, 
    {23,26}, {23,33}, {23,34}, {23,35}, {23,36}, {23,37}, {23,38}, {23,39}, 
    {23,40}, {23,41}, {23,42}, {23,43}, {23,44}, {23,45}, {23,46}, {23,47}, 
    {23,48}, {23,49}, {23,50}, {59,143}, {23,51}, {23,52}, {23,53}, {23,54}, 
    {23,55}, {23,56}, {23,57}, {23,58}, {23,65}, {23,66}, {23,67}, {23,68}, 
    {23,69}, {23,70}, {23,71}, {23,72}, {23,73}, {23,74}, {23,75}, {23,76}, 
    {59,144}, {23,77}, {23,78}, {23,79}, {23,80}, {23,81}, {23,82}, {59,145}, 
    {59,146}, {23,83}, {23,84}, {59,147}, {23,85}, {23,86}, {23,87}, {59,148}, 
    {23,88}, {23,89}, {23,90}, {23,91}, {23,92}, {23,93}, {23,94}, {59,149}, 
    {23,95}, {23,96}, {59,150}, {23,97}, {59,151}, {23,98}, {23,99}, {23,100}, 
    {23,101}, {23,102}, {23,103}, {23,104}, {23,105}, {23,106}, {23,107}, 
    {23,108}, {23,109}, {23,110}, {23,111}, {23,112}, {23,113}, {23,114}, 
    {23,115}, {23,116}, {23,117}, {23,118}, {23,119}, {23,120}, {23,121}, 
    {23,122}, {23,123}, {23,124}, {23,125}, {23,126}, {23,127}, {23,128}, 
    {23,129}, {23,130}, {23,131}, {23,132}, {23,133}, {23,134}, {23,135}, 
    {23,136}, {23,137}, {23,138}, {23,139}, {23,140}, {23,141}, {23,142}, 
    {23,143}, {23,144}, {23,145}, {23,146}, {23,147}, {23,148}, {23,149}, 
    {23,150}, {23,151}, {23,152}, {23,153}, {23,154}, {23,155}, {23,156}, 
    {23,157}, {23,158}, {23,159}, {23,160}, {23,161}, {23,162}, {23,163}, 
    {23,164}, {23,165}, {23,166}, {23,167}, {23,168}, {23,169}, {23,170}, 
    {23,171}, {23,172}, {23,173}, {23,174}, {23,175}, {23,176}, {23,177}, 
    {23,178}, {23,179}, {23,180}, {23,181}, {23,182}, {23,183}, {23,184}, 
    {23,185}, {23,186}, {23,187}, {59,152}, {23,188}, {23,189}, {23,190}, 
    {24,1}, {24,2}, {24,3}, {24,4}, {24,5}, {24,6}, {24,7}, {24,8}, {24,9}, 
    {24,10}, {24,11}, {24,12}, {24,13}, {24,14}, {24,15}, {24,16}, {24,17}, 
    {59,153}, {24,18}, {24,19}, {24,20}, {24,21}, {24,22}, {24,23}, {59,154}, 
    {24,24}, {24,25}, {24,26}, {59,155}, {24,33}, {24,34}, {24,35}, {59,156}, 
    {24,36}, {24,37}, {24,38}, {24,39}, {24,40}, {24,41}, {24,42}, {59,157}, 
    {59,158}, {24,43}, {24,44}, {24,45}, {24,46}, {24,47}, {24,48}, {24,49}, 
    {24,50}, {24,51}, {24,52}, {24,53}, {24,54}, {24,55}, {24,56}, {24,57}, 
    {24,58}, {24,65}, {24,66}, {24,67}, {24,68}, {24,69}, {24,70}, {24,71}, 
    {24,72}, {24,73}, {24,74}, {24,75}, {24,76}, {24,77}, {24,78}, {24,79}, 
    {24,80}, {24,81}, {24,82}, {24,83}, {24,84}, {24,85}, {24,86}, {59,159}, 
    {59,160}, {24,87}, {24,88}, {59,161}, {24,89}, {24,90}, {24,91}, {59,162}, 
    {24,92}, {24,93}, {24,94}, {24,95}, {24,96}, {24,97}, {24,98}, {59,163}, 
    {59,164}, {24,99}, {59,165}, {24,100}, {59,166}, {24,101}, {24,102}, 
    {24,103}, {24,104}, {24,105}, {24,106}, {59,167}, {59,168}, {24,107}, 
    {59,169}, {59,170}, {24,108}, {24,109}, {59,171}, {59,172}, {59,173}, 
    {59,174}, {24,110}, {24,111}, {24,112}, {24,113}, {24,114}, {59,175}, 
    {59,176}, {24,115}, {59,177}, {59,178}, {59,179}, {24,116}, {24,117}, 
    {24,118}, {59,180}, {24,119}, {24,120}, {59,181}, {59,182}, {24,121}, 
    {24,122}, {59,183}, {24,123}, {24,124}, {24,125}, {59,184}, {24,126}, 
    {24,127}, {24,128}, {24,129}, {24,130}, {24,131}, {24,132}, {59,185}, 
    {59,186}, {24,133}, {59,187}, {59,188}, {59,189}, {24,134}, {24,135}, 
    {24,136}, {24,137}, {24,138}, {24,139}, {59,190}, {60,97}, {24,140}, 
    {24,141}, {60,98}, {24,142}, {24,143}, {24,144}, {60,99}, {24,145}, 
    {24,146}, {24,147}, {24,148}, {24,149}, {24,150}, {24,151}, {60,100}, 
    {60,101}, {24,152}, {60,102}, {24,153}, {60,103}, {24,154}, {24,155}, 
    {24,156}, {24,157}, {24,158}, {24,159}, {60,104}, {24,160}, {24,161}, 
    {24,162}, {60,105}, {24,163}, {24,164}, {24,165}, {60,106}, {24,166}, 
    {24,167}, {24,168}, {24,169}, {24,170}, {24,171}, {24,172}, {60,107}, 
    {24,173}, {24,174}, {24,175}, {24,176}, {60,108}, {24,177}, {24,178}, 
    {24,179}, {24,180}, {24,181}, {24,182}, {60,109}, {60,110}, {60,111}, 
    {60,112}, {60,113}, {24,183}, {24,184}, {60,114}, {60,115}, {24,185}, 
    {60,116}, {60,117}, {24,186}, {24,187}, {24,188}, {24,189}, {60,118}, 
    {60,119}, {24,190}, {60,120}, {60,121}, {60,122}, {25,1}, {25,2}, {25,3}, 
    {25,4}, {60,123}, {25,5}, {60,124}, {60,125}, {25,6}, {25,7}, {60,126}, 
    {25,8}, {25,9}, {25,10}, {60,127}, {25,11}, {25,12}, {25,13}, {25,14}, 
    {25,15}, {25,16}, {25,17}, {60,128}, {60,129}, {25,18}, {60,130}, {60,131}, 
    {60,132}, {25,19}, {25,20}, {25,21}, {25,22}, {25,23}, {25,24}, {60,133}, 
    {60,134}, {25,25}, {25,26}, {60,135}, {25,33}, {25,34}, {25,35}, {60,136}, 
    {25,36}, {25,37}, {25,38}, {25,39}, {25,40}, {25,41}, {25,42}, {60,137}, 
    {60,138}, {25,43}, {60,139}, {60,140}, {60,141}, {25,44}, {25,45}, 
    {25,46}, {25,47}, {25,48}, {25,49}, {60,142}, {25,50}, {25,51}, {25,52}, 
    {60,143}, {25,53}, {25,54}, {25,55}, {60,144}, {25,56}, {25,57}, {25,58}, 
    {25,65}, {25,66}, {25,67}, {25,68}, {25,69}, {25,70}, {25,71}, {25,72}, 
    {25,73}, {60,145}, {25,74}, {25,75}, {25,76}, {25,77}, {25,78}, {25,79}, 
    {60,146}, {60,147}, {60,148}, {25,80}, {60,149}, {25,81}, {25,82}, 
    {25,83}, {60,150}, {25,84}, {60,151}, {25,85}, {25,86}, {25,87}, {25,88}, 
    {25,89}, {60,152}, {60,153}, {25,90}, {60,154}, {25,91}, {60,155}, 
    {25,92}, {25,93}, {25,94}, {60,156}, {25,95}, {25,96}, {60,157}, {60,158}, 
    {25,97}, {25,98}, {60,159}, {25,99}, {25,100}, {25,101}, {60,160}, 
    {25,102}, {25,103}, {25,104}, {25,105}, {25,106}, {25,107}, {25,108}, 
    {25,109}, {25,110}, {25,111}, {25,112}, {25,113}, {60,161}, {25,114}, 
    {25,115}, {25,116}, {25,117}, {25,118}, {25,119}, {60,162}, {25,120}, 
    {25,121}, {25,122}, {60,163}, {25,123}, {25,124}, {25,125}, {60,164}, 
    {25,126}, {25,127}, {25,128}, {25,129}, {25,130}, {25,131}, {25,132}, 
    {60,165}, {25,133}, {25,134}, {60,166}, {60,167}, {25,135}, {25,136}, 
    {25,137}, {25,138}, {25,139}, {25,140}, {25,141}, {60,168}, {25,142}, 
    {25,143}, {25,144}, {60,169}, {25,145}, {25,146}, {25,147}, {60,170}, 
    {25,148}, {25,149}, {25,150}, {25,151}, {25,152}, {25,153}, {25,154}, 
    {60,171}, {60,172}, {25,155}, {60,173}, {25,156}, {25,157}, {25,158}, 
    {25,159}, {25,160}, {25,161}, {25,162}, {25,163}, {60,174}, {60,175}, 
    {25,164}, {25,165}, {60,176}, {25,166}, {25,167}, {25,168}, {60,177}, 
    {25,169}, {25,170}, {25,171}, {25,172}, {25,173}, {25,174}, {25,175}, 
    {60,178}, {60,179}, {25,176}, {60,180}, {25,177}, {60,181}, {25,178}, 
    {25,179}, {25,180}, {25,181}, {25,182}, {25,183}, {60,182}, {60,183}, 
    {25,184}, {25,185}, {60,184}, {25,186}, {25,187}, {60,185}, {60,186}, 
    {25,188}, {25,189}, {25,190}, {26,1}, {26,2}, {26,3}, {26,4}, {60,187}, 
    {60,188}, {26,5}, {60,189}, {26,6}, {60,190}, {26,7}, {61,97}, {26,8}, 
    {61,98}, {61,99}, {26,9}, {61,100}, {26,10}, {26,11}, {26,12}, {26,13}, 
    {26,14}, {26,15}, {26,16}, {26,17}, {26,18}, {26,19}, {26,20}, {26,21}, 
    {26,22}, {26,23}, {26,24}, {26,25}, {26,26}, {26,33}, {26,34}, {61,101}, 
    {26,35}, {26,36}, {26,37}, {26,38}, {26,39}, {26,40}, {26,41}, {61,102}, 
    {61,103}, {26,42}, {26,43}, {61,104}, {26,44}, {26,45}, {26,46}, {61,105}, 
    {26,47}, {26,48}, {26,49}, {26,50}, {26,51}, {26,52}, {26,53}, {61,106}, 
    {26,54}, {26,55}, {26,56}, {26,57}, {61,107}, {26,58}, {26,65}, {26,66}, 
    {26,67}, {26,68}, {26,69}, {61,108}, {61,109}, {26,70}, {26,71}, {61,110}, 
    {26,72}, {26,73}, {26,74}, {61,111}, {26,75}, {26,76}, {26,77}, {26,78}, 
    {26,79}, {26,80}, {26,81}, {61,112}, {61,113}, {26,82}, {61,114}, {26,83}, 
    {61,115}, {26,84}, {26,85}, {26,86}, {26,87}, {26,88}, {26,89}, {61,116}, 
    {61,117}, {26,90}, {26,91}, {26,92}, {26,93}, {26,94}, {26,95}, {61,118}, 
    {26,96}, {26,97}, {26,98}, {26,99}, {26,100}, {26,101}, {26,102}, {61,119}, 
    {26,103}, {26,104}, {61,120}, {26,105}, {61,121}, {26,106}, {26,107}, 
    {26,108}, {26,109}, {26,110}, {26,111}, {61,122}, {61,123}, {26,112}, 
    {26,113}, {61,124}, {26,114}, {26,115}, {26,116}, {61,125}, {61,126}, 
    {26,117}, {26,118}, {26,119}, {26,120}, {26,121}, {26,122}, {61,127}, 
    {61,128}, {26,123}, {61,129}, {26,124}, {61,130}, {26,125}, {26,126}, 
    {26,127}, {26,128}, {26,129}, {26,130}, {26,131}, {26,132}, {26,133}, 
    {26,134}, {26,135}, {26,136}, {26,137}, {26,138}, {26,139}, {26,140}, 
    {26,141}, {26,142}, {26,143}, {26,144}, {26,145}, {26,146}, {26,147}, 
    {26,148}, {26,149}, {26,150}, {26,151}, {26,152}, {26,153}, {26,154}, 
    {26,155}, {26,156}, {26,157}, {26,158}, {61,131}, {61,132}, {26,159}, 
    {26,160}, {61,133}, {26,161}, {26,162}, {61,134}, {61,135}, {26,163}, 
    {26,164}, {26,165}, {26,166}, {26,167}, {26,168}, {61,136}, {61,137}, 
    {61,138}, {26,169}, {61,139}, {26,170}, {61,140}, {26,171}, {26,172}, 
    {26,173}, {26,174}, {61,141}, {26,175}, {61,142}, {61,143}, {26,176}, 
    {61,144}, {61,145}, {26,177}, {26,178}, {26,179}, {61,146}, {26,180}, 
    {26,181}, {26,182}, {26,183}, {26,184}, {26,185}, {26,186}, {61,147}, 
    {61,148}, {26,187}, {26,188}, {61,149}, {61,150}, {26,189}, {26,190}, 
    {27,1}, {27,2}, {27,3}, {61,151}, {61,152}, {61,153}, {27,4}, {27,5}, 
    {61,154}, {27,6}, {27,7}, {27,8}, {61,155}, {27,9}, {27,10}, {27,11}, 
    {27,12}, {27,13}, {27,14}, {27,15}, {61,156}, {61,157}, {27,16}, {27,17}, 
    {61,158}, {61,159}, {27,18}, {27,19}, {27,20}, {27,21}, {27,22}, {27,23}, 
    {27,24}, {27,25}, {27,26}, {27,33}, {27,34}, {27,35}, {27,36}, {27,37}, 
    {27,38}, {27,39}, {27,40}, {27,41}, {27,42}, {27,43}, {27,44}, {27,45}, 
    {27,46}, {27,47}, {27,48}, {27,49}, {27,50}, {61,160}, {27,51}, {27,52}, 
    {27,53}, {27,54}, {27,55}, {27,56}, {27,57}, {27,58}, {27,65}, {27,66}, 
    {27,67}, {27,68}, {27,69}, {27,70}, {27,71}, {27,72}, {27,73}, {27,74}, 
    {27,75}, {27,76}, {27,77}, {27,78}, {27,79}, {27,80}, {27,81}, {27,82}, 
    {27,83}, {27,84}, {27,85}, {27,86}, {27,87}, {27,88}, {27,89}, {27,90}, 
    {61,161}, {61,162}, {27,91}, {27,92}, {61,163}, {27,93}, {27,94}, {27,95}, 
    {61,164}, {27,96}, {61,165}, {27,97}, {27,98}, {27,99}, {27,100}, {27,101}, 
    {61,166}, {61,167}, {27,102}, {27,103}, {61,168}, {61,169}, {27,104}, 
    {27,105}, {27,106}, {27,107}, {27,108}, {27,109}, {61,170}, {27,110}, 
    {27,111}, {27,112}, {61,171}, {27,113}, {27,114}, {27,115}, {61,172}, 
    {27,116}, {27,117}, {27,118}, {27,119}, {27,120}, {27,121}, {27,122}, 
    {27,123}, {27,124}, {27,125}, {27,126}, {27,127}, {27,128}, {27,129}, 
    {27,130}, {27,131}, {27,132}, {27,133}, {27,134}, {27,135}, {27,136}, 
    {27,137}, {27,138}, {27,139}, {27,140}, {27,141}, {27,142}, {27,143}, 
    {27,144}, {27,145}, {27,146}, {27,147}, {27,148}, {27,149}, {27,150}, 
    {27,151}, {27,152}, {27,153}, {27,154}, {27,155}, {27,156}, {27,157}, 
    {27,158}, {27,159}, {27,160}, {27,161}, {27,162}, {27,163}, {27,164}, 
    {27,165}, {27,166}, {61,173}, {27,167}, {27,168}, {27,169}, {27,170}, 
    {27,171}, {27,172}, {27,173}, {27,174}, {27,175}, {27,176}, {27,177}, 
    {27,178}, {27,179}, {27,180}, {27,181}, {27,182}, {27,183}, {27,184}, 
    {27,185}, {27,186}, {27,187}, {27,188}, {27,189}, {61,174}, {61,175}, 
    {27,190}, {28,1}, {61,176}, {28,2}, {28,3}, {61,177}, {61,178}, {28,4}, 
    {61,179}, {28,5}, {28,6}, {28,7}, {28,8}, {28,9}, {61,180}, {61,181}, 
    {28,10}, {28,11}, {28,12}, {61,182}, {28,13}, {28,14}, {28,15}, {28,16}, 
    {28,17}, {28,18}, {61,183}, {61,184}, {28,19}, {28,20}, {61,185}, {28,21}, 
    {28,22}, {28,23}, {28,24}, {28,25}, {28,26}, {28,33}, {28,34}, {28,35}, 
    {28,36}, {28,37}, {28,38}, {28,39}, {28,40}, {28,41}, {61,186}, {28,42}, 
    {28,43}, {28,44}, {28,45}, {28,46}, {28,47}, {28,48}, {61,187}, {28,49}, 
    {28,50}, {28,51}, {28,52}, {28,53}, {28,54}, {28,55}, {28,56}, {28,57}, 
    {28,58}, {28,65}, {28,66}, {28,67}, {28,68}, {28,69}, {28,70}, {28,71}, 
    {28,72}, {28,73}, {61,188}, {28,74}, {28,75}, {28,76}, {28,77}, {28,78}, 
    {28,79}, {28,80}, {61,189}, {28,81}, {28,82}, {28,83}, {61,190}, {28,84}, 
    {28,85}, {28,86}, {62,97}, {28,87}, {28,88}, {28,89}, {28,90}, {28,91}, 
    {28,92}, {28,93}, {62,98}, {62,99}, {28,94}, {28,95}, {28,96}, {28,97}, 
    {28,98}, {28,99}, {28,100}, {28,101}, {28,102}, {28,103}, {62,100}, 
    {28,104}, {28,105}, {28,106}, {28,107}, {28,108}, {28,109}, {28,110}, 
    {28,111}, {28,112}, {28,113}, {28,114}, {28,115}, {28,116}, {28,117}, 
    {28,118}, {28,119}, {28,120}, {28,121}, {28,122}, {28,123}, {28,124}, 
    {28,125}, {28,126}, {28,127}, {28,128}, {28,129}, {28,130}, {62,101}, 
    {62,102}, {28,131}, {28,132}, {62,103}, {28,133}, {28,134}, {28,135}, 
    {62,104}, {28,136}, {28,137}, {28,138}, {28,139}, {28,140}, {28,141}, 
    {28,142}, {62,105}, {62,106}, {28,143}, {28,144}, {28,145}, {62,107}, 
    {28,146}, {28,147}, {28,148}, {28,149}, {28,150}, {28,151}, {62,108}, 
    {28,152}, {28,153}, {28,154}, {28,155}, {28,156}, {28,157}, {28,158}, 
    {28,159}, {28,160}, {28,161}, {28,162}, {28,163}, {28,164}, {28,165}, 
    {28,166}, {28,167}, {28,168}, {28,169}, {28,170}, {62,109}, {28,171}, 
    {28,172}, {28,173}, {28,174}, {28,175}, {28,176}, {28,177}, {62,110}, 
    {28,178}, {28,179}, {28,180}, {28,181}, {28,182}, {28,183}, {28,184}, 
    {28,185}, {28,186}, {28,187}, {28,188}, {28,189}, {28,190}, {29,1}, 
    {29,2}, {29,3}, {29,4}, {29,5}, {29,6}, {29,7}, {29,8}, {29,9}, {29,10}, 
    {29,11}, {29,12}, {29,13}, {29,14}, {62,111}, {29,15}, {29,16}, {29,17}, 
    {62,112}, {29,18}, {29,19}, {29,20}, {29,21}, {29,22}, {29,23}, {29,24}, 
    {29,25}, {29,26}, {29,33}, {29,34}, {29,35}, {29,36}, {29,37}, {29,38}, 
    {29,39}, {29,40}, {29,41}, {29,42}, {29,43}, {29,44}, {29,45}, {29,46}, 
    {29,47}, {29,48}, {29,49}, {29,50}, {29,51}, {29,52}, {29,53}, {29,54}, 
    {29,55}, {29,56}, {29,57}, {29,58}, {29,65}, {29,66}, {29,67}, {29,68}, 
    {29,69}, {29,70}, {29,71}, {29,72}, {29,73}, {62,113}, {29,74}, {29,75}, 
    {29,76}, {29,77}, {29,78}, {29,79}, {62,114}, {62,115}, {29,80}, {29,81}, 
    {62,116}, {29,82}, {29,83}, {29,84}, {62,117}, {29,85}, {62,118}, {29,86}, 
    {29,87}, {29,88}, {29,89}, {62,119}, {62,120}, {62,121}, {29,90}, {29,91}, 
    {29,92}, {29,93}, {29,94}, {29,95}, {29,96}, {29,97}, {29,98}, {29,99}, 
    {62,122}, {29,100}, {29,101}, {29,102}, {62,123}, {29,103}, {29,104}, 
    {29,105}, {62,124}, {29,106}, {29,107}, {29,108}, {29,109}, {29,110}, 
    {29,111}, {29,112}, {62,125}, {29,113}, {29,114}, {29,115}, {29,116}, 
    {29,117}, {29,118}, {29,119}, {29,120}, {29,121}, {29,122}, {29,123}, 
    {62,126}, {62,127}, {29,124}, {29,125}, {62,128}, {29,126}, {29,127}, 
    {29,128}, {62,129}, {29,129}, {29,130}, {29,131}, {29,132}, {29,133}, 
    {29,134}, {29,135}, {62,130}, {62,131}, {29,136}, {62,132}, {29,137}, 
    {62,133}, {29,138}, {29,139}, {29,140}, {29,141}, {29,142}, {29,143}, 
    {62,134}, {62,135}, {29,144}, {29,145}, {62,136}, {62,137}, {62,138}, 
    {29,146}, {62,139}, {62,140}, {62,141}, {29,147}, {29,148}, {29,149}, 
    {29,150}, {62,142}, {62,143}, {62,144}, {29,151}, {62,145}, {62,146}, 
    {62,147}, {29,152}, {29,153}, {29,154}, {62,148}, {62,149}, {29,155}, 
    {62,150}, {62,151}, {29,156}, {29,157}, {62,152}, {29,158}, {29,159}, 
    {29,160}, {62,153}, {29,161}, {29,162}, {29,163}, {29,164}, {29,165}, 
    {29,166}, {29,167}, {62,154}, {62,155}, {29,168}, {62,156}, {62,157}, 
    {62,158}, {29,169}, {29,170}, {29,171}, {29,172}, {29,173}, {29,174}, 
    {62,159}, {62,160}, {29,175}, {29,176}, {62,161}, {29,177}, {29,178}, 
    {29,179}, {62,162}, {29,180}, {29,181}, {62,163}, {29,182}, {29,183}, 
    {29,184}, {29,185}, {62,164}, {62,165}, {29,186}, {62,166}, {29,187}, 
    {62,167}, {29,188}, {29,189}, {29,190}, {62,168}, {30,1}, {62,169}, 
    {62,170}, {30,2}, {30,3}, {30,4}, {62,171}, {30,5}, {30,6}, {30,7}, 
    {62,172}, {30,8}, {30,9}, {30,10}, {30,11}, {30,12}, {30,13}, {30,14}, 
    {30,15}, {62,173}, {30,16}, {30,17}, {30,18}, {30,19}, {30,20}, {30,21}, 
    {30,22}, {30,23}, {30,24}, {30,25}, {62,174}, {62,175}, {30,26}, {30,33}, 
    {62,176}, {62,177}, {30,34}, {62,178}, {62,179}, {62,180}, {62,181}, 
    {30,35}, {30,36}, {30,37}, {30,38}, {30,39}, {62,182}, {62,183}, {62,184}, 
    {62,185}, {62,186}, {62,187}, {62,188}, {30,40}, {62,189}, {30,41}, 
    {62,190}, {30,42}, {63,97}, {63,98}, {30,43}, {30,44}, {63,99}, {30,45}, 
    {30,46}, {30,47}, {63,100}, {30,48}, {30,49}, {30,50}, {30,51}, {30,52}, 
    {30,53}, {30,54}, {63,101}, {63,102}, {30,55}, {63,103}, {30,56}, {63,104}, 
    {30,57}, {30,58}, {30,65}, {30,66}, {30,67}, {30,68}, {63,105}, {63,106}, 
    {63,107}, {30,69}, {63,108}, {30,70}, {30,71}, {30,72}, {63,109}, {30,73}, 
    {63,110}, {63,111}, {30,74}, {30,75}, {30,76}, {30,77}, {63,112}, {63,113}, 
    {63,114}, {63,115}, {63,116}, {63,117}, {30,78}, {30,79}, {30,80}, 
    {63,118}, {63,119}, {63,120}, {63,121}, {30,81}, {30,82}, {30,83}, 
    {63,122}, {30,84}, {30,85}, {30,86}, {63,123}, {30,87}, {30,88}, {30,89}, 
    {30,90}, {30,91}, {30,92}, {30,93}, {63,124}, {63,125}, {30,94}, {63,126}, 
    {63,127}, {30,95}, {30,96}, {30,97}, {30,98}, {30,99}, {30,100}, {30,101}, 
    {63,128}, {63,129}, {30,102}, {30,103}, {63,130}, {30,104}, {30,105}, 
    {30,106}, {63,131}, {63,132}, {63,133}, {30,107}, {63,134}, {30,108}, 
    {30,109}, {63,135}, {63,136}, {63,137}, {30,110}, {63,138}, {30,111}, 
    {63,139}, {30,112}, {63,140}, {30,113}, {30,114}, {30,115}, {30,116}, 
    {63,141}, {63,142}, {30,117}, {30,118}, {63,143}, {30,119}, {30,120}, 
    {30,121}, {63,144}, {30,122}, {30,123}, {30,124}, {30,125}, {30,126}, 
    {30,127}, {30,128}, {63,145}, {63,146}, {30,129}, {63,147}, {63,148}, 
    {63,149}, {30,130}, {30,131}, {30,132}, {30,133}, {30,134}, {30,135}, 
    {63,150}, {63,151}, {30,136}, {30,137}, {63,152}, {30,138}, {30,139}, 
    {30,140}, {30,141}, {30,142}, {30,143}, {30,144}, {30,145}, {30,146}, 
    {30,147}, {30,148}, {63,153}, {30,149}, {30,150}, {63,154}, {30,151}, 
    {63,155}, {30,152}, {30,153}, {30,154}, {30,155}, {30,156}, {30,157}, 
    {63,156}, {63,157}, {30,158}, {30,159}, {63,158}, {30,160}, {30,161}, 
    {30,162}, {63,159}, {30,163}, {30,164}, {30,165}, {30,166}, {30,167}, 
    {30,168}, {30,169}, {63,160}, {63,161}, {30,170}, {63,162}, {30,171}, 
    {63,163}, {30,172}, {30,173}, {30,174}, {30,175}, {30,176}, {30,177}, 
    {63,164}, {63,165}, {30,178}, {30,179}, {63,166}, {30,180}, {30,181}, 
    {30,182}, {63,167}, {30,183}, {30,184}, {30,185}, {30,186}, {30,187}, 
    {30,188}, {30,189}, {63,168}, {63,169}, {30,190}, {63,170}, {31,1}, 
    {63,171}, {31,2}, {31,3}, {31,4}, {31,5}, {31,6}, {31,7}, {63,172}, 
    {63,173}, {31,8}, {31,9}, {63,174}, {31,10}, {31,11}, {31,12}, {63,175}, 
    {63,176}, {63,177}, {31,13}, {31,14}, {31,15}, {31,16}, {31,17}, {63,178}, 
    {63,179}, {31,18}, {63,180}, {31,19}, {63,181}, {31,20}, {31,21}, {31,22}, 
    {31,23}, {31,24}, {31,25}, {63,182}, {63,183}, {31,26}, {31,33}, {63,184}, 
    {31,34}, {31,35}, {31,36}, {63,185}, {31,37}, {31,38}, {31,39}, {31,40}, 
    {31,41}, {31,42}, {31,43}, {63,186}, {63,187}, {31,44}, {31,45}, {63,188}, 
    {63,189}, {31,46}, {31,47}, {31,48}, {31,49}, {31,50}, {31,51}, {63,190}, 
    {64,97}, {31,52}, {31,53}, {64,98}, {31,54}, {31,55}, {31,56}, {64,99}, 
    {31,57}, {31,58}, {31,65}, {31,66}, {31,67}, {31,68}, {31,69}, {64,100}, 
    {64,101}, {31,70}, {31,71}, {31,72}, {64,102}, {31,73}, {31,74}, {31,75}, 
    {31,76}, {31,77}, {31,78}, {64,103}, {64,104}, {31,79}, {31,80}, {64,105}, 
    {31,81}, {31,82}, {31,83}, {64,106}, {31,84}, {31,85}, {31,86}, {31,87}, 
    {31,88}, {31,89}, {31,90}, {64,107}, {64,108}, {31,91}, {64,109}, {31,92}, 
    {64,110}, {31,93}, {31,94}, {31,95}, {31,96}, {31,97}, {31,98}, {64,111}, 
    {64,112}, {31,99}, {31,100}, {64,113}, {31,101}, {31,102}, {31,103}, 
    {64,114}, {31,104}, {31,105}, {31,106}, {31,107}, {31,108}, {31,109}, 
    {31,110}, {64,115}, {64,116}, {31,111}, {64,117}, {31,112}, {64,118}, 
    {31,113}, {64,119}, {31,114}, {31,115}, {31,116}, {31,117}, {64,120}, 
    {64,121}, {31,118}, {31,119}, {64,122}, {31,120}, {31,121}, {31,122}, 
    {64,123}, {31,123}, {31,124}, {31,125}, {31,126}, {31,127}, {64,124}, 
    {31,128}, {64,125}, {64,126}, {31,129}, {64,127}, {31,130}, {64,128}, 
    {64,129}, {64,130}, {64,131}, {64,132}, {64,133}, {64,134}, {64,135}, 
    {31,131}, {31,132}, {31,133}, {64,136}, {31,134}, {31,135}, {31,136}, 
    {64,137}, {31,137}, {31,138}, {31,139}, {31,140}, {31,141}, {31,142}, 
    {31,143}, {64,138}, {31,144}, {31,145}, {64,139}, {31,146}, {31,147}, 
    {31,148}, {31,149}, {31,150}, {31,151}, {31,152}, {31,153}, {64,140}, 
    {64,141}, {31,154}, {31,155}, {64,142}, {31,156}, {31,157}, {31,158}, 
    {64,143}, {64,144}, {64,145}, {31,159}, {31,160}, {31,161}, {31,162}, 
    {64,146}, {64,147}, {64,148}, {31,163}, {64,149}, {64,150}, {64,151}, 
    {64,152}, {31,164}, {31,165}, {31,166}, {64,153}, {31,167}, {64,154}, 
    {64,155}, {31,168}, {31,169}, {64,156}, {31,170}, {64,157}, {64,158}, 
    {64,159}, {31,171}, {64,160}, {31,172}, {31,173}, {31,174}, {31,175}, 
    {31,176}, {64,161}, {64,162}, {31,177}, {64,163}, {64,164}, {64,165}, 
    {64,166}, {31,178}, {31,179}, {31,180}, {31,181}, {31,182}, {64,167}, 
    {64,168}, {31,183}, {31,184}, {64,169}, {31,185}, {31,186}, {31,187}, 
    {64,170}, {31,188}, {31,189}, {31,190}, {32,1}, {32,2}, {32,3}, {32,4}, 
    {64,171}, {64,172}, {32,5}, {64,173}, {64,174}, {64,175}, {32,6}, {32,7}, 
    {32,8}, {32,9}, {32,10}, {32,11}, {64,176}, {64,177}, {32,12}, {32,13}, 
    {64,178}, {32,14}, {64,179}, {32,15}, {64,180}, {32,16}, {32,17}, {32,18}, 
    {32,19}, {32,20}, {32,21}, {32,22}, {64,181}, {32,23}, {32,24}, {32,25}, 
    {32,26}, {64,182}, {32,33}, {32,34}, {32,35}, {32,36}, {32,37}, {32,38}, 
    {64,183}, {32,39}, {32,40}, {32,41}, {64,184}, {32,42}, {32,43}, {32,44}, 
    {64,185}, {32,45}, {32,46}, {32,47}, {32,48}, {32,49}, {32,50}, {32,51}, 
    {32,52}, {32,53}, {32,54}, {32,55}, {32,56}, {32,57}, {32,58}, {32,65}, 
    {32,66}, {32,67}, {32,68}, {32,69}, {64,186}, {64,187}, {32,70}, {32,71}, 
    {64,188}, {32,72}, {32,73}, {32,74}, {64,189}, {32,75}, {64,190}, {32,76}, 
    {32,77}, {32,78}, {32,79}, {32,80}, {65,97}, {65,98}, {32,81}, {65,99}, 
    {32,82}, {65,100}, {65,101}, {32,83}, {32,84}, {32,85}, {32,86}, {32,87}, 
    {65,102}, {65,103}, {32,88}, {32,89}, {65,104}, {32,90}, {32,91}, {32,92}, 
    {65,105}, {32,93}, {32,94}, {32,95}, {32,96}, {32,97}, {32,98}, {32,99}, 
    {65,106}, {65,107}, {32,100}, {65,108}, {32,101}, {65,109}, {32,102}, 
    {32,103}, {32,104}, {32,105}, {32,106}, {32,107}, {65,110}, {32,108}, 
    {32,109}, {32,110}, {65,111}, {32,111}, {32,112}, {32,113}, {65,112}, 
    {32,114}, {32,115}, {32,116}, {32,117}, {32,118}, {32,119}, {32,120}, 
    {65,113}, {65,114}, {32,121}, {32,122}, {65,115}, {65,116}, {32,123}, 
    {32,124}, {32,125}, {32,126}, {32,127}, {32,128}, {65,117}, {32,129}, 
    {32,130}, {32,131}, {32,132}, {32,133}, {32,134}, {32,135}, {32,136}, 
    {32,137}, {32,138}, {32,139}, {32,140}, {32,141}, {32,142}, {32,143}, 
    {32,144}, {32,145}, {32,146}, {32,147}, {32,148}, {32,149}, {32,150}, 
    {32,151}, {32,152}, {32,153}, {32,154}, {32,155}, {65,118}, {65,119}, 
    {32,156}, {32,157}, {65,120}, {32,158}, {32,159}, {32,160}, {65,121}, 
    {32,161}, {65,122}, {32,162}, {32,163}, {32,164}, {32,165}, {32,166}, 
    {65,123}, {65,124}, {32,167}, {65,125}, {32,168}, {65,126}, {65,127}, 
    {65,128}, {32,169}, {32,170}, {32,171}, {65,129}, {65,130}, {65,131}, 
    {32,172}, {32,173}, {32,174}, {32,175}, {32,176}, {32,177}, {65,132}, 
    {32,178}, {32,179}, {32,180}, {32,181}, {32,182}, {32,183}, {32,184}, 
    {32,185}, {65,133}, {32,186}, {65,134}, {32,187}, {65,135}, {32,188}, 
    {32,189}, {32,190}, {33,1}, {33,2}, {33,3}, {65,136}, {33,4}, {33,5}, 
    {33,6}, {33,7}, {33,8}, {33,9}, {33,10}, {33,11}, {33,12}, {33,13}, 
    {33,14}, {33,15}, {33,16}, {33,17}, {33,18}, {33,19}, {33,20}, {33,21}, 
    {33,22}, {65,137}, {65,138}, {33,23}, {33,24}, {33,25}, {33,26}, {33,33}, 
    {33,34}, {65,139}, {33,35}, {33,36}, {33,37}, {65,140}, {33,38}, {33,39}, 
    {33,40}, {65,141}, {33,41}, {33,42}, {33,43}, {33,44}, {33,45}, {33,46}, 
    {33,47}, {65,142}, {65,143}, {33,48}, {65,144}, {33,49}, {65,145}, 
    {33,50}, {33,51}, {33,52}, {33,53}, {33,54}, {33,55}, {65,146}, {65,147}, 
    {33,56}, {33,57}, {65,148}, {33,58}, {33,65}, {33,66}, {33,67}, {33,68}, 
    {33,69}, {33,70}, {33,71}, {33,72}, {33,73}, {33,74}, {33,75}, {33,76}, 
    {33,77}, {33,78}, {33,79}, {65,149}, {33,80}, {33,81}, {33,82}, {33,83}, 
    {33,84}, {33,85}, {65,150}, {65,151}, {33,86}, {33,87}, {65,152}, {33,88}, 
    {33,89}, {33,90}, {65,153}, {65,154}, {65,155}, {33,91}, {33,92}, {33,93}, 
    {33,94}, {33,95}, {65,156}, {65,157}, {33,96}, {65,158}, {34,1}, {65,159}, 
    {34,2}, {34,3}, {34,4}, {34,5}, {34,6}, {34,7}, {65,160}, {34,8}, {34,9}, 
    {34,10}, {34,11}, {34,12}, {34,13}, {34,14}, {34,15}, {34,16}, {34,17}, 
    {34,18}, {34,19}, {34,20}, {34,21}, {34,22}, {34,23}, {34,24}, {34,25}, 
    {34,26}, {65,161}, {34,33}, {34,34}, {34,35}, {34,36}, {34,37}, {34,38}, 
    {34,39}, {65,162}, {34,40}, {34,41}, {34,42}, {34,43}, {34,44}, {34,45}, 
    {34,46}, {34,47}, {34,48}, {34,49}, {34,50}, {34,51}, {34,52}, {34,53}, 
    {34,54}, {34,55}, {34,56}, {34,57}, {34,58}, {34,65}, {34,66}, {34,67}, 
    {34,68}, {34,69}, {34,70}, {34,71}, {34,72}, {65,163}, {65,164}, {34,73}, 
    {34,74}, {65,165}, {34,75}, {34,76}, {34,77}, {65,166}, {34,78}, {34,79}, 
    {34,80}, {34,81}, {34,82}, {34,83}, {34,84}, {65,167}, {65,168}, {34,85}, 
    {65,169}, {34,86}, {34,87}, {34,88}, {34,89}, {34,90}, {34,91}, {34,92}, 
    {34,93}, {65,170}, {34,94}, {34,95}, {34,96}, {65,171}, {35,1}, {35,2}, 
    {35,3}, {65,172}, {35,4}, {35,5}, {35,6}, {35,7}, {35,8}, {35,9}, {35,10}, 
    {65,173}, {35,11}, {35,12}, {35,13}, {35,14}, {35,15}, {35,16}, {35,17}, 
    {35,18}, {35,19}, {35,20}, {35,21}, {65,174}, {65,175}, {35,22}, {35,23}, 
    {65,176}, {35,24}, {35,25}, {35,26}, {65,177}, {35,33}, {35,34}, {35,35}, 
    {35,36}, {35,37}, {35,38}, {35,39}, {65,178}, {65,179}, {35,40}, {65,180}, 
    {35,41}, {65,181}, {35,42}, {35,43}, {35,44}, {35,45}, {35,46}, {35,47}, 
    {35,48}, {35,49}, {35,50}, {35,51}, {35,52}, {35,53}, {35,54}, {35,55}, 
    {35,56}, {35,57}, {35,58}, {35,65}, {35,66}, {35,67}, {35,68}, {35,69}, 
    {35,70}, {35,71}, {35,72}, {35,73}, {35,74}, {35,75}, {35,76}, {35,77}, 
    {35,78}, {35,79}, {35,80}, {35,81}, {65,182}, {65,183}, {35,82}, {35,83}, 
    {65,184}, {35,84}, {35,85}, {65,185}, {65,186}, {35,86}, {65,187}, 
    {35,87}, {35,88}, {35,89}, {35,90}, {35,91}, {65,188}, {65,189}, {35,92}, 
    {65,190}, {35,93}, {66,97}, {66,98}, {35,94}, {35,95}, {66,99}, {66,100}, 
    {35,96}, {66,101}, {66,102}, {36,1}, {36,2}, {66,103}, {36,3}, {66,104}, 
    {36,4}, {66,105}, {36,5}, {36,6}, {66,106}, {36,7}, {36,8}, {36,9}, 
    {36,10}, {66,107}, {66,108}, {36,11}, {66,109}, {66,110}, {66,111}, 
    {36,12}, {36,13}, {36,14}, {36,15}, {36,16}, {36,17}, {66,112}, {66,113}, 
    {36,18}, {36,19}, {66,114}, {36,20}, {36,21}, {36,22}, {66,115}, {36,23}, 
    {36,24}, {36,25}, {36,26}, {36,33}, {36,34}, {36,35}, {66,116}, {66,117}, 
    {36,36}, {66,118}, {66,119}, {66,120}, {36,37}, {36,38}, {36,39}, {36,40}, 
    {36,41}, {36,42}, {66,121}, {36,43}, {36,44}, {36,45}, {66,122}, {36,46}, 
    {36,47}, {36,48}, {36,49}, {36,50}, {36,51}, {36,52}, {36,53}, {36,54}, 
    {36,55}, {36,56}, {36,57}, {36,58}, {36,65}, {36,66}, {36,67}, {66,123}, 
    {36,68}, {36,69}, {36,70}, {36,71}, {36,72}, {36,73}, {36,74}, {36,75}, 
    {36,76}, {36,77}, {36,78}, {36,79}, {36,80}, {36,81}, {36,82}, {36,83}, 
    {36,84}, {36,85}, {36,86}, {36,87}, {36,88}, {36,89}, {36,90}, {36,91}, 
    {36,92}, {36,93}, {36,94}, {36,95}, {36,96}, {37,1}, {37,2}, {37,3}, 
    {37,4}, {37,5}, {66,124}, {66,125}, {37,6}, {37,7}, {66,126}, {37,8}, 
    {37,9}, {37,10}, {66,127}, {37,11}, {37,12}, {37,13}, {37,14}, {37,15}, 
    {37,16}, {37,17}, {66,128}, {66,129}, {37,18}, {66,130}, {66,131}, 
    {66,132}, {37,19}, {37,20}, {37,21}, {37,22}, {37,23}, {37,24}, {66,133}, 
    {37,25}, {37,26}, {37,33}, {37,34}, {37,35}, {37,36}, {37,37}, {37,38}, 
    {37,39}, {37,40}, {37,41}, {37,42}, {37,43}, {37,44}, {37,45}, {37,46}, 
    {37,47}, {37,48}, {37,49}, {37,50}, {66,134}, {37,51}, {37,52}, {37,53}, 
    {37,54}, {37,55}, {37,56}, {66,135}, {37,57}, {37,58}, {37,65}, {37,66}, 
    {37,67}, {37,68}, {37,69}, {37,70}, {37,71}, {37,72}, {37,73}, {37,74}, 
    {37,75}, {37,76}, {37,77}, {37,78}, {37,79}, {37,80}, {37,81}, {66,136}, 
    {37,82}, {37,83}, {37,84}, {37,85}, {37,86}, {37,87}, {37,88}, {37,89}, 
    {37,90}, {37,91}, {37,92}, {37,93}, {37,94}, {37,95}, {37,96}, {38,1}, 
    {38,2}, {38,3}, {38,4}, {38,5}, {38,6}, {38,7}, {38,8}, {38,9}, {38,10}, 
    {38,11}, {38,12}, {38,13}, {38,14}, {38,15}, {38,16}, {38,17}, {38,18}, 
    {38,19}, {38,20}, {66,137}, {66,138}, {38,21}, {38,22}, {66,139}, {38,23}, 
    {38,24}, {38,25}, {66,140}, {38,26}, {38,33}, {38,34}, {38,35}, {38,36}, 
    {38,37}, {38,38}, {66,141}, {66,142}, {38,39}, {66,143}, {38,40}, {66,144}, 
    {38,41}, {66,145}, {38,42}, {38,43}, {38,44}, {38,45}, {66,146}, {66,147}, 
    {38,46}, {38,47}, {38,48}, {38,49}, {38,50}, {38,51}, {66,148}, {38,52}, 
    {38,53}, {38,54}, {38,55}, {38,56}, {38,57}, {38,58}, {38,65}, {38,66}, 
    {38,67}, {38,68}, {66,149}, {38,69}, {38,70}, {38,71}, {38,72}, {38,73}, 
    {38,74}, {38,75}, {66,150}, {38,76}, {38,77}, {38,78}, {38,79}, {38,80}, 
    {38,81}, {38,82}, {38,83}, {38,84}, {38,85}, {38,86}, {38,87}, {38,88}, 
    {38,89}, {38,90}, {38,91}, {38,92}, {38,93}, {38,94}, {66,151}, {38,95}, 
    {38,96}, {39,1}, {39,2}, {39,3}, {39,4}, {39,5}, {66,152}, {39,6}, 
    {39,7}, {39,8}, {66,153}, {39,9}, {39,10}, {39,11}, {66,154}, {39,12}, 
    {39,13}, {39,14}, {39,15}, {39,16}, {39,17}, {39,18}, {66,155}, {66,156}, 
    {39,19}, {39,20}, {39,21}, {39,22}, {39,23}, {39,24}, {39,25}, {39,26}, 
    {39,33}, {39,34}, {39,35}, {39,36}, {39,37}, {39,38}, {39,39}, {39,40}, 
    {39,41}, {39,42}, {39,43}, {39,44}, {39,45}, {39,46}, {39,47}, {39,48}, 
    {39,49}, {39,50}, {39,51}, {39,52}, {39,53}, {39,54}, {39,55}, {66,157}, 
    {39,56}, {39,57}, {39,58}, {39,65}, {39,66}, {39,67}, {66,158}, {66,159}, 
    {39,68}, {39,69}, {66,160}, {39,70}, {39,71}, {39,72}, {66,161}, {39,73}, 
    {39,74}, {39,75}, {39,76}, {39,77}, {39,78}, {39,79}, {66,162}, {66,163}, 
    {39,80}, {39,81}, {39,82}, {66,164}, {39,83}, {39,84}, {39,85}, {39,86}, 
    {39,87}, {39,88}, {66,165}, {39,89}, {39,90}, {39,91}, {39,92}, {39,93}, 
    {39,94}, {39,95}, {39,96}, {40,1}, {40,2}, {40,3}, {40,4}, {40,5}, 
    {40,6}, {40,7}, {40,8}, {40,9}, {40,10}, {40,11}, {66,166}, {66,167}, 
    {40,12}, {40,13}, {40,14}, {40,15}, {40,16}, {40,17}, {40,18}, {40,19}, 
    {40,20}, {40,21}, {40,22}, {40,23}, {40,24}, {40,25}, {40,26}, {40,33}, 
    {40,34}, {40,35}, {40,36}, {40,37}, {40,38}, {40,39}, {40,40}, {40,41}, 
    {40,42}, {40,43}, {40,44}, {40,45}, {40,46}, {40,47}, {40,48}, {40,49}, 
    {40,50}, {40,51}, {66,168}, {40,52}, {40,53}, {40,54}, {40,55}, {40,56}, 
    {40,57}, {40,58}, {40,65}, {40,66}, {40,67}, {40,68}, {40,69}, {40,70}, 
    {40,71}, {40,72}, {40,73}, {40,74}, {40,75}, {40,76}, {40,77}, {40,78}, 
    {40,79}, {40,80}, {40,81}, {40,82}, {40,83}, {40,84}, {66,169}, {40,85}, 
    {40,86}, {40,87}, {40,88}, {40,89}, {40,90}, {40,91}, {40,92}, {40,93}, 
    {40,94}, {40,95}, {40,96}, {41,1}, {41,2}, {41,3}, {41,4}, {41,5}, 
    {41,6}, {41,7}, {41,8}, {41,9}, {41,10}, {41,11}, {41,12}, {41,13}, 
    {41,14}, {41,15}, {66,170}, {41,16}, {41,17}, {41,18}, {41,19}, {41,20}, 
    {41,21}, {41,22}, {41,23}, {41,24}, {41,25}, {41,26}, {41,33}, {41,34}, 
    {41,35}, {41,36}, {66,171}, {41,37}, {41,38}, {66,172}, {41,39}, {66,173}, 
    {41,40}, {41,41}, {41,42}, {41,43}, {41,44}, {41,45}, {41,46}, {41,47}, 
    {41,48}, {41,49}, {41,50}, {41,51}, {41,52}, {41,53}, {41,54}, {41,55}, 
    {41,56}, {41,57}, {41,58}, {41,65}, {41,66}, {41,67}, {41,68}, {41,69}, 
    {41,70}, {41,71}, {41,72}, {41,73}, {41,74}, {41,75}, {41,76}, {41,77}, 
    {41,78}, {41,79}, {66,174}, {66,175}, {41,80}, {41,81}, {66,176}, {41,82}, 
    {41,83}, {41,84}, {66,177}, {41,85}, {41,86}, {41,87}, {41,88}, {41,89}, 
    {41,90}, {41,91}, {66,178}, {66,179}, {41,92}, {41,93}, {41,94}, {66,180}, 
    {66,181}, {41,95}, {41,96}, {42,1}, {42,2}, {66,182}, {66,183}, {66,184}, 
    {42,3}, {42,4}, {66,185}, {42,5}, {66,186}, {42,6}, {66,187}, {42,7}, 
    {42,8}, {42,9}, {42,10}, {42,11}, {42,12}, {42,13}, {66,188}, {66,189}, 
    {42,14}, {66,190}, {67,97}, {67,98}, {67,99}, {42,15}, {42,16}, {42,17}, 
    {42,18}, {42,19}, {67,100}, {67,101}, {42,20}, {42,21}, {67,102}, {42,22}, 
    {42,23}, {42,24}, {67,103}, {42,25}, {42,26}, {42,33}, {42,34}, {42,35}, 
    {42,36}, {42,37}, {67,104}, {67,105}, {42,38}, {67,106}, {67,107}, 
    {67,108}, {42,39}, {42,40}, {42,41}, {42,42}, {42,43}, {42,44}, {67,109}, 
    {42,45}, {42,46}, {42,47}, {67,110}, {42,48}, {67,111}, {42,49}, {67,112}, 
    {42,50}, {42,51}, {42,52}, {42,53}, {42,54}, {42,55}, {42,56}, {67,113}, 
    {42,57}, {42,58}, {42,65}, {42,66}, {67,114}, {42,67}, {42,68}, {42,69}, 
    {42,70}, {42,71}, {42,72}, {42,73}, {42,74}, {42,75}, {42,76}, {42,77}, 
    {42,78}, {42,79}, {42,80}, {42,81}, {42,82}, {42,83}, {42,84}, {42,85}, 
    {42,86}, {42,87}, {42,88}, {42,89}, {42,90}, {42,91}, {42,92}, {42,93}, 
    {42,94}, {42,95}, {42,96}, {43,1}, {43,2}, {43,3}, {43,4}, {67,115}, 
    {67,116}, {43,5}, {43,6}, {67,117}, {43,7}, {43,8}, {43,9}, {67,118}, 
    {43,10}, {43,11}, {43,12}, {43,13}, {43,14}, {43,15}, {43,16}, {67,119}, 
    {67,120}, {43,17}, {67,121}, {67,122}, {67,123}, {43,18}, {43,19}, 
    {43,20}, {43,21}, {43,22}, {43,23}, {67,124}, {67,125}, {43,24}, {43,25}, 
    {67,126}, {43,26}, {43,33}, {43,34}, {67,127}, {43,35}, {43,36}, {43,37}, 
    {43,38}, {43,39}, {43,40}, {43,41}, {67,128}, {67,129}, {43,42}, {67,130}, 
    {43,43}, {67,131}, {43,44}, {43,45}, {43,46}, {43,47}, {43,48}, {43,49}, 
    {67,132}, {43,50}, {43,51}, {43,52}, {67,133}, {43,53}, {43,54}, {43,55}, 
    {43,56}, {43,57}, {43,58}, {43,65}, {43,66}, {43,67}, {43,68}, {43,69}, 
    {43,70}, {43,71}, {43,72}, {43,73}, {67,134}, {43,74}, {43,75}, {43,76}, 
    {43,77}, {43,78}, {43,79}, {43,80}, {67,135}, {43,81}, {43,82}, {43,83}, 
    {67,136}, {43,84}, {43,85}, {43,86}, {43,87}, {43,88}, {43,89}, {43,90}, 
    {43,91}, {43,92}, {43,93}, {43,94}, {43,95}, {43,96}, {44,1}, {44,2}, 
    {44,3}, {67,137}, {44,4}, {44,5}, {44,6}, {44,7}, {44,8}, {44,9}, {67,138}, 
    {67,139}, {44,10}, {44,11}, {67,140}, {44,12}, {44,13}, {44,14}, {67,141}, 
    {44,15}, {44,16}, {44,17}, {44,18}, {44,19}, {44,20}, {44,21}, {67,142}, 
    {67,143}, {44,22}, {67,144}, {44,23}, {67,145}, {44,24}, {44,25}, {44,26}, 
    {44,33}, {44,34}, {44,35}, {67,146}, {44,36}, {44,37}, {44,38}, {67,147}, 
    {44,39}, {44,40}, {44,41}, {67,148}, {44,42}, {44,43}, {44,44}, {44,45}, 
    {44,46}, {44,47}, {44,48}, {44,49}, {44,50}, {44,51}, {44,52}, {44,53}, 
    {67,149}, {44,54}, {44,55}, {44,56}, {44,57}, {44,58}, {44,65}, {44,66}, 
    {44,67}, {44,68}, {44,69}, {44,70}, {44,71}, {44,72}, {44,73}, {44,74}, 
    {44,75}, {44,76}, {44,77}, {44,78}, {44,79}, {44,80}, {44,81}, {44,82}, 
    {44,83}, {44,84}, {44,85}, {44,86}, {44,87}, {44,88}, {44,89}, {44,90}, 
    {44,91}, {44,92}, {44,93}, {67,150}, {44,94}, {44,95}, {44,96}, {67,151}, 
    {45,1}, {45,2}, {45,3}, {67,152}, {45,4}, {45,5}, {45,6}, {45,7}, {45,8}, 
    {45,9}, {45,10}, {67,153}, {67,154}, {45,11}, {67,155}, {45,12}, {67,156}, 
    {45,13}, {45,14}, {45,15}, {45,16}, {45,17}, {45,18}, {67,157}, {45,19}, 
    {45,20}, {45,21}, {45,22}, {45,23}, {45,24}, {45,25}, {45,26}, {45,33}, 
    {45,34}, {45,35}, {45,36}, {45,37}, {45,38}, {45,39}, {67,158}, {45,40}, 
    {45,41}, {45,42}, {45,43}, {45,44}, {45,45}, {45,46}, {45,47}, {45,48}, 
    {45,49}, {45,50}, {67,159}, {67,160}, {45,51}, {45,52}, {67,161}, {45,53}, 
    {45,54}, {45,55}, {67,162}, {45,56}, {45,57}, {45,58}, {45,65}, {45,66}, 
    {45,67}, {45,68}, {67,163}, {67,164}, {45,69}, {67,165}, {45,70}, {67,166}, 
    {45,71}, {45,72}, {45,73}, {45,74}, {45,75}, {45,76}, {67,167}, {45,77}, 
    {45,78}, {45,79}, {45,80}, {45,81}, {45,82}, {45,83}, {45,84}, {45,85}, 
    {45,86}, {45,87}, {45,88}, {45,89}, {45,90}, {45,91}, {45,92}, {45,93}, 
    {45,94}, {45,95}, {67,168}, {45,96}, {46,1}, {46,2}, {46,3}, {46,4}, 
    {46,5}, {46,6}, {67,169}, {46,7}, {46,8}, {46,9}, {67,170}, {46,10}, 
    {46,11}, {46,12}, {46,13}, {46,14}, {46,15}, {46,16}, {46,17}, {46,18}, 
    {46,19}, {46,20}, {46,21}, {46,22}, {46,23}, {46,24}, {46,25}, {46,26}, 
    {46,33}, {46,34}, {46,35}, {46,36}, {46,37}, {46,38}, {67,171}, {46,39}, 
    {46,40}, {46,41}, {67,172}, {46,42}, {46,43}, {46,44}, {67,173}, {46,45}, 
    {46,46}, {46,47}, {46,48}, {46,49}, {46,50}, {46,51}, {67,174}, {67,175}, 
    {46,52}, {67,176}, {46,53}, {67,177}, {46,54}, {46,55}, {46,56}, {46,57}, 
    {46,58}, {46,65}, {67,178}, {46,66}, {46,67}, {46,68}, {67,179}, {46,69}, 
    {46,70}, {46,71}, {67,180}, {46,72}, {46,73}, {46,74}, {46,75}, {46,76}, 
    {46,77}, {46,78}, {67,181}, {46,79}, {46,80}, {46,81}, {46,82}, {67,182}, 
    {46,83}, {46,84}, {46,85}, {46,86}, {46,87}, {46,88}, {67,183}, {67,184}, 
    {46,89}, {46,90}, {67,185}, {46,91}, {46,92}, {46,93}, {67,186}, {46,94}, 
    {46,95}, {46,96}, {47,1}, {47,2}, {47,3}, {47,4}, {67,187}, {67,188}, 
    {47,5}, {67,189}, {47,6}, {67,190}, {47,7}, {47,8}, {47,9}, {47,10}, 
    {47,11}, {47,12}, {47,13}, {47,14}, {47,15}, {47,16}, {47,17}, {47,18}, 
    {47,19}, {47,20}, {47,21}, {47,22}, {47,23}, {47,24}, {47,25}, {47,26}, 
    {47,33}, {47,34}, {47,35}, {47,36}, {47,37}, {47,38}, {47,39}, {47,40}, 
    {47,41}, {47,42}, {47,43}, {47,44}, {47,45}, {47,46}, {68,97}, {68,98}, 
    {47,47}, {47,48}, {68,99}, {47,49}, {47,50}, {68,100}, {68,101}, {68,102}, 
    {47,51}, {47,52}, {47,53}, {47,54}, {47,55}, {47,56}, {68,103}, {68,104}, 
    {47,57}, {68,105}, {47,58}, {68,106}, {47,65}, {47,66}, {47,67}, {47,68}, 
    {47,69}, {47,70}, {68,107}, {68,108}, {47,71}, {47,72}, {68,109}, {47,73}, 
    {47,74}, {47,75}, {68,110}, {47,76}, {47,77}, {47,78}, {47,79}, {47,80}, 
    {47,81}, {47,82}, {68,111}, {68,112}, {47,83}, {68,113}, {47,84}, {68,114}, 
    {47,85}, {47,86}, {47,87}, {47,88}, {47,89}, {47,90}, {68,115}, {68,116}, 
    {47,91}, {47,92}, {68,117}, {47,93}, {47,94}, {47,95}, {68,118}, {47,96}, 
    {48,1}, {48,2}, {48,3}, {48,4}, {48,5}, {48,6}, {68,119}, {68,120}, 
    {48,7}, {68,121}, {68,122}, {68,123}, {48,8}, {48,9}, {48,10}, {48,11}, 
    {48,12}, {48,13}, {68,124}, {68,125}, {48,14}, {48,15}, {48,16}, {48,17}, 
    {48,18}, {48,19}, {48,20}, {48,21}, {48,22}, {48,23}, {48,24}, {48,25}, 
    {48,26}, {48,33}, {48,34}, {48,35}, {48,36}, {48,37}, {48,38}, {68,126}, 
    {48,39}, {48,40}, {48,41}, {48,42}, {48,43}, {48,44}, {48,45}, {48,46}, 
    {48,47}, {48,48}, {48,49}, {48,50}, {48,51}, {48,52}, {48,53}, {48,54}, 
    {48,55}, {48,56}, {48,57}, {48,58}, {48,65}, {48,66}, {48,67}, {48,68}, 
    {48,69}, {48,70}, {48,71}, {48,72}, {48,73}, {48,74}, {48,75}, {48,76}, 
    {48,77}, {48,78}, {68,127}, {68,128}, {48,79}, {48,80}, {68,129}, {48,81}, 
    {48,82}, {68,130}, {68,131}, {48,83}, {48,84}, {48,85}, {48,86}, {48,87}, 
    {48,88}, {48,89}, {68,132}, {68,133}, {48,90}, {68,134}, {68,135}, 
    {68,136}, {48,91}, {48,92}, {48,93}, {48,94}, {48,95}, {48,96}, {68,137}, 
    {68,138}, {49,1}, {49,2}, {68,139}, {49,3}, {49,4}, {49,5}, {68,140}, 
    {49,6}, {49,7}, {49,8}, {49,9}, {49,10}, {49,11}, {49,12}, {68,141}, 
    {68,142}, {49,13}, {68,143}, {49,14}, {68,144}, {49,15}, {49,16}, {49,17}, 
    {49,18}, {49,19}, {49,20}, {68,145}, {49,21}, {49,22}, {49,23}, {68,146}, 
    {49,24}, {49,25}, {49,26}, {68,147}, {49,33}, {49,34}, {49,35}, {49,36}, 
    {49,37}, {49,38}, {49,39}, {68,148}, {68,149}, {49,40}, {68,150}, {68,151}, 
    {68,152}, {49,41}, {49,42}, {49,43}, {49,44}, {49,45}, {49,46}, {68,153}, 
    {49,47}, {49,48}, {49,49}, {49,50}, {49,51}, {49,52}, {49,53}, {49,54}, 
    {49,55}, {49,56}, {49,57}, {49,58}, {49,65}, {49,66}, {49,67}, {49,68}, 
    {49,69}, {49,70}, {49,71}, {49,72}, {49,73}, {49,74}, {49,75}, {49,76}, 
    {49,77}, {49,78}, {49,79}, {68,154}, {68,155}, {49,80}, {49,81}, {68,156}, 
    {49,82}, {49,83}, {49,84}, {68,157}, {49,85}, {49,86}, {49,87}, {49,88}, 
    {49,89}, {49,90}, {49,91}, {68,158}, {68,159}, {49,92}, {68,160}, {49,93}, 
    {68,161}, {49,94}, {49,95}, {49,96}, {50,1}, {50,2}, {50,3}, {68,162}, 
    {68,163}, {50,4}, {50,5}, {68,164}, {50,6}, {50,7}, {50,8}, {68,165}, 
    {50,9}, {50,10}, {50,11}, {50,12}, {50,13}, {50,14}, {50,15}, {68,166}, 
    {50,16}, {50,17}, {50,18}, {50,19}, {68,167}, {50,20}, {50,21}, {50,22}, 
    {50,23}, {50,24}, {50,25}, {68,168}, {50,26}, {50,33}, {50,34}, {50,35}, 
    {50,36}, {50,37}, {50,38}, {50,39}, {50,40}, {50,41}, {50,42}, {50,43}, 
    {50,44}, {50,45}, {50,46}, {50,47}, {50,48}, {50,49}, {50,50}, {50,51}, 
    {68,169}, {50,52}, {50,53}, {50,54}, {50,55}, {50,56}, {50,57}, {68,170}, 
    {50,58}, {50,65}, {50,66}, {50,67}, {50,68}, {50,69}, {50,70}, {68,171}, 
    {50,71}, {50,72}, {50,73}, {50,74}, {50,75}, {50,76}, {50,77}, {50,78}, 
    {50,79}, {50,80}, {50,81}, {50,82}, {50,83}, {50,84}, {50,85}, {50,86}, 
    {50,87}, {50,88}, {50,89}, {68,172}, {50,90}, {50,91}, {50,92}, {50,93}, 
    {50,94}, {50,95}, {50,96}, {51,1}, {51,2}, {51,3}, {51,4}, {51,5}, 
    {51,6}, {51,7}, {51,8}, {51,9}, {51,10}, {51,11}, {51,12}, {51,13}, 
    {51,14}, {51,15}, {51,16}, {51,17}, {51,18}, {51,19}, {51,20}, {68,173}, 
    {68,174}, {51,21}, {51,22}, {68,175}, {51,23}, {51,24}, {51,25}, {68,176}, 
    {51,26}, {51,33}, {51,34}, {51,35}, {51,36}, {51,37}, {51,38}, {68,177}, 
    {68,178}, {51,39}, {68,179}, {51,40}, {68,180}, {51,41}, {51,42}, {51,43}, 
    {51,44}, {51,45}, {51,46}, {68,181}, {51,47}, {51,48}, {51,49}, {68,182}, 
    {51,50}, {51,51}, {51,52}, {68,183}, {51,53}, {51,54}, {51,55}, {51,56}, 
    {51,57}, {51,58}, {51,65}, {51,66}, {51,67}, {51,68}, {51,69}, {51,70}, 
    {68,184}, {51,71}, {51,72}, {51,73}, {51,74}, {51,75}, {51,76}, {68,185}, 
    {51,77}, {51,78}, {51,79}, {51,80}, {51,81}, {51,82}, {51,83}, {51,84}, 
    {51,85}, {51,86}, {51,87}, {51,88}, {51,89}, {51,90}, {51,91}, {51,92}, 
    {51,93}, {51,94}, {51,95}, {51,96}, {68,186}, {52,1}, {52,2}, {52,3}, 
    {52,4}, {52,5}, {52,6}, {68,187}, {68,188}, {52,7}, {52,8}, {68,189}, 
    {52,9}, {52,10}, {52,11}, {68,190}, {52,12}, {52,13}, {52,14}, {52,15}, 
    {52,16}, {52,17}, {52,18}, {69,97}, {69,98}, {52,19}, {69,99}, {52,20}, 
    {69,100}, {52,21}, {52,22}, {52,23}, {52,24}, {52,25}, {52,26}, {69,101}, 
    {52,33}, {52,34}, {52,35}, {69,102}, {52,36}, {52,37}, {52,38}, {69,103}, 
    {52,39}, {52,40}, {52,41}, {52,42}, {52,43}, {52,44}, {52,45}, {69,104}, 
    {52,46}, {52,47}, {52,48}, {52,49}, {52,50}, {52,51}, {52,52}, {52,53}, 
    {52,54}, {52,55}, {52,56}, {69,105}, {69,106}, {52,57}, {52,58}, {69,107}, 
    {52,65}, {52,66}, {52,67}, {69,108}, {52,68}, {52,69}, {52,70}, {52,71}, 
    {52,72}, {52,73}, {52,74}, {69,109}, {69,110}, {52,75}, {52,76}, {52,77}, 
    {69,111}, {52,78}, {52,79}, {52,80}, {52,81}, {52,82}, {52,83}, {52,84}, 
    {52,85}, {52,86}, {52,87}, {52,88}, {52,89}, {52,90}, {52,91}, {52,92}, 
    {52,93}, {52,94}, {52,95}, {52,96}, {53,1}, {53,2}, {53,3}, {53,4}, 
    {53,5}, {53,6}, {53,7}, {53,8}, {53,9}, {53,10}, {53,11}, {53,12}, 
    {53,13}, {53,14}, {53,15}, {69,112}, {69,113}, {53,16}, {53,17}, {69,114}, 
    {53,18}, {53,19}, {53,20}, {69,115}, {53,21}, {53,22}, {53,23}, {53,24}, 
    {53,25}, {53,26}, {53,33}, {69,116}, {69,117}, {53,34}, {69,118}, {53,35}, 
    {69,119}, {53,36}, {53,37}, {53,38}, {53,39}, {53,40}, {53,41}, {69,120}, 
    {69,121}, {53,42}, {53,43}, {69,122}, {53,44}, {53,45}, {53,46}, {69,123}, 
    {69,124}, {53,47}, {53,48}, {53,49}, {53,50}, {53,51}, {53,52}, {69,125}, 
    {69,126}, {53,53}, {69,127}, {69,128}, {69,129}, {53,54}, {53,55}, 
    {53,56}, {53,57}, {53,58}, {53,65}, {69,130}, {69,131}, {53,66}, {53,67}, 
    {69,132}, {53,68}, {53,69}, {53,70}, {69,133}, {53,71}, {53,72}, {53,73}, 
    {53,74}, {53,75}, {53,76}, {53,77}, {69,134}, {69,135}, {53,78}, {69,136}, 
    {69,137}, {69,138}, {53,79}, {53,80}, {53,81}, {53,82}, {53,83}, {53,84}, 
    {69,139}, {53,85}, {53,86}, {53,87}, {53,88}, {53,89}, {53,90}, {53,91}, 
    {53,92}, {53,93}, {53,94}, {53,95}, {53,96}, {54,1}, {54,2}, {54,3}, 
    {54,4}, {54,5}, {54,6}, {54,7}, {54,8}, {69,140}, {54,9}, {54,10}, 
    {54,11}, {54,12}, {54,13}, {54,14}, {54,15}, {54,16}, {54,17}, {54,18}, 
    {54,19}, {54,20}, {54,21}, {54,22}, {54,23}, {54,24}, {54,25}, {54,26}, 
    {54,33}, {54,34}, {54,35}, {54,36}, {54,37}, {54,38}, {54,39}, {54,40}, 
    {54,41}, {54,42}, {54,43}, {54,44}, {54,45}, {54,46}, {54,47}, {54,48}, 
    {69,141}, {69,142}, {54,49}, {54,50}, {69,143}, {54,51}, {54,52}, {54,53}, 
    {69,144}, {54,54}, {69,145}, {54,55}, {54,56}, {54,57}, {54,58}, {54,65}, 
    {69,146}, {69,147}, {54,66}, {69,148}, {69,149}, {69,150}, {54,67}, 
    {54,68}, {54,69}, {54,70}, {54,71}, {54,72}, {69,151}, {69,152}, {54,73}, 
    {54,74}, {69,153}, {54,75}, {54,76}, {54,77}, {69,154}, {54,78}, {54,79}, 
    {54,80}, {54,81}, {54,82}, {54,83}, {54,84}, {69,155}, {69,156}, {54,85}, 
    {69,157}, {54,86}, {69,158}, {54,87}, {54,88}, {54,89}, {54,90}, {54,91}, 
    {54,92}, {69,159}, {54,93}, {54,94}, {54,95}, {69,160}, {54,96}, {55,1}, 
    {55,2}, {55,3}, {55,4}, {55,5}, {55,6}, {55,7}, {55,8}, {55,9}, {55,10}, 
    {55,11}, {55,12}, {55,13}, {55,14}, {69,161}, {55,15}, {55,16}, {55,17}, 
    {55,18}, {55,19}, {55,20}, {55,21}, {69,162}, {55,22}, {55,23}, {55,24}, 
    {69,163}, {55,25}, {55,26}, {55,33}, {55,34}, {55,35}, {55,36}, {55,37}, 
    {55,38}, {55,39}, {55,40}, {55,41}, {55,42}, {55,43}, {55,44}, {55,45}, 
    {55,46}, {55,47}, {55,48}, {55,49}, {55,50}, {55,51}, {55,52}, {55,53}, 
    {69,164}, {69,165}, {55,54}, {55,55}, {69,166}, {55,56}, {55,57}, {55,58}, 
    {69,167}, {55,65}, {55,66}, {55,67}, {55,68}, {55,69}, {55,70}, {55,71}, 
    {69,168}, {69,169}, {55,72}, {69,170}, {55,73}, {69,171}, {55,74}, 
    {55,75}, {55,76}, {55,77}, {69,172}, {55,78}, {69,173}, {55,79}, {55,80}, 
    {55,81}, {69,174}, {55,82}, {55,83}, {55,84}, {55,85}, {55,86}, {55,87}, 
    {55,88}, {55,89}, {55,90}, {55,91}, {55,92}, {55,93}, {55,94}, {55,95}, 
    {55,96}, {56,1}, {56,2}, {56,3}, {56,4}, {56,5}, {56,6}, {56,7}, {56,8}, 
    {69,175}, {56,9}, {56,10}, {56,11}, {56,12}, {56,13}, {56,14}, {56,15}, 
    {56,16}, {56,17}, {56,18}, {56,19}, {56,20}, {56,21}, {56,22}, {56,23}, 
    {56,24}, {56,25}, {56,26}, {56,33}, {56,34}, {56,35}, {56,36}, {56,37}, 
    {56,38}, {56,39}, {56,40}, {56,41}, {69,176}, {56,42}, {56,43}, {56,44}, 
    {69,177}, {56,45}, {56,46}, {56,47}, {56,48}, {56,49}, {56,50}, {56,51}, 
    {56,52}, {56,53}, {56,54}, {56,55}, {56,56}, {56,57}, {56,58}, {69,178}, 
    {56,65}, {69,179}, {56,66}, {56,67}, {56,68}, {56,69}, {56,70}, {56,71}, 
    {69,180}, {56,72}, {56,73}, {56,74}, {56,75}, {56,76}, {56,77}, {56,78}, 
    {56,79}, {56,80}, {56,81}, {56,82}, {56,83}, {56,84}, {56,85}, {56,86}, 
    {56,87}, {56,88}, {56,89}, {56,90}, {56,91}, {56,92}, {56,93}, {56,94}, 
    {56,95}, {56,96}, {57,1}, {57,2}, {69,181}, {69,182}, {57,3}, {57,4}, 
    {69,183}, {57,5}, {57,6}, {57,7}, {69,184}, {57,8}, {57,9}, {57,10}, 
    {57,11}, {57,12}, {57,13}, {57,14}, {69,185}, {69,186}, {57,15}, {69,187}, 
    {57,16}, {69,188}, {57,17}, {57,18}, {57,19}, {57,20}, {57,21}, {57,22}, 
    {69,189}, {57,23}, {57,24}, {57,25}, {57,26}, {57,33}, {57,34}, {57,35}, 
    {57,36}, {57,37}, {57,38}, {57,39}, {57,40}, {57,41}, {57,42}, {57,43}, 
    {57,44}, {57,45}, {57,46}, {57,47}, {69,190}, {57,48}, {57,49}, {57,50}, 
    {57,51}, {57,52}, {57,53}, {57,54}, {70,97}, {57,55}, {57,56}, {57,57}, 
    {57,58}, {57,65}, {57,66}, {57,67}, {57,68}, {57,69}, {57,70}, {57,71}, 
    {57,72}, {57,73}, {57,74}, {57,75}, {57,76}, {57,77}, {57,78}, {57,79}, 
    {57,80}, {57,81}, {57,82}, {57,83}, {57,84}, {57,85}, {57,86}, {57,87}, 
    {70,98}, {70,99}, {57,88}, {57,89}, {70,100}, {57,90}, {57,91}, {57,92}, 
    {70,101}, {57,93}, {57,94}, {57,95}, {57,96}, {58,1}, {58,2}, {58,3}, 
    {70,102}, {70,103}, {58,4}, {58,5}, {58,6}, {70,104}, {58,7}, {58,8}, 
    {58,9}, {58,10}, {58,11}, {58,12}, {70,105}, {58,13}, {58,14}, {58,15}, 
    {70,106}, {58,16}, {58,17}, {58,18}, {70,107}, {58,19}, {58,20}, {58,21}, 
    {58,22}, {58,23}, {58,24}, {58,25}, {70,108}, {58,26}, {58,33}, {58,34}, 
    {58,35}, {70,109}, {58,36}, {58,37}, {58,38}, {58,39}, {58,40}, {58,41}, 
    {70,110}, {70,111}, {58,42}, {58,43}, {70,112}, {58,44}, {58,45}, {70,113}, 
    {70,114}, {58,46}, {70,115}, {58,47}, {58,48}, {58,49}, {58,50}, {58,51}, 
    {70,116}, {70,117}, {58,52}, {70,118}, {58,53}, {58,54}, {58,55}, {58,56}, 
    {58,57}, {58,58}, {58,65}, {58,66}, {70,119}, {58,67}, {58,68}, {58,69}, 
    {70,120}, {58,70}, {58,71}, {58,72}, {70,121}, {58,73}, {58,74}, {58,75}, 
    {58,76}, {58,77}, {58,78}, {58,79}, {70,122}, {70,123}, {58,80}, {58,81}, 
    {58,82}, {58,83}, {58,84}, {58,85}, {58,86}, {58,87}, {58,88}, {58,89}, 
    {70,124}, {70,125}, {58,90}, {58,91}, {70,126}, {58,92}, {58,93}, {58,94}, 
    {70,127}, {58,95}, {58,96}, {59,1}, {59,2}, {59,3}, {59,4}, {59,5}, 
    {70,128}, {70,129}, {59,6}, {70,130}, {59,7}, {70,131}, {59,8}, {59,9}, 
    {59,10}, {59,11}, {59,12}, {59,13}, {70,132}, {70,133}, {70,134}, {59,14}, 
    {70,135}, {59,15}, {59,16}, {59,17}, {70,136}, {59,18}, {70,137}, {59,19}, 
    {59,20}, {59,21}, {59,22}, {59,23}, {70,138}, {70,139}, {59,24}, {70,140}, 
    {70,141}, {70,142}, {59,25}, {59,26}, {59,33}, {70,143}, {59,34}, {59,35}, 
    {70,144}, {70,145}, {59,36}, {59,37}, {70,146}, {59,38}, {59,39}, {59,40}, 
    {70,147}, {59,41}, {59,42}, {59,43}, {59,44}, {59,45}, {59,46}, {59,47}, 
    {70,148}, {70,149}, {59,48}, {70,150}, {70,151}, {70,152}, {59,49}, 
    {59,50}, {59,51}, {59,52}, {59,53}, {59,54}, {70,153}, {70,154}, {59,55}, 
    {59,56}, {59,57}, {59,58}, {59,65}, {59,66}, {59,67}, {59,68}, {59,69}, 
    {59,70}, {59,71}, {59,72}, {59,73}, {59,74}, {59,75}, {59,76}, {59,77}, 
    {59,78}, {59,79}, {59,80}, {59,81}, {59,82}, {59,83}, {59,84}, {59,85}, 
    {59,86}, {59,87}, {59,88}, {59,89}, {59,90}, {59,91}, {59,92}, {59,93}, 
    {59,94}, {59,95}, {59,96}, {60,1}, {60,2}, {60,3}, {60,4}, {60,5}, 
    {60,6}, {60,7}, {60,8}, {60,9}, {60,10}, {60,11}, {60,12}, {60,13}, 
    {60,14}, {60,15}, {60,16}, {60,17}, {60,18}, {70,155}, {70,156}, {60,19}, 
    {60,20}, {70,157}, {60,21}, {60,22}, {60,23}, {70,158}, {60,24}, {60,25}, 
    {60,26}, {60,33}, {60,34}, {60,35}, {60,36}, {70,159}, {70,160}, {60,37}, 
    {70,161}, {70,162}, {70,163}, {60,38}, {60,39}, {60,40}, {60,41}, {60,42}, 
    {60,43}, {70,164}, {70,165}, {60,44}, {60,45}, {70,166}, {60,46}, {60,47}, 
    {60,48}, {70,167}, {60,49}, {60,50}, {60,51}, {60,52}, {60,53}, {60,54}, 
    {60,55}, {70,168}, {70,169}, {60,56}, {70,170}, {60,57}, {70,171}, 
    {60,58}, {60,65}, {60,66}, {60,67}, {60,68}, {60,69}, {70,172}, {60,70}, 
    {60,71}, {60,72}, {70,173}, {60,73}, {60,74}, {60,75}, {70,174}, {60,76}, 
    {60,77}, {60,78}, {60,79}, {60,80}, {60,81}, {60,82}, {70,175}, {70,176}, 
    {60,83}, {60,84}, {70,177}, {70,178}, {60,85}, {60,86}, {60,87}, {60,88}, 
    {60,89}, {60,90}, {70,179}, {60,91}, {60,92}, {60,93}, {60,94}, {60,95}, 
    {60,96}, {61,1}, {70,180}, {61,2}, {61,3}, {61,4}, {61,5}, {61,6}, 
    {61,7}, {61,8}, {61,9}, {70,181}, {61,10}, {70,182}, {61,11}, {61,12}, 
    {61,13}, {61,14}, {61,15}, {61,16}, {61,17}, {61,18}, {70,183}, {70,184}, 
    {61,19}, {61,20}, {70,185}, {61,21}, {61,22}, {61,23}, {70,186}, {61,24}, 
    {61,25}, {61,26}, {61,33}, {61,34}, {61,35}, {61,36}, {70,187}, {70,188}, 
    {61,37}, {70,189}, {61,38}, {70,190}, {61,39}, {61,40}, {61,41}, {61,42}, 
    {61,43}, {61,44}, {71,97}, {61,45}, {61,46}, {61,47}, {61,48}, {61,49}, 
    {61,50}, {61,51}, {61,52}, {61,53}, {61,54}, {61,55}, {61,56}, {61,57}, 
    {61,58}, {61,65}, {61,66}, {61,67}, {61,68}, {61,69}, {61,70}, {71,98}, 
    {61,71}, {61,72}, {61,73}, {61,74}, {61,75}, {61,76}, {61,77}, {61,78}, 
    {61,79}, {61,80}, {61,81}, {61,82}, {61,83}, {61,84}, {61,85}, {61,86}, 
    {61,87}, {61,88}, {61,89}, {61,90}, {61,91}, {61,92}, {61,93}, {61,94}, 
    {61,95}, {61,96}, {62,1}, {62,2}, {62,3}, {62,4}, {62,5}, {62,6}, {62,7}, 
    {62,8}, {71,99}, {62,9}, {62,10}, {62,11}, {71,100}, {62,12}, {62,13}, 
    {62,14}, {62,15}, {62,16}, {62,17}, {62,18}, {62,19}, {62,20}, {62,21}, 
    {62,22}, {62,23}, {62,24}, {62,25}, {62,26}, {62,33}, {62,34}, {62,35}, 
    {62,36}, {62,37}, {62,38}, {62,39}, {62,40}, {71,101}, {62,41}, {62,42}, 
    {62,43}, {71,102}, {62,44}, {62,45}, {62,46}, {71,103}, {62,47}, {62,48}, 
    {62,49}, {62,50}, {62,51}, {62,52}, {62,53}, {62,54}, {71,104}, {62,55}, 
    {71,105}, {62,56}, {62,57}, {62,58}, {62,65}, {62,66}, {62,67}, {62,68}, 
    {62,69}, {71,106}, {71,107}, {62,70}, {62,71}, {71,108}, {62,72}, {62,73}, 
    {71,109}, {71,110}, {62,74}, {71,111}, {62,75}, {62,76}, {62,77}, {62,78}, 
    {62,79}, {71,112}, {71,113}, {62,80}, {71,114}, {62,81}, {71,115}, 
    {62,82}, {62,83}, {62,84}, {62,85}, {62,86}, {62,87}, {71,116}, {62,88}, 
    {62,89}, {62,90}, {62,91}, {62,92}, {62,93}, {62,94}, {62,95}, {62,96}, 
    {63,1}, {63,2}, {63,3}, {63,4}, {63,5}, {63,6}, {63,7}, {63,8}, {63,9}, 
    {63,10}, {63,11}, {71,117}, {63,12}, {63,13}, {63,14}, {63,15}, {63,16}, 
    {63,17}, {63,18}, {63,19}, {63,20}, {63,21}, {63,22}, {63,23}, {63,24}, 
    {63,25}, {63,26}, {63,33}, {63,34}, {63,35}, {63,36}, {63,37}, {63,38}, 
    {63,39}, {63,40}, {63,41}, {63,42}, {63,43}, {63,44}, {63,45}, {63,46}, 
    {63,47}, {63,48}, {63,49}, {63,50}, {63,51}, {71,118}, {63,52}, {63,53}, 
    {63,54}, {71,119}, {63,55}, {63,56}, {63,57}, {71,120}, {63,58}, {63,65}, 
    {63,66}, {63,67}, {63,68}, {63,69}, {63,70}, {71,121}, {63,71}, {63,72}, 
    {71,122}, {63,73}, {63,74}, {63,75}, {63,76}, {63,77}, {63,78}, {63,79}, 
    {63,80}, {71,123}, {63,81}, {63,82}, {63,83}, {71,124}, {63,84}, {63,85}, 
    {63,86}, {71,125}, {63,87}, {63,88}, {63,89}, {63,90}, {63,91}, {63,92}, 
    {63,93}, {71,126}, {63,94}, {63,95}, {71,127}, {63,96}, {71,128}, {64,1}, 
    {64,2}, {64,3}, {64,4}, {64,5}, {64,6}, {71,129}, {64,7}, {64,8}, {64,9}, 
    {71,130}, {64,10}, {64,11}, {64,12}, {71,131}, {64,13}, {64,14}, {64,15}, 
    {64,16}, {64,17}, {64,18}, {64,19}, {71,132}, {71,133}, {64,20}, {71,134}, 
    {64,21}, {64,22}, {64,23}, {64,24}, {64,25}, {64,26}, {64,33}, {64,34}, 
    {64,35}, {64,36}, {64,37}, {64,38}, {64,39}, {64,40}, {64,41}, {64,42}, 
    {64,43}, {64,44}, {64,45}, {64,46}, {64,47}, {64,48}, {64,49}, {64,50}, 
    {64,51}, {64,52}, {64,53}, {64,54}, {64,55}, {64,56}, {64,57}, {64,58}, 
    {64,65}, {64,66}, {64,67}, {64,68}, {71,135}, {71,136}, {64,69}, {64,70}, 
    {71,137}, {64,71}, {64,72}, {64,73}, {71,138}, {64,74}, {64,75}, {64,76}, 
    {64,77}, {64,78}, {64,79}, {64,80}, {71,139}, {71,140}, {64,81}, {71,141}, 
    {64,82}, {71,142}, {64,83}, {64,84}, {64,85}, {64,86}, {64,87}, {64,88}, 
    {71,143}, {71,144}, {64,89}, {64,90}, {71,145}, {64,91}, {64,92}, {64,93}, 
    {71,146}, {64,94}, {64,95}, {64,96}, {65,1}, {71,147}, {65,2}, {65,3}, 
    {71,148}, {71,149}, {65,4}, {71,150}, {65,5}, {71,151}, {65,6}, {65,7}, 
    {65,8}, {65,9}, {65,10}, {65,11}, {71,152}, {71,153}, {65,12}, {65,13}, 
    {71,154}, {65,14}, {65,15}, {65,16}, {71,155}, {65,17}, {65,18}, {65,19}, 
    {65,20}, {65,21}, {65,22}, {65,23}, {71,156}, {71,157}, {65,24}, {71,158}, 
    {71,159}, {71,160}, {65,25}, {65,26}, {65,33}, {65,34}, {65,35}, {65,36}, 
    {71,161}, {65,37}, {65,38}, {65,39}, {65,40}, {65,41}, {65,42}, {65,43}, 
    {65,44}, {65,45}, {65,46}, {65,47}, {65,48}, {65,49}, {65,50}, {65,51}, 
    {65,52}, {65,53}, {65,54}, {65,55}, {65,56}, {71,162}, {65,57}, {65,58}, 
    {65,65}, {65,66}, {65,67}, {65,68}, {65,69}, {65,70}, {65,71}, {65,72}, 
    {65,73}, {65,74}, {65,75}, {65,76}, {65,77}, {65,78}, {65,79}, {65,80}, 
    {65,81}, {65,82}, {65,83}, {65,84}, {65,85}, {65,86}, {65,87}, {65,88}, 
    {65,89}, {65,90}, {65,91}, {65,92}, {65,93}, {65,94}, {65,95}, {65,96}, 
    {71,163}, {71,164}, {66,1}, {66,2}, {71,165}, {66,3}, {66,4}, {66,5}, 
    {71,166}, {66,6}, {71,167}, {66,7}, {66,8}, {66,9}, {66,10}, {66,11}, 
    {71,168}, {71,169}, {66,12}, {71,170}, {66,13}, {71,171}, {66,14}, 
    {66,15}, {66,16}, {66,17}, {66,18}, {66,19}, {71,172}, {71,173}, {66,20}, 
    {66,21}, {71,174}, {66,22}, {66,23}, {66,24}, {71,175}, {66,25}, {66,26}, 
    {66,33}, {66,34}, {66,35}, {66,36}, {66,37}, {71,176}, {71,177}, {66,38}, 
    {71,178}, {66,39}, {71,179}, {66,40}, {66,41}, {66,42}, {66,43}, {66,44}, 
    {66,45}, {71,180}, {71,181}, {66,46}, {66,47}, {71,182}, {66,48}, {66,49}, 
    {66,50}, {71,183}, {66,51}, {66,52}, {66,53}, {66,54}, {66,55}, {66,56}, 
    {66,57}, {71,184}, {71,185}, {66,58}, {71,186}, {71,187}, {71,188}, 
    {66,65}, {66,66}, {66,67}, {66,68}, {66,69}, {66,70}, {71,189}, {66,71}, 
    {66,72}, {66,73}, {71,190}, {66,74}, {66,75}, {66,76}, {72,97}, {66,77}, 
    {66,78}, {66,79}, {66,80}, {66,81}, {66,82}, {66,83}, {66,84}, {72,98}, 
    {66,85}, {66,86}, {66,87}, {66,88}, {66,89}, {66,90}, {66,91}, {66,92}, 
    {66,93}, {66,94}, {72,99}, {72,100}, {66,95}, {66,96}, {72,101}, {67,1}, 
    {67,2}, {67,3}, {72,102}, {67,4}, {67,5}, {67,6}, {67,7}, {72,103}, 
    {67,8}, {67,9}, {72,104}, {72,105}, {67,10}, {72,106}, {67,11}, {72,107}, 
    {67,12}, {67,13}, {67,14}, {72,108}, {67,15}, {67,16}, {72,109}, {72,110}, 
    {67,17}, {67,18}, {72,111}, {67,19}, {67,20}, {67,21}, {72,112}, {67,22}, 
    {67,23}, {67,24}, {67,25}, {67,26}, {67,33}, {67,34}, {67,35}, {67,36}, 
    {67,37}, {72,113}, {67,38}, {72,114}, {67,39}, {67,40}, {67,41}, {67,42}, 
    {67,43}, {67,44}, {72,115}, {72,116}, {67,45}, {67,46}, {72,117}, {67,47}, 
    {67,48}, {67,49}, {67,50}, {67,51}, {67,52}, {67,53}, {67,54}, {67,55}, 
    {67,56}, {67,57}, {67,58}, {67,65}, {67,66}, {72,118}, {67,67}, {72,119}, 
    {67,68}, {67,69}, {67,70}, {67,71}, {67,72}, {67,73}, {72,120}, {72,121}, 
    {67,74}, {67,75}, {72,122}, {67,76}, {67,77}, {67,78}, {72,123}, {67,79}, 
    {67,80}, {67,81}, {67,82}, {67,83}, {67,84}, {67,85}, {67,86}, {72,124}, 
    {67,87}, {72,125}, {67,88}, {72,126}, {67,89}, {67,90}, {67,91}, {67,92}, 
    {67,93}, {67,94}, {72,127}, {67,95}, {67,96}, {68,1}, {72,128}, {68,2}, 
    {68,3}, {68,4}, {72,129}, {68,5}, {68,6}, {68,7}, {68,8}, {68,9}, {68,10}, 
    {68,11}, {68,12}, {72,130}, {68,13}, {72,131}, {68,14}, {68,15}, {68,16}, 
    {68,17}, {68,18}, {68,19}, {68,20}, {68,21}, {72,132}, {72,133}, {68,22}, 
    {68,23}, {72,134}, {68,24}, {68,25}, {68,26}, {72,135}, {68,33}, {68,34}, 
    {68,35}, {68,36}, {72,136}, {68,37}, {68,38}, {72,137}, {68,39}, {68,40}, 
    {72,138}, {68,41}, {72,139}, {68,42}, {68,43}, {68,44}, {68,45}, {68,46}, 
    {68,47}, {72,140}, {68,48}, {68,49}, {68,50}, {72,141}, {68,51}, {68,52}, 
    {68,53}, {72,142}, {68,54}, {68,55}, {68,56}, {68,57}, {68,58}, {68,65}, 
    {68,66}, {72,143}, {68,67}, {68,68}, {68,69}, {68,70}, {72,144}, {68,71}, 
    {68,72}, {68,73}, {68,74}, {68,75}, {68,76}, {72,145}, {72,146}, {68,77}, 
    {68,78}, {72,147}, {68,79}, {68,80}, {68,81}, {72,148}, {68,82}, {68,83}, 
    {68,84}, {68,85}, {68,86}, {68,87}, {68,88}, {68,89}, {68,90}, {68,91}, 
    {68,92}, {68,93}, {72,149}, {68,94}, {68,95}, {68,96}, {69,1}, {69,2}, 
    {69,3}, {72,150}, {72,151}, {69,4}, {69,5}, {72,152}, {69,6}, {69,7}, 
    {69,8}, {72,153}, {69,9}, {69,10}, {69,11}, {69,12}, {69,13}, {69,14}, 
    {69,15}, {72,154}, {72,155}, {69,16}, {72,156}, {69,17}, {72,157}, 
    {69,18}, {69,19}, {69,20}, {69,21}, {69,22}, {69,23}, {72,158}, {72,159}, 
    {69,24}, {69,25}, {72,160}, {69,26}, {69,33}, {69,34}, {72,161}, {69,35}, 
    {69,36}, {69,37}, {69,38}, {69,39}, {69,40}, {69,41}, {72,162}, {69,42}, 
    {69,43}, {72,163}, {69,44}, {72,164}, {69,45}, {69,46}, {69,47}, {69,48}, 
    {69,49}, {69,50}, {72,165}, {72,166}, {69,51}, {69,52}, {72,167}, {69,53}, 
    {72,168}, {72,169}, {72,170}, {72,171}, {69,54}, {69,55}, {69,56}, 
    {69,57}, {69,58}, {69,65}, {72,172}, {72,173}, {69,66}, {72,174}, {69,67}, 
    {72,175}, {69,68}, {69,69}, {69,70}, {72,176}, {69,71}, {69,72}, {72,177}, 
    {69,73}, {69,74}, {69,75}, {72,178}, {69,76}, {69,77}, {69,78}, {72,179}, 
    {69,79}, {69,80}, {69,81}, {69,82}, {69,83}, {69,84}, {69,85}, {72,180}, 
    {72,181}, {69,86}, {69,87}, {69,88}, {72,182}, {69,89}, {69,90}, {69,91}, 
    {69,92}, {69,93}, {69,94}, {72,183}, {72,184}, {69,95}, {69,96}, {72,185}, 
    {70,1}, {70,2}, {70,3}, {72,186}, {70,4}, {70,5}, {70,6}, {70,7}, {70,8}, 
    {70,9}, {70,10}, {72,187}, {72,188}, {70,11}, {72,189}, {70,12}, {72,190}, 
    {70,13}, {70,14}, {70,15}, {70,16}, {70,17}, {70,18}, {75,144}, {75,150}, 
    {75,167}, {77,143}, {77,168}, {78,109}, {79,187}, {80,98}, {80,120}, 
    {80,144}, {80,157}, {81,148}, {81,149}, {81,152}, {81,155}, {81,156}, 
    {81,157}, {81,158}, {81,159}, {81,160}, {81,162}, {81,163}, {81,164}, 
    {81,165}, {81,166}, {81,168}, {81,169}, {81,170}, {81,171}, {81,173}, 
    {81,175}, {81,176}, {81,178}, {81,182}, {81,186}, {81,188}, {81,189}, 
    {81,190}, {82,98}, {82,99}, {82,103}, {82,104}, {82,105}, {82,106}, 
    {82,107}, {82,109}, {82,114}, {82,126}, {82,130}, {82,131}, {82,132}, 
    {82,134}, {82,135}, {82,136}, {82,137}, {82,138}, {82,139}, {82,141}, 
    {82,142}, {82,143}, {82,144}, {82,145}, {82,146}, {82,147}, {82,148}, 
    {82,149}, {82,150}, {82,151}, {82,153}, {82,154}, {82,158}, {82,159}, 
    {82,161}, {82,162}, {82,164}, {82,165}, {82,166}, {82,167}, {82,168}, 
    {82,169}, {82,170}, {82,171}, {82,176}, {82,177}, {82,178}, {82,179}, 
    {82,180}, {82,181}, {82,183}, {82,184}, {84,166}, {84,188}, {85,101}, 
    {85,107}, {85,110}, {86,120}, {86,141}, {87,139}, {87,164}, {91,133}, 
    {91,164}, {92,101}, {93,101}, {93,149}, {93,180}, {94,188}, {94,190}, 
    {95,115}, {95,161}, {95,168}, {96,177}, {97,109}, {97,173}, {99,181}, 
    {100,97}, {100,105}, {101,110}, {101,113}, {101,114}, {101,121}, {101,123}, 
    {101,124}, {101,132}, {101,142}, {101,144}, {101,146}, {101,150}, {101,186}, 
    {101,187}, {101,188}, {101,190}, {102,97}, {102,100}, {102,103}, {102,109}, 
    {102,111}, {102,112}, {102,113}, {102,115}, {102,119}, {102,120}, {102,124}, 
    {102,132}, {102,134}, {102,135}, {102,138}, {102,146}, {102,150}, {102,153}, 
    {102,156}, {102,159}, {102,161}, {102,164}, {102,165}, {102,166}, {102,168}, 
    {102,170}, {102,171}, {102,172}, {102,175}, {102,177}, {102,178}, {102,181}, 
    {102,182}, {102,183}, {102,185}, {103,97}, {103,102}, {103,105}, {103,106}, 
    {103,108}, {103,109}, {103,112}, {103,127}, {103,129}, {103,134}, {103,135}, 
    {103,139}, {103,141}, {103,143}, {103,144}, {103,147}, {103,159}, {103,164}, 
    {103,166}, {103,183}, {104,167}, {104,168}, {104,176}, {104,177}, {104,183}, 
    {104,185}, {104,187}, {104,190}, {105,103}, {105,108}, {105,140}, {105,183}, 
    {106,129}, {106,165}, {106,180}, {106,183}, {106,188}, {106,190}, {107,100}, 
    {107,103}, {107,105}, {107,106}, {107,122}, {107,123}, {107,125}, {107,129}, 
    {107,130}, {107,134}, {107,135}, {107,140}, {107,143}, {107,144}, {107,145}, 
    {107,146}, {107,152}, {108,102}, {108,103}, {108,106}, {108,111}, {108,112}, 
    {108,113}, {108,114}, {108,117}, {108,120}, {108,122}, {108,128}, {108,129}, 
    {108,133}, {108,134}, {108,137}, {108,138}, {108,149}, {108,157}, {108,158}, 
    {108,161}, {108,164}, {108,167}, {108,168}, {108,183}, {108,184}, {108,186}, 
    {109,97}, {109,98}, {109,99}, {109,174}, {110,155}, {114,125}, {114,186}, 
    {115,113}, {116,103}, {116,174}, {118,180}, {118,182}, {119,120}, {119,136}, 
    {119,147}, {120,155}, {120,176}, {122,97}, {122,98}, {122,166}, {124,105}, 
    {35,97}, {35,98}, {35,99}, {35,100}, {35,101}, {35,102}, {35,103}, 
    {35,104}, {35,105}, {35,106}, {35,107}, {35,108}, {35,109}, {35,110}, 
    {35,111}, {35,112}, {35,113}, {35,114}, {35,115}, {35,116}, {35,117}, 
    {35,118}, {35,119}, {35,120}, {35,121}, {35,122}, {35,123}, {35,124}, 
    {35,125}, {35,126}, {35,127}, {35,128}, {35,129}, {35,130}, {35,131}, 
    {35,132}, {35,133}, {35,134}, {35,135}, {35,136}, {35,137}, {35,138}, 
    {35,139}, {35,140}, {35,141}, {35,142}, {35,143}, {35,144}, {35,145}, 
    {35,146}, {35,147}, {35,148}, {35,149}, {35,150}, {35,151}, {35,152}, 
    {35,153}, {35,154}, {35,155}, {33,108}, {35,157}, {35,158}, {35,159}, 
    {35,160}, {35,161}, {35,162}, {35,163}, {35,164}, {35,165}, {35,166}, 
    {35,167}, {35,168}, {35,169}, {35,170}, {35,171}, {35,172}, {35,173}, 
    {35,174}, {35,175}, {35,176}, {35,177}, {35,178}, {35,179}, {35,180}, 
    {35,181}, {35,182}, {35,183}, {35,184}, {35,185}, {35,186}, {35,187}, 
    {35,188}, {35,189}, {34,102}, {33,139}, {33,140}, {33,190}, {35,190}, 
    {33,141}, {35,156}
};

/* This returns ERROR if the code point doesn't exist. */
long int cp949_to_unicode(int r, int c)
{
    assert(r >= 0 && r < 128);
    assert(c >= 0 && c < 192);
    return cp949_forward[r][c];
}

/* This one returns 1 on success, 0 if the code point doesn't exist. */
int unicode_to_cp949(long int unicode, int *r, int *c)
{
    int rr, cc;
    long int uu;
    int i, j, k;

    i = -1;
    j = lenof(cp949_backward);
    while (j - i > 1) {
	k = (i + j) / 2;
	rr = cp949_backward[k].r;
	cc = cp949_backward[k].c;
	uu = cp949_forward[rr][cc];
	if (unicode > uu)
	    i = k;
	else if (unicode < uu)
	    j = k;
	else {
	    *r = rr;
	    *c = cc;
	    return 1;
	}
    }
    return 0;
}

/* Functions dealing with the KS X 1001 square subset */
long int ksx1001_to_unicode(int r, int c)
{
    assert(r >= 0 && r < 94);
    assert(c >= 0 && c < 94);
    return cp949_forward[r+0x21][c+0x61];
}

/* This one returns 1 on success, 0 if the code point doesn't exist. */
int unicode_to_ksx1001(long int unicode, int *r, int *c)
{
    int rr, cc;
    if (!unicode_to_cp949(unicode, &rr, &cc))
	return 0;
    rr -= 0x21;
    cc -= 0x61;
    if (rr < 0 || rr >= 94 || cc < 0 || cc >= 94)
	return 0;
    *r = rr;
    *c = cc;
    return 1;
}

#ifdef TESTMODE

#include <stdio.h>

int main(void)
{
    int r, c, rr, cc, ret;
    long int u, uu;

    for (r = 0; r < 128; r++) {
	for (c = 0; c < 192; c++) {
	    u = cp949_to_unicode(r, c);
	    if (u != ERROR) {
		ret = unicode_to_cp949(u, &rr, &cc);
		if (!ret)
		    printf("(%d,%d) -> U-%08lx but didn't convert back\n",
			   r, c, u);
		else if (rr != r || cc != c)
		    printf("(%d,%d) -> U-%08lx -> (%d,%d)\n",
			   r, c, u, rr, cc);
	    }
	}
    }

    for (u = 0; u < 0x10000L; u++) {
	ret = unicode_to_cp949(u, &r, &c);
	if (ret) {
	    uu = cp949_to_unicode(r, c);
	    if (uu == ERROR)
		printf("U-%08lx -> (%d,%d) but didn't convert back\n",
		       u, r, c);
	    else if (uu != u)
		printf("U-%08lx -> (%d,%d) -> U-%08lx\n", u, r, c, uu);
	}
    }

    return 0;
}

#endif
